/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains routines that set up, access and change the
 * mode panel and its associated data structures
 *
 * Kenton Lynne
 *   last update: 02/24/87
 *                04/01/87
 * Nigel Goddard
 *   hacked for X: April, May 1989
 *******************************************************************/

#include "macros.h"
#include "externs.h"
#include "panel.h"
#include "control_panel.h"
#include <X11/Xmu.h>

extern char gi_logfile_name[];

Widget mode_panel;

button_list * gi_mode_buttons; 

int gi_mode_proc();

/* static procedures forward referenced */

/*************************************************************************/
gi_make_mode_panel()
{
/* sets up the mode panel
 */
  Widget temp;

static Arg layout_args[] = {
  { XtNfromHoriz, (XtArgVal)NULL },
  { XtNhorizDistance, (XtArgVal)10 },
  { XtNborderWidth, (XtArgVal) 0},
};

  static char *mode_names[]={" Main "," Link "," Text "," Draw "," Custom ",0};

  /* create the mode panel itself */
  if ((mode_panel = XtCreateManagedWidget("MODE_PANEL", formWidgetClass, gi_tool,
					  NULL, 0)) == NULL)
  {
    fputs("Can't create mode panel\n", stderr);
    exit(1);
  }

  layout_args[0].value = (XtArgVal) AddText(mode_panel,"MODE:",XtJustifyCenter, NULL);
  temp = XtCreateManagedWidget("MODE_RADIO_BOX", formWidgetClass, mode_panel,
			       layout_args, XtNumber(layout_args));
  AddRadio(temp,"mode_radio",gi_mode_proc,mode_names,5,0,&gi_mode_buttons);

  /* items  for logging file */

  strcpy(gi_logfile_name, DFT_LOG_FILE_NAME);
  if ((gi_log_fp = fopen(gi_logfile_name, "w")) == NULL)
    {
      gi_put_error("Can't open log file; no command logging");
      gi_log_fp = NULL;
      gi_log_on = False;
    }
  else
    gi_log_on = True;
}

/*************************************************************************/
static hide_main_items()
{
/* non-display all the items normally displayed
 * only during Main mode
 */

  XtUnmanageChild(gi_main_shell);
}

/*************************************************************************/
static hide_link_items()
{
/* non-display all the items normally displayed
 * only during LINK mode
 */
  XtUnmanageChild(gi_link_shell);
}

/*************************************************************************/
static hide_text_items()
{
/* non-display all the items normally displayed
 * only during TEXT mode
 */
  XtUnmanageChild(gi_text_shell);
}

/*************************************************************************/
static hide_draw_items()
{
/* non-display all the items normally displayed
 * only during DRAW mode
 */
  XtUnmanageChild(gi_draw_shell);

}

/*************************************************************************/
static hide_cust_items()
{
/* non-display all the items normally displayed
 * only during CUSTOM mode
 */
  XtUnmanageChild(gi_custom_shell);
  gi_set_display_mode_to_custom(False); /* remap button translations */
}

/*************************************************************************/
static show_main_items()
{
/* Redisplay all the items normally displayed
 * during Main mode
 */
  XtManageChild(gi_main_shell);
  /* set the cursor */
  gi_last_cursor = gi_main_cursor;
  XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_main_cursor);
  XSelectInput(XtDisplay(gi_tool),gi_gfx,gi_event_mask);
}

/*************************************************************************/
static show_link_items()
{
/* Redisplay all the items normally displayed
 * during LINK mode
 */
  XtManageChild(gi_link_shell);
  /* set the cursor */
  gi_last_cursor = gi_main_cursor;
  XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_main_cursor);
  XSelectInput(XtDisplay(gi_tool),gi_gfx,gi_event_mask);
}

/*************************************************************************/
static show_text_items()
{
/* Redisplay all the items normally displayed
 * during TEXT mode
 */
  XtManageChild(gi_text_shell);
  /* set the cursor */
  gi_last_cursor = gi_text_cursor;
  XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_text_cursor);
  XSelectInput(XtDisplay(gi_tool),gi_gfx,gi_event_mask | KeyPressMask);
}

/*************************************************************************/
static show_draw_items()
{
/* Redisplay all the items normally displayed
 * during DRAW mode
 */
  XtManageChild(gi_draw_shell);
  /* set the cursor */
  gi_last_cursor = gi_draw_cursor;
  XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_draw_cursor);
  XSelectInput(XtDisplay(gi_tool),gi_gfx,gi_event_mask | PointerMotionMask);
}

/*************************************************************************/
static show_cust_items()
{
/* Redisplay all the items normally displayed
 * during CUSTOM mode
 */
  XtManageChild(gi_custom_shell);
  /* set the cursor */
  gi_last_cursor = gi_custom_cursor;
  XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_custom_cursor);
  XSelectInput(XtDisplay(gi_tool),gi_gfx,gi_event_mask);
  gi_set_display_mode_to_custom(True);	/* remap button translations */
}

/*************************************************************************/
int gi_mode_proc(item, value, event)
   Widget item;
   int value;
   caddr_t event;
{
/* called when the user switches modes by clicking over
 * one of the mode buttons, this routine sets up the
 * proper control panel prompts for the new mode and
 * does any necessary cleanup from the mode being left 
 */

  /* if mode has not changed, just return */
  if (value==gi_mode) return;

  /* non-display the old mode control panel items */
  switch (gi_mode)
  {
      case MODE_MAIN: 
           hide_main_items();
           break;

      case MODE_LINK:   
           hide_link_items();
           /* if leaving link mode: make sure everything is reshown */
           gi_reshow_flag |= RESHOW_ALL;
           break;

      case MODE_TEXT:
           hide_text_items();
	   break;

      case MODE_DRAW:
           hide_draw_items();
	   break;

      case MODE_CUSTOM:
           hide_cust_items();
	   break;
   }

  /* if entering link mode, make sure everything is reshown */
  if (value==MODE_LINK) gi_reshow_flag |= RESHOW_ALL;

  /* set new mode */
  gi_mode = value;

  /* show the control panel prompts for new mode */
  switch (gi_mode)
  {
    case MODE_MAIN: 
      show_main_items();
      break;
    case MODE_LINK:
      show_link_items();
      break;
    case MODE_TEXT:
      show_text_items();
      break;
    case MODE_DRAW:
      show_draw_items();
      break;
    case MODE_CUSTOM:
      show_cust_items();
      break;
  }

  /* display mouse button prompts for new mode in message window */
  switch (gi_mode)
  {
    case MODE_MAIN: 
      gi_put_message("show info | move | hide cursor");
      break;
    case MODE_LINK:
      gi_put_message("show info | move | mark link target");
      break;
    case MODE_TEXT:
      gi_put_message( "mark text start | move | delete text");
      break;
    case MODE_DRAW:
      gi_put_message("mark vertex | move | delete drawing"); 
      break;
    case MODE_CUSTOM:
      gi_put_message("menu A | menu B | menu C");
      break;
  }

  /* if mode changes demands it, reshow the entire screen */
  if (gi_reshow_flag & RESHOW_ALL) 
    gi_reshow();

}

/*************************************************************************/
gi_log(string)
 char *string;
{
/* Logs the string to the current log file, period.
 */

  /* write out the requested string to the current log file */
  if (gi_log_on && gi_log_fp)
    fprintf(gi_log_fp,"%s\n",string);
  return(OK);
}

/*************************************************************************/
gi_log_info(at_x, at_y, col)
  int at_x, at_y, col;
{
/* create an info command to show info for
 * a unit at particular display coordinates
 */
  char log_string[MAX_CMD_LENGTH+1];
  
  /* build the info command in the log string */

  (void) sprintf(log_string,"%s %s %1d %1d %1d", 
                     GI_CMD, INFO_CMD_SHORT, 
                     at_x, at_y, col);

  /* write the log string to the log file */
  gi_show_prev(log_string);
  gi_log(log_string);
}
  
/*************************************************************************/
gi_log_delete(at_x, at_y)
  int at_x, at_y;
{
/* creates the move command that would delete
 * an object at the indicated display coordinates 
 */

  char log_string[MAX_CMD_LENGTH+1];
  
  /* build the delete command in the log string */

  (void) sprintf(log_string,"%s %s %1d %1d", 
                     GI_CMD, DEL_CMD_SHORT, 
                     at_x, at_y);

  /* write the log string to the log file */
  gi_show_prev(log_string);
  gi_log(log_string);
}

/*************************************************************************/
gi_log_move(from_x, from_y, to_x, to_y)
  int from_x, from_y, to_x, to_y;
{
/* creates the move command that would move
 * an object at the "from" display coordinates to
 * the "to" display coordinates
 */

  char log_string[MAX_CMD_LENGTH+1];
  
  /* build the move command in the log string */

  (void) sprintf(log_string,"%s %s %1d %1d %1d %1d", 
                     GI_CMD, MOVE_CMD_SHORT, 
                     from_x, from_y, to_x, to_y);

  /* write the log string to the log file */
  gi_show_prev(log_string);
  gi_log(log_string);
}

/*************************************************************************/
gi_log_text(ptr)
  struct txobj *ptr;
{
/* creates a text command that would create the
 * text object passed by the caller and then
 * writes that command to the current log file
 */
  struct saved_fonts *fptr;
  char *lptr, *tptr;
  char log_string[MAX_CMD_LENGTH+1];
  char buf[MAX_CMD_LENGTH+1];
  
  /* initialize log_string to "gi t " */

  (void) sprintf(log_string,"%s %s ",GI_CMD, TEXT_CMD_SHORT);

  lptr = buf;

  /* copy text to a buffer surrouding it
     by double quotes and doubling any
     quotes actually in the string itself 
  */

  tptr = ptr->text_start;
  *lptr++ = DQUOTE;
  
  while (*tptr != '\0')
  {
    *lptr++ = *tptr;
    if (*tptr++ == DQUOTE)
      *lptr++ = DQUOTE;
  }
  *lptr++ = DQUOTE;
  *lptr = '\0';
  
  /* append the text to the log string */

  strcat(log_string,buf);

  /* append the display coordinates */

  (void) sprintf(buf," %1d %1d ",ptr->x_pos,ptr->y_pos);
  strcat(log_string,buf);

  /* search the chain of fonts for the name
     of the correct font and append it to
     the log string
  */

  /* search chain for fonts */
  for (fptr=gi_font_head; fptr!=NULL; fptr=fptr->next)
    {
      if (fptr->font_ptr==ptr->text_font) 
	{
	  /* this is the one: append the font name */
	  strcat(log_string,fptr->font_name);
	  break;
	}
    }

  gi_show_prev(log_string);
  gi_log(log_string);
}
  
/*************************************************************************/
gi_log_draw(ptr)
  struct drobj *ptr;
{
/* creates a draw command that would create the
 * drawn object passed by the caller and then
 * writes that command to the current log file
 */
  int i, *vptr;
  char log_string[MAX_CMD_LENGTH+1];
  char buf[MAX_CMD_LENGTH+1];

  /* initialize string to "gi d" */

  (void) sprintf(log_string,"%s %s ",GI_CMD, DRAW_CMD_SHORT);

  /* append the number of vertices argument  or the  */
  /* indication that this is a bounding box */
  if (ptr->flag & BOUND_BOX)
    (void) sprintf(buf,"%1s",BOUND_BOX_STG);
  else
    (void) sprintf(buf,"%1d",ptr->num_vertices);

  strcat(log_string,buf);

  /* append all the vertex coordinates */

  for (i=0, vptr=ptr->vptr; i<2*ptr->num_vertices; i++, vptr++)
  {
    (void) sprintf(buf," %1d",*vptr);
    strcat(log_string,buf);
  }

  /* write the assembled string to the log file */
  gi_show_prev(log_string);
  gi_log(log_string);

}

