/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*--------------------------------------------------------------------------
  Author: Nigel Goddard
  Date: May 1 1987
----------------------------------------------------------------------------*/

/* the following commands run on the butterfly sim and uniprocessor
   versions only.  Debug commands.
*/

#ifdef BFLY
#  include "bflysim.h"
#else
#  include "uniproc.h"
#  include <sys/time.h>
#endif
#include <ctype.h>

extern char *strcat();

#include "lex.h"

/***** debug *****/
/*---------------------------------------------------------------------------
  Turns debugging on and off, and Auto-Fix on and off.  See help information
  at the end of the function for details.
----------------------------------------------------------------------------*/
/*
Debug_Cmd_debug(argc,argv)
    int argc;
    char ** argv;

{
BFLYCATCH
  switch(argc)
    {
    case 1:
      if (Debug)
	{
	  LOGfprintf(stdout,"in debug mode\n");
	  if (AutoFix)
	    LOGfprintf(stdout,"in auto-fix mode\n");
	  else
	    LOGfprintf(stdout,"not in auto-fix mode\n");
	}
      else
	LOGfprintf(stdout,"not in debug mode\n");
      return 0;
    case 2:
      switch(Lex(argv[1]))
	{
	case OFF:
	case ON:
	  LOGfprintf(stderr,
		     "Cannot switch debug on or off from this interface!\n");
	  break;
	case HELP:
	  Cmd_debug(argc,argv);
	  break;
	default:
	  goto synerror;
	}
      return 0;
    case 3:
      if (Lex(argv[1]) == AUTO)
	switch(Lex(argv[2]))
	  {
	  case ON:
	    AutoFix = TRUE;
	    return 0;
	  case OFF:
	    AutoFix = FALSE;
	    return 0;
	  }
    }

 synerror:
  Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: debug [auto <on|off>]\n\n");
  return 0;
BFLYTHROW("Cmd_debug",0)
} */


/*---------------------------------------------------------------------------
  Debug version of Cmd_quit.  Returns TRUE to exit debug interface.
----------------------------------------------------------------------------*/

int Debug_Cmd_quit(argc,argv)
     int argc;
     char ** argv;

{
  if (argc == 2 && (Lex(argv[1]) == HELP))
    {
      Format = TRUE;
      LOGfprintf(Dispf,"The debug command has a different effect when used from an interface at this level.  From a debug interface (prompt `debug[n]>') it will return you to the previous level interface, but only if you have fixed all the errors that put you at this level.  From the interrupt interface (prompt `interrrupt[n]>') it will cause whatever operation was interrupted to continue at the point at which it was interrupted.\n\n");
      LOGfprintf(Dispf,"You can exit to UNIX from any level by typing control_D.\n\n");
      Format = FALSE;
      return 0;
    }
  else
    return (TRUE);
}


int Debug_Cmd_q(argc,argv)
     int argc;
     char ** argv;

{
  return (Debug_Cmd_quit(argc,argv));
}

/*---------------------------------------------------------------------------
  The  set  command, which is only availible at the debug
  interface, is used to correct errors in unit, site or link
  specifications.  Errors may be fixed with specific values, or the
  simulator default values may be used:  set all default .  The
  list of outstanding errors will be printed in response to:  set
  .  It will also be printed if the user attempts to  quit 
  before all the errors have been corrected.
----------------------------------------------------------------------------*/

Debug_Cmd_set(argc,argv)
     int argc;
     char ** argv;

{
  int type, lx, intval;
  FLINT val;

BFLYCATCH
  switch (argc)
    {
    case 1:
      si_Show_Debug_Values();
      break;
    case 2:
      if (Lex(argv[1]) == HELP)
	goto helpinfo;
      else
	goto synerror;
    case 3:
      switch (type = Lex(argv[1]))
	{
	case ALL:
	  if (Lex(argv[2]) == DEFAULT)
	    si_Set_Debug_Default();
	  else
	    goto synerror;
	  break;
	case IPOT:
	case POT:
	case OUT:
	case WEIGHT:
	  if ((lx = Lex(argv[2])) == INT)
	    val = Yyint;
	  else
	    if (lx == FLOAT)
	      val == Yyfloat;
	    else
	      goto synerror;
	  si_Set_Debug_Value_FLINT(type,val);
	  break;
	case ISTATE:
	case STATE:
	case UNIT:
	case FROM:
	case TO:
	  if ((lx = Lex(argv[2])) == INT)
	    intval = Yyint;
	  else
	    if (lx == FLOAT)
	      intval == Yyfloat;
	    else
	      goto synerror;
	  si_Set_Debug_Value(type,intval);
	  break;
	case FUNC:
	case TYPE:
	case SITE:
	  if (Lex(argv[2]) == IDENT)
	    si_Set_Debug_String(type,argv[2]);
	  else
	    goto synerror;
	  break;
	default:
	  goto synerror;
	}
      break;
    default:
      goto synerror;
    }
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"The set command is used to reset values that are of the wrong type or out of range.  You may reset the potential, output, state, initial potential, type or function for an error when making a unit.  You may reset the site name, unit index or site function when adding a site to a unit.  You may reset the source unit, target unit, target site name, weight, and link function when making a link.  For instance, to correct the function you would type `set func UFsum' perhaps.  You may also reset all of the outstanding errors by typing `set all default'.  They will then be reset to simulator default values.  Typing simply `set' will get you a list of the current settings and notification of any that are incorrect.  Units must be specified by index.\n");

 synerror:
  Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: set <pot|out|state|ipot|istate|unit|to|from|weight> <value>\n       set <func|type|site> <name>\n       set all default\n       set\n");
  return 0;
BFLYTHROW("Debug_Cmd_set",0)
}

/*---------------------------------------------------------------------------
  Returns IGNORE to cause default error correction and exit from command
  interface.
----------------------------------------------------------------------------*/

Debug_Cmd_ignore (argc,argv)
     int argc;
     char ** argv;

{
  if (argc == 2 && (Lex(argv[1]) == HELP))
    {
      LOGfprintf(Dispf,"The ignore command is used to continue from the debug interface (prompt `debug[n]>') as if the unit, site or link which has errors was never specified.  Thus the unit, site, or link will not be made, which may cause other errors further along in the construction process\n\n");
      return 0;
    }
  else
    return(IGNORE);
}

/*---------------------------------------------------------------------------
  Debug version of Cmd_help.  Looks for debug version of command first.
----------------------------------------------------------------------------*/

Debug_Cmd_help(argc,argv)
     int argc;
     char ** argv;

{
  char buff[80];
  func_ptr func;

  (void) sprintf(buff,"%s","Debug_Cmd_");
  if (argc == 1)
    Cmd_help(1,argv);		/* general help info */
  else
    if (argc == 2)
      if (!strcmp(argv[1],"UnitId"))
	Cmd_help(2,argv);
      else
	if ((func = NameToFunc(strcat(buff,argv[1]))) != NULL ||
	    (func = NameToFunc(&buff[6])) != NULL)
	  {
	    argv[0] = argv[1];
	    argv[1][0] = '?';
	    argv[1][1] = '\0';
	    PipeBegin();
	    Format = TRUE;
	    func(argc,argv);
	    Format = FALSE;
	    PipeEnd();
	  }
	else
	  LOGfprintf(stderr,"No such command: %s\n",argv[1]);
    else
      LOGfprintf(stderr,"\nUsage: help [command name]\n");
  return 0;
}
/* end of commands for sim and uniprocessor versions */
