/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains the globals macros and defines for the 
 * Graphics Interface package of the Rochester Connectionist
 * simulator.
 *
 * Kenton Lynne
 *******************************************************************/

#include <stdio.h>
#include <suntool/tool_hs.h>
#include <suntool/panel.h>
#include <suntool/gfxsw.h>
#include <suntool/ttysw.h>
#include <suntool/icon_load.h>
#include <sys/resource.h>

#ifdef FSIM
# define FLINT float
#else
# define FLINT int 
#endif

/******************************************/
/* constants for calls to gi_sim_get_unit */
/******************************************/
#define GET_FIRST  1     /* get 1st matching unit found */
#define GET_NEXT   2     /* get next matching unit found */

/****************************************/
/* constants for universal return codes */
/****************************************/
#define ERROR -1
#define OK 1
#define WARN 0

/****************************************/
/* individual flags for the reshow_flag */
/****************************************/
#define RESHOW_NEEDED        1   /* display needs to be updated */
#define RESHOW_ALL           2   /* all units need to be reshown */
#define SOME_VALS_NEEDED     4   /* some (~VALUE_OK) unit values needed */
#define ALL_VALS_NEEDED      8   /* all unit values need to be updated */
#define SOME_LINKS_NEEDED   16   /* some (~LINK_OK) link values needed */
#define ALL_LINKS_NEEDED    32   /* all link values need to be updated */
#define CLEAR_NEEDED        64   /* screen needs to be cleared */

/***********************/
/* hot spot of cursors */
/***********************/
#define HOTX  7
#define HOTY  7

/*********************/
/* panel size layout */
/*********************/
#define INFO_STARTX  2
#define INFO_STARTY  5
#define INFO_CSPACE  144
#define INFO_RSPACE  18

/*****************/
/* panel numbers */
/*****************/
#define INFO       0
#define CNTL       1
#define GRAPH      2
#define MSG        3
#define CMD        4
#define MODE       5

/******************/
/* mode constants */
/******************/
#define MODE_MAIN    0
#define MODE_LINK    1
#define MODE_TEXT    2
#define MODE_DRAW    3
#define MODE_CUSTOM  4
#define NUMBER_MODES 5

/****************************/
/* limiting value constants */
/****************************/
#define MAX_PANELS 6            /* number of panels in tool */
#define MAX_INT 2147483647      /* largest integer */
#define MAX_CUST_TEXT_LEN 80    /* largest custom button string */
#define MAX_VERTICES  10        /* max vertices for any drawn object */
#define MAX_TEXT_LENGTH 80      /* longest text item string */
#define MAX_INFO_COLS 8         /* number of info columns on info panel */
#define MAX_FONT_LENGTH 80      /* longest allow font name */
#define MAX_CMD_LENGTH 120      /* longest command allowed */
#define MAX_ITEM_LENGTH 20      /* longest panel text item */
#define MAX_NUM_LENGTH 6        /* longest numeric field */
#define MAX_ARGS 24             /* max number of args for parsing */
#define MAX_COORD 99999         /* largest display coordinate */
#define MAX_RANGE 32766         /* largest range allowed */

/***********************/
/* character constants */
/***********************/
#define AUX_CHAR '\\'           /* signifies auxiliary units */
#define DQUOTE    '"'           /* double quote character */ 
#define SITE_SEP_CHAR  '/'      /* seperates unit from sit in target arg */
#define COMMENT_CHAR  '#'       /* indicates line is a comment */
#define BACKSPACE 8             /* backspace character */
#define DELETE  127		/* delete character */
#define CR        13            /* return character */
#define BLANK     ' '           /* blank character */
#define SHOW_CMD_CHAR  's'      /* indicates show command */
#define RESHOW_CMD_CHAR  'r'    /* indicates reshow command */
#define CHANGE_CMD_CHAR  'c'    /* indicates change command */
#define BUT_CMD_CHAR 'b'        /* indicates button command */
#define INFO_CMD_CHAR 'i'       /* indicates info command */
#define MOVE_CMD_CHAR 'm'       /* indicates move command */
#define DEL_CMD_CHAR 'x'        /* indicates delete command */
#define GO_CMD_CHAR  'g'        /* indicates go command */
#define TEXT_CMD_CHAR 't'       /* indicates text command */
#define DRAW_CMD_CHAR 'd'       /* indicates draw command */
#define ERASE_CMD_CHAR  'e'     /* indicates erase command */
#define LOG_CMD_CHAR 'l'        /* indicates a log command */

/******************************/
/* character string constants */
/******************************/
/* defines the default directory for font name lookups */
#define FONT_DIR_STRING "/usr/lib/fonts/fixedwidthfonts/"
#define MAX_NAME_LEN  12        /* longest name displayed on info panel */
#define BOUND_BOX_STG "b"       /* indicates bounding box in draw cmd  */
#define DFT_LOG_FILE_NAME "gi.log"  /* default name of log file */
#define DFT_IMAGE_FILE_NAME "gi.image" /* default image file name */
#define NULL_STRING "*null"     /* indicates null button command */
#define TERM_STRING ";"         /* separator for button commands */
#define ALL_STRING "all"        /* for WHO or HOWMANY prompts */
#define MAX_STRING "max"        /* for numcols prompts */
#define CONT_STRING "--"        /* for continuing button commands */
#define SITE_SEP_STRING "/"     /* separator between unit and site */
#define TO_DIR_STRING ">"       /* indicates LINK/IN on info panel */
#define FROM_DIR_STRING "<"     /* indicates LINK/OUT on infor panel */
#define ANY_SITE "*any"         /* indicates "any" site for target site */
#define UI_SUB_STRING "$u"      /* substitution arg for unit index */
#define X_SUB_STRING  "$x"      /* substitution arg for x coordinate */
#define Y_SUB_STRING  "$y"      /* substitution arg for y coordinate */
#define LOG_ON_STG "on"         /* indicates logging On */
#define LOG_OFF_STG "off"       /* indicates logging off */
#define GI_CMD "gi"             /* string for a gi command */
#define DFT_STRING "*default"   /* indicates default font */
#define SHOW_CMD_SHORT "s"      /* indicates show command */
#define RESHOW_CMD_SHORT "r"    /* indicates reshow command */
#define ERASE_CMD_SHORT "e"     /* indicates erase command */
#define CHANGE_CMD_SHORT "c"    /* indicates changes command */
#define GO_CMD_SHORT "g"        /* indicates go command */
#define DRAW_CMD_SHORT "d"      /* indicates drawn command */
#define TEXT_CMD_SHORT "t"      /* indicates text command */
#define MOVE_CMD_SHORT "m"      /* indicates move command */
#define READ_CMD_SHORT "r"      /* indicates read command */
#define READ_CMD_LONG "read"    /* indicates read command */
#define DEL_CMD_SHORT "x"       /* indicates delete command */
#define BUT_CMD_SHORT "b"       /* indicates button command */
#define INFO_CMD_SHORT "i"      /* indicates info command */
#define RESTART_CMD "restart"	/* indicates restart command */
#define LRANGE_POT   "-1000"    /* default low range for potential */
#define LRANGE_OUT   "-1000"    /* default low range for output */
#define LRANGE_STATE "-10"      /* default low range for state */
#define LRANGE_LINKIN "-1000"   /* default low range for linkin */
#define LRANGE_LINKOUT "-1000"  /* default low range for linkout */
#define LRANGE_DATA  "-1000"    /* default low range for data */
#define HRANGE_POT     "1000"   /* default high range for potential */
#define HRANGE_OUT     "1000"   /* default high range for output */
#define HRANGE_STATE     "10"   /* default high range for state */
#define HRANGE_LINKIN  "1000"   /* default high range for linkin */
#define HRANGE_LINKOUT "1000"   /* default high range for linkout */
#define HRANGE_DATA    "1000"   /* default high range for data */
#define POT_STG    "P"          /* command arg for potential */
#define OUTPUT_STG "O"          /* command arg for output */
#define STATE_STG  "S"          /* command arg for state */
#define DATA_STG   "D"          /* command arg for data */
#define LINKIN_STG   "Li"       /* command arg for linkin */
#define LINKOUT_STG  "Lo"       /* command arg for linkout */

/*******************************************************/
/* defines the chunk of space allocated at a time for  */
/* for string, vertex and pixptr space                 */
/*******************************************************/
#define STRING_SPACE_SIZE 2000
#define VERTEX_SPACE_SIZE 500
#define PIXPTR_SPACE_SIZE 2000

/***************************/
/* miscellaneous constants */
/***************************/
#define UNITSIZE 16                /* size in pixels of default icons */
#define MARKER -1                  /* "unit index" of the marker unit */
#define CUST1_LEN     25           /* length of 1st custom button line */
#define NUM_STD_ICONS 6            /* number of default icons */
#define DFT_NRANGES 16             /* number of icons in default families */
#define DFT_LRANGE_LINKIN (-1000)  /* default low LINKIN/OUT range */
#define DFT_HRANGE_LINKIN   1000   /* default high LININ/OUT range */

/************************/
/* mouse mode constants */
/************************/
#define NORMAL       0
#define MOVE_DROBJ   1
#define MOVE_GROBJ   2
#define MOVE_TXOBJ   3
#define MOVE_DISPLAY 4

/******************************************************************/
/* the following define the array indicies for panel prompt items */
/******************************************************************/
/* item array constants for control panel */
#define MAIN_ITEM_FIRST  0
#define WHO_ITEM         0
#define HOWMANY_ITEM     1
#define HOWMANY_ITEM2    2
#define WHAT_ITEM        3
#define LRANGE_ITEM      4
#define HRANGE_ITEM      5
#define TARGET_ITEM      6
#define HOW_ITEM         7
#define SUBIMAGE_ITEM    8
#define WHERE_ITEM       9
#define WHEREX_ITEM     10
#define WHEREY_ITEM     11
#define SPACEX_ITEM     12
#define SPACEY_ITEM     13
#define NUMCOLS_ITEM    14
#define SHOW_ITEM       15
#define CHANGE_ITEM     16
#define ERASE_ITEM      17
#define MAIN_ITEM_LAST  17

/* for LINK mode: */
#define LINK_ITEM_FIRST  0
#define LDIR_ITEM        0
#define LSITE_ITEM       1
#define LHOW_ITEM        2
#define LLRANGE_ITEM     3
#define LHRANGE_ITEM     4
#define LINK_ITEM_LAST   4

/*for TEXT mode: */
#define TEXT_ITEM_FIRST  0
#define TFONT1_ITEM      0
#define TFONT2_ITEM      1
#define TFONT3_ITEM      2
#define TEXT_ITEM_LAST   2

/* for DRAW mode: */
#define DRAW_ITEM_FIRST  0
#define DTYPE_ITEM       1
#define DRAW_ITEM_LAST   1

/* for CUSTOM mode: */
#define CUST_ITEM_FIRST  0
#define CLBDN1_ITEM      0
#define CLBDN2_ITEM      1
#define CMBDN1_ITEM      2
#define CMBDN2_ITEM      3
#define CRBDN1_ITEM      4
#define CRBDN2_ITEM      5
#define CLBUP1_ITEM      6
#define CLBUP2_ITEM      7
#define CMBUP1_ITEM      8
#define CMBUP2_ITEM      9
#define CRBUP1_ITEM     10
#define CRBUP2_ITEM     11
#define CLOG_ITEM       12
#define CUST_ITEM_LAST  12

/***********************/
/* draw type constants */  
/***********************/
#define LINES       0
#define BOXES       1
#define BOUND       2

/****************************/
/* link direction constants */
/****************************/
#define TO_TARGET   0
#define FROM_TARGET 1

/**********************************************************/
/* argument positions for CHANGE, ERASE and SHOW commands */
/**********************************************************/
#define CMD_TYPE   1
#define CMD_WHO    2
#define CMD_NUM    3
#define CMD_WHAT   4
#define CMD_LRANGE 5
#define CMD_HRANGE 6
#define CMD_TARGET 7 
#define CMD_IMAGE  8
#define CMD_POSX   9
#define CMD_POSY   10
#define CMD_SPAX   11
#define CMD_SPAY   12
#define CMD_CNUM   13

/***********************************************************/
/* constants defining the different unit aspects supported */
/***********************************************************/
#define NOFIND  (-1)
#define POT       1
#define OUTPUT    2
#define STATE     3
#define LINKIN    4
#define LINKOUT   5
#define DATA      6

/***************************************/
/* saves and maintains fonts in memory */
/***************************************/
struct saved_fonts
{
    struct saved_fonts *next;  /* points to next structure on chain */
    char *font_name;           /* points to font name (in string_space) */
    struct pixfont *font_ptr;  /* points to the font itself */
};

/********************************************************/
/* maintains information for a column of the info panel */
/********************************************************/
struct info_unit
{
   int pos;      /* pixel offset of column within the panel */
   int ui;       /* unit index (-1 if not currently used */
   int what;     /* aspect unit is currently tracking */          
   FLINT pot;    /* current unit potential */   
   FLINT out;    /* current unit output */
   FLINT state;  /* current unit state */
   FLINT data;   /* current unit data */
   FLINT link;   /* current unit link value (LINKIN/LINOUT only) */
   int target;   /* current target unit index (LINKIN/LINKOUT only)    */
   char *site;   /* current target site (LINKIN/LINKOUT only)    */
   char name[MAX_NAME_LEN+1];   /* unit name */
   char type[MAX_NAME_LEN+1];   /* unit type */
};

/***********************************************/
/* maintains information for a family of icons */
/***********************************************/
struct gricons
{ 
   struct gricons *next;     /* points to next structure */
   char *name;               /* points to icon name (in string_space) */
   short num;                /* number of icons in the family */
   short size_x;             /* width of icon in pixels */
   short size_y;             /* height of icon in pixels */
   struct pixrect **pix_ptr; /* points to array of pixrect structures */
};

/*****************************************************/
/* maintains information for a unit aspect displayed */
/*****************************************************/
struct grobj
{
   struct grobj *next;   /* points to next grobj structure */
   struct grobj *prev;   /* points to previous grobj structure */
   struct gricons *gptr; /* points to family of icons */ 
   char *site;           /* points to sitename in string_space */
   int x_pos;            /* display space x coordinate */
   int y_pos;            /* display space y coordinate */
   int u_index;          /* unit index */
   int target;           /* target unit index */
   FLINT val;            /* current (main mode) value */
   FLINT link_val;       /* current (link mode) value */
   FLINT lrange;         /* low range for aspect */
   FLINT hrange;         /* high range for aspect */
   short flag;           /* bit flags (see below for constants) */
   short icon_index;     /* index of icon within family (main mode) */ 
   short link_index;     /* index of icon within family (link mode) */
   short u_what;         /* aspect being tracked (main mode) */
};

/***********************************************************/
/* maintains information for a text object being displayed */
/***********************************************************/
struct txobj
{
   struct txobj *next;   /* points to next text object */ 
   struct txobj *prev;   /* points to previous text object */
   char *text_start;     /* points to start of text in string_space */
   struct pixfont *text_font;  /* points to font to use */
   int x_pos;            /* display space x coordinate */
   int y_pos;            /* display space y coordinate */
   short flag;           /* bit flags (see below for constants) */
   short text_len;       /* number of character in text string */
};

/************************************************************/
/* maintains information for a drawn object being displayed */
/************************************************************/
struct drobj
{
   struct drobj *next;   /* points to next drawn object */
   struct drobj *prev;   /* points to previous drawn object */
   int *vptr;            /* points to vertex string in vertex space */
   int left_most;        /* left  most coord of containing rectangle */
   int right_most;       /* right most coord of containing rectangle */
   int top_most;         /* top most coord of containing rectangle */
   int bottom_most;      /* bottom most coord of containing rectangle */
   short flag;           /* bit flags (see below for constants) */
   short num_vertices;   /* number of vertices */
};

/**************************************************/
/* switches for flag of grobj and txobj and drobj */
/**************************************************/
#define TAGGED       1   /* unit tagged as the target unit */
#define NOSHOW       2   /* don't display this unit */
#define DISPLAYED    4   /* this unit currently displayed */
#define MOVING       8   /* object is in process of being moved */
#define LINK_CHG    16   /* link value has changed */
#define VALUE_OK    32   /* value is OK */
#define VALUE_CHG   64   /* value has changed */
#define LINK_OK    128   /* link value is OK */
#define AUX        256   /* auxiliary unit */
#define ON_DISPLAY 512   /* object is located on the display window */
#define BOUND_BOX 1024   /* indicates a bounding box */
