/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains most of the routines related to processing
 * done on or with the info panel.
 *
 * Kenton Lynne
 *******************************************************************/

#include "macros.h"
#include "externs.h"
#include "panel.h"

/* format the icon for the NEXT image */
static short  next_image[16] = {
#include "icon/next"
};
mpr_static(next_icon, 16, 16, 1, next_image);

/* static procedures in this file that are forward referenced */
static next_proc();

static Panel info_panel;

/* info panel items */
static Panel_item  
  ind_item[MAX_INFO_COLS],
  ind_item_val[MAX_INFO_COLS],
  name_item[MAX_INFO_COLS],
  name_item_val[MAX_INFO_COLS],
  type_item[MAX_INFO_COLS],
  type_item_val[MAX_INFO_COLS],
  data_item[MAX_INFO_COLS],
  data_item_val[MAX_INFO_COLS],
  state_item[MAX_INFO_COLS],
  state_item_val[MAX_INFO_COLS],
  pot_item[MAX_INFO_COLS],
  pot_item_val[MAX_INFO_COLS],
  out_item[MAX_INFO_COLS],
  out_item_val[MAX_INFO_COLS],
  link_item[MAX_INFO_COLS],
  link_item_val[MAX_INFO_COLS],
  target_item[MAX_INFO_COLS],
  target_item_val[MAX_INFO_COLS],
  next_info_item;

/*************************************************************************/
gi_make_info_panel()
{
/* creates the info panel 
 */
  struct toolsw *info_panel_sw;
  int i;

  /* create the panel itself */
  if((info_panel_sw = panel_create(gi_tool,
            PANEL_HEIGHT, INFO_HEIGHT,
            PANEL_WIDTH, TOOL_SWEXTENDTOEDGE,
            PANEL_NAME,"Unit Information",
            0)) == NULL)
  {
    fputs("Can't create info_panel\n",stderr);
    exit(1);
  }
  info_panel = info_panel_sw->ts_data;
     
  /* set up info fields for each column of the info display */
  for (i=0; i<MAX_INFO_COLS; i++)
  {
        
     ind_item[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "Index:",
                    PANEL_ITEM_X, INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(0*INFO_RSPACE),
                    0);
     ind_item_val[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "",
                    PANEL_ITEM_X, PANEL_CU(7)+INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(0*INFO_RSPACE),
                    0);
     name_item[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "Name:",
                    PANEL_ITEM_X, INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(1*INFO_RSPACE),
                    0);
     name_item_val[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "",
                    PANEL_ITEM_X, PANEL_CU(6)+INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(1*INFO_RSPACE),
                    0);
     type_item[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "Type:",
                    PANEL_ITEM_X, INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(2*INFO_RSPACE),
                    0);
     type_item_val[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "",
                    PANEL_ITEM_X, PANEL_CU(6)+INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(2*INFO_RSPACE),
                    0);
#ifdef FSIM
     pot_item[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "Pot:",
                    PANEL_ITEM_X, INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(3*INFO_RSPACE),
                    0);
     pot_item_val[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "",
                    PANEL_ITEM_X, PANEL_CU(5)+INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(3*INFO_RSPACE),
                    0);
#else
     pot_item[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "Potential:",
                    PANEL_ITEM_X, INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(3*INFO_RSPACE),
                    0);
     pot_item_val[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "",
                    PANEL_ITEM_X, PANEL_CU(11)+INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(3*INFO_RSPACE),
                    0);
#endif
#ifdef FSIM
     out_item[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "Out:",
                    PANEL_ITEM_X, INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(4*INFO_RSPACE),
                    0);
     out_item_val[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "",
                    PANEL_ITEM_X, PANEL_CU(5)+INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(4*INFO_RSPACE),
                    0);
#else
     out_item[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "Output:",
                    PANEL_ITEM_X, INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(4*INFO_RSPACE),
                    0);
     out_item_val[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "",
                    PANEL_ITEM_X, PANEL_CU(8)+INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(4*INFO_RSPACE),
                    0);
#endif
     state_item[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "State:",
                    PANEL_ITEM_X, INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(5*INFO_RSPACE),
                    0);
     state_item_val[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "",
                    PANEL_ITEM_X, PANEL_CU(7)+INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(5*INFO_RSPACE),
                    0);
     data_item[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "Data:",
                    PANEL_ITEM_X, INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(6*INFO_RSPACE),
                    0);
     data_item_val[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "",
                    PANEL_ITEM_X, PANEL_CU(6)+INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(6*INFO_RSPACE),
                    0);
     link_item[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "Link:",
                    PANEL_ITEM_X, INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(7*INFO_RSPACE),
                    0);
     link_item_val[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "",
                    PANEL_ITEM_X, PANEL_CU(6)+INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(7*INFO_RSPACE),
                    0);
     target_item[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "Target",
                    PANEL_ITEM_X, INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(8*INFO_RSPACE),
                    PANEL_SHOW_ITEM, FALSE,
                    0);
     target_item_val[i] = panel_create_item
                   (info_panel, PANEL_MESSAGE,
                    PANEL_LABEL_STRING, "",
                    PANEL_ITEM_X, PANEL_CU(6)+INFO_STARTX+(i*INFO_CSPACE),
                    PANEL_ITEM_Y, INFO_STARTY+(8*INFO_RSPACE),
                    0);
  } 
        
  /* set up the "NEXT" icons for indicating next column */
  next_info_item = panel_create_item
              (info_panel, PANEL_CHOICE,
               PANEL_FEEDBACK, PANEL_INVERTED,
               PANEL_CHOICE_IMAGES, &next_icon,
                                    &next_icon,
                                    &next_icon,
                                    &next_icon,
                                    &next_icon,
                                    &next_icon,
                                    &next_icon,
                                    &next_icon,
                                    0,
               PANEL_CHOICE_XS, INFO_STARTX+(1*INFO_CSPACE/2)-20,
                                INFO_STARTX+(3*INFO_CSPACE/2)-20,
                                INFO_STARTX+(5*INFO_CSPACE/2)-20,
                                INFO_STARTX+(7*INFO_CSPACE/2)-20,
                                INFO_STARTX+(9*INFO_CSPACE/2)-20,
                                INFO_STARTX+(11*INFO_CSPACE/2)-20,
                                INFO_STARTX+(13*INFO_CSPACE/2)-20,
                                INFO_STARTX+(15*INFO_CSPACE/2)-20,
                                0,
               PANEL_CHOICE_YS, INFO_HEIGHT-16, 0,
               PANEL_NOTIFY_PROC, next_proc,
               0);
}
    
/*************************************************************************/
/*ARGSUSED*/
static next_proc(item, value, event)
   Panel_item item;
   int value;
   struct inputevent *event;
{
/* called when the NEXT icon under an info item is selected
 * sets the NEXT indication to the appropriate
 * information column
 */
 
   if (gi_info_next==value) 
   {
     /* blank out this column */
     gi_info_item[gi_info_next].ui = -1;
     gi_show_info(gi_info_next,TRUE);
   }
   gi_info_next = value;
}

/*************************************************************************/
gi_show_info(col, all_flag)
   int col;
   int all_flag;
{
/* Given an gi_info_item index (column),
 * displays the detailed information 
 * that is contained in the corresponding gi_info_item structure.
 * all_flag : TRUE requests that all fields should be updated
 *          : FALSE requests only changing information
 * NOTE: if ui of info_item is < 0, then column is blanked out
 * Assumes that all info in gi_info_item has been
 * brought up to date by previous call to get_unit_info
 */

  char *dir_ind = TO_DIR_STRING; 
  char buf[MAX_NAME_LEN+1], tbuf[2*MAX_NAME_LEN+1];
  int bold_flag[DATA+1];  /* for deciding which fields in bold */
  int i;
 
  /* if invalid unit number, then blank column out */
  if (gi_info_item[col].ui < 0)
  {
    /* blank out this column */
    panel_set(link_item_val[col], PANEL_LABEL_STRING, "", 0);
    panel_set(ind_item_val[col],  PANEL_LABEL_STRING, "", 0);
    panel_set(name_item_val[col], PANEL_LABEL_STRING, "", 0);
    panel_set(type_item_val[col], PANEL_LABEL_STRING, "", 0);
    panel_set(data_item_val[col], PANEL_LABEL_STRING, "", 0);
    panel_set(pot_item_val[col],  PANEL_LABEL_STRING, "", 0);
    panel_set(out_item_val[col],  PANEL_LABEL_STRING, "", 0);
    panel_set(state_item_val[col], PANEL_LABEL_STRING, "", 0);
    panel_set(target_item_val[col], PANEL_LABEL_STRING, "", 0);
    panel_set(target_item[col], PANEL_SHOW_ITEM, FALSE,0);
    return;
  }
    
  /* initialize all the bold flags to FALSE */
  for (i=POT; i<=DATA; i++) bold_flag[i] = FALSE;

  /* if LINK mode, set up direction and target fields */
  if (gi_mode==MODE_LINK)
  {
    bold_flag[LINKIN] = TRUE;

    if (gi_link_direction!=TO_TARGET)
      dir_ind = FROM_DIR_STRING;

    /* put direction indication and target ui into buffer */
    (void) sprintf(tbuf,"%s%d",dir_ind,gi_cur_link_target);

    /* if site specified, append it to the buffer */
    if (gi_cur_link_site)
    {
      strcat(tbuf,SITE_SEP_STRING);
      strcat(tbuf,gi_cur_link_site);
    }
  }

  /* if main mode, bold font depends on what unit is displaying */
  else 
  {
    switch (gi_info_item[col].what)
    {
      case POT: 
             bold_flag[POT] = TRUE;
             break;
      case OUTPUT:
             bold_flag[OUTPUT] = TRUE;
             break;
      case STATE:
             bold_flag[STATE] = TRUE;
             break;
      case DATA:
             bold_flag[DATA] = TRUE;
             break;
      case LINKOUT:
             dir_ind = FROM_DIR_STRING;
      case LINKIN:
             bold_flag[LINKIN] = TRUE;
             break;
    }
  }

 
  /* if LINK value is to be shown, set up target/site field */
  if (bold_flag[LINKIN])
  {
#ifdef FSIM
    (void) sprintf(buf,"%-1.4g",gi_info_item[col].link);
#else
    (void) sprintf(buf,"%1d",gi_info_item[col].link);
#endif
    panel_set(link_item_val[col], 
            PANEL_LABEL_STRING, buf, 
            PANEL_LABEL_BOLD, TRUE, 0);

    /* if not LINK mode, set up particular target and site */
    if (gi_mode!=MODE_LINK)
    {
      (void) sprintf(tbuf,"%s%d",dir_ind,gi_info_item[col].target);
      if (gi_info_item[col].site)
      {
        strcat(tbuf,SITE_SEP_STRING);
        strcat(tbuf,gi_info_item[col].site);
      }
    }

    /* NOTE: if LINK mode, target/site has been set above */
    panel_set(target_item[col], PANEL_SHOW_ITEM, TRUE, 0); 
    panel_set(target_item_val[col],
              PANEL_LABEL_STRING, tbuf,
              0);
  }
  else /* not LINK value : blank out target item */
  {
    panel_set(link_item_val[col], PANEL_LABEL_STRING, "", 0); 
    panel_set(target_item[col], PANEL_SHOW_ITEM, FALSE, 0); 
    panel_set(target_item_val[col], PANEL_LABEL_STRING, "",0); 
  }

  /* if requested, set up the "static" fields as well */
  if (all_flag)
  {
    /* unit index field */
    (void) sprintf(buf,"%1d",gi_info_item[col].ui);
    panel_set(ind_item_val[col],
              PANEL_LABEL_STRING, buf,
              0);

    /* unit name field */
    panel_set(name_item_val[col],
              PANEL_LABEL_STRING, gi_info_item[col].name,
              0);

    /* unit type field */
    panel_set(type_item_val[col],
             PANEL_LABEL_STRING, gi_info_item[col].type,
             0);
  }
    
  /* now set up the values for the dynamic fields */

#ifdef FSIM  
  (void) sprintf(buf,"%-1.4g",gi_info_item[col].pot);
#else
  (void) sprintf(buf,"%1d",gi_info_item[col].pot);
#endif
  panel_set(pot_item_val[col],
            PANEL_LABEL_STRING,buf,
            PANEL_LABEL_BOLD, bold_flag[POT],
            0);

#ifdef FSIM  
  (void) sprintf(buf,"%-1.4g",gi_info_item[col].out);
#else
  (void) sprintf(buf,"%1d",gi_info_item[col].out);
#endif
  panel_set(out_item_val[col],
            PANEL_LABEL_STRING,buf,
            PANEL_LABEL_BOLD, bold_flag[OUTPUT],
            0);

#ifdef FSIM  
  (void) sprintf(buf,"%1d",(int) gi_info_item[col].state);
#else
  (void) sprintf(buf,"%1d",gi_info_item[col].state);
#endif
  panel_set(state_item_val[col],
            PANEL_LABEL_STRING, buf,
            PANEL_LABEL_BOLD, bold_flag[STATE],
            0);

#ifdef FSIM  
  (void) sprintf(buf,"%-1.4g",gi_info_item[col].data);
#else
  (void) sprintf(buf,"%1d",gi_info_item[col].data);
#endif
  panel_set(data_item_val[col],
            PANEL_LABEL_STRING, buf,
            PANEL_LABEL_BOLD, bold_flag[DATA],
            0);
   
}

/*************************************************************************/
gi_do_info(x, y, log_flag)
  int x, y, log_flag;
{
/* given display coordinates, update
 * the info panel with the value for
 * the unit at that position
 * if no unit is at that position
 * return ERROR else OK
 */
  struct grobj *ptr;
   
  /* check if there is an object at this display coordinate */
  if ((ptr=gi_find_grobj(x,y))!=NULL)
  {
    /* set up the "next" gi_info_item with unit index and what value */
    gi_info_item[gi_info_next].ui = ptr->u_index;
    gi_info_item[gi_info_next].what = ptr->u_what;
    gi_info_item[gi_info_next].target = ptr->target;
    gi_info_item[gi_info_next].site = ptr->site;
 
    /* get the latest values for this unit from the simulator */
    gi_get_info(gi_info_next,TRUE);
 
    /* show the information on the screen */
    gi_show_info(gi_info_next, TRUE);
 
    /* if logging enabled, log this command */
    if (gi_log_on && log_flag) gi_log_info(x, y, gi_info_next+1);
 
    /* calculate next info column */
    gi_info_next = (gi_info_next + 1) % MAX_INFO_COLS;
    if (gi_rects[INFO].r_width < gi_info_item[gi_info_next].pos)
      gi_info_next = 0;
 
    /* rotate NEXT icon right one column */
    panel_set(next_info_item, PANEL_VALUE, gi_info_next, 0);
 
    return(OK);
  }
  return(ERROR);
}
