/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

#include	"sim.h"
#ifdef	FSIM
#	define	BP_ONE	1.0
#	define	BP_ZERO	0.0
#	define	CAST	float
#else
#	define	BP_ONE	1000
#	define	BP_ZERO	0
#	define	CAST	int
#endif
#include "bp.h"

char	 bp_flag = '\0';		/* for keeping track of building */

/** BPmodule is the first thing that must be called before setting up a BP
	network.  The arguments are the name of the network and how many
	layers will be in the network (NOT including input layers or teaching
	layers).  Space is allocated (name_mem) to hold the module name and
	any prefix or suffix that might be needed.  "mod_name" is indexed five
	chars in from the begining (that's where the actual name starts) so
	that it can be refered to without the prefix.  "mod_num" is indexed
	after the name so that layer numbers can be manipulated easily.

	BPmodule first checks to make sure that another module is not being
	built.

	BPmodule enters a name in the nametable with type BP_SYM.  The other
	fields correspond to how many total layers, the completeness of the 
	module (ICMP for incomplete), and the index of the fire unit (starts
	as -1).  It then makes a control unit with a fire site.  Then a
	bias unit (no sites).

	"lastu" is a global variable which is the index of the last unit
	that was created withing the module.  This is set here and then
	used in other routines to make sure no unknown units have been 
	created.  It is updated in all routines that make units.

	A flag is then set to indicate that a module has been started.
**/

BPmodule(name, layers)
char	*name;
int	layers;
{
	int	name_size;

	if(bp_flag)
	{	LOGfprintf(stderr, "BPmodule: still building a module!  Other errors will occur...\n");
		return;
	}
	
	if(layers <= 0)
	{	LOGfprintf(stderr, "BPmodule: there must be at least 1 layer, layers was %d\n", layers);
		return;
	}
	layer_t = layers;
	layer_c = 1;
	fu_index = BP_NO_FIRE;
	
	/* Create enough space to save module name so that there is room
	 * for all necessary prefixes and suffixes (e.g. bias_name, name(42)).
	 * +5 for prefixes, +8 for suffixes, and +1 for \0.
	 */
	name_size = (strlen(name) < NAME_SIZE) ? strlen(name) : NAME_SIZE;
	if((name_mem = (char *)malloc(8+5+name_size+1)) == NULL)
	{	LOGfprintf(stderr, "BPmodule: out of space\n");
		abort();
	}
	mod_name = name_mem + 5;			/* past prefixes */
	mod_num = mod_name + name_size;			/* point to suffix */
	strncpy(mod_name, name, name_size + 1);	        /* add 1 for \0 */

	/* Enter module name in nametable with special field values */
	EnterName(mod_name, BP_SYM, layer_t, ICMP, BP_NO_FIRE);

	/* Make a control unit with type "$control" and name unit
	 * with prefix "cont_" before the module name.  Then add Fire site
	 * with name "$fire".
	 */
	cu_index = MakeUnit("$control", UFcontrol, BP_ZERO, BP_ZERO, (CAST)layer_t, BP_ZERO, QUIET, QUIET);
	strncpy(name_mem, "cont_", 5);
	NameUnit(name_mem, SCALAR, cu_index, 0, 0);
	AddSite(cu_index, "$fire", SFsum, BP_ZERO);
	DeclareState("Quiet", QUIET);				/* !!! */
	DeclareState("Forward", FWD);
	DeclareState("Reverse", REV);

	/* Make a bias unit with type "$bias" and name unit
	 * with prefix "bias_" before the module name.
	 */
	bu_index = MakeUnit("$bias", UFone, BP_ZERO, BP_ZERO, BP_ZERO, BP_ZERO, 0, 0);
	strncpy(name_mem, "bias_", 5);
	NameUnit(name_mem, SCALAR, bu_index, 0, 0);

	lastu = bu_index;
	SMODULE;			       	/* set bp okay flag */
	if(layer_c == layer_t)                 	/* if no hidden layers */
		SNHIDDEN;
}

/** BPname is used internally.  Only used for nested func calls.  Sets
	suffix for "layer."
**/

char *
BPname(layer)
int	layer;						/* layer of unit */
{
	(void) sprintf(mod_num, "(%d)", layer);
	return mod_name;
}

/** BPGetSite is used internally.  Given a sight name and a unit index it
	returns a pointer to the named sight of the unit.  Only used in
	nested func calls (although no reason to limit to that).
**/

Site *
BPGetSite(name, unit)
char	*name;					/* name of site */
int	unit;					/* global unit index */
{
	Site	*sp;
	NameDesc	nd;

	if (FindName(name, &nd))
		for(sp = UnitList[unit].sites; sp != NULL; sp = sp->next)
			if(!strcmp(sp->name, nd.name))
				return sp;
	LOGfprintf(stderr, "BPGetSite: Site %s not found for unit %d\n", name, unit);
	abort();
	/*NOTREACHED*/
}

/** BPlink, given a "from" and "to" layer, a "from" and "to" unit index
	(within a layer) and a weight, will make a link from the "from" unit to
	the "to" unit.  The Link Function Pointer will actually be a pointer
	to the data field of the "$learn" sight of the "from" unit.  
	
	If the from layer is -1 then the from index is interpreted as a global
	unit index and the Link Function Pointer will be set to the data field
	of the "$fire" site of the control unit; this is used when linking
	from the bias unit.

	If the from layer is 0 then the LFP is set as for -1, but the "from"
	index is relative to the input layer (0).

	Debuging is turned off when the link is made so no errors will result
	in suplying MakeLink with a bad LFP.
**/

BPlink(layer_f, index_f, layer_to, index_to, weight)
int	layer_f, index_f,			/* from index and layer #'s */
	layer_to, index_to;			/* to index and layer #'s */
FLINT	weight;					/* weight of link */
{
	register int	from_i,			/* global index FROM unit */
			to_i,			/* global index TO unit */
			s_Debug;
	FLINT	*bp_ptr;			/* ptr to data field in site */

	if(Debug)
	{
		if(!MODULE)
		{	LOGfprintf(stderr, "BPlink: Not called inside module!\n");
			return;
		}
		if(layer_f >= layer_t)
		{	LOGfprintf(stderr, "BPlink: FROM layer is out of range, max %d, chose %d\n", layer_t - 1, layer_f);
			return;
		}
		if(layer_f != -1 && layer_f >= layer_to)
		{	LOGfprintf(stderr, "BPlink: TO layer must be at least one more than FROM layer: %d->%d\n", layer_f, layer_to);
			return;
		}
	}
	if((to_i = NameToInd(BPname(layer_to), index_to, 0)) == -1)
	{	LOGfprintf(stderr, "BPlink: trying to link to non-unit %s[%d]\n", BPname(layer_to), index_to);
		return;
	}
	if(layer_f != -1)
	{	if((from_i = NameToInd(BPname(layer_f), index_f, 0)) == -1)
		{	LOGfprintf(stderr, "BPlink: trying to link from non-unit %s[%d]\n", BPname(layer_f), index_f);
			return;
		}
		if(!layer_f)
			bp_ptr = &((BPGetSite("$fire", cu_index))->data);
		else
			bp_ptr = &((BPGetSite("$learn", from_i))->data);
	} else
	{	from_i = index_f;
		bp_ptr = &((BPGetSite("$fire", cu_index))->data);
	}
	s_Debug = Debug;
	Debug = 0;			/* Debug off so MakeLink wont complain */
	MakeLink(from_i, to_i, "$learn", weight, BP_ZERO, bp_ptr);
	Debug = s_Debug;
}

/** BPinput creates an input level of "nu" units.  First a check is made to
	make sure that this call is within a module, and that this is the fist
	call to BPinput.  The units have no unit funtion.  The name of the
	layer is "<modulename>(0)".  The variable "lastu" is used to check
	that no units were created without using BP commands.  The macro
	SNINPUT makes is so that an input layer cannot be made again.  The
	global unit index is returned.
**/

BPinput(nu)
int	nu;					/* number of units in layer */
{
	int	u, i;

	if(Debug)
	{
		if(!MODULE)
		{	LOGfprintf(stderr, "BPinput: Not called inside module!\n");
			return -1;
		}
		if(NINPUT)
		{	LOGfprintf(stderr, "Badly placed call to BPinput\n");
			return -1;
		}
	}
	SNINPUT;

	u = MakeUnit("$input", NULL, BP_ZERO, BP_ZERO, BP_ZERO, BP_ZERO, 0, 0);
	if(Debug && u != lastu + 1)
		LOGfprintf(stderr, "BPinput: Created some unit without using special commands...going to make layer anyway\n");
	for(i = 1; i < nu; i++)
		MakeUnit("$input", NULL, BP_ZERO, BP_ZERO, BP_ZERO, BP_ZERO, 0, 0);
	lastu = u + nu - 1;
	strcpy(mod_num, "(0)");
	NameUnit(mod_name, VECTOR, u, nu, 0);
	return u;
}

/** BPhidden creates a hidden layer.  Its arguments are as follows: number
	of units, unit function, site funtion, bias flag, bias weight, 
	initial potantial (for units), data field (for units), output,
	initial state, and state (all for units).  First a check is made to 
	make sure that the call is within a module, and that it is legal to
	make hidden layers at this point.  Calls to BPinput are inhibited
	(in case no call was already made).  A layer of units of type
	"$hidden" is produced.  The NO_UNIT and NO_LINK function flags are 
	set, as well as the BP_FORWARD_FLAG (care should be taken that this
	flag, as defined in bp.h, does not conflict with a flag in another
	package).  A learn site is created on each unit.  If the bias flag
	is set (from arg "bias") then the bias unit is linked to each
	unit in this layer with wieght "bias_w".  "lastu" is updated and
	if this is the last legal hidden unit then further calls to 
	BPhidden are prohibited by macro SNHIDDEN.  The global unit index
	to the layer vector is returned.
**/

BPhidden(nu, unit_f, site_f, bias, bias_w, ipot, pot, data, out, istate, state)
int	nu,					/* number of units */
	bias;					/* connect bias unit if true */
FLINT	bias_w, 				/* wieght for bias unit link */
	ipot, pot, data, out;			/* MakeUnit args */
int	istate, state;
FLINT	(*unit_f)(), (*site_f)();		/* unit/site function ptrs */
{
	register int	u, u_tmp,		/* unit indicies */
			i;

	if(Debug)
	{	if(!MODULE)
		{	LOGfprintf(stderr, "BPhidden: Not called inside module!\n");
			return -1;
		}
		if(NHIDDEN)
		{	LOGfprintf(stderr, "Badly placed call to BPhidden\n");
			return -1;
		}
	}
	SNINPUT;				/* no calls to BPinput now */

	u = MakeUnit("$hidden", unit_f, ipot, pot, data, out, istate, state);
	if(Debug && u != lastu + 1)
		LOGfprintf(stderr, "BPhidden: Created some unit without using special commands...making layer anyway\n");
	SetFlag(u, NO_UNIT_FUNC_FLAG);
	SetFlag(u, NO_LINK_FUNC_FLAG);
	SetFlag(u, BP_FORWARD_FLAG);
	AddSite(u, "$learn", site_f, BP_ZERO);
	for(i = 1; i < nu; i++)
	{	u_tmp = MakeUnit("$hidden", unit_f, ipot, pot, data, out, istate, state);
		SetFlag(u_tmp, NO_UNIT_FUNC_FLAG);
		SetFlag(u_tmp, NO_LINK_FUNC_FLAG);
		SetFlag(u_tmp, BP_FORWARD_FLAG);
		AddSite(u_tmp, "$learn", site_f, BP_ZERO);
	}
	(void) sprintf(mod_num, "(%d)", layer_c);
	NameUnit(mod_name, VECTOR, u, nu, 0);
	if(bias)
		for(i = 0; i < nu; i++)
			BPlink(-1, bu_index, layer_c, i, bias_w);
	lastu = u + nu - 1;
	if(++layer_c == layer_t)		/* next layer is output */
		SNHIDDEN;			/* allow no more `hidden' */
	return u;
}

/** BPoutput is similar to BPhidden except that it also puts an "$error"
	site on each unit.  Consequently it take an error site function
	as one of its arguments.
**/

BPoutput(nu, unit_f, lsite_f, esite_f, bias, bias_w, ipot, pot, data, out, istate, state)
int	nu,					/* number of units */
	bias;					/* connect bias unit if true */
FLINT	bias_w, 				/* wieght for bias unit link */
	ipot, pot, data, out;			/* MakeUnit args */
int	istate, state;
FLINT	(*unit_f)(), (*lsite_f)(), (*esite_f)();/* unit/site function ptrs */
{
	register int	u, u_tmp,
			i;

	if(Debug)
	{	if(!MODULE)
		{	LOGfprintf(stderr, "BPoutput: Not called inside module!\n");
			return -1;
		}
		if(!NHIDDEN)
		{	LOGfprintf(stderr, "Badly placed call to BPoutput, need hidden layer(s) first\n");
			return -1;
		}
	}
	
	u = MakeUnit("$output", unit_f, ipot, pot, data, out, istate, state);
	if(Debug && u != lastu + 1)
		LOGfprintf(stderr, "BPoutput: Created some unit without using special commands...making layer anyway\n");
	SetFlag(u, NO_UNIT_FUNC_FLAG);
	SetFlag(u, NO_LINK_FUNC_FLAG);
	SetFlag(u, BP_FORWARD_FLAG);
	AddSite(u, "$learn", lsite_f, BP_ZERO);
	AddSite(u, "$error", esite_f, BP_ZERO);
	for(i = 1; i < nu; i++)
	{	u_tmp = MakeUnit("$output", unit_f, ipot, pot, data, out, istate, state);
		SetFlag(u_tmp, NO_UNIT_FUNC_FLAG);
		SetFlag(u_tmp, NO_LINK_FUNC_FLAG);
		SetFlag(u_tmp, BP_FORWARD_FLAG);
		AddSite(u_tmp, "$learn", lsite_f, BP_ZERO);
		AddSite(u_tmp, "$error", esite_f, BP_ZERO);
	}
	(void) sprintf(mod_num, "(%d)", layer_c);
	NameUnit(mod_name, VECTOR, u, nu, 0);
	if(bias)
		for(i = 0; i < nu; i++)
			BPlink(-1, bu_index, layer_c, i, bias_w);
	layer_c++;
	lastu = nu + u - 1;
	return u;
}

/** BPteach, like BPinput, is optional.  It checks to make sure it is called
	within a module, that no more calls to BPhidden can be made, and
	that a fire unit has not been made already.  It then checks to make
	sure that it is being called in the right order in terms of layers.
	It makes a layer with the same number of units as the output layer.
	The units have no function and no sites.  They are linked to the
	$error sites of their respective output layer units.  The global
	unit index of the unit vector is returned.
**/

BPteach()
{
	NameDesc	out_desc;			/* output unit info */
	int	u, i, utmp,
		out_ind;				/* index output unit */

	if(Debug)
	{	if(!MODULE)
		{	LOGfprintf(stderr, "BPteach: Not called inside module!\n");
			return -1;
		}
		if(!NHIDDEN || FIRE)
		{	LOGfprintf(stderr, "Badly placed call to BPteach\n");
			return -1;
		}
		if(layer_c != layer_t + 1)
		{	LOGfprintf(stderr, "BPteach: current layer is %d, should be %d.  Forgot BPoutput?\n", layer_c, layer_t + 1);
			return -1;
		}
	}

	(void) sprintf(mod_num, "(%d)", layer_t);		/* output layer name */
	out_ind = (FindName(mod_name, &out_desc))->index;
	u = MakeUnit("$teach", NULL, BP_ZERO, BP_ZERO, BP_ZERO, BP_ZERO, 0, 0);
	if(Debug && u != lastu + 1)
		LOGfprintf(stderr, "BPteach: Created some unit without using special commands...making layer anyway\n");
	SetFlag(u, NO_UNIT_FUNC_FLAG);
	SetFlag(u, NO_LINK_FUNC_FLAG);
	MakeLink(u, out_ind++, "$error", BP_ONE, BP_ZERO, NULL);
	for(i = 1; i < out_desc.size; ++i, ++out_ind)
	{	utmp = MakeUnit("$teach", NULL, BP_ZERO, BP_ZERO, BP_ZERO, BP_ZERO, 0, 0);
		SetFlag(utmp, NO_UNIT_FUNC_FLAG);
		SetFlag(utmp, NO_LINK_FUNC_FLAG);
		MakeLink(utmp, out_ind, "$error", BP_ONE, BP_ZERO, NULL); /* ! */
	}
	lastu = out_desc.size + u - 1;
	(void) sprintf(mod_num, "(%d)", layer_c++);
	NameUnit(mod_name, VECTOR, u, out_desc.size, 0);
	return u;
}

/** BPfire creates an optional fire unit with name "fire_<modulename>".  It
	first checks to make sure that it is being called in the proper place.
	The fire unit is linked to the $fire site of the cotnrol unit.
	Further calls to BPfire are inhibited, and the index of the fire unit
	is returned.
**/

BPfire()
{
	if(Debug)
	{	if(!MODULE)
		{	LOGfprintf(stderr, "BPfire: Not called inside module!\n");
			return -1;
		}
		if(!NHIDDEN || layer_c <= layer_t)
		{	LOGfprintf(stderr, "Cannot call BPfire, not done with layers\n");
			return -1;
		}
		if(FIRE)
		{	LOGfprintf(stderr, "BPfire: Already created Fire unit\n");
			return -1;
		}
	}

	fu_index = MakeUnit("$fire_u", UFfire, BP_ZERO, BP_ZERO, BP_ZERO, BP_ZERO, 0, 0);
	if(Debug && fu_index != lastu + 1)
		LOGfprintf(stderr, "BPfire: Created a unit not using special commands...creating unit anyway\n");

	MakeLink(fu_index, cu_index, "$fire", BP_ONE, BP_ZERO, NULL);
	strncpy(name_mem, "fire_", 5);
	*mod_num = '\0';
	NameUnit(name_mem, SCALAR, fu_index, 0, 0);
	SFIRE;
	return fu_index;
}

/** BPendmod closes the module.  It first checks to see that a module was
	started and completed.  It then updates the nametable entry so that
	the module now has CMP (completed) status and the index of the
	fire unit is added (default is BP_NO_FIRE if unit was not made).
**/

BPendmod(name)
char	*name;
{
	*mod_num = '\0';
	if(Debug)
	{	if(!MODULE)
		{	LOGfprintf(stderr, "BPendmod: Not called inside module...more errors will occur\n");
			return;
		}
		if(layer_c <= layer_t)
		{	LOGfprintf(stderr, "BPendmod: Cannot end module yet, need %d more layer(s)\n", (layer_t - layer_c) + 1);
			return;
		}

		/* warn about certain harmless (for now) coding errors... */
		if (name == 0)
			LOGfprintf(stderr, "BPendmod: warning: missing module name\n");
		else if (strcmp(name, mod_name) != 0)
			LOGfprintf(stderr, "BPendmod: warning: module name mismatch ('%s' != '%s')\n", mod_name, name);
	}

	bp_flag = '\0';
	AlterName(mod_name, BP_SYM, layer_t, CMP, fu_index);
	free(name_mem);
}

/** BPsetinput takes a module name, a unit index (within the input layer)
	and a value.  It checks to make sure an input level for that module
	exists, and if it does it checks that the unit index is not out
	of range.  It then sets the output of that unit to "val".

	It finds the index to the input layer vector by constructing a name
	which is the name of the vector ("foo(0)), and then calls FindName
	on that name.
**/

BPsetinput(name, unit, val)
char	*name;
int	unit;
FLINT	val;
{
	NameDesc	nd;
	char	*n;
	
	n = (char *)malloc(strlen(name) + 4);
	(void) sprintf(n, "%s(0)", name);
	if(!FindName(n, &nd))
	{	LOGfprintf(stderr, "BPsetinput: No input layer!\n");
		return;
	}
	if(unit >= nd.size || unit < 0)
	{	LOGfprintf(stderr, "BPsetinput: Unit (%d) out of range, highest is %d\n", unit, nd.size - 1);
		return;
	}
	SetOutput(nd.index + unit, val);
	free(n);	
}

/** BPsetteach does the same thing as BPsetinput except that it acts on
	the teach layer.  It finds the index to the teach layer vector by
	finding out the layer number of the teach layer (first call to
	FindName) and then constructs a new name which is the name of
	the teach layer and calls FindName again.
**/

BPsetteach(name, unit, val)
char	*name;
int	unit;
FLINT	val;
{
	NameDesc	nd;
	register int	l;
	char	*n;

	if(!FindName(name, &nd))
	{	LOGfprintf(stderr, "BPsetteach: No such module \"%s\"\n", name);
		return;
	}
	if(nd.type  != BP_SYM)
	{	LOGfprintf(stderr, "BPsetteach: \"%s\" is not the name of a module.\n", name);
		return;
	}
	l = nd.index + 1;		/* "index" unrelated */
	n = (char *)malloc(strlen(name) + 9);
	(void) sprintf(n, "%s(%d)", name, l);
	if(!FindName(n, &nd))
	{	LOGfprintf(stderr, "BPsetteach: No teach layer!\n");
		return;
	}
	if(unit >= nd.size || unit < 0)
	{	LOGfprintf(stderr, "BPsetteach: Unit (%d) out of range, highest is %d\n", unit, nd.size - 1);
		return;
	}
	SetOutput(nd.index + unit, val);
	free(n);
}

/** BPcycle takes two arguments.  The first is a string representing the name
	of the module, the second is the number of cycles to allow the module
	to execute.  A cycle is a complete forward and backward pass through
	the bp network, so the number of clock steps involved is #layers*2
	(hidden layers + output layer only).  BPcycle sets the output of the
	fire unit to that number.  The fire unit will subtract 1 from its
	output each time it is called.  The control unit will only activate
	a cycle if it is recieving input from the fire unit.  Once a cycle
	is started it will run through until completion.
**/

BPcycle(name, cycles)
char	*name;
int	cycles;
{
	NameDesc	nd;

	if(!FindName(name, &nd))
	{	LOGfprintf(stderr, "BPcycle: No such module \"%s\"\n", name);
		return;
	}

	if(Debug)
	{
		if(nd.type != BP_SYM)
		{	LOGfprintf(stderr, "BPcycle: \"%s\" is not the name of a module.\n", name);
			return;
		}
		if(nd.size != CMP)
		{	LOGfprintf(stderr, "BPcycle: Have not completed module: %s!\n", name);
			return;
		}
		if(nd.length == BP_NO_FIRE)
		{	LOGfprintf(stderr, "BPcycle: No fire unit for modual!\n");
			return;
		}
	}
	SetOutput(nd.length, (CAST)nd.index*2*cycles);/* nd.length = fu_index */
						      /* nd.index = layer_t */
}
