/****************************************************************
**                                                             **
**                        SPIKE_PARMS                          **
**                                                             **
**          By Upinder S. Bhalla, May 1990. Caltech            **
**                                                             **
** This file contains routines for extracting a number of      **
** parameters of a spike that may be useful for comparing      **
** different spike traces. The objective is to get a set of    **
** such parameters which will enable one to evaluate the       **
** validity of a neuronal simulation and to constrain the      **
** parameters used in the simulation.                          **
**                                                             **
****************************************************************/
#include <stdio.h>
#include <math.h>

#define MAXPTS 2000
#define MAXSPIKES 200
#define ENDT 10.0
#define PEAKRANGE 0.005	/* 5 mV should be plenty */
#define DEFAULTMAX -0.02	/* -10 mV as a lower acceptable spike */
#define DEFAULTLO -0.05	/* -50 mV as an upper acceptable AHP */

char *do_spike_parms(argc,argv)
	int argc;
	char	**argv;
{
	float	x[MAXPTS],y[MAXPTS];
	float	dydx[MAXPTS],d2y[MAXPTS],theta[MAXPTS];
	int		peak[MAXSPIKES],valley[MAXSPIKES];
	int		peakno = 0,valleyno=0;
	int		npts = 0;
	int		plotno = 0;
	float	t,dt;
	float	temp;
	int		tempindex;
	float	dtsq,dy1,r,r1;
	float	dy,dx;
	float	startt = 0,endt = ENDT;
	float	peakest = DEFAULTMAX,hyperpolest = DEFAULTLO;
	int		i,k;
	int		RecIndex,KneeIndex,RiseIndex;
	char	*outputfile,*filemode,*notes;

	/* Variables to be evaluated */
	float	PTP=0;
	float	ISI = 0, SpikeWidth = 0, MaxV = 0;
	float	LoT = 0, LoV = 0, LoCurve = 0; 
	float	RiseSlope=0, FallSlope=0;
	float	RecSlope=0,RecT=0,RecV=0;
	float	InflSlope=0,InflT=0,InflV=0;
	float	KneeT=0,KneeV=0,KneeCurve=0;
	float	ThreshT=0,ThreshV=0,ThreshCurve=0;

	float	*allvars[50];
	char	*varnames[50];
	int		nvars;

	if (argc < 2) {
		printf("usage : %s filename [-plot plotno] [-start time] [-end time]\n",argv[0]);
		printf("[-peak V] [-hyperpol V] [-file outputfile mode] [-notes notes]\n");
		printf("Valid filemodes : multifile onefile onefileappend\n");
		printf("The command returns :\n");
		printf("InterSpikeInterval SpikeWidth MaxV LoT LoV LoCurve RiseSlope FallSlope\n");
		printf("RecSlope RecT RecV InflSlope InflT InflV\n");
		printf("KneeT KneeV KneeCurve ThreshT ThreshV ThreshCurve\n");
		return;
	}
	outputfile = NULL;
	filemode = NULL;
	notes = NULL;
	for (i = 2 ; i < argc ; i++) {
		if (strcmp(argv[i],"-plot") == 0) {
			i++; plotno = atoi(argv[i]);
		}
		if (strcmp(argv[i],"-start") == 0) {
			i++; startt = atof(argv[i]);
		}
		if (strcmp(argv[i],"-end") == 0) {
			i++; endt = atof(argv[i]);
		}
		if (strcmp(argv[i],"-peak") == 0) {
			i++; peakest = atof(argv[i]);
		}
		if (strcmp(argv[i],"-hyperpol") == 0) {
			i++; hyperpolest = atof(argv[i]);
		}
		if (strcmp(argv[i],"-notes") == 0) {
			i++; notes = argv[i];
		}
		if (strcmp(argv[i],"-file") == 0) {
			if (argc < (i + 2)) {
				printf("syntax : -file outputfile mode \n");
				return;
			}
			i++; outputfile = argv[i];
			i++; filemode = argv[i];
		}
	}
	nvars = 0;
	allvars[nvars] = &ISI; varnames[nvars] = "ISI"; nvars++;
	allvars[nvars] = &PTP; varnames[nvars] = "PTP"; nvars++;
	allvars[nvars] = &SpikeWidth; varnames[nvars] = "SpikeWidth"; nvars++;
	allvars[nvars] = &MaxV; varnames[nvars] = "MaxV"; nvars++;
	allvars[nvars] = &RiseSlope; varnames[nvars] = "RiseSlope"; nvars++;
	allvars[nvars] = &FallSlope; varnames[nvars] = "FallSlope"; nvars++;
	allvars[nvars] = &LoT; varnames[nvars] = "LoT"; nvars++;
	allvars[nvars] = &LoV; varnames[nvars] = "LoV"; nvars++;
	allvars[nvars] = &LoCurve; varnames[nvars] = "LoCurve"; nvars++;
	allvars[nvars] = &RecT; varnames[nvars] = "RecT"; nvars++;
	allvars[nvars] = &RecV; varnames[nvars] = "RecV"; nvars++;
	allvars[nvars] = &RecSlope; varnames[nvars] = "RecSlope"; nvars++;
	allvars[nvars] = &InflT; varnames[nvars] = "InflT"; nvars++;
	allvars[nvars] = &InflV; varnames[nvars] = "InflV"; nvars++;
	allvars[nvars] = &InflSlope; varnames[nvars] = "InflSlope"; nvars++;
	allvars[nvars] = &KneeT; varnames[nvars] = "KneeT"; nvars++;
	allvars[nvars] = &KneeV; varnames[nvars] = "KneeV"; nvars++;
	allvars[nvars] = &KneeCurve; varnames[nvars] = "KneeCurve"; nvars++;
	allvars[nvars] = &ThreshT; varnames[nvars] = "ThreshT"; nvars++;
	allvars[nvars] = &ThreshV; varnames[nvars] = "ThreshV"; nvars++;
	allvars[nvars] = &ThreshCurve; varnames[nvars] = "ThreshCurve"; nvars++;

	if (!read_plot2(argv[1],plotno,&dt,y,MAXPTS,&npts,startt,endt)) {
		printf("Read plot failed\n");
		return("failed");
	}
	for (i = 0, t = startt ; i < npts ; i++, t+=dt)
		x[i] = t;
	endt = t - dt;
/*
** Construct a dy/dx table
** Construct a table of dot products of sequential line segments
** Run through looking for the Min and Max, identifying peaks and ISI
** Run through identifying the first order parameters and getting their 
**	 values
** Run through for the second order terms.
*/
	for (i = 0 ; i < (npts - 1) ; i++) {
		dy = y[i+1]-y[i];
		dydx[i] = dy/dt;
	}
	dydx[npts -1] = dy/dt;
/* contstructing a theta table using vectors between sequential pts,
** and then normalising */
	dtsq = dt * dt * 1e2; /* scaling so the angles are sensible */
	r = dtsq;
	dy = 0;
	for (i = 0 ; i < (npts-1) ; i++) {
		dy1 = y[i + 1] - y[i];
		r1 = dy1 * dy1 + dtsq;
		/* all the dx are = dt */
		theta[i] = acos((dtsq + dy1 * dy)/(sqrt(r * r1)));
		if (dy > dy1)
			theta[i] = -theta[i];
		r = r1;
		dy = dy1;
	}
	/* fudging the last point */
	theta[npts-1] = theta[npts-2];


	/* Finding the peaks */
	for (i = 1 ; i < (npts - 1) ; i++) {
		if (y[i] > y[i-1] && y[i] > y[i+1] && y[i] > (peakest-PEAKRANGE)){
			peak[peakno] = i;
			MaxV += y[i];
			peakest = y[i];
			peakno++;
		}
		if (y[i] < y[i-1] && y[i] < y[i+1] && y[i] < (hyperpolest+PEAKRANGE)){
			valley[valleyno] = i;
			LoV += y[i];
			hyperpolest = y[i];
			LoT += x[i] - x[peak[valleyno]];
			valleyno++;
		}
	}
	if (peakno == 0 || valleyno == 0) {
		printf("No complete spikes were found : only %d peaks and %d valleys\n",peakno,valleyno);
		return;
	}
	if (peakno != valleyno) {
		printf("An incomplete spike : %d peaks and %d valleys\n",peakno,valleyno);
		return;
	}


	for (i = 0 ; i < peakno ; i++) {
		PTP += y[peak[i]] - y[valley[i]];
		LoCurve += theta[valley[i]];
	}

	for (k = 0 ; k < peakno ; k++) {
	/* Rise Slope */
		for (i = peak[k] ; i > 0 ; i--) {
			if (dydx[i] > dydx[i - 1] && dydx[i] > dydx[i + 1]) {
				RiseSlope += dydx[i];
				RiseIndex = i;
				SpikeWidth -= x[i];
				break;
			}
		}
	/* Fall Slope */
		for (i = peak[k] ; i < valley[k] ; i++) {
			if (dydx[i] < dydx[i - 1] && dydx[i] < dydx[i + 1]) {
				FallSlope += dydx[i];
				SpikeWidth += x[i];
				break;
			}
		}
	/* Thresh Curve */
		for (i = peak[k] ; i > valley[k-1] ; i--) {
			if (theta[i] > theta[i - 1] && theta[i] > theta[i + 1]) {
				ThreshCurve += theta[i];
				ThreshT += x[peak[k]] - x[i];
				ThreshV += y[i];
				break;
			}
		}
	}

	ISI = (x[peak[peakno-1]] - x[peak[0]])/((float)(peakno - 1));
	PTP /= (float)peakno;
	SpikeWidth /= (float)peakno;
	MaxV /= (float)peakno;
	LoT /= (float)peakno;
	LoV /= (float)peakno;
	LoCurve /= (float)peakno;
	RiseSlope /= (float)peakno;
	FallSlope /= (float)peakno;
	ThreshCurve /= (float)peakno;
	ThreshT /= (float)peakno;
	ThreshV /= (float)peakno;
	
	for (k = 0 ; k < (peakno - 1) ; k++) {
	/* Rec Slope */
		for (i = valley[k] ; i < peak[k + 1] ; i++) {
			if (dydx[i] > dydx[i - 1] && dydx[i] > dydx[i + 1]) {
				RecIndex = i;
				RecSlope += dydx[i];
				RecT += x[i] - x[peak[k]];
				RecV += y[i];
				break;
			}
		}
	/* Knee Curve */
	/* If it fails, it will find the peak. */
		for (i = valley[k] ; i < peak[k+1] ; i++) {
			if (theta[i] < theta[i - 1] && theta[i] < theta[i + 1]) {
				KneeIndex = i;
				KneeCurve += theta[i];
				KneeT += x[i] - x[peak[k]];
				KneeV += y[i];
				break;
			}
		}
	/* Infl Slope */
	/* If the knee failed, so will this, and all values will be zero */
		temp = 10e10;
		for (i = KneeIndex ; y[i] < ThreshV ; i++) {
			if (temp > dydx[i]) {
				temp = dydx[i];
				tempindex = i;
			}
		}
		InflSlope += dydx[tempindex];
		InflT += x[tempindex] - x[peak[k]];
		InflV += y[tempindex];
	}

	if (peakno > 1) {
		RecSlope /= (float)(peakno - 1);
		RecT /= (float)(peakno - 1);
		RecV /= (float)(peakno - 1);
		KneeCurve /= (float)(peakno - 1);
		KneeT /= (float)(peakno - 1);
		KneeV /= (float)(peakno - 1);
		InflSlope /= (float)(peakno - 1);
		InflT /= (float)(peakno - 1);
		InflV /= (float)(peakno - 1);
	}

	if (outputfile && strlen(outputfile) > 0) {
		spikeparmoutput(allvars,varnames,nvars,outputfile,filemode,notes);
	}

	printf("%d peaks, %d valleys, over %d pts and from %f to %f sec\n",
		peakno,valleyno,npts,startt,endt);
	if (notes)
		printf("\n%s\n",notes);
	for (i = 0 ; i < nvars ; i++) {
		printf("%s	%g	",varnames[i],*(allvars[i]));
		if ((i%3) == 2)
			printf("\n");
	}	
	printf("\n");

	return("didit");
}

spikeparmoutput(allvars,varnames,nvars,filename,filemode,notes)
	float **allvars;
	char **varnames;
	int	nvars;
	char	*filename;
	char	*filemode;
	char	*notes;
{
	int i;
	static char	parmfile[200];
	FILE	*fp,*fopen();

	if (strcmp(filemode,"multifile") == 0) {
		for (i = 0 ; i < nvars ; i++) {
			sprintf(parmfile,"%s.%s",filename,varnames[i]);
			fp = fopen(parmfile,"a");
			if (notes)
				fprintf(fp,"%s	%g\n",notes,*(allvars[i]));
			else
				fprintf(fp,"%g\n",*(allvars[i]));
			fclose(fp);
		}
	}
	if (strcmp(filemode,"onefile") == 0) {
		fp = fopen(filename,"w");
		if (notes)
			fprintf(fp,"\n%s\n",notes);
		for (i = 0 ; i < nvars ; i++) {
			fprintf(fp,"%s	%g	",varnames[i],*(allvars[i]));
			if ((i%3) == 2)
				fprintf(fp,"\n");
		}	
		fprintf(fp,"\n");
		fclose(fp);
	}
	if (strcmp(filemode,"onefileappend") == 0) {
		fp = fopen(filename,"a");
		if (notes)
			fprintf(fp,"\n%s\n",notes);
		for (i = 0 ; i < nvars ; i++) {
			fprintf(fp,"%s	%g	",varnames[i],*(allvars[i]));
			if ((i%3) == 2)
				fprintf(fp,"\n");
		}	
		fprintf(fp,"\n");
		fclose(fp);
	}
}
