#include <stdio.h>
#include "sim_ext.h"
#include "tools.h"
#include "seg_struct.h"
#include "hh_struct.h"
#include "olf_struct.h"
#include "buf_struct.h"

#define NAMELEN 40
#define NEW_CELL 0x01
#define RELATIVE 0x02
#define POLAR 0x04
#define LAMBDA_WARNING 0x08

#define DATA_MODE 1
#define FLAG_MODE 2
#define COMMENT_MODE 3
#define SCRIPT_MODE 4
#define ERR -1
#define INPUT 0
#define VOLTAGE 0

/*
** These fields are set from the cell definition file
*/
static float RM,CM,RA,EREST_ACT;
static char	*chomp_leading_spaces();

Element *add_compartment(flags,name,link,len,dia)
	int		flags;
	char	*name;
	char	*link;
	float	len;
	float	dia;
{
	int argc;
	char	*argv[10];
	struct compartment_type	*compt;

	if (flags & NEW_CELL) {
		argv[0] = "c_do_copy";
		argv[1] = "/library/compartment";
		argv[2] = name;
		do_copy(3,argv);
		if (strcmp(link,"none") != 0) {
			argv[0] = "c_do_add_msg";
			argv[1] = link;
			argv[2] = name;
			argv[3] = "AXIAL";
			argv[4] = "Vm";
			do_add_msg(5,argv);
	
			argv[1] = name;
			argv[2] = link;
			argv[3] = "RAXIAL";
			argv[4] = "Ra";
			argv[5] = "Vm";
			do_add_msg(6,argv);
		}
	}

	compt = (struct compartment_type *)(GetElement(name));
	if (!compt) {
		fprintf(stderr,"could not find compartment '%s'\n",name);
		return(NULL);
	}
	compt->Cm = CM * len * dia * PI;
	compt->Ra = RA * len /(dia * dia * PI / 4.0);
	compt->Rm = RM / (len * dia * PI);
	compt->Em = EREST_ACT;
	return((Element *)compt);
}


Element *add_channel(name,parent)
	char	*name;
	char	*parent;
{
	int argc;
	char	*argv[10];
	Element	*elm;
	MsgIn	*msg;
	struct ch	*compt;
	static char	source[NAMELEN];
	static char	dest[NAMELEN];
	char	*oname;

	sprintf (source,"/library/%s",name);
	sprintf (dest,"%s/%s",parent,name);
	argv[0] = "c_do_copy";
	argv[1] = source;
	argv[2] = dest;
	do_copy(3,argv);

	if (!(elm = GetElement(dest)))
		return(NULL);

	oname = elm->object->name;

	argv[0] = "c_do_add_msg";
	if (strcmp(oname,"hh_channel") == 0 ||
		strcmp(oname,"channelC2") == 0 ||
		strcmp(oname,"channelA") == 0 ||
		strcmp(oname,"channelB") == 0 ||
		strcmp(oname,"channelC") == 0 ||
		strcmp(oname,"receptor2") == 0) {
		argv[1] = dest;
		argv[2] = parent;
		argv[3] = "CHANNEL";
		argv[4] = "Gk";
		argv[5] = "Ek";
		do_add_msg(6,argv);

		argv[1] = parent;
		argv[2] = dest;
		argv[3] = "VOLTAGE";
		argv[4] = "Vm";
		do_add_msg(5,argv);
		if (strcmp(oname,"vdep_channel") == 0) {
		/*
		** Looking for children of vdep_ch and sending voltage to them
		** if they are gates
		*/
			for (elm = elm->child; elm ; elm = elm->next) {
				if (strcmp(elm->object->name,"vdep_gate") == 0 ||
					strcmp(elm->object->name,"tabgate") == 0) {
					argv[1] = parent;
					argv[2] = elm->name;
					argv[3] = "VOLTAGE";
					argv[4] = "Vm";
					do_add_msg(5,argv);
				}
			}
		}
	} else if (strcmp(oname,"vdep_channel") == 0) {
		argv[1] = dest;
		argv[2] = parent;
		argv[3] = "CHANNEL";
		argv[4] = "Gk";
		argv[5] = "Ek";
		do_add_msg(6,argv);
		/*
		** Looking for children of vdep_ch and sending voltage to them
		** if they do not have other messages in those slots.
		*/
		for (elm = elm->child; elm ; elm = elm->next) {
			if (strcmp(elm->object->name,"vdep_gate") == 0 ||
				strcmp(elm->object->name,"tabgate") == 0) {
				for (msg = elm->msg_in ; msg ; msg = msg->next) {
					if (msg->type == VOLTAGE)
						break;
				}
				if (!msg) { /* No voltage message was found */
					argv[1] = parent;
					argv[2] = elm->name;
					argv[3] = "VOLTAGE";
					argv[4] = "Vm";
					do_add_msg(5,argv);
				}
			}
			if (strcmp(elm->object->name,"table") == 0) {
				for (msg = elm->msg_in ; msg ; msg = msg->next) {
					if (msg->type == INPUT)
						break;
				}
				if (!msg) { /* No INPUT message was found */
					argv[1] = parent;
					argv[2] = elm->name;
					argv[3] = "INPUT";
					argv[4] = "Vm";
					do_add_msg(5,argv);
				}
			}
		}
	} else if (strcmp(oname,"graded") == 0 ||
		strcmp(oname,"spike") == 0) {
		argv[1] = parent;
		argv[2] = dest;
		argv[3] = "INPUT";
		argv[4] = "Vm";
		do_add_msg(5,argv);
	}
	return(elm);
}


parse_compartment(flags,name,parent,x,y,z,d,nargs,ch,dens)
	int	flags;
	char	*name;
	char	*parent;
	float	x,y,z,d;
	int		nargs;
	char	ch[6][20];
	float	*dens;
{
	float nlambda;
	char	*ch_name;
	char	*ch_type;
	float	tx,ty,tz;
	int 	i,j,k;
	float	len;
	Element	*elm, *compt, *parent_compt;

	if (strcmp(parent,"none") == 0) {
		len = sqrt(x * x + y * y + z * z);
		x = y = z = 0.0;
	} else {
		parent_compt = GetElement(parent);
		if (!parent_compt) {
			fprintf(stderr,"could not find parent compt %s\n",parent);
			return;
		}
		if (flags & RELATIVE) {
			len = sqrt(x * x + y * y + z * z);
			x = x + parent_compt->x;
			y = y + parent_compt->y;
			z = z + parent_compt->z;
		} else {
			tx = x - parent_compt->x;
			ty = y - parent_compt->y;
			tz = z - parent_compt->z;
			len = sqrt(tx * tx + ty * ty + tz * tz);
		}
	}
	if (!(compt = add_compartment(flags,name,parent,len,d)))
		return;
	compt->x = x;
	compt->y = y;
	compt->z = z;

	if (flags & LAMBDA_WARNING) {
		nlambda = len / sqrt(RM * d * 0.25 / RA);
		if (nlambda < 0.05)
			printf("WARNING : compartment '%s' is only %f lambdas\n",
				name,nlambda);
		else if (nlambda > 0.5)
			printf("WARNING : compartment '%s' too long: %f lambdas\n",
				name,nlambda);
	}
	for (j = 7,k=0 ; j < nargs ; j += 2,k++) {
		ch_name = ch[k];
		if (!(elm = add_channel(ch_name,name)))
			continue;
		if (strcmp(elm->object->name,"hh_channel") == 0)
			((struct hh_channel_type *)elm)->Gbar =
				dens[k] * len * d * PI;
		else if (strcmp(elm->object->name,"vdep_channel") == 0)
			((struct vdep_channel_type *)elm)->gbar =
				dens[k] * len * d * PI;
		else if (strcmp(elm->object->name,"channelC2") == 0)
			((struct channelC2_type *)elm)->gmax =
				dens[k] * len * d * PI;
		else if (strcmp(elm->object->name,"receptor2") == 0)
			((struct olf_receptor2_type *)elm)->gmax =
				dens[k] * len * d * PI;
		else if (strcmp(elm->object->name,"channelC") == 0)
			((struct channelC_type *)elm)->gmax =
				dens[k] * len * d * PI;
		/* Shortcut for specifying spike thresholds */
		else if (strcmp(elm->object->name,"spike") == 0)
			((struct spike_type *)elm)->thresh = dens[k];
	}
}

do_read_cell(argc,argv)
	int argc;
	char	**argv;
{
	int		i,j;
	int		len;
	char	rawline[201];
	char	*line;
	FILE	*fp,*fopen();
	Element	*elm;
	int		flags = 0;
	int		parse_mode = DATA_MODE;
	float	get_script_float();
	char	*endit;
	char	*argvar[5];

	if (argc < 3) {
		fprintf(stderr,"usage : %s filename cellname\n",argv[0]);
		return;
	}
	if (!(fp = fopen(argv[1],"r"))) {
		fprintf(stderr,"can't open file '%s'\n",argv[1]);
		return;
	}

	fprintf(stderr,"reading '%s'.... \n",argv[1]);
	if ((elm = GetElement(argv[2])) == NULL) {
		flags |= NEW_CELL;
		argvar[0] = "c_do_create";
		argvar[1] = "neutral";
		argvar[2] = argv[2];
		do_create(3,argvar);
	} else {
		flags &= ~NEW_CELL;
	}

	/* getting values from script */
	CM = get_script_float("CM");
	RM = get_script_float("RM");
	RA = get_script_float("RA");
	EREST_ACT = get_script_float("EREST_ACT");

	ChangeWorkingElement(argv[2]);

	for (i = 1, endit = fgets(rawline,200,fp); endit ; endit = fgets(rawline,200,fp), i++ ) {
		line = chomp_leading_spaces(rawline);

		switch (*line) {
			case '\n' :
			case '\r' :
			case '\0' : continue;
			case '/' :
				if (line[1] == '/')
					continue;
				if (line[1] == '*')
					parse_mode = COMMENT_MODE;
				else
					parse_mode = DATA_MODE;
				break;
			case '*' :
				if (line[1] == '/') {
					parse_mode = DATA_MODE;
					continue;
				}
				parse_mode = SCRIPT_MODE;
				break;
			default :
				break;
		}

		switch(parse_mode) {
			case DATA_MODE :
				if (read_data(line,i,flags) == ERR) {
				}
				break;
			/*
			case FLAG_MODE :
				if (read_flag(line,i,flags) == ERR) {
				}
				break;
			*/
			case SCRIPT_MODE :
				if (read_script(line,i,&flags) == ERR) {
				}
				parse_mode = DATA_MODE;
				break;
			case COMMENT_MODE :
				len = strlen(line);
				if (line[len-2] == '*' && line[len-1] == '/')
					parse_mode = DATA_MODE;
				break;
			default :
				break;
		}
	}
	fprintf(stderr,"%s read\n",argv[1]);
	fclose(fp);
}


read_data(line,lineno,flags)
	char	*line;
	int		lineno;
	int		flags;
{
	float	x,y,z,l,d;
	float	r,theta,phi;
	static	char	ch[6][20];
	float	dens[6];
	char	name[NAMELEN],lastname[NAMELEN],parent[NAMELEN];
	int		nargs;

	nargs = sscanf(line,
		"%s%s%f%f%f%f%s%f%s%f%s%f%s%f%s%f",
		name,parent,&x,&y,&z,&d,
		ch[0],&dens[0],ch[1],&dens[1],ch[2],&dens[2],
		ch[3],&dens[3],ch[4],&dens[4]);
	
	if (nargs < 6) {
	/* Not enough inputs matched. */
		fprintf(stderr,"error on line %d :\n",
			lineno);
		fprintf(stderr,"	%s\nFields incorrect\n",line);
		return(ERR);
	}
	/* A shortcut for branches : use '.' instead of full name if
	** the parent is the previous element */
	if (strcmp(parent,".") == 0)
		strcpy(parent,lastname);
	strcpy(lastname,name);
/*
** Cartesian coord mode
**	name parent		x		y		z		dia 	ch dens...
**
** polar coord mode
**	name parent		r		theta	phi		dia		ch dens...
**
*/
	if (flags & POLAR) {
		r = x * 1.0e-6;
		theta = y * PI/180.0;
		phi = z * PI/180.0;
		x = r * sin(phi) * cos(theta);
		y = r * sin(phi) * sin(theta);
		z = r * cos(phi);
	} else {
		x *= 1.0e-6;
		y *= 1.0e-6;
		z *= 1.0e-6;
	}
	d *= 1.0e-6;
	parse_compartment(flags,name,parent,x,y,z,d,nargs,ch,dens);
	return(1);
}

read_script(line,lineno,flags)
	char	*line;
	int		lineno;
	int		*flags;
{
	char	command[NAMELEN];
	char	field[NAMELEN];
	float	value;
	int		nargs;

	nargs = sscanf(line,"%s %s %f",command,field,&value);
	if (nargs == 3 && strcmp(command,"*set_global") == 0) {
		/* setting a global */
		set_script_float(field,value);
		if (strcmp(field,"RM") == 0)
			RM = value;
		if (strcmp(field,"RA") == 0)
			RA = value;
		if (strcmp(field,"CM") == 0)
			CM = value;
		if (strcmp(field,"EREST_ACT") == 0)
			EREST_ACT = value;
	} else if (nargs == 1) /* just setting flags */ {
		if (strcmp(command,"*relative") == 0)
			*flags |= RELATIVE;
		else if (strcmp(command,"*absolute") == 0)
			*flags &= ~RELATIVE;
		else if (strcmp(command,"*polar") == 0)
			*flags |= POLAR;
		else if (strcmp(command,"*cartesian") == 0)
			*flags &= ~POLAR;
		else if (strcmp(command,"*lambda_warn") == 0)
			*flags |= LAMBDA_WARNING;
		else if (strcmp(command,"*lambda_nowarn") == 0)
			*flags &= ~LAMBDA_WARNING;
	}
}

static char *chomp_leading_spaces(line)
	char	*line;
{
	char	*str;

	for(str = line; (*str == ' ' || *str == '	') ; str++);

	return(str);
}

do_write_cell(argc,argv)
	int argc;
	char	**argv;
{
	FILE	*fp,*fopen();
	Element	*neuron;
	Element	*elm,*lastelm;
	Element	*chan;
	Element	*parent,*find_parent_dend();
	ElementList	el;
	Element *elms[MAX_EL_SIZE];
	float	calc_dia();
	float	calc_len();
	float	cond,calc_cond();
	float	get_script_float();
	int		i;
	float	x,y,z,r,theta,phi;
	int		relative_flag = 1, cartesian_flag = 1;
	char	*author;
	long	clock;

	el.nelements = 0;
	el.element = elms;

	if (argc < 3) {
		fprintf(stderr,"usage : %s filename cellname [cartesian/polar] [relative/absolute] [-author author]\n",argv[0]);
		fprintf(stderr,"Default is cartesian relative\n");
		return;
	}
	author = NULL;
	for (i = 3 ; i < argc ; i++) {
		if (strcmp(argv[i],"polar") == 0)
			cartesian_flag = 0;
		if (strcmp(argv[i],"absolute") == 0)
			relative_flag = 0;
		if (strcmp(argv[i],"-author") == 0) {
			i++;
			author = argv[i];
		}
	}

	if (!(fp = fopen(argv[1],"w"))) {
		fprintf(stderr,"can't open file '%s' for writing\n",argv[1]);
		return;
	}

	if ((neuron = GetElement(argv[2])) == NULL) {
		fprintf(stderr,"Can't find cell '%s' for writing\n",argv[2]);
		return;
	}

	/*
	** Finding all dendrites, but not their channels
	*/
	find_all_dends(neuron,0,&el);

	elm = elms[0];
	if (!elm || el.nelements == 0) {
		fprintf(stderr,"Incorrect neuron specification\n");
		return;
	}

	fprintf(stderr,"writing '%s'.... \n",argv[1]);
	/*
	** Writing out the header info
	*/
	fprintf(fp,"//	PARAMETER FILE FOR NEURON '%s'\n",neuron->name);
	if (author)
		fprintf(fp,"//	Author : %s \n",author);
	clock = time(0);
	fprintf(fp,"//	%s\n",ctime(&clock));
	fprintf(fp,"\n//	Format of file :\n");
	fprintf(fp,"// x,y,z,dia are in microns, all other units are SI (Meter Kilogram Second Amp)\n");
	fprintf(fp,"// In polar mode 'r' is in microns, theta and phi in degrees \n");
	fprintf(fp,"// Control line options start with a '*'\n");
	fprintf(fp,"// The format for each compartment parameter line is :\n");

	fprintf(fp,"//name	parent	r	theta	phi	d	ch	dens ...\n");
	fprintf(fp,"//in polar mode, and in cartesian mode :\n");
	fprintf(fp,"//name	parent	x	y	z	d	ch	dens ...\n\n\n");
	/*
	** Specifying coordinate modes
	*/
	fprintf(fp,"//		Coordinate mode\n");
	if (cartesian_flag)
		fprintf(fp,"*cartesian\n");
	else
		fprintf(fp,"*polar\n");
	if (relative_flag)
		fprintf(fp,"*relative\n");
	else
		fprintf(fp,"*absolute\n");

	/*
	** Writing out the constants
	*/
	fprintf(fp,"\n//		Specifying constants\n");
	CM = get_script_float("CM");
	RM = get_script_float("RM");
	RA = get_script_float("RA");
	fprintf(fp,"*set_global	RM	%g\n",RM);
	fprintf(fp,"*set_global	RA	%g\n",RA);
	fprintf(fp,"*set_global	CM	%g\n\n",CM);

	/*
	** Writing out soma
	*/
	fprintf(fp,"%s	none	%g	0	0	%g",elm->name,
		1e6 * calc_len(elm,RM,RA),1e6 * calc_dia(elm,RM,RA));
	for (chan = elm->child ; chan ; chan = chan->next) {
		if ((cond = calc_cond(elm,chan,RM,RA)) > 0) {
			fprintf(fp,"	%s	%g",chan->name,cond);
		}
	}
	fprintf(fp,"\n\n");

	/*
	** loop for writing out each element
	*/
	for (i = 1 ; i < el.nelements ; i++) {
	/* Starting up */
		elm = elms[i];
		parent = find_parent_dend(elm);

	/* Printing name of elm */
		if (elm->index == 0)
			fprintf(fp,"%s	",elm->name);
		else
			fprintf(fp,"%s%d	",elm->name,elm->index);

	/* Printing parent of elm */
		if (parent == lastelm)
			fprintf(fp,".	");
		else if (parent->index == 0)
			fprintf(fp,"%s	",parent->name);
		else
			fprintf(fp,"%s%d	",parent->name,parent->index);
		lastelm = elm;

	/* printing coords of elm */
		if (relative_flag) {
			x = elm->x - parent->x;
			y = elm->y - parent->y;
			z = elm->z - parent->z;
		} else {		/* absolute coordinates */
			x = elm->x;
			y = elm->y;
			z = elm->z;
		}
		x *= 1e6;
		y *= 1e6;
		z *= 1e6;

		if (cartesian_flag) {
			fprintf(fp,"%g	%g	%g	",x,y,z);
		} else { 		/* polar coords : r,theta,phi */
			r = sqrt(x * x + y * y);
			theta = acos(x/r);
			if (y < 0)
				theta = 2 * PI - theta;
			theta *= 180.0/PI;
			
			r = sqrt(x * x + y * y + z * z);

			phi = (acos(z/r)) * 180.0 / PI;
			fprintf(fp,"%g	%g	%g	",r,theta,phi);
		}

	/* printing diameter of elm */
		fprintf(fp,"%g",1e6 * calc_dia(elm,RM,RA));

	/* Loop for printing the channels and their densities */
		for (chan = elm->child ; chan ; chan = chan->next) {
			if ((cond = calc_cond(elm,chan,RM,RA)) > 0) {
				fprintf(fp,"	%s	%g",chan->name,cond);
			}
		}

	/* Finish off the line. Extra line every five, for readability */
		fprintf(fp,"\n");
		if ((i%5) == 0)
		fprintf(fp,"\n");
	}
	fclose(fp);
	fprintf(stderr,"Writing done\n");
}
#undef INPUT
#undef VOLTAGE
