/*
 * hash_fixed.c,v
 * Revision 2.0  1992/04/23  02:49:33  ware
 * First public release.
 *
 * Revision 1.3  1992/02/16  23:52:40  ware
 * Added prototypes for malloc() et al. so it compiles cleanly.
 *
 * Revision 1.2  1991/05/11  18:06:21  pete
 * Increment pointer for simple hash function.
 *
 * Revision 1.1  1991/05/01  13:30:55  pete
 * Initial revision
 *
 */

#include	<stdio.h>
#include	"table.h"
#include	"hash_fixed.h"

#ifdef _USE_PROTO_
static int	hash_fixed_add (char *private, char *datum);
static char	*hash_fixed_find (char *private, char *datum);
static int	hash_fixed_delete (char *private, char *datum, int (*user_free)());
static int	hash_fixed_first (char *private);
static char	*hash_fixed_next (char *private);
static int	hash_fixed_release (char *private);

/*
 * Provide some default functions
 */
static int	hash_fixed_hash (char *datum);
extern int	strcmp (char *s1, char *s2);
extern char	*malloc (int len);
extern void	free (char *p);
extern void	bzero (char *p, int len);
#else /* !_USE_PROTO_ */
static int	hash_fixed_add ();
static char	*hash_fixed_find ();
static int	hash_fixed_delete ();
static int	hash_fixed_first ();
static char	*hash_fixed_next ();
static int	hash_fixed_release ();

/*
 * Provide some default functions
 */
static int	hash_fixed_hash ();
extern int	strcmp ();
extern char	*malloc ();
extern void	free ();
extern void	bzero ();
#endif

#define HASH_DEFAULT_SIZE	513
#define HASH_DEFAULT_COMPARE	strcmp
#define HASH_DEFAULT_FUNC	hash_fixed_hash
#define MAX(a,b)	((a) > (b) ? (a) : (b))

/*
 * A few useful macros.  HASHCALC returns the index for a datum.
 */

#define HASHCALC(ht,d)	((*ht->h_hash) (d) % ht->h_size);

/*
 * Define the hash structure.  We use a simple fixed size hash table
 * with chaining.  The table is an array of pointers to bucket structures.
 * Set t_private to the allocated array.
 */

typedef struct _hash_
{
	int		h_size;		/* size of the table */
	int		h_index;	/* current index for first/next */
	struct _bucket_	*h_bucket;	/* last bucket used */
	int		(*h_compare)(); /* function to compare two elements */
	int		(*h_hash)();	/* return hash value */
	struct _bucket_	**h_table;	/* array of pointers to buckets*/

#ifdef HASHSTATS

#define COLLISIONS(ht) (++ht->h_collisions)
#define CHAINLENGTH(var) (++var)
#define UPDATE_CHAINLENGTH(ht,var) hashtable->h_longest_chain = MAX (hashtable->h_longest_chain, var)
#define MOREBUCKETS(ht) (++ht->h_num_buckets)
#define MORECHAINS(ht)	(++ht->h_chains_used)
#define TEST_DEC_CHAINS(ht,index) if (ht->h_table[index]);else --ht->h_chains_used
#define FEWERBUCKETS(ht) (--ht->h_num_buckets)

	/*
	 * Various statistics
	 */

	int		h_collisions;	/* number of collisions */
	int		h_longest_chain; /* the longest chain */
	int		h_chains_used;	/* number of chains allocated */
	int		h_num_buckets;	/* how many buckets */

#else /* !HASHSTATS */
#define COLLISIONS(ht)
#define CHAINLENGTH(var)
#define UPDATE_CHAINLENGTH(ht,var)
#define MOREBUCKETS(ht)
#define MORECHAINS(ht)
#define TEST_DEC_CHAINS(ht,index)
#define FEWERBUCKETS(ht)

#endif
} Hash;

typedef struct _bucket_
{
	char		*b_datum;	/* the hash data */
	struct _bucket_	*b_next;	/* next item on chain */
} Bucket;

Table *
table_hash_fixed_create (size, user_free, compare, hash_func)
int		size;			/* initial size for hash table */
int		(*user_free)();		/* function for freeing an entry */
int		(*compare)();		/* function for comparing entries */
int		(*hash_func) ();	/* return hash value for entry */
{
	Table		*table;		/* the returned table */
	Hash		*hashtable;	/* the actual hash table used */

	if (!size || size < 0)
	{
		size = HASH_DEFAULT_SIZE;
	}
	if (!compare)
	{
		compare = HASH_DEFAULT_COMPARE;
	}
	if (!hash_func)
	{
		hash_func = HASH_DEFAULT_FUNC;
	}

	table = table_init ();
	if (!table)
	{
		return (Table *) NULL;
	}

	/*
	 * Allocate the private data (i.e. the hash table)
	 */

	table->t_private = malloc (sizeof (Hash));
	hashtable = (Hash *) table->t_private;
	if (!hashtable)
	{
		free ((char *) table);
		return NULL;
	}
	hashtable->h_size = size;
	hashtable->h_index = 0;
	hashtable->h_bucket = NULL;
	hashtable->h_compare = compare;
	hashtable->h_hash = hash_func;
	hashtable->h_table =
		(Bucket **) malloc (sizeof (Bucket *) * hashtable->h_size);
	bzero ((char *) hashtable->h_table,
	       sizeof (Bucket *) * hashtable->h_size);

#ifdef HASHSTATS
	hashtable->h_collisions = 0;
	hashtable->h_longest_chain = 0;
	hashtable->h_chains_used = 0;
	hashtable->h_num_buckets = 0;
#endif

	/*
	 * Initialize the function pointers
	 */

	table->t_add = hash_fixed_add;
	table->t_find = hash_fixed_find;
	table->t_delete = hash_fixed_delete;
	table->t_user_free = user_free;
	table->t_first = hash_fixed_first;
	table->t_next = hash_fixed_next;
	table->t_release = hash_fixed_release;

	return table;
}

void
table_hash_fixed_stats (table)
Table		*table;
{
	Hash	*hashtable;

	if (!table)
		return;
	hashtable = (Hash *) table->t_private;
	if (!hashtable)
		return;
#ifdef HASHSTATS
	printf ("collisions	= %d\n", hashtable->h_collisions);
	printf ("longest chain	= %d\n", hashtable->h_longest_chain);
	printf ("chains used	= %d\n", hashtable->h_chains_used);
	printf ("num chains	= %d\n", hashtable->h_size);
	printf ("num buckets	= %d\n", hashtable->h_num_buckets);
	printf ("avg chain	= %0.2f\n", hashtable->h_num_buckets / (float) hashtable->h_chains_used);
	printf ("norm. collisions= %0.2f\n", hashtable->h_collisions / (float) hashtable->h_num_buckets);
#endif
}

static int
hash_fixed_add (private, datum)
char		*private;		/* private data for hash table */
register char	*datum;			/* data to be inserted */
{
	Hash		*hashtable;	/* the hash table */
	int		index;		/* where in hash table it should go */
	register Bucket	*b;		/* current bucket to insert datum */
	register Bucket	*prev;		/* previous element in chain to b */
	int		chain_length;	/* who many items on this chain */

	hashtable = (Hash *) private;
	index = HASHCALC (hashtable, datum);

	/*
	 * Find the end of the chain but keep prev pointing at the previous
	 * entry so we can update it's b_next field.
	 */

	chain_length = 0;
	prev = NULL;
	for (b = hashtable->h_table[index]; b; b = b->b_next)
	{
		prev = b;
		/*
		 * Not allowed to add duplicates
		 */
		if ((*hashtable->h_compare) (datum, b->b_datum) == 0)
			return ERR;
		COLLISIONS (hashtable);
		CHAINLENGTH (chain_length);
	}

	UPDATE_CHAINLENGTH (hashtable, chain_length);

	/*
	 * Allocate a bucket and set it's b_next to NULL
	 */
	b = (Bucket *) malloc (sizeof (Bucket));
	if (!b)
	{
		return ERR;
	}
	b->b_datum = datum;
	b->b_next = NULL;
	MOREBUCKETS (hashtable);

	/*
	 * Link b to the previous item in the chain.  If prev == NULL it
	 * implies this is the first time on this chain
	 */

	if (!prev)
	{
		hashtable->h_table[index] = b;
		MORECHAINS (hashtable);
	}
	else
	{
		prev->b_next = b;
	}
	return SUCCESS;
}

static char *
hash_fixed_find (private, datum)
char		*private;		/* private data for hash table */
char		*datum;			/* data to find */
{
	Hash		*hashtable;	/* the hash table */
	int		index;		/* where in hash table to find chain */
	register Bucket	*b;		/* current bucket  */
	char		*d;		/* the data */

	hashtable = (Hash *) private;
	index = HASHCALC (hashtable, datum);

	d = NULL;
	for (b = hashtable->h_table[index]; b; b = b->b_next)
	{
		if ((*hashtable->h_compare) (datum, b->b_datum) == 0)
		{
			d = b->b_datum;
			break;
		}
	}
	return d;
}

static int
hash_fixed_delete (private, datum, user_free)
char		*private;		/* private data for hash table */
char		*datum;			/* data to find */
int		(*user_free) ();	/* function to free user's data */
{
	Hash		*hashtable;	/* the hash table */
	int		index;		/* where in hash table to find chain */
	Bucket		*b;		/* current bucket  */
	Bucket		*prev;		/* the previous bucket */

	hashtable = (Hash *) private;
	index = HASHCALC (hashtable, datum);

	prev = NULL;
	for (b = hashtable->h_table[index]; b; b = b->b_next)
	{
		if ((*hashtable->h_compare) (datum, b->b_datum) == 0)
			break;
		else
			prev = b;
	}

	/*
	 * Not found
	 */

	if (!b)
		return ERR;
	/*
	 * Unlink b from the chain
	 */

	if (!prev)
	{
		hashtable->h_table[index] = b->b_next;
		TEST_DEC_CHAINS (hashtable, index);
	}
	else
	{
		prev->b_next = b->b_next;
	}

	/*
	 * Now call the user's function to free the data
	 */

	if (user_free)
		(*user_free) (b->b_datum);
	free ((char *) b);
	FEWERBUCKETS (hashtable);
	return SUCCESS;
}

static int
hash_fixed_first (private)
char		*private;		/* private data for hash table */
{
	Hash		*hashtable;	/* the actual hash table */

	if (!private)
	{
		return ERR;
	}

	hashtable = (Hash *) private;
	hashtable->h_index = 0;
	hashtable->h_bucket = NULL;
	return SUCCESS;
}

static char *
hash_fixed_next (private)
char		*private;		/* private data for hash table */
{
	Hash		*hashtable;	/* the actual hash table */
	Bucket		*b;		/* current bucket */

	if (!private)
	{
		return (char *) NULL;
	}
	hashtable = (Hash *) private;

	if (hashtable->h_bucket)
	{
		b = hashtable->h_bucket->b_next;
	}
	else
	{
		b = NULL;
	}

	while (!b && hashtable->h_index < hashtable->h_size)
	{
		b = hashtable->h_table[hashtable->h_index++];
	}
	if (!b)
	{
		return (char *) NULL;
	}
	else
	{
		hashtable->h_bucket = b;
		return b->b_datum;
	}
}

static int
hash_fixed_release (private)
char		*private;		/* private data for hash table */
{
	Hash		*hashtable;	/* the actual hashtable */

	if (!private)
		return ERR;
	hashtable = (Hash *) private;
	if (hashtable->h_table)
	{
		free ((char *) hashtable->h_table);
		hashtable->h_table = NULL;
		hashtable->h_size = 0;
	}
	free ((char *) hashtable);
	return SUCCESS;
}

static int
hash_fixed_hash (datum)
char		*datum;			/* datum to return a hash value for */
{
	int	value = 0;
	int	i;

	if (!datum)
		return 0;
	for (i = 0; *datum; i++)
	{
		value ^= (*datum++ << (i % sizeof (int)));
	}
	return value;
}
