#include <stdio.h>
#include "build.h"
#include "output.h"
#include "str_util.h"

#define VARINSTANCE	0
#define VARCLASS	1
#define VARCONSTRAINT	2

XoProto (static void, output_instance_structure, (FILE *fptr, class_t *class));
XoProto (static void, output_tex_record_part, (FILE *fptr, class_t *class, char **vars, int type));
XoProto (static void, output_tex_field, (FILE *fptr, field_t *field));
XoProto (static void, output_resource_tex_sum, (FILE *fptr, class_t *class));
XoProto (static void, output_resource_tex_desc, (FILE *fptr, class_t *class));
XoProto (static void, output_constraint_tex_sum, (FILE *fptr, class_t *class));
XoProto (static void, output_constraint_tex_desc, (FILE *fptr, class_t *class));
XoProto (static void, output_actions_tex, (FILE *fptr, class_t *class));
XoProto (static void, output_translation_tex, (FILE *fptr, class_t *class));
XoProto (static void, output_tex_quoted, (FILE *fptr, char *str));
XoProto (static char *, output_tex_quoted_str, (char *str, char *buffer));
XoProto (static int, has_constraints, (class_t *class));

/*
 * Write out the class resource documentation file, res_<class>.tex.
 */

int
output_doc (options, class)
options_t	*options;
class_t		*class;			/* the class to describe */
{
	FILE		*fptr;		/* where to write */
	char		old[2048];	/* name of the backup file */
	char		*purpose;	/* description of the widget */
	char		filename[2048];
	char		buffer[BUFSIZ];
	char		name[128];	/* name of the class variable */
	int		diff = FALSE;

	if (!options)
	{
		fprintf (stderr, "Options are NULL\n");
		return (diff);
	}
	if (!options->opt_dodoc)
	{
		return (diff);
	}
	if (!class)
	{
		fprintf (stderr, "Class is NULL\n");
		return (diff);
	}
	if (!class->cl_name || !*class->cl_name)
	{
		fprintf (stderr, "Class has no name\n");
		return (diff);
	}
	sprintf (filename, "%s/res_%s.tex", options->opt_docdir, class->cl_name);
	fptr = output_open (filename, old);
	if (!fptr) 
	{
		return (diff);
	}

	/*
	 * 1. Write out the initial filler.
	 */
	fprintf (fptr, "%%\n%% ");
	fprintf (fptr, "Only included if it is the original definition.\n");
	fprintf (fptr, "%%\n");
	fprintf (fptr, "\\iforig\n");
#define REF_PREFIX	"def:"
	fprintf (fptr, "\\label{%s%s}\n", REF_PREFIX, class->cl_name);
	if ((purpose = var_lookup (class->cl_vars, "purpose")) != NULL)
	{
		fprintf (fptr, "%s\n\n", purpose);
	}

	/*
	 * 2. Write out how the include file and class variable are
	 *    declared.
	 */
	fprintf (fptr, "The following include file is needed.\n");
	fprintf (fptr, "The class variable is declared in this file and\n\
should be used in the call to \\code{XtCreateWidget()} or\n\
\\code{XtCreateManagedWidget()}.\n\n");
	fprintf (fptr, "\\begin{verbatim}\n");
	output_cpp_include (fptr, class, class->cl_name, "", FALSE);
	output_var (class, "WidgetClass", name);
	fprintf (fptr, "extern WidgetClass %s;\n", name);
	fprintf (fptr, "\\end{verbatim}\n");
	fprintf (fptr, "\\fi\n");	/* finish from \\iforig in 1. */

	/*
	 * 3. Write out the summary table of resources for this class
	 */

	fprintf (fptr, "\\begin{resources}{%s}\n", class->cl_name);
	fprintf (fptr, "\
%% Name		& Class		& Type		& Default\n");
	output_resource_tex_sum (fptr, class);
	fprintf (fptr, "\\end{resources}\n");

	/*
	 * 4. Write out the full description for each of the resources.
	 */
	  
	fprintf (fptr, "%%\n%% ");
	fprintf (fptr, "Only included if it is the original definition.\n");
	fprintf (fptr, "%%\n");
	fprintf (fptr, "\\iforig\n\\begin{resourcedescription}\n");
	output_resource_tex_desc (fptr, class);
	fprintf (fptr, "\\end{resourcedescription}\n\\fi\n");

	/*
	 * 5. Write similar information, summary and description, for
	 *    each of the constraint resources.
	 */

	if (has_constraints (class))
	{
		fprintf (fptr, "The following resources are associated with\n");
		fprintf (fptr, "child of this widget.\n");
		fprintf (fptr, "\\begin{resources}{%s}\n", class->cl_name);
		fprintf (fptr, "\
%% Name		& Class		& Type		& Default\n");
		output_constraint_tex_sum (fptr, class);
		fprintf (fptr, "\\end{resources}\n");
		
		/*
		 * 4. Write out the full description for each of the resources.
		 */
		
		fprintf (fptr, "%%\n%% ");
		fprintf (fptr, "Only included if it is the original definition.\n");
		fprintf (fptr, "%%\n");
		fprintf (fptr, "\\iforig\n\\begin{resourcedescription}\n");
		output_constraint_tex_desc (fptr, class);
		fprintf (fptr, "\\end{resourcedescription}\n\\fi\n");
	}

	/*
	 * 6. Write out documentation for the actions
	 */
	if (class->cl_actions && class->cl_actions->al_count > 0)
	{
		output_actions_tex (fptr, class);
	}

	/*
	 * 7. Write out documentation for the translations
	 */
	if (class->cl_translations && class->cl_translations->tl_count > 0)
	{
		output_translation_tex (fptr, class);
	}

	/*
	 * 8. Write out the instance record.
	 */
	/* Same as item (7) in output_includep() */
	output_type (class, "Rec", name);
	fprintf (fptr, "\\begin{tgrind}\n\\hsize 7in\\par\n");
	fprintf (fptr, "\\L{\\LB{typedef struct \\_%s \\{}}\n",
		 output_tex_quoted_str (name,  buffer));
	output_instance_structure (fptr, class);
	output_type (class, "Rec", name);
	fprintf (fptr, "\\L{\\LB{\\} %s,}",
		 output_tex_quoted_str (name, buffer));
	output_type (class, "Widget", name);
	fprintf (fptr, "\\Tab{24}{\\K{* %s};}}\n",
		 output_tex_quoted_str (name, buffer));
	fprintf (fptr, "\\end{tgrind}\n");

	diff = output_close (fptr, filename, old);
	return diff;
}

static void
output_instance_structure (fptr, class)
	FILE		*fptr;
	class_t		*class;
{
	char		name[BUFSIZ];
	char		buffer[BUFSIZ];
	char		buffer2[BUFSIZ];

	if (!fptr || !class)
		return;
	output_instance_structure (fptr, class_lookup (class->cl_superclass));
	fprintf (fptr, "\\L{\\LB{}\\Tab{4}{typedef struct \\{}}\n");
	output_tex_record_part (fptr, class, class->cl_instance_vars, VARINSTANCE);
	output_type (class, "Part", name);
	fprintf (fptr, "\\L{\\LB{}\\Tab{4}{\\}} \\K{%s},\\Tab{16}{\\K{%s};}}\n",
		 output_tex_quoted_str (name, buffer),
		 output_tex_quoted_str (class->cl_inst_field,  buffer2));
}

static void
output_tex_record_part (fptr, class, vars, type)
FILE		*fptr;			/* where to write to */
class_t		*class;			/* this class */
char		**vars;			/* list of variables */
int		type;			/* whether a class or instance var */
{
	char		**ptr;		/* list of instance fields */
	field_t		dummy;		/* dummy field */
	field_t		*field;
	int		override;
	int		one = FALSE;

	for (ptr = vars; ptr && *ptr; ptr++)
	{
		if (type == VARINSTANCE)
			field = lookup_instance_var (class, *ptr, &override);
		else if (type == VARCLASS)
			field = lookup_class_var (class, *ptr, &override);
		else if (type == VARCONSTRAINT)
			field = lookup_constraint_var (class, *ptr, &override);
		else
			field = NULL;
		/*
		 * Do not output anything if this is an override field
		 */
		if (!is_override (field))
		{
			one = TRUE;
			output_tex_field (fptr, field);
		}
	}
	if (!one)
	{
		dummy.vf_type = "int";
		dummy.vf_field = "dummy";
		dummy.vf_comment = "keep compiler happy";
		dummy.vf_default = "0";
		output_tex_field (fptr, &dummy);
		return;
	}
}

static void
output_tex_field (fptr, field)
FILE		*fptr;			/* where to write to */
field_t		*field;			/* what to write */
{
	char		buf[256];

	if (!field || !fptr)
		return;
#ifdef notdef
	fprintf (fptr, "\\hbox{\\vbox{\\hskip .1in\\hskip .1in\n\t");
	fprintf (fptr, "\\hbox to 1.5in{{\\bf %s}\\hfil}%%\n\t",
		 output_tex_quoted_str (field->vf_type, buf));
	fprintf (fptr, "\\hbox to 1.5in{%s;\\hfil}%%\n\t",
		 output_tex_quoted_str (field->vf_field, buf));
	if (field->vf_comment && *field->vf_comment)
		fprintf (fptr, "\\hbox to 3.0in{/* {\\it %s} \\hfil*/}",
			 output_tex_quoted_str (field->vf_comment, buf));
	fprintf (fptr, "}}\n");
#endif
	fprintf (fptr, "\\L{\\LB{}\\Tab{8}{\\K{%s}}",
		 output_tex_quoted_str (field->vf_type, buf));
	fprintf (fptr, "\\Tab{24}{\\V{%s};}",
		 output_tex_quoted_str (field->vf_field, buf));
	if (field->vf_comment && *field->vf_comment)
		fprintf (fptr, "\\Tab{40}{\\C{}/\\* %s \\*/\\CE{}}",
			 output_tex_quoted_str (field->vf_comment, buf));
	fprintf (fptr, "}\n");
}

/*
 * Output a tabular form for LaTeX documentation.
 */
static void
output_constraint_tex_sum (fptr, class)
FILE		*fptr;			/* where to write to */
class_t		*class;			/* class to document */
{
	field_t		*field;
	char		**ptr;
	int		override;
	int		has_override = 0;
	int		named = 0;
	char		**reslist;

	if (!fptr)
		return;
	if (!class)
		return;
	output_constraint_tex_sum (fptr, class_lookup (class->cl_superclass));
	reslist = str_list_copy (class->cl_constraint_vars);
	str_list_sort (reslist);
	for (ptr = reslist; ptr && *ptr; ptr++)
	{
		if (!named)
		{
			/*
			 * Print a reference to the parent class.
			 */
			fprintf (fptr, "\\multicolumn{4}{|c|}{{\\bf %s} (see page \\pageref{%s%s})}\\\\ \\hline\n",
				 class->cl_name, REF_PREFIX, class->cl_name);
			named = 1;
		}
		field = lookup_constraint_var (class, *ptr, &override);
		if (is_override (field))
		{
			field = lookup_constraint_override (class, field);
			override = 1;
			has_override = 1;
		}
		if (!field || !field->vf_name || !*field->vf_name)
			continue;
		fprintf (fptr, "%s\t& %s\t& %s\t& ",
			 field->vf_name, field->vf_class, field->vf_type);
		output_tex_quoted (fptr, field->vf_default);
		fprintf (fptr, "%s \\\\ \\hline\n", override?"*":"");
		if (override)
			instance_var_free (field);
	}
	str_list_free (reslist);
}
/*
 * Output a tabular form for LaTeX documentation.
 */
static void
output_resource_tex_sum (fptr, class)
FILE		*fptr;			/* where to write to */
class_t		*class;			/* class to document */
{
	field_t		*field;
	char		**ptr;
	int		override;
	int		has_override = 0;
	int		named = 0;
	char		**reslist;

	if (!fptr)
		return;
	if (!class)
		return;
	output_resource_tex_sum (fptr, class_lookup (class->cl_superclass));
	reslist = str_list_copy (class->cl_instance_vars);
	str_list_sort (reslist);
	for (ptr = reslist; ptr && *ptr; ptr++)
	{
		if (!named)
		{
			/*
			 * Print a reference to the parent class.
			 */
			fprintf (fptr, "\\multicolumn{4}{|c|}{{\\bf %s} (see page \\pageref{%s%s})}\\\\ \\hline\n",
				 class->cl_name, REF_PREFIX, class->cl_name);
			named = 1;
		}
		field = lookup_instance_var (class, *ptr, &override);
		if (is_override (field))
		{
			field = lookup_instance_override (class, field);
			override = 1;
			has_override = 1;
		}
		if (!field || !field->vf_name || !*field->vf_name)
			continue;
		fprintf (fptr, "%s\t& %s\t& %s\t& ",
			 field->vf_name, field->vf_class, field->vf_type);
		output_tex_quoted (fptr, field->vf_default);
		fprintf (fptr, "%s \\\\ \\hline\n", override?"*":"");
		if (override)
			instance_var_free (field);
	}
	str_list_free (reslist);
}

/*
 * Output the description of each resource for LaTeX documentation.
 */
static void
output_resource_tex_desc (fptr, class)
FILE		*fptr;			/* where to write to */
class_t		*class;			/* class to document */
{
	field_t		*field;
	char		**ptr;
	int		override;
	char		**reslist;
	int		count;

	if (!fptr)
		return;
	if (!class)
		return;

	reslist = str_list_copy (class->cl_instance_vars);
	str_list_sort (reslist);
	count = 0;
	/*
	 * Describe all the resources
	 */
	for (ptr = reslist; ptr && *ptr; ptr++)
	{
		field = lookup_instance_var (class, *ptr, &override);
		if (!field)
		{
			fprintf (stderr, "No instance variable for %s\n",
				 *ptr);
			continue;
		}
		if (field->vf_name && *field->vf_name)
		{
			++count;
			fprintf (fptr, "\\item[XtN%s]\n", field->vf_name);
			if (field->vf_description)
				fprintf (fptr, "%s\n", field->vf_description);
			else
				fprintf (fptr, "No documentation.\n");
			fprintf (fptr, "\\resname{");
			output_tex_quoted (fptr, field->vf_field);
			fprintf (fptr, "}\n\n");
		}
	}
	/*
	 * Now add all the instance variables that are not resources.
	 */
	for (ptr = reslist; ptr && *ptr; ptr++)
	{
		field = lookup_instance_var (class, *ptr, &override);
		if (!field)
		{
			fprintf (stderr, "No instance variable for %s\n",
				 *ptr);
			continue;
		}
		if (!field->vf_name || !*field->vf_name)
		{
			/*
			 * See if this is an instance variable that has some
			 * description that should be documented
			 */
			++count;
			fprintf (fptr, "\\ifinternal\n");
			fprintf (fptr, "\\item[\\code{ ");
			output_tex_quoted (fptr, field->vf_field);
			fprintf (fptr, "}]\n");
			if (field->vf_description && *field->vf_description)
			{
				fprintf (fptr, "%s\n", field->vf_description);
			}
			else if (field->vf_comment && *field->vf_comment)
			{
				fprintf (fptr, "%s.\n", field->vf_comment);
			}
			else
			{
				fprintf (fptr, "No documentation.\n");
			}
			fprintf (fptr, "\\fi\n\n");
		}
	}
	str_list_free (reslist);
	if (!count)
	{
		fprintf (fptr, "\\item[NONE]\nThere are no resources.\n\n");
	}
}

/*
 * Output the description of each resource for LaTeX documentation.
 */
static void
output_constraint_tex_desc (fptr, class)
FILE		*fptr;			/* where to write to */
class_t		*class;			/* class to document */
{
	field_t		*field;
	char		**ptr;
	int		override;
	char		**reslist;
	int		count;

	if (!fptr)
		return;
	if (!class)
		return;

	reslist = str_list_copy (class->cl_constraint_vars);
	str_list_sort (reslist);
	count = 0;
	/*
	 * Describe all the resources
	 */
	for (ptr = reslist; ptr && *ptr; ptr++)
	{
		field = lookup_constraint_var (class, *ptr, &override);
		if (!field)
		{
			fprintf (stderr, "No constraint variable for %s\n",
				 *ptr);
			continue;
		}
		if (field->vf_name && *field->vf_name)
		{
			++count;
			fprintf (fptr, "\\item[XtN%s]\n", field->vf_name);
			if (field->vf_description)
				fprintf (fptr, "%s\n", field->vf_description);
			else
				fprintf (fptr, "No documentation.\n");
			fprintf (fptr, "\\resname{");
			output_tex_quoted (fptr, field->vf_field);
			fprintf (fptr, "}\n\n");
		}
	}
	/*
	 * Now add all the constraint variables that are not resources.
	 */
	for (ptr = reslist; ptr && *ptr; ptr++)
	{
		field = lookup_constraint_var (class, *ptr, &override);
		if (!field)
		{
			fprintf (stderr, "No constraint variable for %s\n",
				 *ptr);
			continue;
		}
		if (!field->vf_name || !*field->vf_name)
		{
			/*
			 * See if this is an constraint variable that has some
			 * description that should be documented
			 */
			++count;
			fprintf (fptr, "\\ifinternal\n");
			fprintf (fptr, "\\item[\\code{ ");
			output_tex_quoted (fptr, field->vf_field);
			fprintf (fptr, "}]\n");
			if (field->vf_description && *field->vf_description)
			{
				fprintf (fptr, "%s\n", field->vf_description);
			}
			else if (field->vf_comment && *field->vf_comment)
			{
				fprintf (fptr, "%s.\n", field->vf_comment);
			}
			else
			{
				fprintf (fptr, "No documentation.\n");
			}
			fprintf (fptr, "\\fi\n\n");
		}
	}
	str_list_free (reslist);
	if (!count)
	{
		fprintf (fptr, "\\item[NONE]\nThere are no resources.\n\n");
	}
}

static int
has_constraints (class)
	class_t		*class;
{
	if (!class)
		return (0);
	if (class->cl_cons_field && *class->cl_cons_field
	    && class->cl_instance_vars && *class->cl_instance_vars)
		return (1);
	else
		return (has_constraints (class_lookup (class->cl_superclass)));
}

static void
output_actions_tex (fptr, class)
FILE		*fptr;			/* where to write the docs */
class_t		*class;			/* the class */
{
	int		i;
	action_t	*a;

	if (!class)
		return;
	if (!class->cl_actions || !class->cl_actions->al_list
	    || class->cl_actions->al_count <= 0)
		return;
	fprintf (fptr, "\\iforig\n");
	fprintf (fptr, "The following actions are defined by this widget.\n");
	fprintf (fptr, "\\begin{resourcedescription}\n");
	a = class->cl_actions->al_list;
	for (i = 0; i <  class->cl_actions->al_count; i++)
	{
		fprintf (fptr, "\\item[%s]\n", a[i].a_name);
		if (a[i].a_description && *a[i].a_description)
			fprintf (fptr, "%s\n\n", a[i].a_description);
		else
			fprintf (fptr, "No documentation.\n\n");
	}
	fprintf (fptr, "\\end{resourcedescription}\n");
	fprintf (fptr, "\\fi\n");
}

static void
output_translation_tex (fptr, class)
FILE		*fptr;			/* where to write the code */
class_t		*class;			/* the class */
{
	int	i;
	translation_t	*t;
	translation_list_t	*tlist;

	if (!class || !class->cl_translations ||
	    !class->cl_translations->tl_list || 
	    class->cl_translations->tl_count <= 0)
		return;
	tlist = class->cl_translations;
	t = tlist->tl_list;
	fprintf (fptr, "\\iforig\n");
	fprintf (fptr, "The following translations are used by this widgets.\n\n");
	for (i = 0; i < tlist->tl_count; i++)
	{
		fprintf (fptr, "\\action{");
		output_tex_quoted (fptr, t[i].t_events);
		fprintf (fptr, "}{");
		output_tex_quoted (fptr, t[i].t_actions);
		fprintf (fptr, "}");
		if (t[i].t_description && *t[i].t_description)
		{
			fprintf (fptr, "%%\n\t{\n\t%s\n\t}\n",
				 t[i].t_description);
		}
		else
		{
			fprintf (fptr, "{}\n");
		}
	}
	fprintf (fptr, "\\fi\n");
}

static void
output_tex_quoted (fptr, str)
	FILE	*fptr;
	char	*str;
{
	char	buffer[256];

	fprintf (fptr, "%s", output_tex_quoted_str (str, buffer));
}

static char *
output_tex_quoted_str (str, buf)
char		*str;
char		*buf;
{
	char		*buffer = buf;

	if (!str || !buffer)
		return ((char *) NULL);
	while (*str)
	{
		if (*str == '_' || *str == '#')
			*buffer++ = '\\';
		else if (*str == '<' || *str == '>')
			*buffer++ = '$';
		*buffer++ = *str;
		if (*str == '<' || *str == '>')
			*buffer++ = '$';
		str++;
	}
	*buffer = '\0';
	return buf;
}
