/************************************************************
 *                                                          *
 *  Permission is hereby granted  to  any  individual   or  *
 *  institution   for  use,  copying, or redistribution of  *
 *  the xgobi code and associated documentation,  provided  *
 *  that   such  code  and documentation are not sold  for  *
 *  profit and the  following copyright notice is retained  *
 *  in the code and documentation:                          *
 *     Copyright (c) 1990,1991,1992,1993 Bellcore           *
 *                                                          *
 *  We welcome your questions and comments, and request     *
 *  that you share any modifications with us.               *
 *                                                          *
 *    Deborah F. Swayne            Dianne Cook              *
 *     dfs@bellcore.com      dcook@stat.rutgers.edu         *
 *      (201) 829-4263                                      *
 *                                                          *
 ************************************************************/

#include <stdio.h>
#include <signal.h>
#include "xincludes.h"
#include "xgobitypes.h"
#include "xgobivars.h"

Boolean gotone = False;

static void
stdin_missing()
{
  if (!gotone)
  {
    fprintf(stderr, "xgobi requires a filename or some data from stdin\n");
    exit(0);
  }
}

void
read_array(data_in, xg)
  char *data_in;
  xgobidata *xg;
{
  register int ch;
  int i, j, jrows, jcols, fs;
  int nitems;
  float row1[NCOLS];
  int nblocks;
  char fname[100];
  FILE *fopen(), *fp;
  int ungetc();
  int fclose();

/*
 * Check file exists and open it - for stdin no open needs to be done
 * only assigning fp to be stdin.
*/
  if (strcmp(data_in,"stdin") == 0)
  {
    fp = stdin;
  }
  else
  {
    if ((fp = fopen(data_in, "r")) == NULL)
    {
      strcpy(fname, data_in);
      strcat(fname, ".dat");
      if ((fp = fopen(fname, "r")) == NULL)
      {
        (void) fprintf(stderr,
          "Neither the file %s nor %s exists\n", data_in, fname);
        exit(0);
      }
    }
  }
/*
 * If reading from stdin, set an alarm.  If after 5 seconds, no data
 * has been read, print an error message and exit.     dfs
*/
  if (fp == stdin)
  {
    alarm((unsigned int) 5);
    signal(SIGALRM, stdin_missing);
  }

/*
 * Read in the first row of the data file and calculate ncols.
*/
  xg->ncols_used = 0;
  while ( (ch = getc(fp)) != '\n')
  {
    if (!gotone)
      gotone = True;                /* dfs */
    if (ch == '\t' || ch == ' ')
      ;
    else
    {
      if ( ungetc(ch, fp) == EOF ||
         fscanf(fp, "%g", &row1[xg->ncols_used++]) < 0 )
      {
        fprintf(stderr,
          "read_array: error in reading first row of data");
        exit(0);
      }
    }
  }

  xg->ncols = xg->ncols_used + 1;
/*
 * Allocate space for 500 rows, leaving two extra slots for the
 * extra variables; one sometimes mapped, one never mapped.
*/
  xg->raw_data = (float **) XtMalloc(
    (Cardinal) 500 * sizeof(float *));
  for (i=0; i<500; i++)
    xg->raw_data[i] = (float *) XtMalloc(
      (Cardinal) xg->ncols * sizeof(float));

/*
 * Fill in the first row
*/
  for (j=0; j<xg->ncols_used; j++)
    xg->raw_data[0][j] = row1[j];

/*
 * Read data, reallocating whenever 500 more rows are read in.
 * Determine nrows.
*/
  nblocks = 1;
  nitems = xg->ncols_used;
  xg->nrows = jrows = 1;
  jcols = 0;
  while (1)
  {
    fs = fscanf(fp, "%g", &xg->raw_data[xg->nrows][jcols]);
    jcols++;

    if (fs == EOF)
      break;
    else if (fs < 0)
    {
      fprintf(stderr, "read_array: error in fscanf\n");
      exit(1);
    }
    else
    {
      nitems++;

      if (jcols == xg->ncols_used)
      {
        jcols = 0;
        xg->nrows++;
        jrows++;
      }
      if (jrows == 500)
      {
        jrows = 0;
        nblocks++;

        xg->raw_data = (float **) XtRealloc((char *)
          xg->raw_data,
          (Cardinal) (nblocks * 500) * sizeof(float *));
        for (i=500*(nblocks-1); i<500*nblocks; i++)
          xg->raw_data[i] = (float *) XtMalloc(
            (Cardinal) xg->ncols * sizeof(float));
      }
    }
  }

/*
 * Close the data file
*/
  if (fclose(fp) == EOF)
    fprintf(stderr, "read_array: error in fclose");

  if ( nitems != xg->nrows * xg->ncols_used )
  {
    (void) fprintf(stderr, "read_array: nrows*ncols != nitems\n");
    /*return(0);*/
    exit(1);
  }
  else
  {
    /*
     * One last XtFree and XtRealloc to make raw_data take up exactly
     * the amount of space it needs.
    */
    for (i=xg->nrows; i<500*nblocks; i++)
      XtFree((XtPointer) xg->raw_data[i]);
    xg->raw_data = (float **) XtRealloc((XtPointer) xg->raw_data,
      (Cardinal) xg->nrows * sizeof(float *));

    /*
     * If the data contains only one column, add a second,
     * the numbers 1:nrows.
    */
    if (xg->ncols_used == 1)
    {
      xg->ncols_used = 2;
      xg->ncols = 3;
      for (i=0; i<xg->nrows; i++)
      {
        xg->raw_data[i] = (float *) XtRealloc(
          (XtPointer) xg->raw_data[i],
          (Cardinal) 3 * sizeof(float));
        xg->raw_data[i][1] = (float) (i+1) ;
      }
    }
  }
}

int
Sread_array(data_in, xg)
  char *data_in;
  xgobidata *xg;
{
  int i, j, nitems = 0;
  int ok = 0;
  FILE *fopen(), *fp;
  int unlink();

  if ((fp = fopen(data_in, "r")) == NULL)
    (void) fprintf(stderr,
      "Sread_array: data file %s does not exist\n", data_in);

  else
  {
    xg->raw_data = (float **) XtMalloc(
			(Cardinal) xg->nrows * sizeof(float *));
    for (i=0; i<xg->nrows; i++)
      xg->raw_data[i] = (float *) XtMalloc(
				(Cardinal) xg->ncols * sizeof(float));

    for (i=0; i<xg->nrows; i++)
      for (j=0; j<xg->ncols_used; j++)
        nitems = nitems + fscanf(fp, "%g", &xg->raw_data[i][j]);

    /*
     * Test the number of items read against the dimension
     * of the array.
    */
    if (nitems == xg->nrows * xg->ncols_used)
    {
      ok = 1;
      if (unlink(data_in) != 0)
        fprintf(stderr, "Sread_array: error in unlink");
    }
    else
    {
      ok = 0;
      (void) fprintf(stderr, "Sread_array: nrows*ncols != nitems\n");
      /*return(0);*/
      exit(1);
    }

    /*
     * Now fill some data into the extra column, which will be
     * used if groups are defined by brushing.
    */
    for (i=0; i<xg->nrows; i++)
      xg->raw_data[i][xg->ncols_used] = 1.0;
  }

  return(ok);
}


void
fill_extra_column(xg)
  xgobidata *xg;
{
  int i;

  for (i=0; i<xg->nrows; i++)
    xg->raw_data[i][xg->ncols_used] = 1.0 ;
}

void
find_root_name_of_data(fname, title)
  char *fname, *title;
/*
 * Strip off preceding directory names and find the final filename
 * to use in resetting the title and iconName.
*/
{
  char *pf;
  int j = 0;

  pf = fname;
  while (fname[j] != '\0')
  {
    if (fname[j] == '/')
      pf = &fname[j+1];
    j++;
  }

  (void) strcpy(title, pf);
}

void
set_title_and_icon(fname, xg)
  char *fname;
  xgobidata *xg;
/*
 * If the user hasn't specified a title in a general resource
 * file, a data-specific resource file, or on the command
 * line, and if the data is being read in from a file, then
 * use the name of the file to compose the title and iconName
 * resources.
*/
{
  char fulltitle[150], usertitle[120], rootfname[100];
  char *str_type[50];
  XrmValue value;

  if (XrmGetResource(display->db, "xgobi.title", "XGobi.Title",
      str_type, &value))
  {
    (void) strncpy(usertitle, value.addr, (int) value.size);
  }

  if (strcmp(fname, "stdin") == 0)  /* If input is from stdin */
    strcpy(fulltitle, "XGobi");

  else  /* If reading from a file */
  {
    strcpy(fulltitle, "XGobi: ");

    /* usertitle is by default XGobi, the class name */
    if (usertitle && strlen(usertitle) > 0)
    {
      /* If the title is exactly XGobi, find the data name and
       * add it to fulltitle
      */
      if (strcmp(usertitle, "XGobi") == 0)
      {
        find_root_name_of_data(fname, rootfname);
        strcat(fulltitle, rootfname);
      }
      /* If the title contains XGobi, just use it */
      else if (strncmp("XGobi", usertitle, 5) == 0)
        strcpy(fulltitle, usertitle);

      /* else add it to fulltitle */
      else
        strcat(fulltitle, usertitle);
    }
    else  /* If usertitle is NULL */
    {
      find_root_name_of_data(fname, rootfname);
      strcat(fulltitle, rootfname);
    }
  }

  XtVaSetValues(xg->shell,
    XtNtitle, (String) fulltitle,
    XtNiconName, (String) fulltitle,
    NULL);
}

int
read_vgroups(data_in, xg)
/*
 * Read in the grouping numbers for joint scaling of variables
*/
  char *data_in;
  xgobidata *xg;
{
  char lab_file[115], suffix[15];
  int itmp, i, j, found = 0;
  int id, newid, maxid;
  FILE *fopen(), *fp;

  xg->vgroup_ids = (int *) XtMalloc(
    (Cardinal) xg->ncols * sizeof(int));

  (void) strcpy(suffix, ".vgroups");

  if (data_in != NULL && strcmp(data_in, "stdin") != 0)
  {
    i = 0;
    (void) strcpy(lab_file, data_in);
    (void) strcat(lab_file, suffix);
    if ( (fp = fopen(lab_file,"r")) != NULL)
      found = 1;
  }

  if (found)
  {
    i = 0;
    while ((fscanf(fp, "%d", &itmp) != EOF) && (i < xg->ncols_used))
      xg->vgroup_ids[i++] = itmp;

    /*
     * Add the vgroup_id value for the extra column.
    */
    if (i == xg->ncols_used)
    {
      xg->vgroup_ids[i] = i;
    }

    if (i < xg->ncols_used)
    {
      (void) fprintf(stderr,
        "Number of variables and number of group types do not match.\n");
      (void) fprintf(stderr,
        "Creating extra generic groups.\n");
      for (j=i; j<xg->ncols; j++)
        xg->vgroup_ids[j] = j;
    }

/*
 * Find maximum vgroup id.
*/
    maxid = 0;
    for (i=1; i<xg->ncols; i++)
      if (xg->vgroup_ids[i] > maxid)
        maxid = xg->vgroup_ids[i];
/*
 * Find minimum vgroup id, set it to 0.  Find next, set it to 1; etc.
*/
    id = 0;
    newid = -1;
    while (id <= maxid)
    {
      found = 0;
      for (j=0; j<xg->ncols; j++)
      {
        if (xg->vgroup_ids[j] == id)
        {
          newid++;
          found = 1;
          break;
        }
      }
      if (found)
        for (j=0; j<xg->ncols; j++)
          if (xg->vgroup_ids[j] == id)
            xg->vgroup_ids[j] = newid;
      id++;
    }
  }

  else
  {
    for (i=0; i<xg->ncols; i++)
      xg->vgroup_ids[i] = i;
  }
  return(1);
}

int
read_erase(data_in, xg)
/*
 * Read in the grouping numbers for joint scaling of variables
*/
  char *data_in;
  xgobidata *xg;
{
  char lab_file[115], suffix[15];
  int itmp, i, found = 0;
  FILE *fopen(), *fp;

  xg->erased = (unsigned short *) XtMalloc(
     (Cardinal) xg->nrows * sizeof(unsigned short));

  (void) strcpy(suffix,".erase");

  if (data_in != NULL && strcmp(data_in, "stdin") != 0)
  {
    i = 0;
    (void) strcpy(lab_file, data_in);
    (void) strcat(lab_file, suffix);
    if ( (fp = fopen(lab_file,"r")) != NULL)
      found = 1;
  }

  if (found)
  {
    i = 0;
    while ((fscanf(fp, "%d", &itmp) != EOF) && (i < xg->nrows))
      xg->erased[i++] = (unsigned short) itmp;

    if (i < xg->nrows)
    {
      (void) fprintf(stderr, "Problem in reading file %s; \n", lab_file);
      (void) fprintf(stderr, "not enough rows\n");
    }
  }
  else
    for (i=0; i<xg->nrows; i++)
      xg->erased[i] = 0;

  return(1);
}

int
read_extra_resources(data_in)
/*
 * Read in the data-specific resource file.
*/
  char *data_in;
{
  char lab_file[115], suffix[15];
  int found = 0;
  int ok = 1;
  FILE *fopen(), *fp;

  (void) strcpy(suffix,".resources");

  if (strcmp(data_in, "stdin") != 0)
  {
    (void) strcpy(lab_file, data_in);
    (void) strcat(lab_file, suffix);
    if ( (fp = fopen(lab_file,"r")) != NULL)
      found = 1;
  }

  if (found)
  {
    XrmDatabase newdb;

    if ( newdb = XrmGetFileDatabase(lab_file) )
      XrmMergeDatabases(newdb, &display->db);
    else
    {
      ok = 0;
      fprintf(stderr,
        "read_extra_resources: problem reading data-specific resource file\n");
      exit(1);
    }

  /*
   * Close the data file
  */
    if (fclose(fp) == EOF)
      fprintf(stderr, "read_extra_resources: error in fclose");
  }
  return(ok);
}


