/*
 * $Id: udpath.c,v 2.10 1991/07/31 20:29:57 steve Exp $
 *
 * $__Header$
 */

/*LINTLIBRARY*/

#include "udposix.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <limits.h>
#include <assert.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "udpath.h"
#include "udalloc.h"

#ifndef PATH_MAX
#   define PATH_MAX	_POSIX_PATH_MAX
#endif


/*
 *	Return the access mode given a mode mask and access bits.
 */

    static unsigned
get_access(mode, readbit, writebit, executebit)
    unsigned	mode;
    unsigned	readbit, writebit, executebit;
{
    unsigned	accessmode	= 0;

    if (mode & readbit)
	accessmode	|= UD_FREAD;
    if (mode & writebit)
	accessmode	|= UD_FWRITE;
    if (mode & executebit)
	accessmode	|= UD_FEXECUTE;

    return accessmode;
}


/*
 *	Return the type of a file.  Return values are defined in "udpath.h":
 *
 *	File type:
 *		UD_FACCESS	Error.  Couldn't access file.
 *		UD_FDEV		Character-oriented device
 *		UD_FREG		Regular file
 *		UD_FDIR		Directory
 *		UD_FFIFO	Pipe or FIFO special file
 *		UD_FBLK		Block-oriented device
 *		UD_FOTHER	Something else
 *
 *	Access mode:
 *		UD_FREAD	Readable.
 *		UD_FWRITE	Writable.
 *		UD_FEXECUTE	Executable.
 *
 *	Function returns 1 on success and 0 if it can't stat(2) the file.
 *
 *	This function won't work on relative VMS names (e.g. "[-]").
 */

    int
udpath(path, type, mode)
    char	*path;		/* Input name of file */
    int		*type;		/* Returned type of file */
    int		*mode;		/* Returned access mode of file */
{
    int		success	= 0;		/* Return status */
#   ifdef vms
    int		lenpath	= strlen(path);
    char	buf[PATH_MAX+1];
#   endif
    struct stat	stat_buf;

    assert(path != NULL);

#   ifdef vms
    if (path[lenpath-1] == ']') {		/* if directory specification */
	char	*cp	= strrchr(path, '.');
	
	if (cp != NULL) {
	    int		nchr	= cp - path;

	    (void)memcpy(buf, path, nchr);
	    buf[nchr++]	= ']';
	    (void)strcpy(buf+nchr, ++cp);
	    (void)strcpy(buf+lenpath-1, ".dir");
	    path	= buf;
	}
    }
#   endif

    if (stat(path, &stat_buf) == 0) {

	if (S_ISREG(stat_buf.st_mode)) {	/* It's a file */
	    *type	= UD_FREG;
	} else if (S_ISDIR(stat_buf.st_mode)) {	/* It's a directory */
	    *type	= UD_FDIR;
	} else if (S_ISCHR(stat_buf.st_mode)) {	/* It's a character-device */
	    *type	= UD_FCHR;
	} else if (S_ISBLK(stat_buf.st_mode)) {	/* It's a block-device */
	    *type	= UD_FBLK;
	} else if (S_ISFIFO(stat_buf.st_mode)) {/* It's a pipe or FIFO */
	    *type	= UD_FFIFO;
	} else {				/* It's something else */
	    *type	= UD_FOTHER;
	}

#	ifdef	vms
	stat_buf.st_uid	&= 0177777;		/* mask out gid part */
#	endif	/* vms */

	*mode	= 0;				/* Default = no access */

	if (geteuid() == (unsigned)stat_buf.st_uid)
	    *mode	|= get_access(stat_buf.st_mode, 
		S_IRUSR, S_IWUSR, S_IXUSR);

	if (getegid() == (unsigned)stat_buf.st_gid)
	    *mode	|= get_access(stat_buf.st_mode, 
		S_IRGRP, S_IWGRP, S_IXGRP);

	*mode	|= get_access(stat_buf.st_mode, S_IROTH, S_IWOTH, S_IXOTH);

	success	= 1;
    }

    return success;
}


/*
 *	Extract the head of a path specification (i.e. the specification of 
 *	the parent directory.
 *
 *	The function returns a pointer to malloc()ed space containing the
 *	specification.  It should be free()d by the user.
 */

    char*
udhead(path)
    const char	*path;
{
    char	*buf	= NULL;
    char	*udstrdup();

    if (path != NULL && (buf = udstrdup(path)) != NULL) {

#ifdef vms

	if (buf[strlen(buf)-1] == ']') {
	    /* Path is pure directory specification.  Remove last component. */

	    char	*endp	= strrchr(buf, '.');

	    if (endp == NULL) {
		/* No directory components in path.  Use device if possible. */

		if ((endp = strrchr(buf, ':')) == NULL) {
		    /* No device in path.  Use current directory. */

		    if (strlen(path) > 2) {
			(void)strcpy(buf, "[]");
		    } else {
			FREE(buf);
			buf	= udstrdup("[]");
		    }

		} else {
		    /* Device in path.  Use it. */
		    endp[1]	= 0;
		}
	    } else {
		/* Trailing directory component exists.  Remove it. */
		(void)strcpy(endp, "]");
	    }

	} else {
	    /* Path is not pure directory specification. */

	    char	*endp	= strrchr(buf, ']');

	    if (endp == NULL) {
		/* No directory in path.  Look for device. */

		if ((endp = strrchr(buf, ':')) == NULL) {
		    /* No device in path.  Use current directory. */

		    if (strlen(path) > 2) {
			(void)strcpy(buf, "[]");
		    } else {
			FREE(buf);
			buf	= udstrdup("[]");
		    }

		} else {
		    /* Device in path.  Use it. */
		    endp[1]	= 0;
		}

	    } else {
		/* Path has directory. Remove trailing file specification. */
		endp[1]	= 0;
	    }
	}

#else	/* !vms */

	char	*endp	= buf + strlen(buf);

	/* Remove trailing '/'s */
	while (endp > buf+1 && endp[-1] == '/')
	    --endp;
	*endp	= 0;

	/* Locate last '/' */
	endp	= strrchr(buf, '/');

	if (endp == NULL) {

	    /* No '/' in path.  Use current directory. */
	    if (strlen(path) > 1) {
		(void)strcpy(buf, ".");
	    } else {
		FREE(buf);
		buf	= udstrdup(".");
	    }

	} else {

	    /* Remove trailing '/'s */
	    while (endp > buf+1 && endp[-1] == '/')
		--endp;
	    *endp	= 0;
	}

#endif	/* !vms */

    }

    return buf;
}
