/****************************************************************************/
/*                                                                          */
/*  VolVis is a volume visualization system for investigating, manipulating */
/*  and rendering geometric and volumetric data.                            */
/*                                                                          */
/*  Copyright (C) 1993 by the Research Foundation of the State University   */
/*                            of New York                                   */
/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 1, or (at your option)     */
/*  any later version.                                                      */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this program; if not, write to the Free Software             */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/*                                                                          */
/*  For information on VolVis, contact us at:                               */
/*                                                                          */
/*                volvis@cs.sunysb.edu                         (email)      */
/*                                                                          */
/*                Lisa Sobierajski & Ricardo Avila             (US Mail)    */
/*                Department of Computer Science                            */
/*                State University of New York at Stony Brook               */
/*                Stony Brook, New York  11794-4400                         */
/*                                                                          */
/****************************************************************************/




/*
 *                  File: MOTIF_ide_ui.c
 *                Author: He, Taosong
 *                  Date: 1/19/93
 *           Description: MOTIF routines for input device UI creation
 *  Modification History: 
 *
 *         who?         when?           why?
 *    -----------------------------------------------------------
 *
 */

/******************************************************************/
/*                The standard C include file                     */
/******************************************************************/

# include <stdio.h>
# include <stdlib.h>
# include <string.h>

/******************************************************************/
/*               The MOTIF include file                           */
/******************************************************************/


# include <X11/keysym.h>
# include <Xm/Xm.h>
# include <Xm/Form.h>
# include <Xm/Frame.h>
# include <Xm/RowColumn.h>
# include <Xm/ToggleBG.h>
# include <Xm/ToggleB.h>
# include <Xm/PushB.h>
# include <Xm/PushBG.h>
# include <Xm/BulletinB.h>
# include <Xm/DialogS.h>
# include <Xm/Text.h>
# include <Xm/TextF.h>
# include <Xm/LabelG.h>
# include <Xm/DrawingA.h>
# include <Xm/CascadeB.h>
# include <Xm/MessageB.h>

/******************************************************************/
/*                  VolVis include file                           */
/******************************************************************/

# include "C_ide.h"
# include "MOTIF_windefs.h"

static Widget button1[3], button2[3], button3[1], button5[3];
static Widget mode_pulldown, mode_options[3], mode_menu;

/*
 *        Procedure Name: C_ide_quit
 *          Return Value: void
 *       Input Variables: client_data: the widget to exit
 *      Output Variables:
 *      Update Variables:
 *      Global Variables:
 *           Description: A call back function which unmanage the widget
 */
void C_ide_quit(w, client_data, call_data)
Widget w;
Widget client_data;
caddr_t call_data;
{
           XtUnmanageChild(client_data);
}

/*        Global variable: redisplay
 *                          This variable is used to control the display 
 *                          of Virtual device panel. When we finished
 *                          adding a virtual device, the virtual device 
 *                          panel should be redisplayed to show the change.
 */

static int  redisplay = 0;

/*
 *        Procedure Name: C_ide_quit_destroy
 *          Return Value: void
 *       Input Variables: client_data: the widget to destroy
 *      Output Variables:
 *      Update Variables:
 *      Global Variables:
 *           Description: A call back function which destory the widget
 */

void VD_callback();
void C_ide_quit_destroy(w, client_data, call_data)
Widget w;
Widget client_data;
caddr_t call_data;
{

           XtDestroyWidget(client_data);
           VD_callback(NULL, NULL);
}

/*
 *        Procedure Name: MOTIF_event_get_display
 *          Return Value: void
 *       Input Variables: 
 *      Output Variables:
 *      Update Variables: A keyboard event
 *      Global Variables:
 *           Description: Assign the display value for the keyboard event
 */

void MOTIF_event_get_display(event)

XKeyEvent *event;
{
           extern Widget vv_toplevel;

           event->display = XtDisplay(vv_toplevel);
}

/*         Global variable: IDE_panel
 *                           This variable is used to represent the control
 *                           panels for different input devices.
 */

Widget IDE_panel[C_MAX_RD_NUMBER];

/*
 *        Procedure Name: initialize_RD_panel
 *          Return Value: void
 *       Input Variables:
 *      Output Variables:
 *      Update Variables: IDE_panel           
 *      Global Variables: IDE_panel;
 *                        vv_toplevel;
 *                        window_defs;
 *           Description: Initializae the control panels for different input 
 *                        devices.
 */

void initialize_RD_panel()
{
   extern Widget vv_toplevel;
   extern void create_mouse_ui();
   extern void create_keyboard_ui();
/*
   extern void create_isotrak_ui();
*/
   extern void create_spaceball_ui();
   extern void create_dataglove_ui();
   extern C_WindowDefs    window_defs;


   int valid;

   Arg                   args[20];
   int                   n, i;
   XmString              string;

   for (i = 0; i< C_CURRENT_RD_NUMBER; i++)
     {

       if (!IDE_panel[i]) 
         {
            n = 0;
            XtSetArg( args[n], XmNautoUnmanage, False ); n++;
            C_Set_Color_Dialog( args, n, window_defs );
            IDE_panel[i] = XmCreateBulletinBoardDialog(
                           vv_toplevel, "", args, n);
            valid = 1;

            switch (i)
             {
               case C_MOUSE: 
                create_mouse_ui(IDE_panel[i]);
                string = XmStringCreate( "Mouse Control Panel", 
                                         XmSTRING_DEFAULT_CHARSET);
                break;
               case C_KEYBOARD:
                create_keyboard_ui(IDE_panel[i]);
                string = XmStringCreate( "Keyboard Control Panel", 
                                         XmSTRING_DEFAULT_CHARSET);
                break;
               case C_ISOTRAK:
/*
                create_isotrak_ui(IDE_panel[i]);
                string = XmStringCreate( "Isotrak Control Panel", 
                                         XmSTRING_DEFAULT_CHARSET);
*/
                valid = 0;
                break;
               case C_SPACEBALL:
                create_spaceball_ui(IDE_panel[i]);
                string = XmStringCreate( "Spaceball Control Panel",
                                         XmSTRING_DEFAULT_CHARSET);  
                break;
               case C_DATAGLOVE:
/*
                create_dataglove_ui(IDE_panel[i]);
                string = XmStringCreate( "DataGlove Control Panel", 
                                         XmSTRING_DEFAULT_CHARSET);
*/
                valid = 0;
                break;
             }
            if (valid)
             {
             n =0;
             XtSetArg( args[n], XmNdialogTitle, string ); n++;
             XtSetValues(IDE_panel[i], args, n);
             XmStringFree(string);
 
             XtUnmanageChild(IDE_panel[i]);
             }
       }
     }
}
               
                
/*
 *        Procedure Name: show_panel_callback
 *          Return Value: void
 *       Input Variables: The index of device panel
 *      Output Variables:
 *      Update Variables: 
 *      Global Variables: IDE_panel;
 *           Description: Call back function. Called to pop up the control 
 *                        panel for the specific input device.
 */

void show_panel_callback(w, client_data, call_data)
Widget w;
int client_data;
caddr_t call_data;
{
  if (client_data != C_ISOTRAK) 
    {
     if (IDE_panel[client_data])
     XtManageChild(IDE_panel[client_data]);
    }
}

/*
 *        Procedure Name: RD_callback
 *          Return Value: void
 *       Input Variables: 
 *      Output Variables:
 *      Update Variables:
 *      Global Variables: window_defs;
 *           Description: Call back function. Called to pop up the 
 *                        Device Control panel menu.
 *                         Button 1: Mouse.
 *                         Button 2: Isotrak.
 *                         Button 3: Spaceball.
 *                         Button 4: DataGlove.
 *                         Button 5: Keyboard.
 *                         Button 6: Close.
 *                         Button 2-4 currently are not supported. 
 */

void RD_callback(w, call_data)
Widget w;
caddr_t call_data;
{
        extern C_WindowDefs    window_defs;

        Widget                 rd_bbdialog;

        Arg             args[15];       /* Argument List */
        int             n;              /* Argument Count */
        int             i;

        XmString        string;

        Widget rc, button[C_MAX_RD_NUMBER+1];

        n=0;
        XtSetArg( args[n], XmNautoUnmanage, False ); n++;
        C_Set_Color_Dialog( args, n, window_defs );
        rd_bbdialog = XmCreateBulletinBoardDialog(w,
                                "Device Control Panel", args, n);


        n = 0;
        XtSetArg( args[n], XmNx, 10 ); n++;
        XtSetArg( args[n], XmNy, 10 ); n++;
        XtSetArg( args[n], XmNorientation, XmVERTICAL); n++;
        XtSetArg( args[n], XmNresizeHeight, TRUE ); n++;
        C_Set_Color_RowColumn( args, n, window_defs );
        rc = XmCreateRowColumn (rd_bbdialog, "rc", args, n);

        for ( i=0; i<C_CURRENT_RD_NUMBER; i++)
         {
          switch (i)
           { 
             case C_MOUSE:
               string = XmStringCreate ("Mouse", XmSTRING_DEFAULT_CHARSET);
               break;
             case C_ISOTRAK:
               string = XmStringCreate ("Isotrak", XmSTRING_DEFAULT_CHARSET);
               break;
             case C_SPACEBALL:
               string = XmStringCreate ("Spaceball", XmSTRING_DEFAULT_CHARSET);
               break;
             case C_DATAGLOVE:
               string = XmStringCreate ("DataGlove", XmSTRING_DEFAULT_CHARSET);
               break;
             case C_KEYBOARD:
               string = XmStringCreate ("Keyboard", XmSTRING_DEFAULT_CHARSET);
               break;
            }
           n=0;
           XtSetArg (args[n], XmNlabelString, string); n++;
           C_Set_Color_Button( args, n, window_defs );
           button[i] = XmCreatePushButton(rc, "Mouse",args,n);
           XtAddCallback(button[i], XmNactivateCallback, 
                         show_panel_callback, i);
           XmStringFree (string);
          }

          n=0;
          string = XmStringCreate ("Close", XmSTRING_DEFAULT_CHARSET);
          XtSetArg (args[n], XmNlabelString, string); n++;
          C_Set_Color_Button( args, n, window_defs );
          button[C_CURRENT_RD_NUMBER] = 
            XmCreatePushButton(rc, "Close", args, n);
          XtAddCallback(button[C_CURRENT_RD_NUMBER], 
            XmNactivateCallback, C_ide_quit, rd_bbdialog);
          XmStringFree (string);

          XtManageChildren(button, 6);
          XtManageChild(rc);
          XtManageChild(rd_bbdialog);
}

/*
 *        Procedure Name: assign_VD_callback 
 *          Return Value: void
 *       Input Variables: 
 *      Output Variables:
 *      Update Variables:
 *      Global Variables:
 *           Description: Call back function. Called when the user want to 
 *                        assign virtual device number to a drawingarea.
 */

void assign_VD_callback(w, call_data)
Widget w;
caddr_t call_data;
{
        extern void C_set_VD_change_mode();

        C_set_VD_change_mode(0);
}

/*
 *        Procedure Name:  change_device
 *          Return Value:
 *       Input Variables:  Device name
 *      Output Variables:
 *      Update Variables:
 *      Global Variables:
 *           Description:  Call back function. Change the current device
 *                         to the device name. Used when configure the
 *                         virtual device.
 */

void change_device(w,device,call_data)
Widget w;
int device;
caddr_t call_data;
{
        extern void C_set_device();

        C_set_device(device);

}

void reset_vd_config(w, client_data, call_data)
Widget w;
int client_data;
caddr_t call_data;
{
	extern XColor  IDE_set_color[C_MAX_RD_NUMBER];
        extern C_IDE_VD_configuration IDE_VD_configuration[C_MAX_VD_NUMBER];
        extern int C_check_allocate_device();
        extern void C_get_allocated_device();
        extern void C_init_vd_default();
        extern void C_init_selection();
        extern void C_update_selection_label();

        void display_orientation_buttons();

        Arg          args[10];
        int          n;

        int i;

        int allocated_device;

        if (C_check_allocate_device()) return;

        C_get_allocated_device(&allocated_device);

        C_init_vd_default(allocated_device, client_data);

        C_init_selection(allocated_device, client_data);

         n = 0;
         XtSetArg(args[n], XmNforeground,
          IDE_set_color[allocated_device].pixel); n++;
         XtSetValues(button1[0], args, n);

         n = 0;
         XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].pos2dx].pixel);  n++;
         XtSetValues(button1[1], args, n);

         n = 0;
         XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].pos2dy].pixel);  n++;
         XtSetValues(button1[2], args, n);

         n = 0;
         XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].pos3dx].pixel);  n++;
         XtSetValues(button2[0], args, n);

         n = 0;
         XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].pos3dy].pixel);  n++;
         XtSetValues(button2[1], args, n);

         n = 0;
         XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].pos3dz].pixel);  n++;
         XtSetValues(button2[2], args, n);

         n = 0;
         XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].ori].pixel); n++;
         XtSetValues(button3[0], args, n);

         n = 0;
         XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].oriazi].pixel); n++;
         XtSetValues(button5[0], args, n);

         n = 0;
         XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].oriele].pixel); n++;
         XtSetValues(button5[1], args, n);

         n = 0;
         XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].orirol].pixel); n++;
         XtSetValues(button5[2], args, n);
    
         n = 0;
         XtSetArg (args[n], XmNmenuHistory,
           mode_options[IDE_VD_configuration[client_data].orientation_mode]);
         n++;
         XtSetValues(mode_menu, args, n);
 
         display_orientation_buttons(client_data);

         C_update_selection_label();

         
}



/*
 *        Procedure Name:  change_config_callback
 *          Return Value:
 *       Input Variables:  An index indicate which group of parameters
 *                         should be changed.
 *      Output Variables:
 *      Update Variables:
 *      Global Variables:  IDE_set_color;
 *                         IDE_VD_configuration;
 *           Description:  Call back function. Used to change the device for
 *                         a group of parameters.
 */

void change_config_callback(w,index,call_data)
Widget w;
int    index;
caddr_t call_data;
{
        extern C_WindowDefs    window_defs;
        extern int C_check_allocate_device();
        extern void C_set_VD_configuration_2dx();
        extern void C_set_VD_configuration_2dy();
        extern void C_set_VD_configuration_3dx();
        extern void C_set_VD_configuration_3dy();
        extern void C_set_VD_configuration_3dz();
        extern void C_set_VD_configuration_ori();
        extern void C_set_VD_configuration_oriazi();
        extern void C_set_VD_configuration_oriele();
        extern void C_set_VD_configuration_orirol();

        extern XColor  IDE_set_color[C_MAX_RD_NUMBER]; 
        extern C_IDE_VD_configuration IDE_VD_configuration[C_MAX_VD_NUMBER];

        Arg          args[10];
        int          n;

        int allocated_device;

        if (C_check_allocate_device()) return;

/*      The meaning of the index should be refrenced to the configuration of
 *      virtual device allocation panel.
 */

        switch (index)
         {
           case 0: C_set_VD_configuration_2dx(&allocated_device);
                   break;
           case 1: C_set_VD_configuration_2dy(&allocated_device);
                   break;
           case 2: C_set_VD_configuration_3dx(&allocated_device);
                   break;
           case 3: C_set_VD_configuration_3dy(&allocated_device);
                   break;
           case 4: C_set_VD_configuration_3dz(&allocated_device);
                   break;
           case 5: C_set_VD_configuration_ori(&allocated_device);
                   break;
           case 6: C_set_VD_configuration_oriazi(&allocated_device);
                   break;
           case 7: C_set_VD_configuration_oriele(&allocated_device);
                   break;
           case 8: C_set_VD_configuration_orirol(&allocated_device);
                   break;
         }

           n = 0;
           XtSetArg(args[n], XmNforeground, 
                    IDE_set_color[allocated_device].pixel); n++;
           XtSetValues(w, args, n);

           n = 0;
           C_Set_Color_Button(args, n, window_defs );
           XtSetValues(button1[0], args, n);
}

/*
 *        Procedure Name:  VD_allocate_callback
 *          Return Value:  void
 *       Input Variables: 
 *      Output Variables:
 *      Update Variables:
 *      Global Variables:  window_defs;
 *                         IDE_VD_configuration;
 *                         IDE_set_color;                       
 *                         button.
 *           Description:  Pop up the Virtual device allocation panel
 */

void VD_allocate_callback(w,client_data,call_data)
Widget w;
int     client_data;
caddr_t call_data;
{
         extern int C_VD_allocate();
         extern C_WindowDefs    window_defs;

         void                   C_create_parameter();
         void                   C_create_device();
         void                   C_create_device_selection();
         void                   create_help_exit();

         Widget                 par_label, sel_label, dev_label;
         Widget                 multi_dialog;

         XmString               string;
         XmString               string1, string2;

         char char_string[20];

         Arg                    args[20];
         int                    n;

         int                    allocated_device; 

        if (!C_VD_allocate(&client_data, &redisplay)) return;

        n=0;
        XtSetArg( args[n], XmNautoUnmanage, False ); n++;
        C_Set_Color_Dialog( args, n, window_defs );
        if (client_data == 0) 
          string = XmStringCreate("Configuration    Default Virtual Device", 
                                    XmSTRING_DEFAULT_CHARSET);
        else  
          {
             sprintf(char_string, "Configuration    Virtual Device %d", 
                     client_data);
             string = XmStringCreate(char_string, XmSTRING_DEFAULT_CHARSET);
          }           
        XtSetArg( args[n], XmNdialogTitle, string ); n++;
        multi_dialog = XmCreateBulletinBoardDialog(w, 
                                      "Virtual Device Allocation", args, n);
        XmStringFree(string);

        n = 0;
        XtSetArg( args[n], XmNx, 10 ); n++;
        XtSetArg( args[n], XmNy, 10 ); n++;
        C_Set_Color_Label( args, n, window_defs );
        par_label = XmCreateLabelGadget(multi_dialog, "Parameters", args, n);
        XtManageChild(par_label);
        C_create_parameter(multi_dialog, client_data);

        n = 0;
        XtSetArg( args[n], XmNx, 310); n++;
        XtSetArg( args[n], XmNy, 10 ); n++;
        C_Set_Color_Label( args, n, window_defs );
        dev_label = XmCreateLabelGadget(multi_dialog, "Devices", args, n);
        XtManageChild(dev_label); 
        C_create_device(multi_dialog);
        
        create_help_exit(multi_dialog, client_data);

        XtManageChild(multi_dialog);

}

void create_help_exit(parent, client_data)
Widget parent;
int client_data;
{
        extern void C_create_selection_map();
        extern C_WindowDefs    window_defs;
        
        Widget frame, rc;
        Widget button[3];

        int n;
        Arg args[20];
        XmString string;

        n = 0;
        XtSetArg( args[n], XmNx, 10); n++;
        XtSetArg( args[n], XmNy, 260); n++;
        C_Set_Color_Frame( args, n, window_defs );
        frame = XmCreateFrame( parent, "frame", args, n );
        XtManageChild(frame);

        n =0;
        C_Set_Color_RowColumn( args, n, window_defs );
        XtSetArg(args[n], XmNorientation, XmVERTICAL); n++;
        XtSetArg(args[n], XmNpacking, XmPACK_COLUMN); n++;
        XtSetArg(args[n], XmNnumColumns, 3); n++;
        rc = XmCreateRowColumn(frame, "rc", args, n);
        XtManageChild (rc);

        n=0;
        XtSetArg( args[n], XmNx, 150 ); n++;
        XtSetArg( args[n], XmNy, 260); n++;
        C_Set_Color_Button( args, n, window_defs );
        button[0] = XmCreatePushButton(rc, "Selection", args, n );
        XtAddCallback(button[0], XmNactivateCallback,
                      C_create_selection_map, client_data);

        n=0;
        C_Set_Color_Button( args, n, window_defs );
        button[1] = XmCreatePushButton( rc, "Help", args, n);
        
        n=0;
        C_Set_Color_Button( args, n, window_defs );
        button[2] = XmCreatePushButton( rc, "Close", args, n);
        XtAddCallback(button[2], XmNactivateCallback, 
                      C_ide_quit_destroy, parent);
 
        XtManageChildren(button, 3);
}

static Widget ori_euler_rc[C_MAX_VD_NUMBER];
static Widget ori_other_rc[C_MAX_VD_NUMBER];

void display_orientation_buttons(device)
int device;
{
     extern C_IDE_VD_configuration
               IDE_VD_configuration[C_MAX_VD_NUMBER];

     if (IDE_VD_configuration[device].orientation_mode == C_EULER)
        {
         XtUnmanageChild ( ori_other_rc[device]);
         XtManageChild ( ori_euler_rc[device]);
        }
     else
        {
         XtUnmanageChild ( ori_euler_rc[device]);
         XtManageChild ( ori_other_rc[device]);
        }
}

void vd_orientation_mode_change(w, client_data, call_data)
Widget w;
int client_data;
caddr_t call_data;
{
     extern C_IDE_VD_configuration
               IDE_VD_configuration[C_MAX_VD_NUMBER];

     int device, mode;

        device = client_data/10;
        mode = client_data-device*10;

        IDE_VD_configuration[device].orientation_mode = mode;

        display_orientation_buttons(device);

}

void C_create_parameter(parent, client_data)
Widget parent;
int client_data;
{
        
        extern C_WindowDefs    window_defs;
        extern C_IDE_VD_configuration     
               IDE_VD_configuration[C_MAX_VD_NUMBER];     
        extern void C_get_allocated_device();
        extern XColor                 IDE_set_color[C_MAX_RD_NUMBER];

        Widget frame, form;
        Widget rc1, rc2, rc4;



        int n;
        XmString string;
        Arg args[20];

        n = 0;
        XtSetArg (args[n], XmNx, 10); n++;
        XtSetArg (args[n], XmNy, 40); n++;
        C_Set_Color_Frame( args, n, window_defs );
        frame = XmCreateFrame (parent, "frame", args, n);
        XtManageChild (frame);

        n = 0;
        C_Set_Color_RowColumn( args, n, window_defs );
        form = XmCreateForm(frame, "form", args, n);
        XtManageChild (form);

        n =0;
        C_Set_Color_RowColumn( args, n, window_defs );
        XtSetArg(args[n], XmNtopAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNleftPosition, 1); n++;
        XtSetArg(args[n], XmNrightPosition, 99); n++;
        XtSetArg(args[n], XmNtopPosition,1); n++;
        XtSetArg(args[n], XmNbottomPosition, 24); n++;
        XtSetArg(args[n], XmNorientation, XmHORIZONTAL); n++;
        XtSetArg(args[n], XmNpacking, XmPACK_COLUMN); n++;
        XtSetArg(args[n], XmNnumColumns, 1); n++;
        rc1 = XmCreateRowColumn(form, "rc1", args, n);
        XtManageChild (rc1);

        n =0;
        C_Set_Color_RowColumn( args, n, window_defs );
        XtSetArg(args[n], XmNtopAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNleftPosition, 1); n++;
        XtSetArg(args[n], XmNrightPosition, 99); n++;
        XtSetArg(args[n], XmNtopPosition,26); n++;
        XtSetArg(args[n], XmNbottomPosition, 50); n++;
        XtSetArg(args[n], XmNorientation, XmHORIZONTAL); n++;
        XtSetArg(args[n], XmNpacking, XmPACK_COLUMN); n++;
        XtSetArg(args[n], XmNnumColumns, 1); n++;
        rc2 = XmCreateRowColumn(form, "rc2", args, n);
        XtManageChild (rc2);

        n =0;
        C_Set_Color_RowColumn( args, n, window_defs );
        XtSetArg(args[n], XmNtopAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNleftPosition, 1); n++;
        XtSetArg(args[n], XmNrightPosition, 99); n++;
        XtSetArg(args[n], XmNtopPosition, 51); n++;
        XtSetArg(args[n], XmNbottomPosition, 74); n++;
        XtSetArg(args[n], XmNorientation, XmHORIZONTAL); n++;
        ori_other_rc[client_data] = XmCreateRowColumn(form, "rc", args, n);

        n =0;
        C_Set_Color_RowColumn( args, n, window_defs );
        XtSetArg(args[n], XmNtopAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNleftPosition, 1); n++;
        XtSetArg(args[n], XmNrightPosition, 99); n++;
        XtSetArg(args[n], XmNtopPosition, 51); n++;
        XtSetArg(args[n], XmNbottomPosition, 74); n++;
        XtSetArg(args[n], XmNorientation, XmVERTICAL); n++;
        XtSetArg(args[n], XmNpacking, XmPACK_COLUMN); n++;
        XtSetArg(args[n], XmNnumColumns, 3); n++;
        ori_euler_rc[client_data] = XmCreateRowColumn(form, "rc", args, n);

        n =0;
        C_Set_Color_RowColumn( args, n, window_defs );
        XtSetArg(args[n], XmNtopAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg(args[n], XmNleftPosition, 1); n++;
        XtSetArg(args[n], XmNrightPosition, 99); n++;
        XtSetArg(args[n], XmNtopPosition, 76); n++;
        XtSetArg(args[n], XmNbottomPosition,99); n++;
        XtSetArg(args[n], XmNorientation, XmHORIZONTAL); n++;
        rc4 = XmCreateRowColumn(form, "rc4", args, n);
        XtManageChild (rc4);

        n = 0;
        string = XmStringCreate ("Reset", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button1[0]=XmCreatePushButton(rc1, "Reset", args,n);
        XtAddCallback(button1[0], XmNactivateCallback, 
                      reset_vd_config, client_data);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Pos2d_x", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg(args[n], XmNforeground,
         IDE_set_color[IDE_VD_configuration[client_data].pos2dx].pixel);  n++;
        button1[1]=XmCreatePushButton(rc1, "Pos2d_x", args,n);
        XtAddCallback(button1[1], XmNactivateCallback,
                      change_config_callback, 0);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Pos2d_y", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg(args[n], XmNforeground,
         IDE_set_color[IDE_VD_configuration[client_data].pos2dy].pixel);  n++;
        button1[2]=XmCreatePushButton(rc1, "Pos2d_y", args,n);
        XtAddCallback(button1[2], XmNactivateCallback,
                      change_config_callback, 1);
        XmStringFree (string);

        XtManageChildren(button1, 3);

        n = 0;
        string = XmStringCreate ("Pos3d_x", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg(args[n], XmNforeground,
         IDE_set_color[IDE_VD_configuration[client_data].pos3dx].pixel);  n++;
        button2[0]=XmCreatePushButton(rc2, "Pos3d_x", args,n);
        XtAddCallback(button2[0], XmNactivateCallback,
                      change_config_callback, 2);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Pos3d_y", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg(args[n], XmNforeground,
         IDE_set_color[IDE_VD_configuration[client_data].pos3dy].pixel);  n++;
        button2[1]=XmCreatePushButton(rc2, "Pos3d_y", args,n);
        XtAddCallback(button2[1], XmNactivateCallback,
                      change_config_callback, 3);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Pos3d_z", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].pos3dz].pixel);  n++;
        button2[2]=XmCreatePushButton(rc2, "Pos3d_z", args,n);
        XtAddCallback(button2[2], XmNactivateCallback,
                      change_config_callback, 4);
        XmStringFree (string);

        XtManageChildren(button2, 3);

        n = 0; 
        string = XmStringCreate ("Orientation", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].ori].pixel); n++;
        button3[0] = XmCreatePushButton(ori_other_rc[client_data], 
                                       "Orientation", args, n);
        XtAddCallback(button3[0], XmNactivateCallback,
                      change_config_callback, 5);
        XmStringFree (string);

        XtManageChildren(button3, 1);

        n = 0;
        string = XmStringCreate ("Azimuth", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].oriazi].pixel); n++;
        button5[0] = XmCreatePushButton(ori_euler_rc[client_data], 
                                       "Orientation", args, n);
        XtAddCallback(button5[0], XmNactivateCallback,
                      change_config_callback, 6);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Elevator", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].oriele].pixel); n++;
        button5[1] = XmCreatePushButton(ori_euler_rc[client_data], 
                                        "Orientation", args, n);
        XtAddCallback(button5[1], XmNactivateCallback,
                      change_config_callback, 7);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Roll", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg(args[n], XmNforeground,
          IDE_set_color[IDE_VD_configuration[client_data].orirol].pixel); n++;
        button5[2] = XmCreatePushButton(ori_euler_rc[client_data],
                                        "Orientation", args, n);
        XtAddCallback(button5[2], XmNactivateCallback,
                      change_config_callback, 8);
        XmStringFree (string);

        XtManageChildren(button5, 3);


        n = 0;
        C_Set_Color_PullDown( args, n, window_defs );
        mode_pulldown = XmCreatePulldownMenu(rc4, "pulldown", args, n);

        n = 0;
        string = XmStringCreate ( "Euler Angles", 
                                   XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        mode_options[0] = XmCreatePushButtonGadget(
                           mode_pulldown, "", args, n);
        XtAddCallback (mode_options[0], XmNactivateCallback,
                  vd_orientation_mode_change, client_data*10+C_EULER);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ( "Quaternion",
                                        XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        mode_options[1] = XmCreatePushButtonGadget(
                           mode_pulldown, "", args, n);
        XtAddCallback (mode_options[1], XmNactivateCallback,
                 vd_orientation_mode_change, client_data*10+C_QUATERNION);
        XmStringFree (string);
 
        n = 0;
        string = XmStringCreate ( "Matrix", 
                                        XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        mode_options[2] = XmCreatePushButtonGadget(
                          mode_pulldown, "", args, n);
        XtAddCallback (mode_options[2], XmNactivateCallback,
                  vd_orientation_mode_change, client_data*10+C_MATRIX);
        XmStringFree (string);
        XtManageChildren(mode_options, 3);


        n = 0;
        string = XmStringCreate ("Orientation mode:", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        XtSetArg (args[n], XmNsubMenuId, mode_pulldown); n++;
        XtSetArg (args[n], XmNmenuHistory, 
           mode_options[IDE_VD_configuration[client_data].orientation_mode]); 
        n++;
        C_Set_Color_OptionMenu( args, n, window_defs );
        mode_menu = XmCreateOptionMenu( rc4, "mode_menu", args, n);
        XtManageChild(mode_menu);
        XmStringFree (string);

        display_orientation_buttons(client_data);

}

void C_create_device(parent)
Widget parent;
{
        extern C_WindowDefs    window_defs;
        extern C_IDE_VD_configuration
               IDE_VD_configuration[C_MAX_VD_NUMBER];
        extern void C_get_allocated_device();
        extern XColor                 IDE_set_color[C_MAX_RD_NUMBER];

        Widget frame; 

        Widget rc;

        Widget button[5];

        int n;
        XmString string;
        Arg args[20];

        n = 0;
        XtSetArg (args[n], XmNx, 310); n++;
        XtSetArg (args[n], XmNy, 40); n++;
        C_Set_Color_Frame( args, n, window_defs );
        frame = XmCreateFrame (parent, "frame", args, n);
        XtManageChild (frame);

        n = 0;
        C_Set_Color_RowColumn( args, n, window_defs );
        XtSetArg(args[n], XmNorientation, XmHORIZONTAL); n++;
        XtSetArg( args[n], XmNpacking, XmPACK_COLUMN); n++;
        XtSetArg(args[n], XmNnumColumns, 5); n++;
        rc = XmCreateRowColumn(frame, "rc", args, n);
        XtManageChild (rc);

        n = 0;
        string = XmStringCreate ("Mouse", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg (args[n], XmNforeground, IDE_set_color[C_MOUSE].pixel);n++;
        button[0]=XmCreatePushButton(rc, "Mouse", args,n);
        XtAddCallback(button[0], XmNactivateCallback, change_device,C_MOUSE);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Keyboard", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg (args[n], XmNforeground, IDE_set_color[C_KEYBOARD].pixel);n++;
        button[1]=XmCreatePushButton(rc, "Keyboard", args,n);
        XtAddCallback(button[1], XmNactivateCallback, change_device,C_KEYBOARD);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Isotrak", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg (args[n], XmNforeground,
                  IDE_set_color[C_ISOTRAK].pixel);n++;
        button[2] = XmCreatePushButton(rc, "Isotrak", args,n);
/*
        XtAddCallback(button[2],XmNactivateCallback, change_device,C_ISOTRAK);
*/
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Spaceball", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg (args[n], XmNforeground,
                  IDE_set_color[C_SPACEBALL].pixel);n++;
        button[3]=XmCreatePushButton(rc, "Spaceball", args,n);
        XtAddCallback(button[3], XmNactivateCallback,
                      change_device,C_SPACEBALL);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("DataGlove", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        XtSetArg (args[n], XmNforeground,
                  IDE_set_color[C_DATAGLOVE].pixel); n++;
        button[4]=XmCreatePushButton(rc, "DataGlove", args,n);
/*
        XtAddCallback(button[4], XmNactivateCallback,
                      change_device, C_DATAGLOVE);
*/
        XmStringFree (string);

        XtManageChildren(button, 5);
}

void C_create_device_selection(parent)
Widget parent;
{
        extern C_WindowDefs    window_defs;
        extern C_IDE_VD_configuration
               IDE_VD_configuration[C_MAX_VD_NUMBER];
        extern void C_get_allocated_device();
        extern XColor                 IDE_set_color[C_MAX_RD_NUMBER];

        Widget frame, rc;

        Widget button[4];

        int n;
        XmString string;
        Arg args[20];

        n = 0;
        XtSetArg (args[n], XmNx, 10); n++;
        XtSetArg (args[n], XmNy, 300); n++;
        C_Set_Color_Frame( args, n, window_defs );
        frame = XmCreateFrame (parent, "frame", args, n);
        XtManageChild (frame);

        n =0;
        C_Set_Color_RowColumn( args, n, window_defs );
        XtSetArg(args[n], XmNorientation, XmVERTICAL); n++;
        XtSetArg(args[n], XmNpacking, XmPACK_COLUMN); n++;
        XtSetArg(args[n], XmNnumColumns, 4); n++;
        rc = XmCreateRowColumn(frame, "rc", args, n);
        XtManageChild (rc);

        n = 0;
        string = XmStringCreate ("Mouse", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button[0]=XmCreatePushButton(rc, "Mouse", args,n);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Keyboard", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button[1]=XmCreatePushButton(rc, "Keyboard", args,n);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Spaceball", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button[2] =XmCreatePushButton(rc, "Spaceball", args,n);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("DataGlove", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button[3]=XmCreatePushButton(rc, "DataGlove", args,n);
        XmStringFree (string);

        XtManageChildren(button, 4);
}

void C_ide_output_unit_toggle(w, client_data, call_data)
Widget w;
int client_data;
XmToggleButtonCallbackStruct *call_data;
{
  extern C_IDE_simulation_method C_ide_simulation_method;

  if (call_data->set)
     if (client_data == 0) {
        C_ide_simulation_method.unit = C_PIXEL;
     }
     else {
        C_ide_simulation_method.unit = C_NORMALIZED;
     }
}

void C_ide_output_mode_toggle(w, client_data, call_data)
Widget w;
int client_data;
XmToggleButtonCallbackStruct *call_data;
{
  extern C_IDE_simulation_method C_ide_simulation_method;

  if (call_data->set)
     if (client_data == 0) {
        C_ide_simulation_method.mode = C_SCREEN;
                           }
     else {
        C_ide_simulation_method.mode = C_WINDOW;
          }
}

void C_ide_create_mode(parent)
Widget parent;
{
    
    extern C_WindowDefs     window_defs;

    Widget flabel;
    Widget frame;
    Widget radiobox;
    Widget toggle[2];
    Arg args[20];
    XmString string;

    int n,i;

    n = 0;
    XtSetArg( args[n], XmNx, 10 ); n++;
    XtSetArg( args[n], XmNy, 10 ); n++;
    string = XmStringCreate ("Active Mode:", XmSTRING_DEFAULT_CHARSET);
    XtSetArg (args[n], XmNlabelString, string); n++;
    C_Set_Color_Label( args, n, window_defs );
    flabel = XmCreateLabelGadget(parent, "Output Unit:", args, n);
    XtManageChild(flabel);
    XmStringFree (string);

    n = 0;
    XtSetArg (args[n], XmNx, 120);  n++;
    XtSetArg (args[n], XmNy, 10);  n++;
    C_Set_Color_Frame( args, n, window_defs );
    frame = XmCreateFrame (parent, "frame", args, n);
    XtManageChild (frame);

    n = 0;
    XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
    C_Set_Color_RadioBox( args, n, window_defs );
    radiobox = XmCreateRadioBox (frame, "radiobox", args, n);
    XtManageChild (radiobox);

    /* create the toggles */

    n = 0;
    string = XmStringCreate ("Screen", XmSTRING_DEFAULT_CHARSET);
    XtSetArg (args[n], XmNlabelString, string); n++;
    C_Set_Color_ToggleButton( args, n, window_defs );
    toggle[0] = XtCreateWidget ("Screen",xmToggleButtonWidgetClass,
                                radiobox, args, n);
    XtAddCallback(toggle[0], XmNvalueChangedCallback,
                             C_ide_output_mode_toggle, 0);
    XmStringFree (string);

    n = 0;
    string = XmStringCreate ("Window        ", XmSTRING_DEFAULT_CHARSET);
    XtSetArg (args[n], XmNlabelString, string); n++;
    C_Set_Color_ToggleButton( args, n, window_defs );
    toggle[1] = XtCreateWidget ("Screen",xmToggleButtonWidgetClass,
                                radiobox, args, n);
    XtAddCallback(toggle[1], XmNvalueChangedCallback,
                             C_ide_output_mode_toggle, 1);
    XmStringFree (string);

    
    n = 0;
    XtSetArg( args[n], XmNset, TRUE ); n++;
    XtSetValues (toggle[0], args, n); /* "SCREEN" is initially set */

    XtManageChildren (toggle, 2);
}

void C_ide_create_unit(parent)
Widget parent;
{
    extern C_WindowDefs     window_defs;

    Widget flabel;
    Widget frame;
    Widget radiobox;
    Widget toggle[2];
    Arg args[20];
    XmString string;

    int n,i;

    n = 0;
    XtSetArg( args[n], XmNx, 10 ); n++;
    XtSetArg( args[n], XmNy, 70 ); n++;
    string = XmStringCreate ("Output Unit:", XmSTRING_DEFAULT_CHARSET);
    XtSetArg (args[n], XmNlabelString, string); n++;
    C_Set_Color_Label( args, n, window_defs );
    flabel = XmCreateLabelGadget(parent, "Output Unit:", args, n);
    XtManageChild(flabel);
    XmStringFree (string);

    n = 0;
    XtSetArg (args[n], XmNx, 120);  n++;
    XtSetArg (args[n], XmNy, 70);  n++;
    C_Set_Color_Frame( args, n, window_defs );
    frame = XmCreateFrame (parent, "frame", args, n);
    XtManageChild (frame);
 
    n = 0;
    XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
    C_Set_Color_RadioBox( args, n, window_defs );
    radiobox = XmCreateRadioBox (frame, "radiobox", args, n);
    XtManageChild (radiobox);

    /* create the toggles */

    n = 0;
    string = XmStringCreate ("Pixel ", XmSTRING_DEFAULT_CHARSET);
    XtSetArg (args[n], XmNlabelString, string); n++;
    C_Set_Color_ToggleButton( args, n, window_defs );
    toggle[0] = XtCreateWidget ("Pixel ",xmToggleButtonWidgetClass,
                                radiobox, args, n);
    XtAddCallback(toggle[0], XmNvalueChangedCallback,
                             C_ide_output_unit_toggle, 0);
    XmStringFree (string);

    n = 0;
    string = XmStringCreate ("Normalized   ", XmSTRING_DEFAULT_CHARSET);
    XtSetArg (args[n], XmNlabelString, string); n++;
    C_Set_Color_ToggleButton( args, n, window_defs );
    toggle[1] = XtCreateWidget ("Normalized     ",xmToggleButtonWidgetClass,
                                radiobox, args, n);
    XtAddCallback(toggle[1], XmNvalueChangedCallback,
                             C_ide_output_unit_toggle, 1);
    XmStringFree (string);


    n = 0;
    XtSetArg( args[n], XmNset, TRUE ); n++;
    XtSetValues (toggle[1], args, n); /* "Window" is initially set */

    XtManageChildren (toggle, 2);
}
    
    


        
/*
 *        Procedure Name: VD_callback 
 *          Return Value: void
 *       Input Variables: 
 *      Output Variables:
 *      Update Variables:
 *      Global Variables: window_defs;
 *                        IDE_current_VD_number;
 *           Description: Pop up the virtual device board
 */

void VD_callback(w, call_data)
Widget w;
caddr_t call_data;
{
        extern C_WindowDefs    window_defs;
        extern void            C_set_VD_change_mode();
        extern int             IDE_current_VD_number;
        extern Widget          vv_toplevel;

        Widget  rc, button[C_MAX_VD_NUMBER+1];
        Widget  rc1, button1[C_MAX_VD_NUMBER+1];
        static  Widget  parent;
        static  Widget  vd_bbdialog;

        Widget frame1, frame2;

        Arg             args[10];       /* Argument List */
        int             n;              /* Argument Count */
        XmString        string, string1;

        int i;

        char       temp_string[10];


        if (w!=NULL) parent = vv_toplevel;

        if ((redisplay == 0) && (w==NULL))
          return;

        C_set_VD_change_mode(-1); 
        redisplay = 0;

        if (vd_bbdialog) XtDestroyWidget(vd_bbdialog);

        n=0;
        XtSetArg( args[n], XmNautoUnmanage, False ); n++;
        C_Set_Color_Dialog( args, n, window_defs );
        string = XmStringCreate("Virtual Device Control", 
                       XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNdialogTitle, string); n++;
        vd_bbdialog = XmCreateBulletinBoardDialog(parent,
                                                  "Virtual Device", args, n);
        XmStringFree(string);

        C_ide_create_unit(vd_bbdialog);
  
        C_ide_create_mode(vd_bbdialog);

        n=0;
        XtSetArg( args[n], XmNx, 10 ); n++;
        XtSetArg( args[n], XmNy, 140 ); n++;
        C_Set_Color_Frame( args, n, window_defs );
        frame1 = XmCreateFrame( vd_bbdialog, "frame", args, n );
        XtManageChild(frame1);
 
        n =0;
        XtSetArg( args[n], XmNorientation, XmVERTICAL); n++;
        XtSetArg( args[n], XmNpacking, XmPACK_COLUMN); n++;
        XtSetArg( args[n], XmNnumColumns, 4); n++;
        C_Set_Color_RowColumn( args, n, window_defs );
        rc = XmCreateRowColumn (frame1, "rc", args, n);


        n=0;
        string = XmStringCreate ("Default", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button[0] = XmCreatePushButton(rc, "Default", args,n);
        XtAddCallback(button[0], XmNactivateCallback, VD_allocate_callback, 0);
        XmStringFree (string);

        for (i=1; i<=IDE_current_VD_number; i++)
        {
          n=0;
          sprintf(temp_string, "VD %d", i);
          string = XmStringCreate (temp_string, XmSTRING_DEFAULT_CHARSET);
          XtSetArg (args[n], XmNlabelString, string); n++;
          C_Set_Color_Button( args, n, window_defs );
          button[i] = XmCreatePushButton(rc, "VD", args,n);
          XtAddCallback(button[i], XmNactivateCallback, 
                        VD_allocate_callback, i);
          XmStringFree (string);
         }

        
        n = 0;
        XtSetArg( args[n], XmNx, 10 ); n++;
        XtSetArg( args[n], XmNy, (int)(IDE_current_VD_number/4)*50+200); n++;
        C_Set_Color_Frame( args, n, window_defs );
        frame2 = XmCreateFrame( vd_bbdialog, "frame", args, n );
        XtManageChild(frame2);

        n=0;
        XtSetArg( args[n], XmNorientation, XmHORIZONTAL); n++;
        XtSetArg( args[n], XmNpacking, XmPACK_COLUMN); n++;
        XtSetArg( args[n], XmNresizeWidth, TRUE ); n++;
        XtSetArg( args[n], XmNresizeHeight, TRUE ); n++;
        C_Set_Color_RowColumn( args, n, window_defs );
        rc1 = XmCreateRowColumn (frame2, "rc", args, n);


        n=0;
        string = XmStringCreate ("Assign", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button1[0] = XmCreatePushButton(rc1, "Assign VD", args,n);
        XtAddCallback(button1[0], XmNactivateCallback, 
                      assign_VD_callback,NULL);
        XmStringFree (string);

        n=0;
        string = XmStringCreate ("Add", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button1[1] = XmCreatePushButton(rc1, "Add VD", args,n);
        XtAddCallback(button1[1], XmNactivateCallback, 
                      VD_allocate_callback, C_MAX_VD_NUMBER+1);
        XmStringFree (string);

        n=0;
        string = XmStringCreate ("Help", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button1[2] = XmCreatePushButton(rc1, "help",args,n);
        XmStringFree (string);

        n=0;
        string = XmStringCreate ("Close", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button1[3] = XmCreatePushButton(rc1, "Close",args,n);
        XtAddCallback(button1[3], XmNactivateCallback,
                      C_ide_quit, vd_bbdialog);
        XmStringFree (string);

        XtManageChildren(button, IDE_current_VD_number+1);
        XtManageChild(rc);
        XtManageChildren(button1, 4);
        XtManageChild(rc1);
        XtManageChild(vd_bbdialog);

}

/*
 *        Procedure Name: create_ide_window 
 *          Return Value:
 *       Input Variables: 
 *      Output Variables:
 *      Update Variables:
 *      Global Variables: vv_toplevel;
 *                        ide_bbdialog;
 *                        window_defs;
 *           Description: Pop up the main Input device menu.
 */

void create_ide_window()
{
        extern void X_define_token_color();
        extern Widget vv_toplevel;
        extern Widget ide_bbdialog;
        extern C_WindowDefs    window_defs;

        Widget button[3];
        Widget rc;

        XmString string;
        Arg             args[20];       /* Argument List */
        int             n;

        initialize_RD_panel();
        X_define_token_color(XtDisplay(vv_toplevel));
  
        n=0;
        XtSetArg( args[n], XmNautoUnmanage, False ); n++;
        C_Set_Color_Dialog( args, n, window_defs );
        string = XmStringCreate("Input Devices", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNdialogTitle, string); n++;
        ide_bbdialog = XmCreateBulletinBoardDialog(vv_toplevel,
                                "input device", args, n);

        n=0;
        XtSetArg( args[n], XmNx, 10 ); n++;
        XtSetArg( args[n], XmNy, 10 ); n++;
        XtSetArg( args[n], XmNorientation, XmVERTICAL); n++;
        XtSetArg(args[n], XmNorientation, XmVERTICAL); n++;
        XtSetArg(args[n], XmNpacking, XmPACK_COLUMN); n++;
        XtSetArg(args[n], XmNnumColumns, 1); n++;
        C_Set_Color_RowColumn( args, n, window_defs );
        rc = XmCreateRowColumn(ide_bbdialog, "Input Device", args, n);

        n = 0;
        string = XmStringCreate ("Device Panel", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button[0]=XmCreatePushButton(rc, "Device Panel", args,n);
        XtAddCallback(button[0], XmNactivateCallback, RD_callback, NULL);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Virtual Device", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button[1]=XmCreatePushButton(rc, "Virtual Device", args,n);
        XtAddCallback(button[1],XmNactivateCallback, VD_callback, NULL);
        XmStringFree (string);

        n = 0;
        string = XmStringCreate ("Close", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_Button( args, n, window_defs );
        button[2]=XmCreatePushButton(rc, "Close", args,n);
        XtAddCallback(button[2], XmNactivateCallback, C_ide_quit, ide_bbdialog);
        XmStringFree(string);

        XtManageChildren(button, 3);
        XtManageChild(rc);
 }
 

/*
 *        Procedure Name:  create_dialog
 *          Return Value:  void
 *       Input Variables:  Widget parent
 *                         char *text
 *                         char *label
 *      Output Variables:  none
 *      Update Variables:  none
 *      Global Variables:  none
 *           Description:  Given text to be displayed, label for the dialog, and a
 *                         parent, it creates an informationdialog and puts it on the screen
 */
void create_dialog(text,parent,label)
Widget parent;
char *text,*label;
{
        extern C_WindowDefs    window_defs;
        Widget dialog;
        XmString tstring,lstring;
        Arg args[20];
        int n;

        n = 0;
        tstring = XmStringCreate (text, XmSTRING_DEFAULT_CHARSET);
        XtSetArg(args[n], XmNmessageString,tstring); n++;
        lstring = XmStringCreate(label, XmSTRING_DEFAULT_CHARSET);
        XtSetArg( args[n], XmNdialogTitle, lstring ); n++;
        C_Set_Color_Dialog(args,n,window_defs);
        dialog = XmCreateInformationDialog(parent,"info",args,n);
        XtManageChild(dialog);
        XtPopup(XtParent(dialog),XtGrabNone);
        XmStringFree(tstring);
        XmStringFree(lstring);
}

