/****************************************************************************/
/*                                                                          */
/*  VolVis is a volume visualization system for investigating, manipulating */
/*  and rendering geometric and volumetric data.                            */
/*                                                                          */
/*  Copyright (C) 1993 by the Research Foundation of the State University   */
/*                            of New York                                   */
/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 1, or (at your option)     */
/*  any later version.                                                      */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this program; if not, write to the Free Software             */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/*                                                                          */
/*  For information on VolVis, contact us at:                               */
/*                                                                          */
/*                volvis@cs.sunysb.edu                         (email)      */
/*                                                                          */
/*                Lisa Sobierajski & Ricardo Avila             (US Mail)    */
/*                Department of Computer Science                            */
/*                State University of New York at Stony Brook               */
/*                Stony Brook, New York  11794-4400                         */
/*                                                                          */
/****************************************************************************/




/*
 *		File:  C_mea_voxel.c
 *	      Author:  Hui Chen
 *		Date:  Feb. 6, 1992
 * 	 Description:  This file contains routine: voxel_method(),
 *		       which calculates the surface area by counting 
 *		       the number of voxels on the surface of the object.	
 *		       It is device_independent.
 * Modification History:
 * 
 *	who?		when?		why?
 * -----------------------------------------------
 *
 */


#include <math.h>
#include "C_volvis.h"
#include "C_measure.h"

/****************************************/
/*					
/*	Procedure Name:  voxel_method
/*	  Return Value:  int 
/*     Input Variables:  C_Voxel_8bit	*data	- pointer to raw data array
/*			 int		xsize	- x resolution of data array
/*			 int		ysize	- y resolution of data array
/*			 int		zsize	- z resolution of data array
/*			 int		low	- low threshold value used for
/*						  segmentation
/*			 int		high	- high threshold value used for
/*						  segmentation
/*                       float rx, ry, rz       - units/voxels ratios in
/*                                                x, y, z axis, respectively
/*    Output Variables:  none
/*    Update Variables:  none
/*    Global Variables:  none 
/*     	   Description:  This routine calculates the surface area by counting 
/*		         the number of voxels on the surface of the object.	
 */


int voxel_method(data,xsize,ysize,zsize,low,high,rx,ry,rz)
C_Voxel_8bit	*data;
int xsize,ysize,zsize,low,high;
float rx, ry, rz;
{
	C_Voxel_8bit *dptr;		/* pointer to raw data		*/
	int	xstep;		/* increment for a step in x	*/
	int	ystep;		/* increment for a step in y	*/
	int	zstep;		/* increment for a step in z	*/
	int	x, y, z;	/* integer position loop variables */
	int	sur_count;	/* count of voxels on the surface  */

	/* set up the increaments */

	zstep = xsize*ysize;
	ystep = xsize;
	xstep = 1;

	/* initialize the count to 0 */

	sur_count = 0;

	/* now start counting the voxels on the surface */

	for (z=1; z<(zsize-1); z++) {
	    for (y=1; y<(ysize-1); y++)
	    {
		/* initialize dptr for this row */
		dptr = data + (z*zstep) + (y*ystep) + xstep;
		for (x=1; x<(xsize-1); x++)
		{
		    if (C_In_Range( *dptr, low, high))
		       if ( C_On_Surface(dptr,low,high,
				         xstep,ystep,zstep)) {
			   sur_count++;
			}
		    dptr += xstep;
		}
	    }
	}
	return(sur_count);
}

/****************************************/
/*					
/*	Procedure Name:  s_seed_voxel
/*	  Return Value:  int 
/*     Input Variables:  C_Voxel_8bit	*data	- pointer to raw data array
/*			 int		xsize	- x resolution of data array
/*			 int		ysize	- y resolution of data array
/*			 int		zsize	- z resolution of data array
/*			 int		low	- low threshold value used for
/*						  segmentation
/*			 int		high	- high threshold value used for
/*						  segmentation
/*                       float rx, ry, rz       - units/voxels ratios in
/*                                                x, y, z axis, respectively
/*			 C_MeaVoxelList  *mea_surface_voxels[] - surface list
/*    Output Variables:  none
/*    Update Variables:  none
/*    Global Variables:  none 
/*     	   Description:  This routine calculates the surface area by counting 
/*		         the number of voxels on the surface list.	
 */

int s_seed_voxel(mea_surface_voxels, data,xsize,ysize,zsize,low,high,rx,ry,rz)
C_MeaVoxelList  *mea_surface_voxels[C_MEA_HASH_MAX];
C_Voxel_8bit  *data;
int      xsize,ysize,zsize;
int      low, high;
float    rx, ry, rz;
{
	C_MeaVoxelList  *vptr, *tempptr;
	int	sur_count;	/* count of voxels on the surface  */
	int     i;
	
	/* initialize the count to 0 */
	sur_count = 0;

	for (i=0; i<C_MEA_HASH_MAX; i++) {
            vptr = mea_surface_voxels[i];
            while (vptr != NULL) {
		sur_count++;
		tempptr = vptr;
                vptr = vptr->next_voxel;
                free(tempptr);
	    }
	}
	return(sur_count);
}

/****************************************/
/*					
/*	Procedure Name:  mea_distance
/*	  Return Value:  float
/*     Input Variables:  
/*			 C_FPosition s1 - intersection point
/*			 C_FPosition s2 - ray origin 
/*    Output Variables:  none
/*    Update Variables:  none
/*    Global Variables:  none 
/*     	   Description:  This routine calculates the distance between
/*			 a ray origin and an intersection point. 
*/

float mea_distance(s1, s2)
C_FPosition s1;
C_FPosition s2;
{
	float	distance;

	distance = (float) sqrt((double)(C_Square(s1.x - s2.x) 
				       + C_Square(s1.y - s2.y) 
				       + C_Square(s1.z - s2.z)));

	return(distance);
}
