/****************************************************************************/
/*                                                                          */
/*  VolVis is a volume visualization system for investigating, manipulating */
/*  and rendering geometric and volumetric data.                            */
/*                                                                          */
/*  Copyright (C) 1993 by the Research Foundation of the State University   */
/*                            of New York                                   */
/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 1, or (at your option)     */
/*  any later version.                                                      */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this program; if not, write to the Free Software             */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/*                                                                          */
/*  For information on VolVis, contact us at:                               */
/*                                                                          */
/*                volvis@cs.sunysb.edu                         (email)      */
/*                                                                          */
/*                Lisa Sobierajski & Ricardo Avila             (US Mail)    */
/*                Department of Computer Science                            */
/*                State University of New York at Stony Brook               */
/*                Stony Brook, New York  11794-4400                         */
/*                                                                          */
/****************************************************************************/



 
/*
 *		File:  C_mea_v_cell2.c
 *	      Author:  Hui Chen
 *		Date:  Dec. 7, 1991
 * 	 Description:  This file contains routines: v_cell_method2().
 *		       This is device_independent.
 *
 *		       v_cell_method2() is the routine which calculates the
 *		       volume by "marching cubes" algorithm.
 *
 * Modification History:
 * 
 *	who?		when?		why?
 * -----------------------------------------------
 *
 */


#include "C_volvis.h"
#include "C_measure.h"

/****************************************/
/*					
/*	Procedure Name:  v_cell_method2
/*	  Return Value:  none
/*     Input Variables:  C_Voxel_8bit	*data	- pointer to raw data array
/*			 int		xsize	- x resolution of data array
/*			 int		ysize	- y resolution of data array
/*			 int		zsize	- z resolution of data array
/*			 int		low	- low threshold value used for
/*						  segmentation
/*			 int		high	- high threshold value used for
/*						  segmentation
/*                       float rx, ry, rz       - units/voxels ratios in
/*                                                x, y, z axis, respectively
/*    Output Variables:  none
/*    Update Variables:  none
/*    Global Variables:  none
/*     	   Description:  This routine calculates the volume 
/*			 by "marching cubes" algorithm.
 */


void v_cell_method2(data,xsize,ysize,zsize,low,high,rx,ry,rz)
C_Voxel_8bit  *data;
int	 xsize,ysize,zsize;
int      low, high;
float	 rx, ry, rz;
{
	extern void rotate_cube();	/* rotate the cell(cube) */
	extern float volume_cal2();     /* calculate the volume inside the cell */

	extern int cube_table[256][4];

	C_Voxel_8bit *dptr;		/* pointer to raw data		*/
	int	xstep;		/* increment for a step in x	*/
	int	ystep;		/* increment for a step in y	*/
	int	zstep;		/* increment for a step in z	*/
	int	x, y, z;	/* integer position loop variables */
	int	cell_index;	/* the index value of the cell     */
	int	i;		/* loop control variable */

	float   vol_one_cell; 	/* volume of a single full ocupied cell */

	float	volume;		/* variable contains calculated volume */
	

	vol_one_cell = rx * ry * rz * 6.0; /* since we will do /6.0 to the
                                              sum result at the end */

	/* set up the increaments */

	zstep = xsize*ysize;
	ystep = xsize;
	xstep = 1;

	volume = 0.;		/* initialization */

	/* now start calculating the volume */

	if ((rx == ry) && (rx == rz)) {
	    for (z=0; z<(zsize-1); z++)
	        for (y=0; y<(ysize-1); y++)
	        {		
		/* initialize dptr for this row */

		    dptr = data + (z*zstep) + (y*ystep);

		    for (x=0; x<(xsize-1); x++)
		    {
		    /* calculate the index value of the cell depending on
		       the intensities of the eight vertices
		    */
		        index_cal((dptr),low,high,xstep,ystep,zstep,cell_index);

		    /* in case of surface cut through the cell,
		       rotate the cell to match one of the standard 
		       patterns and then calculate the volume 
		       inside the cell according the way specified for
		       that standard pattern
		    */
		        if (cell_index == 255) 
			    volume += vol_one_cell;
			else if (cell_index != 0)
		        {
			/* rotate the cell */
			    rotate_cube(cube_table[cell_index][0],
				    (dptr),
				    (dptr+xstep),
				    (dptr+ystep+xstep), 
				    (dptr+ystep),
				    (dptr+zstep),
				    (dptr+zstep+xstep),
				    (dptr+zstep+ystep+xstep),
				    (dptr+zstep+ystep));

			/* calculate the volume */
			    if (cube_table[cell_index][3] == 0)
			       volume += volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  rx, ry, rz);
			    else
			       volume += (vol_one_cell - 
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  rx, ry, rz));
		        }
		        dptr += xstep;
		    }
	        }
	}
	else {
	    for (z=0; z<(zsize-1); z++)
	        for (y=0; y<(ysize-1); y++)
	        {		
		/* initialize dptr for this row */

		    dptr = data + (z*zstep) + (y*ystep);

		    for (x=0; x<(xsize-1); x++)
		    {
		        index_cal((dptr),low,high,xstep,ystep,zstep,cell_index);

		    /* in case of surface cut through the cell,
		       rotate the cell to match one of the standard 
		       patterns and then calculate the volume 
		       inside the cell according the way specified for
		       that standard pattern
		    */
		        if (cell_index == 255) 
			    volume += vol_one_cell;
			else if (cell_index != 0)
		        {
			/* rotate the cell */
			    rotate_cube(cube_table[cell_index][0],
				    (dptr),
				    (dptr+xstep),
				    (dptr+ystep+xstep), 
				    (dptr+ystep),
				    (dptr+zstep),
				    (dptr+zstep+xstep),
				    (dptr+zstep+ystep+xstep),
				    (dptr+zstep+ystep));

			/* calculate the volume */
			    if (cube_table[cell_index][3] == 0) {
			    switch(cube_table[cell_index][1]) {
				case 0: volume += 
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  rx, ry, rz);
					break;
				case 1: volume += 
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  ry, rx, rz);
					break;
				case 2: volume += 
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  rx, rz, ry);
					break;
				case 3: volume += 
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  rz, ry, rx);
					break;
				case 4: volume += 
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  ry, rz, rx);
					break;
				case 5: volume += 
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  rz, rx, ry);
					break;
				default: 
					printf("error in data file: axis_case is invalid.\n");
					break;
			    }
			    }
			    else {
			    switch(cube_table[cell_index][1]) {
				case 0: volume += (vol_one_cell -
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  rx, ry, rz));
					break;
				case 1: volume += (vol_one_cell -
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  ry, rx, rz));
					break;
				case 2: volume += (vol_one_cell -
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  rx, rz, ry));
					break;
				case 3: volume += (vol_one_cell -
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  rz, ry, rx));
					break;
				case 4: volume += (vol_one_cell -
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  ry, rz, rx));
					break;
				case 5: volume += (vol_one_cell -
					 volume_cal2(low, high, 
						     cube_table[cell_index][2],
						  rz, rx, ry));
					break;
				default: 
					printf("error in data file: axis_case is invalid.\n");
					break;
			    }
			    }
		        }
		        dptr += xstep;
		    }
	        }
	}
        /* since I didn't do "devided by 6. i.e. /6." in each tetrahedron
           volume calculations in order to save time, I need to do it for
           the sum here.
        */
	volume = volume / 6.0;
	printf("Volume = %f in cube of unit.\n", volume);
}


