/****************************************************************************/
/*                                                                          */
/*  VolVis is a volume visualization system for investigating, manipulating */
/*  and rendering geometric and volumetric data.                            */
/*                                                                          */
/*  Copyright (C) 1993 by the Research Foundation of the State University   */
/*                            of New York                                   */
/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 1, or (at your option)     */
/*  any later version.                                                      */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this program; if not, write to the Free Software             */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/*                                                                          */
/*  For information on VolVis, contact us at:                               */
/*                                                                          */
/*                volvis@cs.sunysb.edu                         (email)      */
/*                                                                          */
/*                Lisa Sobierajski & Ricardo Avila             (US Mail)    */
/*                Department of Computer Science                            */
/*                State University of New York at Stony Brook               */
/*                Stony Brook, New York  11794-4400                         */
/*                                                                          */
/****************************************************************************/



/*
 *                  File: C_typedefs.h
 *                Author: Lisa Sobierajski & Rick Avila
 *                  Date:
 *           Description:
 *  Modification History:
 *
 *         who?         when?           why?
 *    -----------------------------------------------------------
 *
 */

#ifdef C_MOTIF
#include <Xm/Xm.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#endif
#ifdef C_OL
#include <xview/xview.h>
#include <X11/Xlib.h>
#endif
/************************************************************************/
/*                                                                      */
/*	                      Typedefs					*/
/*                                                                      */
/************************************************************************/


/*	Definition of a byte						*/
typedef unsigned char   C_Byte;

/*	A C_GraphWin, for Starbase, is simple an integer, used to store */
/*	the starbase file descriptor for a window			*/
#ifdef C_STARBASE
typedef Display		C_X_Display;
typedef Window		C_X_Window;
typedef GC		C_X_GC;
typedef Pixmap		C_X_Pixmap;
#endif

#ifdef C_GL
typedef Display		C_X_Display;
typedef Window		C_X_Window;
typedef GC		C_X_GC;
typedef Pixmap		C_X_Pixmap;
#endif

#ifdef C_X_GRAPHICS
typedef Display		C_X_Display;
typedef Window		C_X_Window;
typedef GC		C_X_GC;
typedef Pixmap		C_X_Pixmap;
#endif


#ifndef C_STARBASE
#ifndef C_GL
#ifndef C_X_GRAPHICS
typedef int		C_X_Display;
typedef int		C_X_Window;
typedef int		C_X_GC;
typedef int		C_X_Pixmap;
#endif
#endif
#endif

#ifdef C_MOTIF
typedef Widget		C_UIWin;
typedef Colormap 	C_Colormap;
#endif
#ifdef C_OL
typedef Xv_Window	C_UIWin;
typedef Colormap 	C_Colormap;
#endif

#ifndef C_MOTIF
#ifndef C_OL
typedef int	C_UIWin;
typedef int 	C_Colormap;
#endif
#endif
/*	Definition of a matrix.  This is a 4x4 array of floats,		*/
/*	defining a transformation in a homogenous coordinate		*/
/*	system.								*/
typedef struct
{
	float   matrix[4][4];
} C_Matrix;

/*	Definition of a plane.  A plane is described by its plane	*/
/*	equation, ax + by + cz + d = 0.					*/
typedef struct
{
	float   a;
	float   b;
	float   c;
	float   d;
} C_Plane;

/*	This is the definition of a floating point position.  Our 	*/
/*	world space is assumed to range from -INFINITY to INFINITY 	*/
/* 	along each axis.						*/
typedef struct
{
	float   x;
	float   y;
	float   z;
} C_FPosition;

/*	This is the definition of an integer position.  Our world space	*/
/*	is assumed to range from -MAX_INT to MAX_INT along each axis.	*/
typedef struct
{
	int	x;
	int	y;
	int	z;
} C_IPosition;

/*	This is the definition of a normal.  The three float		*/
/*	values represent the three components of the normal.		*/
typedef struct
{
	float	dx;
	float	dy;
	float	dz;
} C_Normal;


/*	This is the definition of a float vector in 3-space.  The	*/
/*	three float values represent the three components of the	*/
/*	vector.								*/
typedef struct
{
	float	x;
	float	y;
	float	z;
} C_FVector;

/*	This is the definition of an integer vector in 3-space.  The	*/
/*	three integer values represent the three components of the 	*/
/*	vector.								*/
typedef struct
{
	int	x;
	int	y;
	int	z;
} C_IVector;



/*	This is the definition of a float vector in plane.  The	*/
/*	two float values represent the two components of the	*/
/*	vector.				           		*/
typedef struct
{
	float	x;
	float	y;
} C_FPosition2D;

/*      This is the definition of float Euler Angles representation    */ 
/*      of 3D orientation.                                             */
typedef struct
{
        float   a;
        float   b;
        float   r;
} C_FOrientation;

/*      This is the definition of float Quaternion representation      */
/*      of 3D orientation.                                             */
typedef struct
{
        float   x;
        float   y;
        float   z;
        float   w;
} C_FQuaternion;


/*      This is the definition of a color.  It has a red, green		*/
/*	and blue component.						*/
typedef struct
{
	unsigned char	red;
	unsigned char	green;
	unsigned char	blue;
} C_Color;


/************************************************************************/
/*                                                                      */
/*	The Interface between appliaction program and input device      */
/*      This structure include all the information application needs    */
/*                                                                      */
/************************************************************************/

typedef struct
{

        C_FPosition2D      pos2d;
        C_FPosition        pos3d;
        C_FOrientation     orientation;
        C_FQuaternion      quaternion;
        C_Matrix           rotation;

} C_IDE_token;

/************************************************************************/
/*                                                                      */
/*	           	Data & Viewing Typedefs				*/
/*                                                                      */
/************************************************************************/

/*	Voxel definition.		 				*/
/*	Change C_VOXEL_SIZE constant in C_consts.h if this definition	*/
/*	is changed!!!							*/
typedef	unsigned char	C_Voxel_8bit;

/*	pointer to one of 256 different geometric objects. If unsigned 	*/
/*	char is changed to unsigned short, then there can be up to 	*/
/*	65,536 different geometric objects				*/
typedef unsigned char	C_GeoIndex;

/*	Definition of a simple range of values. Primarily used for 	*/
/*	segmentation purposes.						*/
typedef struct
{
	int		low,
			high;
} C_SimpleRange;

/*	Definition of an array of opacity values, one for each possible	*/
/*	voxel value.							*/
typedef struct
{
	float		opacity[C_VOXEL_SIZE];
} C_OpacityArray;

/*	Definition of an array of color values, one for each possible	*/
/*	voxel value.							*/
typedef struct
{
	C_Color		color[C_VOXEL_SIZE];
} C_ColorArray;

/*	Definition of a local shading model. This is only PART of the	*/
/*	entire shading model, please see C_WorldShade also! At the 	*/
/*	local level only the diffuse and specular coefficients and the	*/
/*	specular power are specified. The diffuse color is the color of	*/
/*	the object, which may be obtained several ways(textures...)	*/
typedef struct
{
	float		diffuse_coef;
	float		specular_coef;
	float		specular_power;
	float		transmit_coef;
	float		transmissitivity;
	float		index_of_refraction;
} C_SimpleShade;

/*	Definition of an array shading model.  The object color is 	*/
/*	used as an index (3 bit red, 3 bit green, 2 bit blue) into 	*/
/*	this array to determine the diffuse and specular coefficients,  */
/*	as well as the specular power.					*/
typedef struct
{
	float		diff_coef_array[C_BYTE_SIZE];
	float		spec_coef_array[C_BYTE_SIZE];
	float		spec_pow_array[C_BYTE_SIZE];
	float		tran_coef_array[C_BYTE_SIZE];
	float		ref_index_array[C_BYTE_SIZE];
	float		transmiss_array[C_BYTE_SIZE];
} C_ArrayShade;	
	

/*	Definition of a volume texture.					*/

typedef struct
{
	int		references;
	int		xres;
	int		yres;
	int		zres;
	C_Byte		*red;
	C_Byte		*green;
	C_Byte		*blue;
	char		*texture_name;
} C_VolTexture;

/*	Definition of the scanned data voxel format. This type of data 	*/
/*	contains segmentation data and type, color data and type, and 	*/
/*	a pointer to the scanned data array.				*/
typedef struct
{
	C_SegType	seg_type;

	struct
	{
		float		isovalue;

		C_OpacityArray	*s_opacity;
	} seg;

	int		references;

	C_Voxel_8bit	*scalar_data;
} C_ScalarData_8bit;


typedef struct
{
	C_Plane		plane;
	int		num_vertices;
	C_FPosition	*vertices;
} C_Polygon;

typedef struct
{
	int		num_polygons;
	C_Polygon	*polygon;
} C_MultiPoly;

typedef struct
{
	float		radius;
	float		spread;
} C_Torus;

typedef struct
{
	C_GeoType	geo_type;

	union
	{
		C_Polygon	*polygon;
		C_MultiPoly	*multi_poly;
		C_Torus		*torus;
	} geo_object;

} C_GeoData;

 

/*	Definition of a coordinate system. A coordinate system consists	*/
/*	of a floating point 3D origin, and a floating point vector for	*/
/*	each of the axes. NOTE: These vectors should always be of unit	*/
/*	length (1.0)!!! 						*/
typedef struct
{
	C_FPosition	origin;

	C_FVector	x_axis,
			y_axis,
			z_axis;
} C_CoordSys;


/****************************************/
/*                                      */
/*      Parc Subdivision Structures     */
/*                                      */
/****************************************/
typedef struct C_ParcSubNode
{
        unsigned int            index;  /* SubVolume Array Index */
	unsigned char		subnode_faces;
        C_FVector               normal; /* Approximate Surface Normal */
        struct C_ParcSubNode    *next;  /* Pointer To Next Node  */
} C_ParcSubNode;

typedef struct
{
        float		sub_thresh_low;
        int             sub_level;
        int             sub_count;
        C_ParcSubNode   *sub_list;
} C_ParcInfo;


/*	Definition of a subnode list structure */
typedef struct C_NavSubNode
{
        C_IPosition     position;
        C_Normal        normal;
	struct C_NavSubNode	*next;
} C_NavSubNode;

typedef struct
{
        int             sub_level;
	float           sub_x_size;
	float           sub_y_size;
	float           sub_z_size;
        int             sub_count;
        C_NavSubNode   *sub_list;
} C_NavObject;


/*      Definition of a voxel list structure */
typedef struct
{
        C_IPosition     position;
        C_Normal        normal;
        int             xend;
} C_VoxelElement;

struct voxlist
{
        int             voxel_count;
        C_VoxelElement  voxel[C_NUM_VOXELS];
        struct voxlist  *next_voxel_list;
};

typedef struct voxlist C_VoxelList;

typedef struct
{
        C_VoxelList    *visible_surface;
	int             block_size;
} C_VoxelObject;

typedef struct
{
        C_CutType       cut_type;

        union
        {
                C_Plane         *plane;
		float		*box_cut;
        } cut;

	C_CutShadeType 	cut_shade;

} C_CutGeom;

typedef struct
{
        C_MeasureFlag   flag;
        int             value_voxels;
        float           value_units;
} C_MeaValue;


/*	Definition of a volume.  Each volume has an original and 	*/
/*	current coordinate system. The original coordinate system is 	*/
/*	the origin of the data when first read into the system. The 	*/
/*	current coordinate system is the product of the original 	*/
/*	coordinate system and the transform matrix. The volume also	*/
/*	specifies its size in units and voxels. The variable unit_type	*/
/*	specifies the measurement system used for the data, for example */
/*	: meters, millimeters or microns. The matrix wtol_voxel		*/
/*	transforms a point from world to local coordinates in voxels.	*/
/* 	The other three matrices (ltow_voxel, wtol_units, ltow_voxel)	*/
/*	follow accordingly. The plane equation array stores the plane	*/
/*	equations for each of the six faces of the volume. The 		*/
/*	local_shade variable describes the coefficients of the local 	*/
/*	shading model used. A data type and data are also given along 	*/
/*	with texture type and texture data.				*/
typedef struct
{
	C_CoordSys	orig_c_sys;	/* Starting Coord System */
	C_CoordSys	c_sys;		/* Current Coord System  */

	C_Matrix	transform;

	unsigned short  x_size_voxels,
			y_size_voxels,
			z_size_voxels;

	float		x_size_units,
			y_size_units,
			z_size_units;

	C_UnitType	unit_type;

	C_Matrix	wtol_voxels,	/* w = World, l = Local */
			ltow_voxels,
			wtol_units,
			ltow_units;

	C_Plane		plane_equation[C_NUM_FACES];

	C_DataType	data_type;

	union
	{
		C_ScalarData_8bit	*scalar_data_8bit;
		C_GeoData		*geo_data;
	} data;

	C_Voxel_8bit	*icon_8bit;
	unsigned int	*icon_24bit;

        int             num_cuts;

        C_CutGeom       *local_cut[C_MAX_LOCAL_CUTS];

	C_ColorType	color_type;

	struct	
	{
		C_Color		*single_color;
		C_ColorArray	*color_array;
	} color_data;


	C_TextureType	texture_type;

	union
	{
		C_VolTexture	*voltex;
		C_VolTexture	*radtex;
	} texture;


	C_ShadeType	shade_type;

	struct
	{
		C_SimpleShade	*simple_shade;
		C_ArrayShade	*array_shade;
	} local_shade;

	char			*volume_name;
	char			*file_name;

	int			visible;
	int			modifiable;

	C_DataOrigin		data_origin;
	C_DataModification 	data_modification;

        C_ParcInfo      	*p_info;

	C_MeaValue      mea_values[2][3];

} C_Volume;

/*	Definition of a point light source. The original and current	*/
/*	coordinate system and a transformation matrix are given.	*/
/*	In addition a light color is specified.				*/
typedef struct
{
	C_FPosition	orig_light_pos;
	C_FPosition	light_pos;

	C_Matrix	transform;

	C_Color		light_color;

	float		light_intensity;

} C_PointLight;

/*	Definition of a light source. The light_type specifies which	*/
/*	light type may be used from the several available.		*/
typedef struct
{
	C_LightType	light_type;

	union
	{
		/* Fill In With Your Favorite Light Flavors!!! */
		C_Volume	*light_volume;

		C_PointLight	*light_point;
	} light;

	char		*light_name;
	char		*file_name;

	int		visible;
	int		modifiable;

} C_Light;

/*	Definition of a world shading model. This is only PART of the	*/
/*	entire shading model, please see C_LocalShade also! The world	*/
/*	shading model determines the ambient color and coefficient and	*/
/*	a background color for the world.				*/
typedef struct
{
	float		ambient_coef;
	C_Color		ambient_color;
	C_Color		bkgrnd_color;		
} C_WorldShade;

/*	Definition of a world. Consists of an original coordinate 	*/
/*	system - this is ALWAYS current! the number of volumes and 	*/
/*	lights are given as well as an array of volumes and lights.	*/
/*	Finally, a world shanding model is given.			*/
typedef struct   
{
	C_CoordSys	orig_c_sys;	/* Static Coordinate System */

        int             num_lights,
                        num_volumes,
                        num_cuts;

	C_Volume	*volume[C_MAX_VOLUMES];
	C_Light		*light[C_MAX_LIGHTS];
        C_CutGeom       *global_cut[C_MAX_GLOBAL_CUTS];


	C_WorldShade	world_shade;

} C_World;

/*	Definition of a viewing window. A viewing window consists of a	*/
/*	original and current coordinate system and a transformation	*/
/*	matrix for getting to the current system from the original.	*/
/*	A field of view is specified for perspective projections where	*/
/*	a zero value represents a parallel projection. The width and	*/
/*	height in pixels and units are also given.			*/
typedef struct
{
	C_CoordSys		orig_c_sys;	/* Starting Coord System */
	C_CoordSys		c_sys;		/* Current Coord System */

	C_Matrix		transform;

	C_Matrix		wtol_pixels,	/* w = World, l = Local */
				ltow_pixels,
				wtol_units,
				ltow_units;

	float			fov;		/* Field Of View */
						/* 0 = Parallel  */
	unsigned short		width_pixels,
				height_pixels;

	float			width_units,
				height_units;

	C_ProjectionType	projection_type;

	C_SuperSampleType	supersample_type;

	int			display_during_projection;

	int			max_reflect_ray_level;
	int			max_transmit_ray_level;
	int			max_indirect_light_level;
	int			shadows_enabled;
	C_ImageAccuracy		image_accuracy;

	int			modifiable;
} C_View;





/************************************************************************/
/*									*/
/*			Image Control Typedefs				*/
/*									*/
/************************************************************************/

typedef struct
{
	C_RenderingType render_type;
        C_ImageAccuracy image_accuracy;
        float           compute_time;
        int             compute_shadows;        
        int             levels_of_reflection;
        int             levels_of_transmission;
	int		levels_of_indirect_light;
} C_ImageInfo;



/*	Definition of an image. Specifies the width and height in 	*/
/*	pixels and a depth in bytes. A depth of 3 is a 24 bit image.	*/
/*	The red, green and blue pointers point to a sequential array	*/
/*	of red, green and blue pixel components.			*/
typedef struct
{
	int		width,
			height,
			depth;

	C_ImageInfo 	image_info;

	C_Byte		*red,
			*green,
			*blue;
} C_Image;


/*	A C_ImageEntry is a single entry in the C_ImageTable.  It	*/
/*	has a field for the image name (which is displayed		*/
/*	in the pulldown menu of the Image Viewer), a field for the 	*/
/*	file name (where this file was read from and/or is written to)	*/
/*	and a field for the image itself.				*/
typedef struct
{
	char		*image_name;
	char		*file_name;
	C_Image		*image;
	C_UIWin		ui_win;
	C_UIWin		parent_win;
	C_UIWin		time_label;
	GC		graph_win;
} C_ImageEntry;

/*	The C_ImageTable is a table with C_MAX_IMAGES entries.  Each	*/
/*	entry is a C_ImageEntry.  The table keeps track of the total	*/
/*	number of images, as well as the active_image entry.		*/
typedef struct
{
	int		num_images;
	int		active_image;
	C_Colormap	colormap;
	C_ImageEntry	*image_entry[C_MAX_IMAGES];
} C_ImageTable;



/************************************************************************/
/*                                                                      */
/*	           	Navigator Typedefs				*/
/*                                                                      */
/************************************************************************/


typedef struct
{
        float   minx;
        float   miny;
        float   minz;
        float   maxx;
        float   maxy;
        float   maxz;
} C_World_Extent;

 
/* Timing Structure */
typedef struct
{
        long    sec;
        long    usec;
} C_Time_Val;

/* User preference Structure */
typedef struct
{
        C_UserType  user;

} C_Preference;


