/*
 * Copyright (c) 1990,1,2 Mark Nitzberg
 * and President and Fellows of Harvard College
 * All rights reserved.
 * 
 * Match T's and corners and add continuation contours to an edge map
 */
#include <math.h>
#include <stdio.h>
#include <malloc.h>
#include <hvision.h>
#include "vector.h"
#include "edgemap.h"
#include "region.h"

char    Usage[] = "\
usage: %s [-s sigma] [-w window_size] [-t tmaxerr] [-k kfactor]\n\
	  [-c continfile [-n nu] [-a alpha]] [input_file]\n\
\n\
   Smooths sampled curves of the form produced by \"curves\"\n\
   by fitting window_size points at a time, using a gaussian weighting\n\
   function of std dev Sigma (default 5); window_size defaults to 2*sigma.\n\
\n\
   Without the -c argument, T-junctions are identified before\n\
   smoothing by best-fitting pairs of ends to a single line or\n\
   circle.  If a pair yields a fit with average error < tmaxerr\n\
   (default Sigma), then the pair with the least such error is\n\
   smoothed together as a single contour.  Error is given by the\n\
   best fit error in distance units, plus kfactor (default 0) times\n\
   the square of curvature of the best fit circle.\n\
   Use -t 0 to defeat the interpretation of Ts.\n\
\n\
   With -c, T junctions are instead interpreted according to the\n\
   continuations specified in continfile.  The program adds the\n\
   continuation contours and produces `.ec<n>' and `.r<n>' files.\n\
\n\
        -s      standard deviation of gaussian weights, default 5\n\
        -w      window size -- # pixels to fit at a time, default 4*sigma\n\
        -t      cutoff for avg error for T disambiguation, default sigma\n\
	-k	junction error = best fit error + kfactor*curvature\n\
	-c	specify continuations file.\n\
	    -n	specify nu, the length penalty param\n\
	    -a  specify alpha, the curvature squared penalty parameter\n";

/* parameters */
real    Sigma;
int     WindowSize;
real    TMaxError = -1;
real    KFactor;
char   *ContinFile = NULL;
char   *ImageFile = NULL;
IMAGE  *InputImage;
real    Alpha;			/* curvature square penalty */

#define DEFAULT_ALPHA 10
real    Nu;			/* length penalty */

#define DEFAULT_NU 1

#define MAXPAIR 500
typedef struct Pairing {
    short   npair;
    struct pair {
	int     c1, end1;
	int     c2, end2;
    }       pairs[MAXPAIR];
} Pairing;
Pairing P;

/* for region wizardry */
real    MinPixel, MaxPixel;


/* prototypes */

#ifdef ANSI
int     main(int, char **);
int     ReadPairing(char *);
int     MarkTJunctions(void);
real    JunctionError(int, int, int, int);
int     FillPL(struct PointList *, int, int, struct PointList *, int, int, int);
int     Smooth(void);
int     Pad(int, int, struct PointList *, int, int);
int     EliminateSwitchbacks(struct PointList *, int, int);
real    EndDistance(int, int, int, int);
int     ReJoinJunctions(void);
vector *IntersectLines(vector, vector, vector, vector);
int     ReJoinCorner(struct Junction *);
int     FirstPoint(int, int, vector *);
int     FirstTwoPoints(int, int, vector *, vector *);
int     SecondAndThird(int, int, vector *, vector *);
int     ReJoinT(struct Junction *);
int     ReJoinCenterOfMass(struct Junction *);
int     AddContinContours(void);
int     Output(char *, char *, char *);
vector *IntersectSegments(vector, vector, vector, vector);
real    AddElastica(double, double, double, double, double, double, double, double);
int     LayDownContour(struct PointList *);
int     IntersectEverywhere(struct PointList **, int *);
struct PointList *AllocCopyPointList(struct PointList *, int, int);
real    AngleFix(double);
struct PointList *InsertPointIntoPointList(struct PointList *, int, double, double);
int     SplitContourAfter(int, int, double, double);

#else				/* ANSI */

int     main();
int     ReadPairing();
int     MarkTJunctions();
real    JunctionError();
int     FillPL();
int     Smooth();
int     Pad();
int     EliminateSwitchbacks();
real    EndDistance();
int     ReJoinJunctions();
vector *IntersectLines();
int     ReJoinCorner();
int     FirstPoint();
int     FirstTwoPoints();
int     SecondAndThird();
int     ReJoinT();
int     ReJoinCenterOfMass();
int     AddContinContours();
int     Output();
vector *IntersectSegments();
real    AddElastica();
int     LayDownContour();
int     IntersectEverywhere();
struct PointList *AllocCopyPointList();
real    AngleFix();
struct PointList *InsertPointIntoPointList();
int     SplitContourAfter();

#endif				/* ANSI */

extern real RefinePoint( /* x,y,w,n,i,x2,y2,tan_angle,k */ );
extern void RefineSampledCurve( /* x,y,n,m,sigma,newx,newy,tan_an,k,err */ );
real    JunctionError( /* int cno1, int end1, int cno2, int end2 */ );


/* ---------- the program ------------ */
main(argc, argv)
int     argc;
char  **argv;
{
    char    c;
    char    imagename[512];

    extern char *optarg;
    extern int optind;

    PgmName = argv[0];

    while ((c = getopt(argc, argv, "s:w:t:k:c:n:a:")) != -1) {

	switch (c) {

	case '?':		/* getopt() arg not in list */
	    fprintf(stderr, Usage, PgmName);
	    exit(1);

	case 's':
	    Sigma = atof(optarg);
	    break;
	case 'w':
	    WindowSize = atoi(optarg);
	    break;
	case 't':
	    TMaxError = atof(optarg);
	    break;
	case 'k':
	    KFactor = atof(optarg);
	    break;
	case 'c':
	    ContinFile = optarg;
	    break;
	case 'n':
	    Nu = atof(optarg);
	    break;
	case 'a':
	    Alpha = atof(optarg);
	    break;
	}
    }

    if (optind < argc) {
	CurrentFile = argv[optind];
	if (freopen(CurrentFile, "r", stdin) == NULL) {
	    fprintf(stderr, "Can't open %s for input\n", CurrentFile);
	    exit(1);
	}
    }
    if (!ReadContours(stdin))
	exit(Status);

    if (ContinFile != NULL && !ReadPairing(ContinFile))
	exit(Status);

    if (ImageFile == NULL) {
	ImageFile = imagename;
	strcpy(imagename, FileNameNoNumber(CurrentFile, ".im"));
    }
    if ((InputImage = hvReadImage(ImageFile)) == NULL) {
	fprintf(stderr, Usage, PgmName);
	exit(1);
    }
    if (InputImage->pixeltype != REALPIX)
	hvConvert(InputImage, InputImage, REALPIX);

    hvMinMax(InputImage, 0, &MinPixel, &MaxPixel);

    /* set parameters */
    if (Sigma == 0)
	Sigma = 5;
    if (Nu == 0)
	Nu = DEFAULT_NU;
    if (Alpha == 0)
	Alpha = DEFAULT_ALPHA;
    if (WindowSize == 0)
	WindowSize = (int) (0.5 + 4 * Sigma);
    if (TMaxError == -1)
	TMaxError = Sigma;

    /* now starts the code */
    MarkTJunctions();

    Smooth();

    ReJoinJunctions();

    if (ContinFile != NULL) {

	AddContinContours();

    } else {

	/* output */
	if (strcmp(CurrentFile, STDIN_STRING) == 0)
	    PrintContours(stdout, "Smoothed contours");
	else {
	    /* save in file.es */
	    char   *name = FileName(CurrentFile, ".es");
	    FILE   *f = fopen(name, "w");

	    if (f == NULL) {
		fprintf(stderr,
			"%s: Can't open `%s' for output\n",
			PgmName, name);
		exit(3);
	    } else {
		PrintContours(f, "Smoothed contours");
		fclose(f);
		fprintf(stderr,
			"%s: wrote output in `%s'\n",
			PgmName, name);
	    }
	}
    }

    exit(Status);
}

ReadPairing(fname)
char   *fname;
{
    FILE   *f;
    char    buf[512];

    if ((f = fopen(fname, "r")) == NULL) {
	fprintf(stderr, "%s: can't read continuation file %s\n",
		PgmName, fname);
	Status = 5;
	return FALSE;
    }
    P.npair = 0;
    while (fgets(buf, 512, f) != NULL) {
	if (buf[0] == '#' || buf[0] == '\n')
	    continue;
	if (P.npair == MAXPAIR) {
	    fprintf(stderr,
		    "%s: file %s has > %d continuations--aborting\n",
		    PgmName, fname, MAXPAIR);
	    Status = 7;
	    return FALSE;
	}
	if (sscanf(buf, "%d%d%d%d",
		   &P.pairs[P.npair].c1,
		   &P.pairs[P.npair].end1,
		   &P.pairs[P.npair].c2,
		   &P.pairs[P.npair].end2) != 4) {
	    fprintf(stderr,
		    "%s: file %s needs 4 numbers on each non-comment line\n",
		    PgmName, fname);
	    Status = 6;
	    return FALSE;
	}
	P.npair++;
    }
    return TRUE;
}


#define HUGE_ERROR 10000	/* ridiculous avg dist. error */

MarkTJunctions()
{
    Junction *J;
    int     i, j, besti, bestj;
    real    err, besterr;
    int     nT, nBoundary;

    /* for each T found, set "isT" to TRUE & associate 2 curves in J */
    nT = 0;
    for (J = &Junctions[0]; J <= &Junctions[NJunctions - 1]; J++) {

	/* a T needs 3 */
	if (J->n != 3)
	    continue;

	/* first try to use the continuation list */
	if (ContinFile != NULL) {
	    for (i = 0; i < P.npair; i++) {

		/*
		 * if one of the 3 contours at J is continued, the other two
		 * are assumed to be a T
		 */
		for (j = 0; j < 3; j++) {
		    if ((J->c[j].cno == P.pairs[i].c1
			 && J->c[j].end == P.pairs[i].end1)
			|| (J->c[j].cno == P.pairs[i].c2
			    && J->c[j].end == P.pairs[i].end2)) {
			J->isT = TRUE;
			nT++;
			if (j == 1) {
			    J->c1 = 0;
			    J->c2 = 2;
			} else {
			    J->c1 = (j + 1) % 3;
			    J->c2 = (j + 2) % 3;
			}
			goto nextj;
		    }
		}
	    }

    nextj: continue;
	}
	/* if 2 contours are boundary, case closed */
	nBoundary = 0;
	for (i = 0; i < 3; i++)
	    if (Contours[J->c[i].cno].p->boundary) {
		/* (c1,c2) = (0,1), (0,2) or (1,2) */
		J->c1 = J->c2;
		J->c2 = i;
		nBoundary++;
	    }
	if (nBoundary == 2) {
	    J->isT = TRUE;
	    /* nT++;  -- why mislead the user? */
	    continue;
	}
	/* use error of fit to circle */
	besterr = HUGE_ERROR;

	/* for each pair (i,j) = (0,1),(0,2),(1,2) get curvature */
	for (i = 0; i <= 1; i++) {
	    for (j = i + 1; j <= 2; j++) {
		err = JunctionError(J->c[i].cno, J->c[i].end,
				    J->c[j].cno, J->c[j].end);

		if (err < besterr) {
		    besterr = err;
		    besti = i;
		    bestj = j;
		}
	    }
	}

	if (besterr < TMaxError) {

	    /* contour besti to bestj associate for smoothing */
	    J->isT = TRUE;
	    J->c1 = besti;
	    J->c2 = bestj;

	    nT++;
	}
    }
    fprintf(stderr, "Found %d T junctions\n", nT);
}

/* Error = (best fit error) + KFactor*(curvature^2) */
real
JunctionError(cno1, end1, cno2, end2)
int     cno1, end1, cno2, end2;
{
    static PointList *plist;	/* temporary -- ends of 2 curves */
    static real *Gaussian;	/* 2*Sigma, 2*WindowSize */
    static int center;
    int     left, actualsize;
    real    newx, newy, newtheta, newk;
    real    bestfiterror;

    if (plist == NULL) {
	/* odd, isn't it? */
	int     odd = 2 * WindowSize - 1;

	plist = NewPointList(odd);
	plist->n = odd;
	Gaussian = AllocReal(odd);
	DrawGauSpline1D(Gaussian, odd, 2 * Sigma);

	/* NB Gaussian[WindowSize - 1] is the center */
	center = WindowSize - 1;
    }

    /*
     * fill plist from the center to the left starting at end1 of cno1,
     * WindowSize points, and from the center (again) to the right, starting
     * at end2 of cno2
     */

    actualsize = FillPL(Contours[cno1].p, EndIndex(cno1, end1), Direction(end1),
			plist, center, BACKWARD,
			WindowSize);

    /* put some limit on how short a stub we call a T! */
    if (actualsize < WindowSize / 2)
	return HUGE_ERROR;

    actualsize -= 1;		/* the repeated centerpoint */
    left = center - actualsize;

    actualsize += FillPL(Contours[cno2].p, EndIndex(cno2, end2), Direction(end2),
			 plist, center, FORWARD,
			 WindowSize);

    if (actualsize < WindowSize)
	return HUGE_ERROR;

    bestfiterror = RefinePoint(plist->x + left, plist->y + left,
			       Gaussian + left,
			       actualsize, center - left,
			       &newx, &newy, &newtheta, &newk);
    return bestfiterror + KFactor * (newk * newk);
}

/*
 * FillPL copies a specified segment of one pointlist into a specified
 * segment of another, in a specified direction.
 * 
 * WITHOUT PROPER HANDLING OF OVERLAP.  Thank you.
 * 
 * Returns the number of points copied.
 */
int
FillPL(p1, i1, di1, p2, i2, di2, n)
/* from	       to    n	 */
PointList *p1, *p2;
int     i1, di1, i2, di2, n;
{
    int     count = 0;

    while (i1 >= 0 && i1 < p1->size
	   && i2 >= 0 && i2 < p2->size

	   && count < n) {

	p2->x[i2] = p1->x[i1];
	p2->y[i2] = p1->y[i1];
	p2->theta[i2] = p1->theta[i1];
	p2->k[i2] = p1->k[i1];

	i1 += di1;
	i2 += di2;
	count++;
    }
    return count;
}

/*
 * Blur everywhere with a vengeance: within contours across the associated
 * parts of T's across head/tail boundaries of closed contours
 * 
 * In case a contour should be continued across a T or around a loop, we pad
 * with (WindowSize+1)/2.
 * 
 * MINOR OVERSIGHT WHICH I'm NOT GOING TO FIX: One side of a T will be smoothed
 * before the other side, so that the second side uses the _smoothed_ data
 * from the first side to do its fitting.  This is just too bad.
 */
Smooth()
{
    Contour *C;
    int     c;
    PointList *p, *q;
    real   *err;
    int     length, pad, left, actualsize;

    /* find longest contour */
    length = 0;

    for (C = &Contours[0]; C <= &Contours[NContours - 1]; C++)
	if (length < C->p->n)
	    length = C->p->n;

    pad = (WindowSize + 3) / 2;	/* on each side */

    p = NewPointList(length + 2 * pad);
    p->n = p->size;
    q = NewPointList(length + 2 * pad);
    q->n = q->size;
    err = AllocReal(length + 2 * pad);

    /* smooth each contour */
    for (c = 0, C = &Contours[0]; C <= &Contours[NContours - 1]; c++, C++) {

	/* the image boundary stays as is */
	if (C->p->boundary)
	    continue;

	/* copy contour into p with pad points extra on left */
	FillPL(C->p, HEAD, FORWARD, p, pad, FORWARD, C->p->n);

	/* pad each end where needed */
	left = pad;
	actualsize = Pad(c, HEAD, p, HEAD, pad);
	left -= actualsize;

	actualsize += Pad(c, TAIL, p, TAIL, pad);
	actualsize += C->p->n;

	RefineSampledCurve(p->x + left, p->y + left, actualsize, WindowSize, Sigma,
	       q->x + left, q->y + left, q->theta + left, q->k + left, err);

	C->p->n = EliminateSwitchbacks(q, left, C->p->n);	/* may decrease */

	FillPL(q, pad, FORWARD, C->p, 0, FORWARD, C->p->n);
    }

    FreePointList(p);
    FreePointList(q);
    free(err);
}

/*
 * pad end end of pointlist p (which has "pad" padding points on each side)
 * according to end end of contour C; return # pad points
 */
int
Pad(c, end, p, endp, pad)
int     c;
int     end;
PointList *p;
int     endp;
int     pad;
{
    /* is padding necessary ? */
    Contour *C = &Contours[c];
    Junction *J = &Junctions[C->junc[end]];
    int     pindex, pdir;

    if (endp == HEAD) {
	pindex = pad - 1;
	pdir = BACKWARD;
    } else {
	pindex = C->p->n + pad;
	pdir = FORWARD;
    }

    /* closed */
    if (C->p->closed && !C->p->withcorner) {

	/* wrap going in from the other end */
	return FillPL(C->p, EndIndex(c, OtherEnd(end)), Direction(OtherEnd(end)),
		      p, pindex, pdir,
		      pad);
    }
    /* one of the two "associated" contours at a T */
    if (J->isT && (c == J->c[J->c1].cno || c == J->c[J->c2].cno)) {

	/* wrap by following the associate */
	int     a = (c == J->c[J->c1].cno ? J->c2 : J->c1);
	int     ac = J->c[a].cno;
	int     aend = J->c[a].end;
	int     aindex = EndIndex(ac, aend);
	PointList *ap = Contours[ac].p;

	/* don't include the duplicate point */
	if (aend == HEAD)
	    aindex++;
	else
	    aindex--;

	return FillPL(ap, aindex, Direction(aend),
		      p, pindex, pdir,
		      pad);
    }
    /* no padding needed */
    return 0;
}

/*
 * assure contour never turns <= 90 degrees in one step or repeats a pixel.
 * Pack points in the old-fashioned cobol style. returns (n - # pts
 * eliminated)
 */


EliminateSwitchbacks(p, offset, n)
PointList *p;
int     offset, n;
{
    int     from, to;

    if (p->boundary)
	return n;

    to = 1 + offset;
    from = 1 + offset;
    while (from < n + offset) {

	/*      to
	 *	 V   from
	 *     , ' ,   V     , ,
	 *   '       ' ' , '
	 *  --> increasing index -->
	 *
	 * (to-1) - from  gives vector "left"
	 * (from+1) - from  gives vector "right"
	 *
	 * left.right = |left| |right| cos(angle)
	 *	      >= 0 if angle <= 90 deg
         */
	while (from < n - 1 + offset) {
	    real    leftx = (p->x[to - 1] - p->x[from]);
	    real    lefty = (p->y[to - 1] - p->y[from]);
	    real    rightx = (p->x[from + 1] - p->x[from]);
	    real    righty = (p->y[from + 1] - p->y[from]);

	    /*
	     * the angle left - here - right > 90 degrees the the points here
	     * and right are different
	     */
	    if (rightx * rightx + righty * righty > 0
		&& leftx * rightx + lefty * righty < 0)
		break;

	    from++;
	}

	if (from != to) {
	    p->x[to] = p->x[from];
	    p->y[to] = p->y[from];
	    p->theta[to] = p->theta[from];
	    p->k[to] = p->k[from];
	}
	from++;
	to++;
    }

    if (from - to > 0)
	fprintf(stderr, "Contour: %d points trimmed to %d points\n",
		n, n - (from - to));

    return n - (from - to);
}


real
EndDistance(c1, end1, c2, end2)
int     c1, end1, c2, end2;
{
    PointList *p1 = Contours[c1].p;
    PointList *p2 = Contours[c2].p;
    int     i1 = EndIndex(c1, end1);
    int     i2 = EndIndex(c2, end2);
    vector  point1, point2;

    point1.x = p1->x[i1];
    point1.y = p1->y[i1];

    point2.x = p2->x[i2];
    point2.y = p2->y[i2];

    return v_distance(point1, point2);
}

#define POINT_TOLERANCE 0.1	/* close to the same point */

ReJoinJunctions()
{
    int     jno;

    /*
     * 0.  Any junction with all offsets within POINT_TOLERANCE can be
     * replaced by the average location of all points involved.
     */
    for (jno = 0; jno < NJunctions; jno++) {

	Junction *J = &Junctions[jno];
	int     within = TRUE;
	int     i, j;

	if (J->n <= 1)
	    continue;		/* nothing to rejoin */

	/* if pairwise within POINT_TOLERANCE, just average */
	for (i = 0; i < J->n - 1; i++) {
	    for (j = i + 1; j < J->n; j++) {
		if (EndDistance(J->c[i].cno, J->c[i].end,
				J->c[j].cno, J->c[j].end)
		    > POINT_TOLERANCE) {

		    within = FALSE;
		}
	    }
	}

	if (within)
	    ReJoinCenterOfMass(J);

	/* otherwise ... */
	else if (J->n == 2)
	    ReJoinCorner(J);

	else if (J->n == 3 && J->isT)
	    ReJoinT(J);

	/* do it even though they're not so close */
	else
	    ReJoinCenterOfMass(J);
    }
}

/*
 * Returns a pointer to a static vector containing the intersection of two
 * lines or NULL if they are parallel. Takes four points specifying the two
 * lines; cowboy style no checks for singularities etc.
 */
vector *
IntersectLines(head1, tail1, head2, tail2)
vector  head1, tail1, head2, tail2;
{
    vector  d1, d2, dhead;
    static vector intersection;
    real    denom;

    d1 = v_difference(head1, tail1);	/* tail1 - head1 */
    d2 = v_difference(head2, tail2);
    dhead = v_difference(head1, head2);

    denom = v_dot(d1, v_perp(d2));

    if (denom == 0) {

	/* they're parallel */
	return NULL;
    } else {

	/* line intersection is at head1 + t1 d1 */
	real    t1 = v_dot(dhead, v_perp(d2)) / denom;

	/* real t2 = v_dot(dhead, v_perp(d1))/denom;  */

	intersection = v_sum(head1, v_scalar_mult(t1, d1));
    }

    return &intersection;
}


ReJoinCorner(J)
Junction *J;
{

    /*
     * The `p's are the endpoint locations, called "head"s below. The arrows
     * show the way towards the next point on the contour.
     * 
     * a) non-overlap    b) non-intersecting  c) intersecting overlap overlap
     * p--->		  p			p p		  | p--->
     * | |		  |	 	      p-+---> v		  v
     * v
     */

    /* get the two line segments; head = endpoint */
    vector  head1, tail1, head2, tail2, newpos, *intersection;
    int     i;

    SecondAndThird(J->c[0].cno, J->c[0].end, &head1, &tail1);
    SecondAndThird(J->c[1].cno, J->c[1].end, &head2, &tail2);

    intersection = IntersectLines(head1, tail1, head2, tail2);

    if (intersection == NULL)	/* parallel -- just take midpt */
	newpos = v_scalar_mult(0.5, v_sum(head1, head2));
    else
	newpos = *intersection;

    /* put newpos at both places */
    for (i = 0; i < 2; i++) {
	int     cno = J->c[i].cno;
	PointList *p = Contours[cno].p;
	int index = EndIndex(cno, J->c[i].end);

	p->x[index] = newpos.x;
	p->y[index] = newpos.y;
	p->n = EliminateSwitchbacks(p, 0, p->n);
    }
}

FirstPoint(cno, end, pt)
int     cno, end;
vector *pt;
{
    PointList *p = Contours[cno].p;
    int index = EndIndex(cno, end);

    pt->x = p->x[index];
    pt->y = p->y[index];
}

FirstTwoPoints(cno, end, endpt, nextpt)
int     cno, end;
vector *endpt, *nextpt;
{
    PointList *p = Contours[cno].p;
    int index = EndIndex(cno, end);
    int     dir = Direction(end);

    endpt->x = p->x[index];
    endpt->y = p->y[index];
    index +=dir;

    nextpt->x = p->x[index];
    nextpt->y = p->y[index];
}

SecondAndThird(cno, end, endpt, nextpt)
int     cno, end;
vector *endpt, *nextpt;
{
    PointList *p = Contours[cno].p;
    int index = EndIndex(cno, end);
    int     dir = Direction(end);

    if (p->n > 2)
	index +=dir;

    endpt->x = p->x[index];
    endpt->y = p->y[index];
    index +=dir;

    nextpt->x = p->x[index];
    nextpt->y = p->y[index];
}

#define BIG_GAP	5.0		/* too far to jump to intersection */

ReJoinT(J)
Junction *J;
{

    /*
     *	a) standard break-ups		b) assorted crossovers
     *
     *	    ^		    ^		    ^		     ^
     *	    |		    |		    |		     |
     *	    p		    p		    |		<----+-p
     *	<-----p p-->	<--p  p-->	<---+-p p-->	  p--+---->
     *					    |		     |
     *
     *  Take the line connecting next-door neighbors of the associate p's,
     *  intersected with the line connecting the 1st 2 points of the stem.
     */

    vector  dummy, p1, p2;	/* the two associate points 1st neighbors */
    vector  stem1, stem2;	/* 1st two points of the stem */
    vector *intersection, newpos;
    int     i;

    /* the two associates */
    int     cno1 = J->c[J->c1].cno;
    int     end1 = J->c[J->c1].end;

    int     cno2 = J->c[J->c2].cno;
    int     end2 = J->c[J->c2].end;

    /* J->c1 and J->c2 are 0,1 0,2 or 1,2; so 3-(sum) gives stem */
    int     stem = 3 - (J->c1 + J->c2);
    int     cnostem = J->c[stem].cno;
    int     endstem = J->c[stem].end;

    FirstTwoPoints(cnostem, endstem, &stem1, &stem2);
    FirstTwoPoints(cno1, end1, &dummy, &p1);
    FirstTwoPoints(cno2, end2, &dummy, &p2);

    intersection = IntersectLines(p1, p2, stem1, stem2);

    if (intersection == NULL) {	/* parallel? */
	fprintf(stderr,
	  "A T with a parallel stem & top ?  Back to the drawing board.\n");

	/* take center of mass */
	newpos = v_scalar_mult(1.0 / 3.0, v_sum(p1, v_sum(p2, stem1)));
    } else
	newpos = *intersection;

    /* check that nobody jumps too far */
    for (i = 0; i < 3; i++) {
	int     cno = J->c[i].cno;
	PointList *p = Contours[cno].p;
	int index = EndIndex(cno, J->c[i].end);

	/* distance check */
	if (v_distance(newpos, v_vec(p->x[index], p->y[index]))
	    > BIG_GAP) {
	    ReJoinCenterOfMass(J);
	    return;
	}
    }

    /* put newpos at all places */
    for (i = 0; i < 3; i++) {
	int     cno = J->c[i].cno;
	PointList *p = Contours[cno].p;
	int index = EndIndex(cno, J->c[i].end);

	p->x[index] = newpos.x;
	p->y[index] = newpos.y;
	p->n = EliminateSwitchbacks(p, 0, p->n);
    }
}

/* join by taking center of mass--not optimal, but OK for now */
ReJoinCenterOfMass(J)
Junction *J;
{
    int     i;

    /* compute center of mass */
    real    avgx = 0.0, avgy = 0.0;

    for (i = 0; i < J->n; i++) {
	int     cno = J->c[i].cno;
	PointList *p = Contours[cno].p;
	int index = EndIndex(cno, J->c[i].end);

	avgx += p->x[index];
	avgy += p->y[index];
    }
    avgx /= J->n;
    avgy /= J->n;

    /* substitute (avgx,avgy) at all points */
    for (i = 0; i < J->n; i++) {
	int     cno = J->c[i].cno;
	PointList *p = Contours[cno].p;
	int index = EndIndex(cno, J->c[i].end);

	p->x[index] = avgx;
	p->y[index] = avgy;
	p->n = EliminateSwitchbacks(p, 0, p->n);
    }
}

/*
 * add continuations
 */
AddContinContours()
{
    int     i;
    char    expl[512];
    real    AddElastica();

    /* fill length, ksq, endtan and endavg for each contour */
    for (i = 0; i < NContours; i++)
	/* needs a std dev. for length scale--not our huge Sigma */
	FillFields(i, 1.25);

    /* add contours from pairing */
    for (i = 0; i < P.npair; i++) {
	int     c1 = P.pairs[i].c1;
	int     end1 = P.pairs[i].end1;
	int     c2 = P.pairs[i].c2;
	int     end2 = P.pairs[i].end2;
	int     ei1 = EndIndex(c1, end1);
	int     ei2 = EndIndex(c2, end2);
	PointList *p1 = Contours[c1].p;
	PointList *p2 = Contours[c2].p;

	/* this can introduce many more contours if there are intersections */
	(void) AddElastica(p1->x[ei1], p1->y[ei1], Contours[c1].endtan[end1],
		   p2->x[ei2], p2->y[ei2], Contours[c2].endtan[end2] + M_PI,
			   Alpha, Nu);
    }

    /* re-calculate region map */
    TraceRegions();
    SetContourRegions();

    /* save it all! */
    sprintf(expl, "Continuations with alpha=%g nu=%g",
	    Alpha, Nu);
    Output(".ec", ".r", expl);
}

/*
 * save the edgemap to file.<edgemapext> if non-NULL save region map to
 * file.<regionext> if non-NULL expl is optional explanation for edgemap
 * file's header comment
 */
Output(edgemapext, regionext, expl)
char   *edgemapext, *regionext, *expl;
{
    if (strcmp(CurrentFile, STDIN_STRING) == 0) {
	if (edgemapext != NULL)
	    PrintContours(stdout, "Continuations added");
	PrintRegions(stdout);
    } else {
	/* save in file.es */
	char   *name;
	FILE   *f;

	if (edgemapext != NULL) {
	    name = FileName(ContinFile, edgemapext);
	    if ((f = fopen(name, "w")) == NULL) {
		fprintf(stderr,
			"%s: Can't open `%s' for output\n",
			PgmName, name);
		Status = 4;
	    } else {
		PrintContours(f, expl);
		fclose(f);
		fprintf(stderr,
			"%s: wrote edge map to `%s'\n",
			PgmName, name);
	    }
	}
	if (regionext != NULL) {
	    name = FileName(ContinFile, regionext);
	    if ((f = fopen(name, "w")) == NULL) {
		fprintf(stderr,
			"%s: Can't open `%s' for output\n",
			PgmName, name);
		Status = 5;
	    } else {
		PrintRegions(f);
		fclose(f);
		fprintf(stderr,
			"%s: wrote region map to `%s'\n",
			PgmName, name);
	    }
	}
    }
}

/*
 * Returns a pointer to a static vector containing the intersection of two
 * segments or NULL if they do not truly intersect, or if they are parallel.
 * Takes four points specifying the two lines; cowboy style no checks for
 * singularities etc.
 */
vector *
IntersectSegments(head1, tail1, head2, tail2)
vector  head1, tail1, head2, tail2;
{
    vector  d1, d2, dhead;
    static vector intersection;
    real    denom;

    d1 = v_difference(head1, tail1);	/* tail1 - head1 */
    d2 = v_difference(head2, tail2);
    dhead = v_difference(head1, head2);

#ifdef paranoid
    if ((dhead.x == 0 && dhead.y == 0)	/* heads meet */
	||(tail1.x == tail2.x && tail1.y == tail2.y)	/* tails meet */
	||(head1.x == tail2.x && head1.y == tail2.y)	/* head1 meets tail2 */
	||(tail1.x == head2.x && tail1.y == head2.y)) {	/* head2 meets tail1 */
	fprintf(stderr,
		"IntersectSegments( (%g,%g)-(%g,%g) (%g,%g)-(%g,%g) )\n",
		head1.x, head1.y, tail1.x, tail1.y,
		head2.x, head2.y, tail2.x, tail2.y);
	fprintf(stderr,
		"   intersect at endpoint.\n");
	abort();
    }
#endif				/* paranoid */

    denom = v_dot(d1, v_perp(d2));

    if (denom == 0) {

	/* they're parallel */
	return NULL;
    } else {

	/* line intersection is at head1 + t1 d1 */
	real    t1 = v_dot(dhead, v_perp(d2)) / denom;
	real    t2 = v_dot(dhead, v_perp(d1)) / denom;

	if (t1 < 0 || t1 > 1 || t2 < 0 || t2 > 1)
	    return NULL;

	intersection = v_sum(head1, v_scalar_mult(t1, d1));

	return &intersection;
    }
}

/*
 * lay down an elastica, splitting wherever there is an intersection. Expects
 * the region map to contain all other contour points.
 */
#define NELASTPTS 50

real
AddElastica(x1, y1, theta1, x2, y2, theta2, Alpha, Nu)
real    x1, y1, theta1, x2, y2, theta2, Alpha, Nu;
{
    PointList *pelast;
    real    energy;
    real    BuildElast();

    if (NContours >= MAXCONTOUR) {
	fprintf(stderr, "Out of contours for adding elastica.\n");
	return 0;
    }
    pelast = NewPointList(NELASTPTS);
    pelast->n = NELASTPTS;

    energy = BuildElast(pelast, x1, y1, theta1, x2, y2, theta2, Alpha, Nu);

    /* BuildElast may fudge just a bit on endpoints */
    pelast->x[0] = x1;
    pelast->y[0] = y1;
    pelast->x[pelast->n - 1] = x2;
    pelast->y[pelast->n - 1] = y2;

    LayDownContour(pelast);

    return energy;
}

#define MAXSEG	200		/* how many contours we could ever cross */

/*
 * find all intersections and split all contours that intersect; then lay
 * down this contour in pieces
 */
LayDownContour(p)
PointList *p;
{
    Contour *C;
    int     seg[MAXSEG];
    int     nseg, i;

    /*
     * seg[0],seg[1] is interval (point indeces) of 1st segment seg[1],seg[2]
     * ... second segment, etc
     */
    nseg = IntersectEverywhere(&p, seg);

    /* lay down each segment individually */
    for (i = 0; i < nseg; i++) {
	if (NContours >= MAXCONTOUR) {
	    fprintf(stderr, "%s: out of contour space in LayDownContour()\n",
		    PgmName);
	    exit(3);
	}
	C = &Contours[NContours];
	C->p = AllocCopyPointList(p, seg[i], seg[i + 1]);
	FillFields(NContours, 1.25);
	C->p->contin = TRUE;
	C->junc[HEAD] = FindOrAddJunction(C->p->x[0], C->p->y[0], NContours, HEAD);
	C->junc[TAIL] = FindOrAddJunction(C->p->x[C->p->n - 1],
				     C->p->y[C->p->n - 1], NContours, TAIL);
	NContours++;
    }
}

/* ASSUMES p is not self-intersecting */
IntersectEverywhere(pp, seg)
PointList **pp;			/* may need to free it and alloc bigger one */
int     seg[];
{
    int     segno = 1;
    int     i, j, cno;
    real    fromx, fromy, tox, toy;
    real    fromx2, fromy2, tox2, toy2;
    vector *intersection;
    PointList *p = *pp;

    seg[0] = 0;

    /* split other contours where p intersects them */
    tox = p->x[0];
    toy = p->y[0];
    for (i = 1; i < p->n; i++) {
	fromx = tox;
	fromy = toy;
	tox = p->x[i];
	toy = p->y[i];

	/* for all other contours */
	for (cno = 0; cno < NContours; cno++) {
	    PointList *p2 = Contours[cno].p;

	    if (p2->boundary)
		continue;

	    tox2 = p2->x[1];
	    toy2 = p2->y[1];
	    for (j = 2; j < p2->n - 1; j++) {
		fromx2 = tox2;
		fromy2 = toy2;
		tox2 = p2->x[j];
		toy2 = p2->y[j];

		intersection =
		    IntersectSegments(
				      v_vec(fromx, fromy), v_vec(tox, toy),
				  v_vec(fromx2, fromy2), v_vec(tox2, toy2));
		if (intersection == NULL)
		    continue;

		/* got one! */
		SplitContourAfter(cno, j - 1,
				  intersection->x, intersection->y);
		/* p2 may have been freed */
		p2 = Contours[cno].p;
		p = InsertPointIntoPointList(p, i - 1,
					  intersection->x, intersection->y);
		seg[segno++] = i;
		i++;
		j++;
	    }
	}
    }

    seg[segno] = p->n - 1;
    *pp = p;
    return segno;
}

/* copy p from from to to into an allocated point list */
PointList *
AllocCopyPointList(p, from, to)
PointList *p;
int     from, to;
{
    int     n;
    PointList *newp;

    if (from < 0 || from >= p->size
	|| to < 0 || to >= p->size
	|| from > to) {
	fprintf(stderr, "AllocCopyPointList called with bad args--fail\n");
	abort();
    }
    n = to - from + 1;
    newp = NewPointList(n);
    newp->n = n;
    /* newp->{closed, withcorner, contin} all left FALSE */

    CopyPointList(p, from, newp, 0, n);
    return newp;
}

/*
 * return an angle between -pi and pi
 */
real
AngleFix(a)
real    a;
{
    while (a > M_PI)
	a -= 2 * M_PI;
    while (a < -M_PI)
	a += 2 * M_PI;

    return a;
}

/*
 * Make a new copy of pointlist porig with point (x,y) inserted after point
 * pno.  Frees porig.
 */
PointList *
InsertPointIntoPointList(porig, pno, x, y)
PointList *porig;
int     pno;
real    x, y;
{
    PointList *p;

    p = NewPointList(porig->n + 1);
    p->n = porig->n + 1;
    CopyPointList(porig, 0, p, 0, pno + 1);
    CopyPointList(porig, pno + 1, p, pno + 2, porig->n - pno - 1);
    p->x[pno + 1] = x;
    p->y[pno + 1] = y;
    p->theta[pno + 1] = porig->theta[pno];
    p->k[pno + 1] = porig->k[pno];
    p->closed = porig->closed;
    p->withcorner = porig->withcorner;
    p->contin = porig->contin;
    p->boundary = porig->boundary;
    FreePointList(porig);
    return p;
}


/*
 * add a point (x,y) after point pno in contour cno; then split at that point
 * UNLESS (x,y) is point pno.  Then just split there
 */
SplitContourAfter(cno, pno, x, y)
int     cno, pno;
real    x, y;
{
    Contour *C = &Contours[cno];

    if (C->p->x[pno] == x && C->p->y[pno] == y) {
	if (pno == 0 || pno == C->p->n - 1) {
	    fprintf(stderr, "Split contour at %s, bookkeeping error\n",
		    pno == 0 ? "HEAD" : "TAIL");
	    abort();
	}
	SplitContour(cno, pno + 1);
    } else {
	C->p = InsertPointIntoPointList(C->p, pno, x, y);
	SplitContour(cno, pno + 1);
    }
}
