/***
 * Copyright (c) 1990,1,2 Mark Nitzberg
 * and President and Fellows of Harvard College
 * All rights reserved.
 *
 * Functions to best-fit circles or lines to sets of points.
 *
 * NOTE: all the functions deal in type "real" (typedef just below).
 *	 Just for good measure, internal calculations in the
 *	 primitive fitcl() are carried out in double.
 */
#include <math.h>
#include <stdio.h>
#include <malloc.h>

#define HUGE_ERROR 10000

typedef float real;

void RefineSampledCurve();
real RefinePoint();
real    fitcl();


/* for all intents and purposes */
#define	ESSENTIALLY_ZERO	(1.0e-8)


/*
 * compile with -DTESTING to get a standalone test program
 */

#ifdef TESTING
#define N 1000
#define DEG(angle_in_radians) (180.0*(angle_in_radians)/M_PI)

char    Usage[] = "\
usage: %s [-s sigma] [-w window_size] [input_file]\n\
\n\
Works in one of two ways:\n\
1. IF THE FIRST INPUT LINE has 3 numbers, then reads a list of points\n\
   with weights (3 numbers per line) and produces\n\
   mathematica input so you can see how well\n\
   these fitting function works, showing a single best-fit\n\
   to all the points with the given weights;\n\
\n\
2. OTHERWISE the first line must have two numbers (width height)\n\
   in the form produced by \"curves\".  Produces a list of refined\n\
   curves by fitting window_size points at a time,\n\
   using a gaussian weighting function of std dev Sigma (default 2.5).\n\
   The output can be printed using pscurves.  window_size defaults to 4*sigma.\n";

main(argc, argv)
int     argc;
char  **argv;
{
    real    x[N], y[N], w[N], newx[N], newy[N], tanangle[N], k[N], err[N];
    real    cl[4];		/* coefficients of circle or line eqn */
    char    buf[512];
    real    sigma = 2.5;
    int     m = 0;
    int     i, n;
    char    c;
    real    width, height;
    real    err1;

    extern char *optarg;
    extern int optind;

    char   *PgmName = argv[0];

    while ((c = getopt(argc, argv, "s:w:")) != -1) {

	switch (c) {

	case '?':		/* getopt() arg not in list */
	    fprintf(stderr, Usage, PgmName);
	    exit(1);

	case 's':
	    sigma = atof(optarg);
	    break;
	case 'w':
	    m = atof(optarg);
	    break;
	}
    }

    if (optind < argc) {
	if (freopen(argv[optind], "r", stdin) == NULL) {
	    fprintf(stderr, "Can't open %s for input\n", argv[optind]);
	    exit(1);
	}
    }
    if (m == 0)
	m = (int) (0.5 + 4 * sigma);

    /* read in a list of points */
    n = 0;
    while (gets(buf) != NULL) {
	if (sscanf(buf, "%g%g%g", &x[n], &y[n], &w[n]) != 3)
	    break;
	n++;
    }

    if (n > 0) {

	/* just test the fitting function */
	printf("(* %d points as follows: *)\n", n);

	/* mathematica list of points */
	printf("points={\n");
	for (i = 0; i < n; i++) {
	    printf(" {%.17f,%.17f (*w %g*)}%s\n",
		   x[i], y[i],
		   w[i],
		   i < n - 1 ? "," : "");
	}
	printf("}\n");

	/* find best circle or line */
	err1 = fitcl(x, y, w, n, cl);

	MathematicaShow(x, y, w, n, cl, err1);

	exit(0);
    }

    /*
     * refine all the curves
     */

    /* input width height */
    while (buf[0] == '#' || buf[0] == '\n' || buf[0] == '\0')
	if (gets(buf) == NULL)
	    exit(0);

    if (sscanf(buf, "%g%g", &width, &height) != 2) {
	fprintf(stderr, Usage, PgmName);
	exit(1);
    }
    printf("# width height\n%g %g\n\n", width, height);
    printf("# Curves, separated by blank lines:\n# x y tangent_angle k displacement\n");

    while (!feof(stdin)) {

	n = ReadCurve(stdin, x, y, N);
	if (n < 3) {
	    if (n > 0)
		fprintf(stderr,
			"Dropping %d-point curve at (%g,%g).\n",
			n, x[0], y[0]);
	    continue;
	}
	/* update the points with the given sigma and m */
	RefineSampledCurve(x, y, n, m, sigma, newx, newy, tanangle, k, err);

	printf("# CONTOUR %d POINTS\n", n);
	for (i = 0; i < n; i++) {
	    printf("%g %g %g %g %g\n",
		   newx[i], newy[i], DEG(tanangle[i]), k[i], err[i]);
	}
	printf("\n");
    }
}

int
ReadCurve(stream, xarr, yarr, maxpoints)
FILE   *stream;
real   *xarr, *yarr;
int     maxpoints;
{
    char   *fgets();
    char    buf[512];
    int     point;

    if (feof(stream))
	return 0;

    point = 0;

    while (fgets(buf, 512, stream) != NULL) {

	if (buf[0] == '#' || buf[0] == '\n' || buf[0] == '\0')
	    break;
	if (point == maxpoints) {
	    fprintf(stderr, "Point %d on a curve--too many, truncating here\n");
	    break;
	}
	if (sscanf(buf, "%g%g", &xarr[point], &yarr[point]) != 2) {
	    fprintf(stderr, "Point %d on a curve--not understood\n");
	    return 0;
	}
	point++;
    }
    return point;
}


/*
 * show an eigenvector cl produced by fit circle-line algorithm
 */
MathematicaShow(x, y, w, n, cl, err)
real   *x, *y, *w, *cl;
int     n;
{
    void    vecminmax();
    real    minx, maxx, miny, maxy, range;
    real    meanx, meany, wtot;
    real    a, b, c, d, A, B, R;
    int     i;

    printf("(* fitcl() gives\n%g  +  %g x  +  %g y  +  %g (x^2+y^2)  =  0, err = %g *)\n",
	   cl[0], cl[1], cl[2], cl[3], err);




    a = cl[0];
    b = cl[1];
    c = cl[2];
    d = cl[3];

    /* put a border around graphs */
    vecminmax(y, n, &miny, &maxy);
    vecminmax(x, n, &minx, &maxx);
    range = maxx - minx;
    minx -= range / 5;
    maxx += range / 5;
    miny -= range / 5;
    maxy += range / 5;

    if (fabs(d) < ESSENTIALLY_ZERO) {

	/* straight line bx + cy + a = 0 for our purposes! */
	printf("Show[ListPlot[points, Prolog->{PointSize[.02]}, AspectRatio->Automatic]\n");
	if (c > ESSENTIALLY_ZERO) {
	    /* y = -b/c x - a */
	    printf(",Plot[%.17f x + %.17f, {x,%.17f,%.17f}, AspectRatio->Automatic]]\n",
		   -b / c, -a, minx, maxx);
	} else {
	    /* x = -c/b y - a */
	    printf(",Plot[%.17f y + %.17f, {y,%.17f,%.17f}, AspectRatio->Automatic]]\n",
		   -c / b, -a, miny, maxy);
	}
    } else {
	real    thetamin, thetamax;

	/* circle.  Compute center (A,B) and radius R */
	A = -b / (2 * d);
	B = -c / (2 * d);
	R = (-a / d + A * A + B * B);
	if (R < 0) {
	    printf("Negative radius!\n");
	    R = -R;
	}
	R = sqrt(R);

	/* find point on circle closest to center of {(xi,yi)} */
	meanx = meany = wtot = 0;
	for (i = 0; i < n; i++) {
	    meanx += w[i] * x[i];
	    meany += w[i] * y[i];
	    wtot += w[i];
	}
	meanx /= wtot;
	meany /= wtot;

	/* no, use thetamin the x,y of least angle and thetamax of most angle */
	thetamin = atan2(y[0] - B, x[0] - A);
	thetamax = atan2(y[n - 1] - B, x[n - 1] - A);
	if (thetamin > thetamax) {
	    real    temp = thetamin;

	    thetamin = thetamax;
	    thetamax = temp;
	}
	/* draw an arc */
	printf("Show[ListPlot[points, Prolog->{PointSize[.02]}, AspectRatio->Automatic]\n");
	printf(",Graphics[Circle[{%.17f,%.17f},%.17f,{%.17f,%.17f}]],AspectRatio->Automatic]\n",
	       A, B, R, thetamin, thetamax);
    }
}


void
vecminmax(v, n, minp, maxp)
real   *v;
int     n;
real   *minp, *maxp;
{
    int     i;
    real    min, max;

    if (n == 0)
	return;
    min = v[0];
    max = v[0];

    for (i = 1; i < n; i++) {
	if (min > v[i])
	    min = v[i];
	if (max < v[i])
	    max = v[i];
    }
    *minp = min;
    *maxp = max;
}

#endif	/* TESTING */





/*
 * Library functions for sampled curve smoothing and segmentation
 */

/*
 * RefinePoint(x,y,w,n,i,x2,y2,tan_angle,k) -- localize point & get
 * tan_angle. & curv.
 * 
 * Given n points (x[],y[]) with weights w[] finds the best weighted fit to a
 * circle or line using fitcl() and computes a refined location, tangent and
 * curvature for the single point (x[i],y[i]), putting it in parameters
 * (x2,y2). (x2,y2) is the point on the circle or line closest to
 * (x[i],y[i]). tan_angle gets the angle of the tangent to the circle or line
 * at (x2,y2) and k gets the curvature (1/radius of circle, or 0 if line).
 * Returns the weighted average error (distance units).
 */

real
RefinePoint(x, y, w, n, i, x2, y2, tan_angle, k)
real    x[], y[], w[];
int     n, i;
real   *x2, *y2, *tan_angle, *k;
{
    real    cl[4];
    real    a, b, c, d;
    double  dx, dy, err;

    err = fitcl(x, y, w, n, cl);

    a = cl[0];
    b = cl[1];
    c = cl[2];
    d = cl[3];

    /* line or circle? */
    if (fabs(d) <= ESSENTIALLY_ZERO) {

	/* straight line bx + cy + a = 0 */
	real    temp;

	*k = 0;			/* zero curvature */

	/* point on line closest to x[i],y[i] */
	temp = b * b + c * c;
	*x2 = (-(a * b) + c * c * x[i] - b * c * y[i]) / temp;
	*y2 = (-(a * c) - b * c * x[i] + b * b * y[i]) / temp;

	/* tangent vector is (c,-b) */
	*tan_angle = atan2(-b, c);

    } else {

	/* circle.  Compute center (A,B) and radius R */
	double  A, B, R, Rovernorm;
	real    angle1, angle2;

	A = -b / (2 * d);
	B = -c / (2 * d);
	R = (-a / d + A * A + B * B);
	if (R < 0) {
	    fprintf(stderr, "circle fit has negative radius!\n");
	    R = -R;
	}
	R = sqrt(R);

	/* abs(curvature)  Choose sign below */
	*k = 1.0 / R;		/* curvature = 1/radius */

	/*
	 * sign of curvature is + if the ray (A,B) to the ith point must
	 * sweep counterclockwise to meet the ray (A,B) to the i+1st point
	 */
	if (i == n - 1) {

	    /* get arg( (i-1)st point - (A,B)) and for ith point */
	    angle1 = atan2(B - y[i - 1], A - x[i - 1]);
	    angle2 = atan2(B - y[i], A - x[i]);
	} else {

	    /* get arg(ith point - (A,B)) and for (i+1)st point */
	    angle1 = atan2(B - y[i], A - x[i]);
	    angle2 = atan2(B - y[i + 1], A - x[i + 1]);
	}

	/* Assure they don't jump the ray at Pi (= -Pi) (just add Pi to both) */
	if (angle1 * angle2 < 0 && M_PI - fabs(angle1) + M_PI - fabs(angle2) < M_PI) {
	    angle1 -= M_PI;
	    if (angle1 < -M_PI)
		angle1 += 2 * M_PI;
	    angle2 -= M_PI;
	    if (angle2 < -M_PI)
		angle2 += 2 * M_PI;
	}
	/* negative curvature if sweep was clockwise */
	if (angle2 < angle1)
	    *k = -(*k);

	/* find point on circle closest to (x[i],y[i]) */
	dx = x[i] - A;
	dy = y[i] - B;
	Rovernorm = R / sqrt(dx * dx + dy * dy);
	*x2 = x[i] * Rovernorm + A * (1 - Rovernorm);
	*y2 = y[i] * Rovernorm + B * (1 - Rovernorm);

	/* tangent to circle at that point (vector -dy,dx) */
	*tan_angle = atan2(dx, -dy);
    }
    return err;
}

/*
 * RefineSampledCurve(x,y,n,m,sigma,newx,newy,tan_angle,k,err)
 * 
 * This produces a complete new curve (newx[i],newy[i]) with tangents
 * tan_angle[i], curvatures k[i] and the weighted average errors err[i] of
 * each fit. m (<= n) is the number of points to fit at a time, and Sigma is
 * the standard deviation of a gaussian weighting function used to fit nearer
 * points better. With m = 8 Sigma, the weighting function is 0 at the
 * borders of the m-pixel window.  Maybe m = 2 Sigma will work best.
 * 
 * ENDPOINTS:  simulate reflection at the boundaries. Build a weighting function
 * in 2m+1 pixels, and use different m-point portions of it depending on how
 * close we are to boundaries.
 */
void
RefineSampledCurve(x, y, n, m, sigma, newx, newy, tan_angle, k, err)
real    x[], y[];
int     n, m;
real    sigma;
real    newx[], newy[], tan_angle[], k[], err[];
{
    real   *w;
    int     i, stdoff;

    /* need at least 3 points */
    if (n < 3) {
	fprintf(stderr,
		"RefineSampledCurve() not bothering, curve < 3 pts.\n");
	return;
    }
    /* now build the weight function as if we have a long curve */

    /* odd size */
    if ((m & 1) == 0) {
	if (m == n)
	    m--;
	else
	    m++;
    }
    /* real the size and add one */
    w = (real *) calloc(2 * m + 1, sizeof (real));
    DrawGauSpline1D(w, 2 * m + 1, sigma);

    /* now m is the center of the weight function */

    /*
     * stdoff -- standard offset in both directions from ith point in x, and
     * from center in w Note that m must be <= n
     */
    if (m > n) {
	m = n;
	if ((m & 1) == 0)
	    m--;
    }
    stdoff = (m - 1) / 2;

    for (i = 0; i < n; i++) {
	int     xoff, woff;

	xoff = i - stdoff;
	woff = m - stdoff;

	/* clip at left boundary */
	if (xoff < 0) {
	    /* move both offsets right */
	    woff += -xoff;
	    xoff = 0;
	}
	/* clip at right boundary */
	if (xoff + m > n) {

	    /* move both offsets left */
	    woff -= (xoff + m) - n;
	    xoff -= (xoff + m) - n;
	}
	err[i] =
	    RefinePoint(x + xoff, y + xoff, w + woff,
			m, i - xoff,
			&newx[i], &newy[i], &tan_angle[i], &k[i]);
    }
    free(w);
}


/*
 * local functions for fitcl() not available outside this file
 */
static double *vector ();
static double **matrix();
static void free_vector();
static void free_matrix();
static void nrerror();
static void jacobi();

/*
 * fitcl(x,y,w,n,cl) --
 * 
 * Fills cl with the 4 values  a,b,c,d  so that the
 * 
 * a + bx + cy + d(x^2+y^2)
 * 
 * is a "best fit" to the points (x[i],y[i]) i=0..n-1 with strength of fit to
 * each point weighted by w[i]. If you don't care about these strengths, let
 * w[i]=1.
 * 
 * RETURNS the weighted average error in distance units.
 */
real
fitcl(x, y, w, n, cl)
real   *x, *y;
real   *w;			/* weights for the (x[i],y[i]) */
int     n;			/* size of (0-based) x,y,w */
real    cl[4];			/* the resulting a,b,c,d = cl[0,1,2,3] */
{
    double **tAwA;		/* 4x4 matrix comprised of A transpose * A */
    double *r;			/* x[i]^2 + y[i]^2 */
    double **O1, **O2;		/* eigenvector orthogonals */
    double **E1, **E2;		/* intermediate 4x4's */
    double **A2inv, **temp;
    double *D1, *D2;		/* eigenvalue diagonals */
    double  wsum;		/* sum of the weights w[i] */
    real    err;		/* return value */
    int     i, j, k, nrot;

    if (n < 3) {
	fprintf(stderr,
		"fitcl() can't fit %d (< 3) points\n", n);
	return HUGE_ERROR;
    }

    /*
     * /  1    xi     yi     ri  \ |			   |   each entry
     * mult. by wi, |  xi  xi^2   xi*yi  xi*ri |   then summed over i tAwA =
     * |			   | |  yi  yi*xi  yi^2   yi*ri |   "xi" =
     * x[i] |			   |   "ri" = (xi^2 + yi^2) \  ri  ri*xi
     * ri*yi  ri^2  /
     * 
     */

    tAwA = matrix(1, 4, 1, 4);
    r = vector (0, n - 1);

    for (i = 0; i < n; i++)
	r[i] = x[i] * x[i] + y[i] * y[i];

    /* fill upper triangle first, then reflect */
    for (i = 0; i < n; i++) {

	tAwA[1][1] += w[i];
	tAwA[1][2] += w[i] * x[i];
	tAwA[1][3] += w[i] * y[i];
	tAwA[1][4] += w[i] * r[i];

	tAwA[2][2] += w[i] * x[i] * x[i];
	tAwA[2][3] += w[i] * x[i] * y[i];
	tAwA[2][4] += w[i] * x[i] * r[i];

	tAwA[3][3] += w[i] * y[i] * y[i];
	tAwA[3][4] += w[i] * y[i] * r[i];

	tAwA[4][4] += w[i] * r[i] * r[i];
    }

    /* reflect upper triangle to lower */
    for (i = 2; i <= 4; i++)
	for (j = 1; j < i; j++)
	    tAwA[i][j] = tAwA[j][i];

    /* sum w[0..n-1] */
    wsum = tAwA[1][1];

    /*
     * tAwA  = O1 D1 tO1
     */
    O1 = matrix(1, 4, 1, 4);
    D1 = vector (1, 4);

    jacobi(tAwA, 4, D1, O1, &nrot);

    /*
     * take the inverse square root of the diagonals
     */
    for (i = 1; i <= 4; i++) {
	if (fabs(D1[i]) == ESSENTIALLY_ZERO) {

	    /* exact fit to line / circle; return eigenvector (column) */
	    j = i;
	    for (i = 1; i <= 4; i++)
		cl[i - 1] = O1[i][j];
	    return 0.0;
	}
	/* don't muck with negatives that come from lost precision */
	D1[i] = 1 / sqrt(fabs(D1[i]));
    }

    /*
     * now D1 = sqrt(D1^-1).
     * 
     * /  0  0  0 -2 \ E1 =  |  0  1  0  0 | |  0  0  1  0 | \ -2  0  0  0 /
     * 
     * Find E2 = A2inv E1 A2inv   where A2inv = O1 D1 tO1
     * 
     */
    E1 = matrix(1, 4, 1, 4);
    E1[1][4] = -2;
    E1[2][2] = 1;
    E1[3][3] = 1;
    E1[4][1] = -2;

    A2inv = matrix(1, 4, 1, 4);

    for (i = 1; i <= 4; i++)
	for (j = 1; j <= 4; j++)
	    for (k = 1; k <= 4; k++)
		A2inv[i][j] += O1[i][k] * D1[k] * O1[j][k];

    E2 = matrix(1, 4, 1, 4);
    temp = matrix(1, 4, 1, 4);

    /* temp = A2inv E1 */
    for (i = 1; i <= 4; i++)
	for (j = 1; j <= 4; j++)
	    for (k = 1; k <= 4; k++)
		temp[i][j] += A2inv[i][k] * E1[k][j];

    /* E2 = temp A2inv */
    for (i = 1; i <= 4; i++)
	for (j = 1; j <= 4; j++)
	    for (k = 1; k <= 4; k++)
		E2[i][j] += temp[i][k] * A2inv[k][j];

    /* get O2,D2 eigen{values,vectors} of E2 */
    O2 = matrix(1, 4, 1, 4);
    D2 = vector (1, 4);

    jacobi(E2, 4, D2, O2, &nrot);

    j = -1;			/* get index of largest pos eigenvalue */
    for (i = 1; i <= 4; i++)
	if (D2[i] > 0 && (j == -1 || D2[i] > D2[j]))
	    j = i;

    if (j == -1) {
	fprintf(stderr,
		"fitcl() no positive eigenvalues.\n");
	return HUGE_ERROR;
    }
    /* zero result array */
    for (i = 1; i <= 4; i++)
	cl[i - 1] = 0;

    /* Apply A2inv to jth eigenvector (column); result in 0-based array cl */
    for (i = 1; i <= 4; i++)
	/* j fixed at column eigenvector */
	for (k = 1; k <= 4; k++)
	    cl[i - 1] += A2inv[i][k] * O2[k][j];

    /* the jth eigenvalue gives the error */
    err = 1.0 / sqrt(D2[j] / wsum);	/* err has units of distance */

    free_vector(r, 0, n - 1);
    free_vector(D1, 1, 4);
    free_vector(D2, 1, 4);
    free_matrix(tAwA, 1, 4, 1, 4);
    free_matrix(O1, 1, 4, 1, 4);
    free_matrix(E1, 1, 4, 1, 4);
    free_matrix(A2inv, 1, 4, 1, 4);
    free_matrix(O2, 1, 4, 1, 4);
    free_matrix(E2, 1, 4, 1, 4);
    free_matrix(temp, 1, 4, 1, 4);

    return err;
}

/*
 * numerical recipes utilities
 */
static void
nrerror(error_text)
char    error_text[];
{
    fprintf(stderr, "Numerical Recipes run-time error...\n");
    fprintf(stderr, "%s\n", error_text);
    fprintf(stderr, "...now exiting to system...\n");
    exit(1);
}



static double
       *vector (nl, nh)
int     nl, nh;
{
    double *v;

    v = (double *) calloc((unsigned) (nh - nl + 1), sizeof (double));
    if (!v)
	nrerror("allocation failure in vector()");
    return v - nl;
}


static double
      **
matrix(nrl, nrh, ncl, nch)
int     nrl, nrh, ncl, nch;
{
    int     i;
    double **m;

    m = (double **) calloc((unsigned) (nrh - nrl + 1), sizeof (double *));
    if (!m)
	nrerror("allocation failure 1 in matrix()");
    m -= nrl;

    for (i = nrl; i <= nrh; i++) {
	m[i] = (double *) calloc((unsigned) (nch - ncl + 1), sizeof (double));
	if (!m[i])
	    nrerror("allocation failure 2 in matrix()");
	m[i] -= ncl;
    }
    return m;
}


static void
free_vector(v, nl, nh)
double *v;
int     nl, nh;
{
    free((char *) (v + nl));
}

static void
free_matrix(m, nrl, nrh, ncl, nch)
double **m;
int     nrl, nrh, ncl, nch;
{
    int     i;

    for (i = nrh; i >= nrl; i--)
	free((char *) (m[i] + ncl));
    free((char *) (m + nrl));
}

/*
 * jacobi(a,n,d,v,nrot) -- diagonalize find d,v such that
 * 
 * a = tv d v      for nxn matrix a.
 * 
 * Puts number of jacobi rotations required into *nrot.
 */
#define ROTATE(a,i,j,k,l) g=a[i][j];h=a[k][l];a[i][j]=g-s*(h+g*tau);\
     a[k][l]=h+s*(g-h*tau);

static void
jacobi(a, n, d, v, nrot)
double **a, d[], **v;
int     n, *nrot;
{
    int     j, iq, ip, i;
    double  tresh, theta, tau, t, sm, s, h, g, c, *b, *z;

    b = vector (1, n);
    z = vector (1, n);

    for (ip = 1; ip <= n; ip++) {
	for (iq = 1; iq <= n; iq++)
	    v[ip][iq] = 0.0;
	v[ip][ip] = 1.0;
    }
    for (ip = 1; ip <= n; ip++) {
	b[ip] = d[ip] = a[ip][ip];
	z[ip] = 0.0;
    }
    *nrot = 0;
    for (i = 1; i <= 50; i++) {
	sm = 0.0;
	for (ip = 1; ip <= n - 1; ip++) {
	    for (iq = ip + 1; iq <= n; iq++)
		sm += fabs(a[ip][iq]);
	}
	if (sm == 0.0) {
	    free_vector(z, 1, n);
	    free_vector(b, 1, n);
	    return;
	}
	if (i < 4)
	    tresh = 0.2 * sm / (n * n);
	else
	    tresh = 0.0;
	for (ip = 1; ip <= n - 1; ip++) {
	    for (iq = ip + 1; iq <= n; iq++) {
		g = 100.0 * fabs(a[ip][iq]);
		if (i > 4 && fabs(d[ip]) + g == fabs(d[ip])
		    && fabs(d[iq]) + g == fabs(d[iq]))
		    a[ip][iq] = 0.0;
		else if (fabs(a[ip][iq]) > tresh) {
		    h = d[iq] - d[ip];
		    if (fabs(h) + g == fabs(h))
			t = (a[ip][iq]) / h;
		    else {
			theta = 0.5 * h / (a[ip][iq]);
			t = 1.0 / (fabs(theta) + sqrt(1.0 + theta * theta));
			if (theta < 0.0)
			    t = -t;
		    }
		    c = 1.0 / sqrt(1 + t * t);
		    s = t * c;
		    tau = s / (1.0 + c);
		    h = t * a[ip][iq];
		    z[ip] -= h;
		    z[iq] += h;
		    d[ip] -= h;
		    d[iq] += h;
		    a[ip][iq] = 0.0;
		    for (j = 1; j <= ip - 1; j++) {
			ROTATE(a, j, ip, j, iq)
		    }
		    for (j = ip + 1; j <= iq - 1; j++) {
			ROTATE(a, ip, j, j, iq)
		    }
		    for (j = iq + 1; j <= n; j++) {
			ROTATE(a, ip, j, iq, j)
		    }
		    for (j = 1; j <= n; j++) {
			ROTATE(v, j, ip, j, iq)
		    }
		    ++(*nrot);
		}
	    }
	}
	for (ip = 1; ip <= n; ip++) {
	    b[ip] += z[ip];
	    d[ip] = b[ip];
	    z[ip] = 0.0;
	}
    }
    nrerror("Too many iterations in routine JACOBI");
}

#undef ROTATE
