/*
 * Copyright (c) 1990,1,2 Mark Nitzberg
 * and President and Fellows of Harvard College
 * All rights reserved.
 * 
 * curves -- trace edges and complete across gaps.
 * 
 * Every function and macro here takes x,y rather than i,j coordinates.
 */

char    Usage[] = "\
usage: %s [-a angl] [-b B] [-c ct] [-C npix] [-g G] [-m M]\n\
	  [-o] [-p] [-s Sdev] [-t Lo Hi] [-x lev] image-file\n\
\n\
Traces curves and writes to image-file with extension change to .e\n\
\n\
 -a angl  growing endpoints can bend +/- angl degrees (def. 15)\n\
 -b B	  delete contours < B pixels before gap-jumping, default 2*SDev\n\
 -c ct	  cornerness threshold is ct%%ile, default 98\n\
 -C npix  cornerness radius (default 4*Sdev)\n\
 -g G	  maximum gap to jump is G pixels, default 3.2*Sdev\n\
 -m M	  minimum self-contained contour length is M pixels, default 2*SDev\n\
 -o	  draw an outline around the image before edge-tracing\n\
 -p	  prune crack tips\n\
 -s Sdev  std dev of gaussian (default 1.25) for smoothing and length scales\n\
 -t Lo Hi edge strength thresholds in %%ile, default lo=30, hi=60\n\
 -x lev	  output level 1..10, 10 means print every step. See DBUG in the code\n";

#include <hvision.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include "vector.h"

/*
 * DEBUG MACROS:
 * 
 * Note that the assert() calls produce nice complaints with source line numbers
 * and can be defeated, producing *no* code, by compiling with -DNDEBUG
 */

#ifndef NDEBUG
#define	warn(string)	warning(string)
#define assert(ex)	if (!(ex)) {\
			    fprintf(stderr,\
				"Assertion failed: file \"%s\", line %d\n",\
				__FILE__, __LINE__);\
			    abort();\
			}
#else
#define warn(string) 0
#define assert(ex)
#endif

/*
 * Debug print statements again, eliminated by -DNDEBUG NOTE:  use
 * double-parentheses so macro preproc. can handle varargs
 */
int     DebugLevel = 0;

#ifndef NDEBUG
#define DBUG(level,x)	{ if (DebugLevel >= level) { printf x; fflush(stdout); }}
#else
#define	DBUG(level,x)
#endif


/*
 * parameters that might as well be arguments.
 */
#define	LENGTH_SCALE_IN_SDEVS 4.0	/* window size for curvature
					 * computation, etc. */
#define MIN_CLOSED_CONTOUR (12.0*SDev)	/* # pixels in the minimum closed
					 * contour (no little squares) */
#define TOO_SOON_TO_DOUBLE_BACK MIN_CLOSED_CONTOUR-1	/* don't run into
							 * ourself within this
							 * many steps */
#define	MAX_JUMP_IN_SDEVS 3.2	/* length of largest gap to jump, measured in
				 * SDev's */

#define IROUND(x)	((int) ((x)+0.5))

/* degrees <-> radians */
#define	DEG(x)	((x) * 180. / M_PI)
#define	RAD(x)	((x) * M_PI / 180.)

/* pixel colors for edge "bitmap" */
#define BLACK 0
#define WHITE 255

/* pixel position type */
typedef struct PixelPos {
    int     x, y;
}       PixelPos;

/***
 * neighbor shorthand:  0 for center
 *	4 3 2
 *	5 0 1
 *	6 7 8
 */
typedef unsigned char Neighbor;

/* NeighborPos[neighbor] gives a PixelPos of neighbor's relative pos. */
PixelPos NeighborPos[] = {
    {0, 0},
    {1, 0}, {1, 1}, {0, 1}, {-1, 1},
    {-1, 0}, {-1, -1}, {0, -1}, {1, -1}
};

#define NEIGHBORS (sizeof(NeighborPos)/sizeof(*NeighborPos))

/* opposites of neighbor directions */
Neighbor AntiNeighbor[] = {
    0, 5, 6, 7, 8, 1, 2, 3, 4
};


/*
 * Contour: length, first and last pixel positions
 */
typedef struct Contour {
    struct Contour *R;		/* next in linked list */

    int     length;		/* number of pixels involved */
    short   closed;		/* TRUE if closed curve */
    short   split;		/* TRUE if closed & HEAD is corner point */
    short   boundary;		/* TRUE if contour is along image boundary */

    /* the two ends */
    PixelPos end[2];

#define	HEAD	0
#define	TAIL	1

} Contour;

/* linked list of contours */
Contour *Contours;
int     NContours = 0;


/*
 * Information kept for each pixel in the pixel map: type, contours that run
 * through here
 * 
 * WARNING:  read the comment about n[]
 */
typedef struct ContourPoint {
    Contour *con;		/* contour to which this is associated */
    real    angle;		/* (radians) edge tangent angle */
    Neighbor n[2];		/* pixel directions of up to two neighbors
				 * Contours are oriented: go from
				 * Contour->end[HEAD] by tracing the n[0]'s,
				 * etc.  At ends (T, X, corner, END) note
				 * that n[0] OR n[1] holds the neighbor
				 * direction; the other has value 0 */

#define FORWARD 0		/* n[0] takes you forward along con */
#define BACKWARD 1

}       ContourPoint;

/***
 * Types of pixels:
 *
 *  P_EMPTY	not on any contour
 *  P_POINT	one contour consists of this isolated point
 *  P_END		one contour ends here
 *  P_MIDDLE	middle of a contour somewhere
 *  P_CORNER	contours end here, different tangents
 *  P_T	 3 contours end here, 2 w/same tangent
 *  P_Y	 3 contours end here, 2 w/same tangent
 *  P_X	 4 contours end here, 2 pairs of matching tangents
 */

typedef enum PixelType {
    P_EMPTY, P_POINT, P_END, P_MIDDLE, P_CORNER, P_T, P_Y, P_X,
}       PixelType;

typedef struct PixelInfo {

    PixelType type;

    vector  pos;		/* possibly refined position */
    real    k;			/* curvature if this is a P_MIDDLE point */

    /***
     * conpt[] -- up to four contours can hit each pixel.
     *  P_EMPTY	all conpt[i].n[j] are 0
     *  P_POINT	conpt[i].n[j] all 0; conpt[0].con is trivial contour
     *  P_END	conpt[0].n[0 or 1], whichever!=0, tells which way to go
     *  P_MIDDLE 	conpt[0].n[0 and 1] tell 2 ways to go
     *  P_CORNER	conpt[0,1].n[0 or 1] tell 2 ways to go
     *  P_T or P_Y	conpt[0,1,2].n[0 or 1] tell 3 ways to go
     *		in case of a T, conpt[0] is the "stem"
     *  P_X		conpt[0,1,2,3].n[0 or 1] tell 4 ways to go
     */
    ContourPoint conpt[4];
}       PixelInfo;

int     NPixelInfos = 0;

/* string to print contour type */
#define TYPE_STRING(t)\
	((t)==P_EMPTY ? "EMPTY" :\
	 (t)==P_POINT ? "POINT" :\
	 (t)==P_END ? "END" :\
	 (t)==P_MIDDLE ? "MIDDLE" :\
	 (t)==P_CORNER ? "CORNER" :\
	 (t)==P_T ? "T" :\
	 (t)==P_Y ? "Y" :\
	 (t)==P_X ? "X" :\
	 "<unknown-type>")

/*
 * shorthand to skip through a contour, point by point
 */
/* which conpt in a pixinfo structure == con ? if none, return NO_CONTOUR */
#define NO_CONTOUR (-1)
#define CONPT(pixinfo, c) \
       ((c) == (pixinfo)->conpt[0].con ? 0 :\
	(c) == (pixinfo)->conpt[1].con ? 1 :\
	(c) == (pixinfo)->conpt[2].con ? 2 :\
	(c) == (pixinfo)->conpt[3].con ? 3 : NO_CONTOUR)

/* which conpt in a pixinfo structure == con ? if none, fail */
#define CONPT_force(pixinfo, c) \
       ((c) == (pixinfo)->conpt[0].con ? 0 :\
	(c) == (pixinfo)->conpt[1].con ? 1 :\
	(c) == (pixinfo)->conpt[2].con ? 2 :\
	(c) == (pixinfo)->conpt[3].con ? 3 : warn("Pixel did not belong to contour"))

/* angle given pixel and contour */
#define C_ANGLE(pixinfo, c)  ((pixinfo)->conpt[CONPT_force(pixinfo, c)].angle)

/*
 * Pixel maps: edge strength, x and y components of edge direction, and
 * cornerness measure
 */
IMAGE  *Strength, *Edgex, *Edgey, *Cornerness;

/* avoid possible macro conflicts */

#ifdef PIXEL
#undef PIXEL
#endif

/* access any pixel of an image as any type, using x,y */
#define __PIXEL(I,x,y,t)	((t **)((I)->twod))[y][x]

#define	BYTE(I,x,y)	__PIXEL(I,x,y,unsigned char)
#define	PIX(I,x,y)	__PIXEL(I,x,y,real)
#define	EDGE_X(x,y)	PIX(Edgex, x, y)
#define	EDGE_Y(x,y)	PIX(Edgey, x, y)
#define	EDGE_STRENGTH(x,y) PIX(Strength, x, y)
#define	CORNERNESS(x,y)	PIX(Cornerness, x, y)

/*
 * The Pixel map:  Pixel Info at each pixel, with subtleties.
 * 
 * PMap(x,y) points to an ContourPoint structure nearer to pixel x,y than any
 * other pixel. NULL if there are none. The area associated to a pixel is a
 * circle of radius 1/sqrt(2), with only randomness to handle the
 * ambiguities.
 */
IMAGE  *PixelMap;

#define	PMap(x,y)	__PIXEL(PixelMap, (x), (y), PixelInfo *)


/* TRUE if two contours meeting at their endpoints join into one */
int     AlwaysMergeCorners = FALSE;

/* function types */
char   *calloc(), *malloc();
char   *FileName();

ContourPoint *NewContourPoint();

PixelInfo *NewPixelInfo();
PixelInfo *DistanceFromEndP();
PixelInfo *NthFromEndP();

real    ThresholdPercentile();
real    Compatible();
real    Curvature();
real    AngleFix();
real    ReflectAngle();

vector  Reflect();
vector  pptovec();

PixelPos vectopp();



/*
 * hello.
 */

/* command-line parameters */
char   *PgmName;
char   *InputFileName;
real    SDev = 1.25;
real    LoPercentile = 30.;
real    HiPercentile = 60.;
real    SweepAngle = 15;	/* degrees +/- that growing contours can bend */
real    CornerRadius = 0.0;	/* radius of nbhd for local max corner det. */
real    CornerPercentile = 98.;	/* cornerness threshold */
int     MaxJumpGap = 0;
int     MinBeforeGrowth = 0;
int     MinStandaloneContour = 0;
int     PruneCrackTips = FALSE;
int     OutlineBorder = FALSE;

/* the length scale of interest */
real    LengthScale;

/* the size of the input image & all intermediates */
int     Height, Width;

/* thresholds */
real    Lo, Hi;			/* edge strength cutoffs */
real    CornerThresh;		/* cornerness */


main(argc, argv)
int     argc;
char   *argv[];
{
    char    c;
    extern char *optarg;
    extern int optind;
    IMAGE  *I;


    /*
     * usage: %s [-a angl] [-b B] [-c ct] [-C npix] [-g G] [-m M] [-o] [-p]
     * [-s Sdev] [-t Lo Hi] [-x lev] image-file
     */

    PgmName = argv[0];

    while ((c = getopt(argc, argv, "a:b:c:C:g:m:ops:t:x:")) != -1) {

	switch (c) {

	case '?':
	    goto badargs;	/* getopt() arg not in list */
	case 'a':
	    SweepAngle = atof(optarg);
	    break;
	case 'b':
	    MinBeforeGrowth = atoi(optarg);
	    break;
	case 'c':
	    CornerPercentile = atof(optarg);
	    break;
	case 'C':
	    CornerRadius = atof(optarg);
	    break;
	case 'g':
	    MaxJumpGap = atoi(optarg);
	    break;
	case 'm':
	    MinStandaloneContour = atoi(optarg);
	    break;
	case 'o':
	    OutlineBorder = TRUE;
	    break;
	case 'p':
	    PruneCrackTips = TRUE;
	    break;
	case 's':
	    SDev = atof(optarg);
	    break;
	case 't':
	    if (optind + 1 >= argc)
		goto badargs;
	    else {
		LoPercentile = atof(optarg);
		HiPercentile = atof(argv[optind]);
		optind++;
	    }
	    break;

	case 'x':
	    DebugLevel = atoi(optarg);
	    break;

	}
    }

    if (optind >= argc) {
badargs:
	fprintf(stderr, Usage, PgmName);
	exit(1);
    }
    InputFileName = argv[optind];



    /* ------------------ main program code ------------------ */


    /* read image in question */
    if ((I = hvReadImage(InputFileName)) == NULL)
	exit(1);

    if (I->pixeltype != REALPIX)
	(void) hvConvert(I, I, REALPIX);

    SetParameters(I);

    /* compute edgeness, cornerness and edge direction */
    GradientComputations(I);
    WriteImage(Cornerness, "Cornerness", ".corner");

    /* set thresholds */
    Lo = ThresholdPercentile((real *) Strength->im, Height * Width, LoPercentile);
    Hi = ThresholdPercentile((real *) Strength->im, Height * Width, HiPercentile);

    CornerThresh = ThresholdPercentile((real *) Cornerness->im,
				       Height * Width, CornerPercentile);

    /* ------------ Trace edges, linking up ContourPoints ------------ */

    /*
     * The PixelMap is a copy of the original image, but with each pixel a
     * pointer.  At edge points, this pointer points to the PixelInfo
     * structure. Non-edge points contain a NULL pointer.
     */
    PixelMap = hvMakeImage(Height, Width, 1, SEQUENTIAL, INTPIX);

    if (OutlineBorder)
	DrawBorder(0);

    /* AlwaysMergeCorners = TRUE; */
    BuildPixelMap(1);

    PrintContourStatistics();

    DeleteShortContours(MinBeforeGrowth);
    MergeCorners();
    PrintContourStatistics();
    PrintContours("After deleting short contours, before growing by reflection", ".e1");
    AlwaysMergeCorners = FALSE;

    /* split contours at corners */
    FindCorners(CornerThresh, CornerRadius);

    /*
     * reflect at ends to join up with neighbors.
     */
    GrowContours(MaxJumpGap);
    PrintContourStatistics();
    PrintContours("After growing by reflection, before pruning", ".e2");

    if (MinStandaloneContour > 1)
	DeleteStandaloneContours(MinStandaloneContour);

    /* and delete stray contours that end off in nowhere */
    if (PruneCrackTips)
	DeleteStrayContours(MinStandaloneContour);

    /* corner detection */
    MergeCorners();
    FindCorners(CornerThresh, CornerRadius);
    PrintContours("After pruning and corner detection", ".e");

    return 0;			/* success */
}


/*
 * Set parameters given image I
 */
SetParameters(I)
IMAGE  *I;
{
    Height = I->height;
    Width = I->width;

    SweepAngle = RAD(SweepAngle);

    /* diameter of neighborhood for gap-jumping */
    if (MaxJumpGap == 0)
	MaxJumpGap = IROUND(SDev * MAX_JUMP_IN_SDEVS);

    /* min contour before growth phase */
    if (MinBeforeGrowth == 0)
	MinBeforeGrowth = IROUND(SDev * 2);

    if (MinStandaloneContour == 0)
	MinStandaloneContour = MinBeforeGrowth;

    /* length scale for computing curvature, etc. */
    LengthScale = SDev * LENGTH_SCALE_IN_SDEVS;
    if (LengthScale < 2)
	LengthScale = 2;

    if (CornerRadius == 0)
	CornerRadius = LengthScale;

    /* PrintParameters(stderr, ""); */
}


/*
 * Compute edge strength, cornerness, and edge direction using the
 * Nitzberg/Shiota filtering idea by summing Ix^2, Iy^2, and IxIy over a
 * window (here we just blur with Gaussian). Shorter eigenvalue gives
 * cornerness; corresp. eigenvector gives edge direction; and sqrt(sum of
 * eigenvalues) gives edge strength.
 */
GradientComputations(I)
IMAGE  *I;
{
    int     y, x;
    double  disc, a, b, c, e1, e2;
    double  vx, vy, length;
    IMAGE  *Gradx = NULL, *Grady = NULL;
    IMAGE  *Ix2, *IxIy, *Iy2;	/* assorted products thereof */

    fprintf(stderr,
	    "Computing eigenvalues of ( Ix^2  IxIy / IxIy  Iy^2 ) ...\n");

    /* compute Ix^2, Iy^2, and IxIy */
    hvGradient(I, NULL, &Gradx, &Grady, REALPIX);
    Ix2 = hvMakeImage(Height, Width, 1, 0, REALPIX);
    Iy2 = hvMakeImage(Height, Width, 1, 0, REALPIX);
    IxIy = hvMakeImage(Height, Width, 1, 0, REALPIX);

    Strength = hvMakeImage(Height, Width, 1, 0, REALPIX);
    Edgex = hvMakeImage(Height, Width, 1, 0, REALPIX);
    Edgey = hvMakeImage(Height, Width, 1, 0, REALPIX);
    Cornerness = hvMakeImage(Height, Width, 1, 0, REALPIX);

    hvPixOp2(Gradx, Gradx, Ix2, 0, BOP_MUL, CLIP_OUTPUT);
    hvPixOp2(Grady, Grady, Iy2, 0, BOP_MUL, CLIP_OUTPUT);
    hvPixOp2(Gradx, Grady, IxIy, 0, BOP_MUL, CLIP_OUTPUT);

    hvFreeImage(Gradx);
    hvFreeImage(Grady);

    /* blur them with splined gaussian */
    GaussConvolve(Ix2, SDev);
    GaussConvolve(Iy2, SDev);
    GaussConvolve(IxIy, SDev);

    /* get eigenvalues & fill edge & corner arrays */
    for (y = 0; y < Height; y++) {
	for (x = 0; x < Width; x++) {

	    a = PIX(Ix2, x, y);
	    b = PIX(IxIy, x, y);
	    c = PIX(Iy2, x, y);

	    /* eigenvalues */
	    disc = sqrt(a * a + 4 * b * b - 2 * a * c + c * c);
	    e1 = (a + c - disc) / 2;
	    e2 = (a + c + disc) / 2;

	    /* assure e1 < e2 (paranoia of near-0 errors) */
	    if (e1 > e2) {
		vx = e1;
		e1 = e2;
		e2 = vx;
	    }
	    /* |shorter eigenvalue| indicates cornerness */
	    CORNERNESS(x, y) = fabs(e1);

	    /* sqrt(e1/e2) gives eccentricity (ignore) */

	    /* sqrt(a + c) gives edge strength */
	    EDGE_STRENGTH(x, y) = sqrt(a + c);

	    /* tangent to edge =~ +/- eigenvector corresp to e1, normalized */
	    vx = -b;
	    vy = a - e1;
	    length = sqrt(vx * vx + vy * vy);
	    if (length != 0) {
		vx /= length;
		vy /= length;
	    } else
		vx = vy = 0;

	    EDGE_X(x, y) = vx;
	    EDGE_Y(x, y) = vy;
	}
    }

    hvFreeImage(Ix2);
    hvFreeImage(Iy2);
    hvFreeImage(IxIy);
}


/*
 * ThresholdPercentile(array, nelt, percentile) -- real *array, percentile
 * 
 * find V for which percentile% of the array < V, i.conpt., for which # { i |
 * array[i] < V } ~= nelt * percentile/100
 */
#define	HISTOGRAM	1000	/* # of intervals for real-line histogram */

real
ThresholdPercentile(array, nelt, percentile)
real   *array;
int     nelt;
real    percentile;
{
    int     histogram[HISTOGRAM];
    int     i, interval;
    real    thresh;
    int     marker, sum;
    real    min, max;

    /* get min & max values */
    min = max = array[0];
    for (i = 1; i < nelt; i++) {
	if (min > array[i])
	    min = array[i];
	if (max < array[i])
	    max = array[i];
    }

    /*
     * histogram divides [min,max] into HISTOGRAM equal intervals [min,m1)
     * [m1,m2) [m2,m3) ... [m{HISTOGRAM-1},max]
     */

    for (i = 0; i < HISTOGRAM; i++)
	histogram[i] = 0;

    for (i = 1; i < nelt; i++) {
	interval = (int) 0.5 +
	    (array[i] - min) * HISTOGRAM / (max - min);
	if (interval == HISTOGRAM)
	    interval--;
	histogram[interval]++;
    }

#ifdef	PRINTHIST
    /* draw a 23hx50w graph of the histogram */
    {
#define	SMALLHIST	50
#define	MAG		(HISTOGRAM/SMALLHIST)

	static int beenhere = FALSE;
	int     smallhist[SMALLHIST], i, j;
	real    maxy;

	if (!beenhere)
	    beenhere = TRUE;
	else
	    goto skip;

	maxy = 0;
	for (i = 0; i < SMALLHIST; i++) {
	    smallhist[i] = 0;
	    for (j = 0; j < MAG; j++)
		smallhist[i] += histogram[i * MAG + j];

	    if (maxy < smallhist[i])
		maxy = smallhist[i];
	}

	for (i = 23; i >= 1; i--) {
	    DBUG(1, (stderr, "%8d |", (int) (i * (maxy / 23.))));
	    for (j = 0; j < SMALLHIST; j++) {
		if (smallhist[j] >= i * (maxy / 23.))
		    DBUG(1, ("*"));
		else
		    DBUG(1, (" "));
	    }
	    DBUG(1, ("\n"));
	}
	DBUG(1, ("         +"));
	for (i = 0; i < SMALLHIST; i++)
	    DBUG(1, ("-"));
	DBUG(1, ("\n        %g", min));
	for (i = 0; i < SMALLHIST - 8; i++)
	    DBUG(1, (" "));
	DBUG(1, ("%g\n\t\t\t\tHistogram\n\n", max));
    }
skip:
#endif	/* PRINTHIST */

    /* mark %ile by number of pixels not to be exceeded */
    marker = nelt * percentile / 100.0;

    for (i = 0, sum = 0; i < HISTOGRAM && sum < marker; i++)
	sum += histogram[i];

    thresh = min + (i - 1) * (max - min) / HISTOGRAM;

    return thresh;
}

#undef HISTOGRAM

/*
 * build a closed contour that is the border of the image
 * 
 * the border is "pad" pixels in from the actual image boundary.
 */
DrawBorder(pad)
int     pad;
{
    Contour *c, *NewContour();
    int     x, y;


    x = pad;
    y = pad;
    c = NewContour(x, y);
    c->boundary = TRUE;		/* image boundary */

    /* across the top */
    for (x++; x < Width - pad; x++)
	AddPointToContour(c, HEAD, (real) x, (real) y);
    x--;

    /* down the right */
    for (y++; y < Height - pad; y++)
	AddPointToContour(c, HEAD, (real) x, (real) y);
    y--;

    /* across the bottom */
    for (x--; x >= pad; x--)
	AddPointToContour(c, HEAD, (real) x, (real) y);
    x++;

    /* finally, up the left side */
    for (y--; y >= pad; y--)
	AddPointToContour(c, HEAD, (real) x, (real) y);

    /* should produce a closed contour */
    PrintContourStatistics();
}


BuildPixelMap(pad)
int     pad;
{
    int     x, y;

    fprintf(stderr, "Tracing edges ...\n");

    for (y = pad; y < Height - pad; y++) {
	for (x = pad; x < Width - pad; x++) {
	    if (PMap(x, y) == NULL && LocalMax(x, y, Hi)) {
		BuildContour(x, y);
	    }
	}
    }
}

/* boolean */
LocalMax(ax, ay, thresh)
int     ax, ay;
real    thresh;
{
    /* must be local max in +/- grad direction and Strength >= thresh */
    real    x, y, mag;
    real    a, b, c, d, adjusted, maxcoeff, mincoeff, absx, absy;

    mag = EDGE_STRENGTH(ax, ay);

    if (mag <= thresh)
	return FALSE;

    if (ay == 0 || ay == Height - 1 || ax == 0 || ax == Width - 1)
	return FALSE;

    /* since Edgex, Edgey give tangent, we take perp */
    x = PIX(Edgey, ax, ay);
    y = -PIX(Edgex, ax, ay);
    absx = ABS(x);
    absy = ABS(y);

    /***
     * There are four cases (O is current point):
     *
     *  (1)      (2)       (3)      (4)
     *
     * a b        b a     a            d
     *   O        O       b O c    b O c
     *   c d    d c           d    a
     */

    if (absx < absy) {
	maxcoeff = absy;
	mincoeff = absx;
	b = PIX(Strength, ax, ay - 1);
	c = PIX(Strength, ax, ay + 1);
    } else {
	maxcoeff = absx;
	mincoeff = absy;
	b = PIX(Strength, ax - 1, ay);
	c = PIX(Strength, ax + 1, ay);
    }
    if (x * y > 0) {		/* same sign */
	/* case 1 or 3 - \ */
	a = PIX(Strength, ax - 1, ay - 1);
	d = PIX(Strength, ax + 1, ay + 1);
    } else {
	if (absx < absy) {
	    /* case 2 - steep / */
	    a = PIX(Strength, ax + 1, ay - 1);
	    d = PIX(Strength, ax - 1, ay + 1);
	} else {
	    /* case 4 - shallow / */
	    a = PIX(Strength, ax - 1, ay + 1);
	    d = PIX(Strength, ax + 1, ay - 1);
	}
    }

    /* finally */
    adjusted = (maxcoeff + mincoeff) * mag;
    if (adjusted > maxcoeff * b + mincoeff * a
	&& adjusted > maxcoeff * c + mincoeff * d)
	return TRUE;

    return FALSE;
}

PixelInfo *
NewPixelInfo(x, y)
int     x, y;
{
    PixelInfo *p;
    real    cos, sin;

    assert(PMap(x, y) == NULL);	/* don't need 2nd one */

    p = (PixelInfo *) calloc(sizeof (PixelInfo), 1);
    if (p == NULL) {
	fprintf(stderr, "Not enough memory at %d-th PixelInfo\n",
		NPixelInfos);
	exit(2);
    }
    p->type = P_EMPTY;
    p->pos = v_vec((real) x, (real) y);
    p->k = 0;
    cos = EDGE_X(x, y);
    sin = EDGE_Y(x, y);
    if (cos != 0 || sin != 0)
	p->conpt[0].angle = atan2(sin, cos);
    else
	p->conpt[0].angle = 0;

    NPixelInfos++;
    return p;
}

/* make a 1-point contour at x,y and add to Contours */
Contour *
NewContour(x, y)
int     x, y;
{
    Contour *con;
    PixelInfo *pixelinfo;

    con = (Contour *) calloc(sizeof (Contour), 1);
    con->end[HEAD].x = x;
    con->end[HEAD].y = y;
    con->end[TAIL].x = x;
    con->end[TAIL].y = y;
    con->length = 1;
    con->closed = FALSE;	/* not a closed curve */
    con->split = FALSE;		/* not a kinked closed curve */
    con->boundary = FALSE;	/* not on image border */

    if (PMap(x, y) == NULL) {
	pixelinfo = PMap(x, y) = NewPixelInfo(x, y);
	pixelinfo->type = P_POINT;
	pixelinfo->conpt[0].con = con;
	/* conpt[i].n[j] all i,j "next" pointers are 0 */
    }
    /* else the caller must take care of that */

    con->R = Contours;
    Contours = con;
    NContours++;

    return con;
}

/*
 * Build a new contour structure starting at point x,y add to the Contours
 * list.
 * 
 * Walk along neighboring edge points that are above the Lo threshold, and whose
 * edge tangent is near parallel to the tail.
 * 
 * After it's all over, go the other way.
 */
BuildContour(x, y)
int     x, y;
{
    Contour *con;
    ContourPoint *conpt;
    PixelInfo *p;

    con = NewContour(x, y);
    TraceContour(con, 0);	/* last direction = none */

    if (!con->closed) {
	ReverseContour(con);

	/* last direction = delta from 2nd to 1st pt */
	GetEnd(con, HEAD, &conpt, &p);
	GetNext(con, HEAD, &conpt, &p);

	if (conpt == NULL)
	    return;		/* a 1-point contour */

	TraceContour(con, (int) conpt->n[BACKWARD]);
    }

    /*
     * kill the contour if the whole thing is next to the image boundary
     */
    if (OutlineBorder)
	KillIfBoundaryNeighbor(con);
}

KillIfBoundaryNeighbor(c)
Contour *c;
{
    ContourPoint *conpt;
    PixelInfo *p;
    real    totaldist = 0.0;
    int     npoints = 0;

    for (GetEnd(c, HEAD, &conpt, &p); conpt != NULL;
	 GetNext(c, FORWARD, &conpt, &p)) {
	real    dtop, dbot, dleft, dright, dmin;

	dtop = MAX((p->pos.y - 0) - 1, 0);
	dbot = MAX((Height - 1 - p->pos.y) - 1, 0);
	dleft = MAX((p->pos.x - 0) - 1, 0);
	dright = MAX((Width - 1 - p->pos.x) - 1, 0);

	dmin = dtop;
	if (dmin > dbot)
	    dmin = dbot;
	if (dmin > dleft)
	    dmin = dleft;
	if (dmin > dright)
	    dmin = dright;
	totaldist += dmin;
	npoints++;
    }

    /* clung too close to the boundary ? */
    if (totaldist < SDev * npoints)
	DeleteContour(c);
}

/* convert pixelpos to vector */
vector
pptovec(pixelpos)
PixelPos pixelpos;
{
    vector  v;

    v.x = pixelpos.x;
    v.y = pixelpos.y;
    return v;
}

/* convert vector to PixelPos */
PixelPos
vectopp(v)
vector  v;
{
    PixelPos pixelpos;

    pixelpos.x = IROUND(v.x);
    pixelpos.y = IROUND(v.y);
    return pixelpos;
}

/* an impossibly large penalty */
#define HUGE_PENALTY	1.0e10

/*
 * Trace a contour, "backwards", i.e., adding to the HEAD. curdir is the
 * neighbor direction in which we last moved. Con must be structurally
 * consistent.
 */
TraceContour(con, curdir)
Contour *con;
int     curdir;
{
    int     x, y, dir, bestdir, n;
    real    compat, bestcompat;	/* compatibility measures: lower is better */
    PixelPos dirvec, curpixpos;
    PixelInfo *p, *curp;	/* pixel info structures for candidate &
				 * current */
    ContourPoint *curconpt;	/* "current" point */

    /* the 8 or 5 neighbors to visit given you just moved in dir curdir */
    static char neighbors[NEIGHBORS][NEIGHBORS] = {
	1, 2, 3, 4, 5, 6, 7, 8, 0,	/* 1st pixel looks all around */
	7, 8, 1, 2, 3, 0, 0, 0, 0,	/* Note:	 */
	8, 1, 2, 3, 4, 0, 0, 0, 0,	/* 4 3 2	 */
	1, 2, 3, 4, 5, 0, 0, 0, 0,	/* 5 0 1	 */
	2, 3, 4, 5, 6, 0, 0, 0, 0,	/* 6 7 8	 */
	3, 4, 5, 6, 7, 0, 0, 0, 0,	/* Example:	 */
	4, 5, 6, 7, 8, 0, 0, 0, 0,	/* if you last moved 1 (right) */
	5, 6, 7, 8, 1, 0, 0, 0, 0,	/* you may next move 7,8,1,2,3 */
	6, 7, 8, 1, 2, 0, 0, 0, 0,
    };

    curpixpos = con->end[HEAD];

    /* set "current" pointer into pixel map, and its contour point */
    GetEnd(con, HEAD, &curconpt, &curp);
    DBUG(7, ("Tracing from %d-pt con head %s(%d,%d)\n",
	     con->length, TYPE_STRING(curp->type),
	     con->end[HEAD].x, con->end[HEAD].y));

    for (;;) {

	GetEnd(con, HEAD, &curconpt, &curp);

	/* link to best nearest neighbor with near parallel tan */
	bestdir = -1;
	bestcompat = HUGE_PENALTY;

	/* cycle over plausible neighbors given we're headed curdir */
	for (n = 0; (dir = neighbors[curdir][n]) != 0; n++) {

	    dirvec = NeighborPos[dir];
	    x = dirvec.x + curpixpos.x;
	    y = dirvec.y + curpixpos.y;

	    /*
	     * non-maximum supression
	     */
	    if (!LocalMax(x, y, Lo))
		continue;

	    /*
	     * Can't double back on ourself too quickly
	     */
	    if ((p = PMap(x, y)) != NULL && CONPT(p, con) != NO_CONTOUR
	    && StepsFromHereToEnd(p, HEAD, con) <= TOO_SOON_TO_DOUBLE_BACK) {

		DBUG(7, ("Skipping neighbor (%d,%d) due to double-back\n", x, y));
		continue;
	    }

	    /*
	     * compatibility: [0,1 or so]  lower is better tangent at
	     * curpixpos, and tangent at candidate.
	     */
	    compat = Compatible(pptovec(dirvec),	/* vector cur to next */
	    /* tan at cur */
				v_vec(EDGE_X(curpixpos.x, curpixpos.y),
				      EDGE_Y(curpixpos.x, curpixpos.y)),
	    /* tan at next */
				v_vec(EDGE_X(x, y), EDGE_Y(x, y)));

	    if (bestdir == -1 || compat < bestcompat) {
		bestdir = dir;
		bestcompat = compat;
	    }
	}

	if (bestdir == -1) {
	    return;		/* nothing to do */
	}
	/* dir = neighbor number of new candidate */
	dir = bestdir;

	/* set curpixpos and x,y to the new candidate neighbor */
	x = curpixpos.x += NeighborPos[dir].x;
	y = curpixpos.y += NeighborPos[dir].y;

	/*
	 * Finish inserting -- returns TRUE if hit end
	 */
	if (AddPointToContour(con, HEAD, (real) x, (real) y))
	    return;

	curdir = dir;		/* remember which way we just went */
    }
}


/*
 * add pixel (x,y) to end end of contour con TRUE if this created a Y, T,
 * Corner or closed contour rather than an End, FALSE otherwise
 */
AddPointToContour(con, end, rx, ry)
Contour *con;
int     end;
real    rx, ry;
{
    ContourPoint *conpt, *endconpt;
    PixelInfo *p, *endp;
    int     dir;		/* neighbor direction from (x,y) to end of
				 * con */
    int     ix = IROUND(rx);
    int     iy = IROUND(ry);
    PixelPos pos;

    /*
     * get neighbor direction from end of con to (ix,iy)
     */
    pos.x = ix;
    pos.y = iy;
    dir = DirectionToNeighbor(con->end[end], pos);
    if (dir == 0) {
	/* Indeed, we are not a neighbor */
	DBUG(0, ("AddPointToContour(%g, %g, contour %d(%d,%d), %s) Non-neighbor\n",
		 rx, ry, con->length, con->end[end].x, con->end[end].y,
		 end == HEAD ? "HEAD" : "TAIL"));
	return TRUE;		/* this contour ends here */
    }
    /* check for pixel info at new candidate point */
    if ((p = PMap(ix, iy)) != NULL && p->type != P_EMPTY) {

	/*
	 * We've just run into another contour (or self)
	 * 
	 * NOTE:  trashes the rx, ry for whatever location was already there.
	 */
	JoinContour(con, end, p);
	return TRUE;
    }
    /* new point */
    if (p == NULL)		/* don't make new if it's already there &
				 * empty */
	p = PMap(ix, iy) = NewPixelInfo(ix, iy);

    GetEnd(con, end, &endconpt, &endp);

    /* update this point before doing next */
    if (endp->type == P_POINT)
	endp->type = P_END;
    else if (endp->type == P_END)
	endp->type = P_MIDDLE;
    else			/* ?adding to a contour that ended in a Y? */
	assert(0);

    /*
     * point end of contour towards new pixel, and make new pixel new end of
     * con
     */
    endconpt->n[end == HEAD ? BACKWARD : FORWARD] = dir;
    p->type = P_END;
    p->pos.x = rx;
    p->pos.y = ry;

    con->end[end] = pos;	/* this becomes the new end */
    con->length++;
    conpt = &p->conpt[0];
    conpt->con = con;		/* contour pointer is con */

    /* point new pixel to old end */
    conpt->n[end == HEAD ? FORWARD : BACKWARD] = AntiNeighbor[dir];

#ifdef NOTUSED
    /* point conpt->angle in that direction as well */
    PointAngle(&conpt->angle, pptovec(NeighborPos[dir]));

    /* if this was the 2nd ContourPoint, do 1st as well */
    if (con->length == 2) {
	enddir = endconpt->n[end == HEAD ? FORWARD : BACKWARD];
	if (end == HEAD)
	    enddir = AntiNeighbor[enddir];
	PointAngle(&endconpt->angle, pptovec(NeighborPos[enddir]));
    }
#endif				/* NOTUSED */

    return FALSE;
}


/* assure that angle points in the same direction as vector v */
PointAngle(angle, v)
real   *angle;
vector  v;
{
    double  sin, cos;

    sincos(*angle, &sin, &cos);

    /* flip 180 degrees if dot prod.<0 */
    if (v.x * cos + v.y * sin < 0)
	*angle = AngleFix(M_PI + *angle);
}

/*
 * count contour lengths; check for inconsistencies
 */
PrintContourStatistics()
{
    int    *lengths, *closedlengths;
    int     maxlength, i;
    Contour *c;

    maxlength = 0;
    for (c = Contours; c != NULL; c = c->R)
	if (c->length > maxlength)
	    maxlength = c->length;

    lengths = (int *) calloc(maxlength + 1, sizeof (int));
    closedlengths = (int *) calloc(maxlength + 1, sizeof (int));
    if (lengths == NULL)
	return;			/* not enough memory */
    if (closedlengths == NULL) {
	free(lengths);
	return;
    }
    for (c = Contours; c != NULL; c = c->R) {
	if (c->closed)
	    closedlengths[c->length]++;
	lengths[c->length]++;
    }

    fprintf(stderr, "Contours:  %d, lengths: ", NContours);
    for (i = 0; i <= maxlength; i++) {
	if (lengths[i] != 0) {
	    fprintf(stderr, "%d", lengths[i]);
	    if (closedlengths[i] != 0)
		fprintf(stderr,
			"(%d closed)", closedlengths[i]);
	    fprintf(stderr, "@%d ", i);
	}
    }
    fprintf(stderr, "\n");
    free(lengths);
    free(closedlengths);

#ifndef	NDEBUG
    CheckContourConsistency();
#endif	/* NDEBUG */
}

/***
 * Compatibility of two points with tangents.  Given vectors v, v1, and v2
 *
 *        ^            ^
 *       /t1  v       /
 *      *----------->*
 *     /          t2/
 *    /v1          /v2
 *
 * The length of v, |v|, is the distance between the points.
 * The sin of the angle between v1 and v2 is sinv1v2.
 *
 * Penalty value is given by the following:
 *
 * For nearby points the penalty is
 *
 *	nearby penalty = |sinv1v2| + SPL*|v|
 *
 * SPL is a tradeoff constant that gives sin of angle per unit length.
 * Thus a point farther away is more compatible
 * only if the angle between its tangent and the candidate's is
 * closer by arcsin(SPL)*(extra distance).
 *
 * Nearby is when |v| <= LengthScale/2
 *
 */

#define	SPL .25881904514	/* sin(15 degrees): pay 15 degrees per unit
				 * distance */

real
Compatible(v, v1, v2)
vector  v, v1, v2;
{
    real    lengthv, sinv1v2;
    real    nearby_penalty;

    lengthv = v_norm(v);
    if (lengthv < 0.00000001 || v_normsq(v1) == 0 || v_normsq(v2) == 0)
	return HUGE_PENALTY;	/* big penalty:  same pixel! */

    sinv1v2 = fabs(v_dot(v_perp(v1), v2));

    nearby_penalty = sinv1v2 + SPL * lengthv;

    return nearby_penalty;

    /* distant_penalty left as an exercise */
}

/*
 * Turn every "corner" that's not associated to closed contours into a
 * "middle".
 */
MergeCorners()
{
    Contour *c, *c1, *c2;
    ContourPoint *conpt;
    PixelInfo *p;
    int     end;
    int     merged;

    DBUG(9, ("Merging contours that meet at corners ...\n"));

    do {

	merged = FALSE;
	for (c = Contours; c != NULL; c = c->R) {

	    c->split = FALSE;	/* automatic merge */

	    for (end = 0; end <= 1; end++) {
		GetEnd(c, end, &conpt, &p);

		c1 = p->conpt[0].con;
		c2 = p->conpt[1].con;
		if (p->type == P_CORNER
		    && c1->closed == FALSE
		    && c2->closed == FALSE
		/* both FALSE or both TRUE */
		    && c1->boundary == c2->boundary) {

		    /* Merge adds c2 to c1 and deletes c2 */
		    MergeCorner(p);
		    merged = TRUE;	/* drop out & start w/1st contour */
		    break;
		}
	    }
	    if (merged)
		break;
	}

    } while (merged);
}

/***
 * Merge two contours by killing the corner at p,
 * and delete the second, p->conpt[1].con, from Contours
 *
 * 1. MAY REVERSE CONTOURS
 * 2. MAY ADD CLOSED CONTOURS
 */
MergeCorner(p)
PixelInfo *p;
{
    PixelInfo *pt;
    ContourPoint *conpt;
    Contour *c1 = p->conpt[0].con;
    Contour *c2 = p->conpt[1].con;
    int     closed = FALSE;

    assert(p->type == P_CORNER);

    /* make p the tail of c1 and the head of c2 */
    GetEnd(c1, HEAD, &conpt, &pt);
    if (pt == p)
	ReverseContour(c1);

    GetEnd(c2, HEAD, &conpt, &pt);
    if (pt != p) {
	ReverseContour(c2);
	GetEnd(c2, HEAD, &conpt, &pt);
    }
    DBUG(4, ("Merging %d(%d,%d) at CORNER(%g,%g) with %d(%d,%d)\n",
	     c1->length, c1->end[HEAD].x, c1->end[HEAD].y,
	     p->pos.x, p->pos.y,
	     c2->length, c2->end[TAIL].x, c2->end[TAIL].y));

    /* if the last point in c2 is p, build a closed contour */
    if (c1->end[HEAD].x == c2->end[TAIL].x &&
	c1->end[HEAD].y == c2->end[TAIL].y) {

	/* crazy case:  c1 == c2 */
	if (c1->length == 2) {
	    DeleteContour(c1);
	    return;
	}
	if (c2->length == 2) {
	    DeleteContour(c2);
	    return;
	}
	/* just lop off the last point */
	LopOff(c2, TAIL, 1);
	closed = TRUE;
    }
    /* fetch 2nd point in c2 (1st point was the corner p) */
    GetNext(c2, FORWARD, &conpt, &pt);

    /* make p the middle of c1 */
    p->type = P_MIDDLE;
    p->conpt[0].n[FORWARD] = p->conpt[1].n[FORWARD];

    p->conpt[1].con = NULL;	/* null out the forward pointer in p */
    p->conpt[1].n[FORWARD] = 0;
    p->conpt[1].n[BACKWARD] = 0;

    /* for each pixel in c2, change its c2 contour pointer to c1 */
    while (conpt != NULL) {

	conpt->con = c1;

	/*
	 * go to next in c2 -- see GetNext to be sure this works with
	 * conpt->con changed
	 */

	GetNext(c2, FORWARD, &conpt, &pt);
    }
    c1->length += (c2->length - 1);
    c1->end[TAIL] = c2->end[TAIL];
    c1->closed = closed;

    /* and delete c2 from the list */
    ExcludeContour(c2);
    free(c2);			/* why keep it around? */
    NContours--;

    /* don't leave a tiny closed contour */
    if (c1->closed && c1->length < MIN_CLOSED_CONTOUR)
	DeleteContour(c1);
}



/*
 * At the end of each contour, head out in a reflectwardsly direction to see
 * if we run into another contour before travelling maxgap from the end. If
 * not, try perturbing the angle of reflection a bit each way. See
 * GrowContour.
 */
GrowContours(maxgap)
int     maxgap;
{
    Contour *c;
    int     changed;

    fprintf(stderr, "Growing and linking contours across %d-pixel gaps\n", maxgap);

    /*
     * Each time through the Contours list, if we attach anything, we've
     * modified the Contours list & have to start from scratch.
     */
    do {
	changed = FALSE;

	/* for each contour ... */
	for (c = Contours; c != NULL; c = c->R) {

	    if (!c->closed && c->length >= maxgap /* + LengthScale / 2.0 */)
		if (GrowContour(c, maxgap)) {
		    changed = TRUE;
		    break;
		}
	}
    } while (changed);
}

/*
 * Grow each endpoint of a contour by reflection, making a mess of the
 * PixelMap, just to see what reflection will give.
 * 
 * Reflect in the line perpendicular to the tangent "near" the endpoint, and if
 * that fails, try +/- SweepAngle.
 * 
 * Added later:  jump directly to a "corner" if within the trajectory
 */
GrowContour(c, maxgap)
Contour *c;
int     maxgap;
{
    ContourPoint *conpt;
    PixelInfo *p, *q;
    real    traj, reflectangle;
    double  si, co;
    int     end;

    /* for each of the two ends of the contour ... */
    for (end = 0; end <= 1; end++) {

	GetEnd(c, end, &conpt, &p);

	/* should grow T's and corners as well, but too bad for now */
	if (p->type != P_END)
	    continue;

	/* use tangent by going in LengthScale/2.0 from this end */
	if ((q = NthFromEndP(c, end, IROUND(LengthScale / 3.0))) == NULL)
	    /* give it a second chance */
	    if ((q = NthFromEndP(c, end, IROUND(LengthScale / 4.0))) == NULL)
		return FALSE;	/* not that long a contour */

	traj = C_ANGLE(q, c);

	/* first try going outward at ~angle traj to find a corner */
	sincos(traj, &si, &co);	/* point traj away from q */
	if (si * (q->pos.y - p->pos.y) + co * (q->pos.x - p->pos.x) > 0)
	    traj = AngleFix(traj + M_PI);
	if (JumpToCorner(c, end, traj, maxgap * M_SQRT2))
	    return TRUE;

	/* if not, grow by reflection */
	reflectangle = AngleFix(traj + M_PI / 2.0);

	/* try at theta */
	if (GrowEndpoint(c, end, maxgap, reflectangle))
	    return TRUE;

	/* then at theta+SweepAngle and theta-SweepAngle */
	if (GrowEndpoint(c, end, maxgap, AngleFix(reflectangle + SweepAngle)))
	    return TRUE;

	if (GrowEndpoint(c, end, maxgap, AngleFix(reflectangle - SweepAngle)))
	    return TRUE;

    }
    return FALSE;
}


real
distance(x, y, x2, y2)
real    x, y, x2, y2;
{
    real    dx = x2 - x, dy = y2 - y;

    return sqrt(fabs(dx * dx + dy * dy));
}

/***
 * Joinable(con, end, x, y) -- returns integer > 0, bigger is better.
 *
 *	0 if (x,y) is an impossible point to join to end end of con
 *	1 if (x,y) is an unused point (NULL or EMPTY pixelinfo)
 *    n+1 if (x,y) runs into contours the longest of which is length n,
 *		   to which it could be joined.
 *		   If the contour is con itself, then n is the length
 *		   of the longer of the two parts after it will be joined.
 *
 * Used for growing an endpoint, growth prefers to continue along unused points
 * rather than stop short; but better still is to connect to a contour,
 * particularly not itself; the longer the better.
 */
int
Joinable(con, end, x, y)
Contour *con;
int     end;
int     x, y;
{
    int     i, n, n2;
    PixelInfo *p;

    if (x < 0 || x >= Width
	|| y < 0 || y >= Height)
	return 0;

    if ((p = PMap(x, y)) == NULL || p->type == P_EMPTY)
	return 1;

    if (p->type == P_X)
	return 0;

    if (CONPT(p, con) != NO_CONTOUR) {

	/* p is a point on con */
	n = StepsFromHereToEnd(p, end, con);
	if (n <= TOO_SOON_TO_DOUBLE_BACK)
	    return 0;
	if (con->length - n > n)
	    n = con->length - n;
	n++;			/* # points = # steps + 1 */

    } else {

	/* take largest stretch attached here */
	n = 0;
	for (i = 0; i < 4; i++)
	    if (p->conpt[i].con != NULL) {
		n2 = p->conpt[i].con->length;
		if (n2 > n)
		    n = n2;
	    }
    }
    /* add 1 so it's always better than an empty point */
    return n + 1;
}

/*
 * Grow the end of a contour by reflection.
 * 
 * As it grows, look around at the neighbors as well.
 * 
 * returns TRUE if it connects to another contour, false otherwise.
 */

#define	DEAD_END		0
#define	ADDED_NEW_POINT		1
#define	JOINED_TO_SOMETHING	2

/* externs for line drawing */
static Contour *__c;
static int __end;
static int *__Ppointno;
static real __reflectedangle;
static int __lastx, __lasty;

GrowEndpoint(c, end, maxgap, reflectangle)
Contour *c;
int     end;
int     maxgap;
real    reflectangle;
{
    ContourPoint *conpt, *stepinconpt;
    PixelInfo *stepin, *p;
    vector  endpoint;
    vector  from, to;		/* assure sequence of neighbors */
    PixelPos ifrom, ito;
    int     direction = (end == HEAD ? FORWARD : BACKWARD);
    int     i, pointno, joined;
    int     AddBestNeighbor();

    GetEnd(c, end, &stepinconpt, &stepin);
    endpoint = stepin->pos;

    /* the draw line function doesn't take all these args */
    __c = c;
    __end = end;
    __Ppointno = &pointno;

    /* as we step inwards with "stepin", reflect and add outwards */
    joined = FALSE;
    for (pointno = 0; pointno < maxgap;) {

	GetEnd(c, end, &conpt, &p);
	from = p->pos;
	ifrom = vectopp(from);

	GetNext(c, direction, &stepinconpt, &stepin);
	if (stepinconpt == NULL)/* ran out of points to reflect */
	    break;

	/* find reflected point */
	to = Reflect(stepin->pos, endpoint, reflectangle);
	ito = vectopp(to);

	/* don't get carried away */
	if (ito.x < 0 || ito.x > Width - 1
	    || ito.y < 0 || ito.y > Height - 1)
	    break;

	if (ifrom.x == ito.x && ifrom.y == ito.y)
	    continue;		/* just skip if we hit same pt twice */

	/* Draw a line to the next point */
	__reflectedangle = ReflectAngle(stepinconpt->angle, reflectangle);

	i = ApplyToLineWhile(ifrom.x, ifrom.y,
			     ito.x, ito.y,
			     AddBestNeighbor,
			     ADDED_NEW_POINT);

	if (i == DEAD_END)
	    break;

	if (i == JOINED_TO_SOMETHING) {
	    joined = TRUE;
	    break;		/* hit another contour */
	}
    }

    /* if we just grew but didn't join anything, shrink back */
    if (!joined && pointno > 0)
	LopOff(c, end, pointno);

    return joined;
}

/*
 * AddBestNeighbor(x,y)
 * 
 * add the most joinable neighbor of x,y (itself preferred) to end end of
 * contour c for the above function.  Arguments c, end, reflectedangle and a
 * pointer to a point counter are all passed through externals to keep the
 * line-drawing function general.
 * 
 * If it finds a more joinable neighbor than itself, adds itself first if
 * necessary. Increments *(point counter) with each point added.
 * 
 * If x,y happens to be the end of the contour being grown, and there are no
 * contours nearby to connect, fakes growing 1 point. This tells the
 * line-drawing function to continue.
 * 
 * In the unfortunate situation where we are asked to grow from a non-neighbor
 * of the end being grown, complain and return dead-end (stop growth). This
 * is a bug that I'm not going to worry about for the time.
 * 
 * When choosing among equals, prefer the anti-neighbor of the contour endpoint.
 * if x,y itself is not in the running.
 * 
 * returns	DEAD_END if no good neighbors were available, ADDED_NEW_POINT if it
 * tacked a point or two on the end but didn't join up or
 * _TO_SOMETHING if joined up with something.
 */
AddBestNeighbor(x, y)
int     x, y;
{
    int     ix, iy, bestdir, bestval, dir;
    int     val, joinability[NEIGHBORS];
    ContourPoint *conpt;
    PixelInfo *p;
    PixelPos pp;
    Contour *c = __c;
    int    *pn = __Ppointno;
    int     end = __end;
    int     dirConEndToXY;

    bestval = 0;
    for (dir = 0; dir < NEIGHBORS; dir++) {

	ix = x + NeighborPos[dir].x;
	iy = y + NeighborPos[dir].y;
	val = Joinable(c, end, ix, iy);
	joinability[dir] = val;
	if (val > bestval) {
	    bestval = val;
	    bestdir = dir;
	}
    }

    DBUG(10, ("AddBestNeighbor (%d,%d):\n%d %d %d\n%d %d %d\n%d %d %d\n",
	      x, y,
	      joinability[4], joinability[3], joinability[2],
	      joinability[5], joinability[0], joinability[1],
	      joinability[6], joinability[7], joinability[8]));

    if (bestval == 0)
	return DEAD_END;

    /* special case:  are we trying to "add" the endpoint of c? */
    if (bestval == 1 && c->end[end].x == x && c->end[end].y == y)
	return ADDED_NEW_POINT;	/* fake */

    pp.x = x;
    pp.y = y;
    dirConEndToXY = DirectionToNeighbor(c->end[end], pp);

    /* prefer center if we have a choice */
    if (bestval == joinability[0])
	bestdir = 0;

    /* otherwise prefer antineighbor of endpoint */
    else if (joinability[0] == 0
	     && bestval == joinability[dirConEndToXY])
	bestdir = dirConEndToXY;

    pp.x = x + NeighborPos[bestdir].x;
    pp.y = y + NeighborPos[bestdir].y;

    /*
     * see if best neighbor happens to be a direct neighbor of end end of
     * contour c
     */
    if (DirectionToNeighbor(pp, c->end[end]) != 0) {

	DBUG(7, ("adding (%d,%d)\n", pp.x, pp.y));

	/* yes, just add or join */
	if (AddPointToContour(c, end, (real) pp.x, (real) pp.y))
	    return JOINED_TO_SOMETHING;
	goto NewPoint;
    }
    /* not direct neighbor, use x,y first */
    if (bestdir == 0) {
	/* thought this shouldn't happen */
	DBUG(5, ("Not growing %d(%d,%d): AddBestNeighbor called with non-neighbor.\n",
		 c->length, c->end[HEAD].x, c->end[HEAD].y));
	return DEAD_END;	/* pity--it might have got somewhere
				 * important */
    }
    if (joinability[0] == 0) {	/* x,y itself is not a good go-between */

	DBUG(5, ("Not growing %d(%d,%d): best choice not a neighbor.\n",
		 c->length, c->end[HEAD].x, c->end[HEAD].y));
	return DEAD_END;
    }
    DBUG(7, (" add(%d,%d):", x, y));

    if (AddPointToContour(c, end, (real) x, (real) y))
	return JOINED_TO_SOMETHING;
    (*pn)++;
    GetEnd(c, end, &conpt, &p);
    conpt->angle = __reflectedangle;

    if (AddPointToContour(c, end, (real) pp.x, (real) pp.y))
	return JOINED_TO_SOMETHING;
NewPoint:
    /* this is the new end, set its angle */
    GetEnd(c, end, &conpt, &p);
    conpt->angle = __reflectedangle;

    (*pn)++;
    return ADDED_NEW_POINT;
}


/* Like AddBestNeighbor, but just add x,y to __c */
AddNeighbor(x, y)
int     x, y;
{
    Contour *c = __c;
    int    *pn = __Ppointno;
    int     end = __end;

    /* special case:  are we trying to "add" the endpoint of c? */
    if (c->end[end].x == x && c->end[end].y == y)
	return ADDED_NEW_POINT;	/* fake */

    DBUG(7, (" add(%d,%d):", x, y));

    if (AddPointToContour(c, end, (real) x, (real) y))
	return JOINED_TO_SOMETHING;
    (*pn)++;
    return ADDED_NEW_POINT;
}

/* Like AddNeighbor, but just checks that x,y is clear */
CheckNeighbor(x, y)
int     x, y;
{
    Contour *c = __c;
    int     end = __end;
    PixelInfo *p;

    /* special case:  are we trying to "add" the endpoint of c? */
    if (c->end[end].x == x && c->end[end].y == y)
	return ADDED_NEW_POINT;	/* fake */

    if ((p = PMap(x, y)) != NULL && p->type != P_EMPTY) {
	__lastx = x;
	__lasty = y;
	return JOINED_TO_SOMETHING;
    }
    return ADDED_NEW_POINT;
}

int
JumpToCorner(c, end, traj, dist)
Contour *c;
int     end;
real    traj;			/* angle */
real    dist;			/* max */
{
    int     left, right, bottom, top;
    real    x, y, distance();
    PixelInfo *p, *qbest;
    ContourPoint *conpt;
    int     AddNeighbor();

    /*
     * how do we check pixels starting at c, end within the cone between
     * traj-SweepAngle and traj+SweepAngle radius between 0 and dist ?
     */
    GetEnd(c, end, &conpt, &p);

    left = right = IROUND(p->pos.x);
    bottom = top = IROUND(p->pos.y);
    /* get bounding box of cone */
    x = IROUND(p->pos.x + dist * cos(traj - SweepAngle));
    y = IROUND(p->pos.y + dist * sin(traj - SweepAngle));
    if (x < 0)
	x = 0;
    if (x > Width - 1)
	x = Width - 1;
    if (y < 0)
	y = 0;
    if (y > Height - 1)
	y = Height - 1;
    if (left > x)
	left = x;
    if (right < x)
	right = x;
    if (bottom > y)
	bottom = y;
    if (top < y)
	top = y;
    x = IROUND(p->pos.x + dist * cos(traj + SweepAngle));
    y = IROUND(p->pos.y + dist * sin(traj + SweepAngle));
    if (x < 0)
	x = 0;
    if (x > Width - 1)
	x = Width - 1;
    if (y < 0)
	y = 0;
    if (y > Height - 1)
	y = Height - 1;
    if (left > x)
	left = x;
    if (right < x)
	right = x;
    if (bottom > y)
	bottom = y;
    if (top < y)
	top = y;


    /* now we have a bounding box, just look for corners therein */
    qbest = NULL;
    for (y = bottom; y <= top; y++) {
	for (x = left; x <= right; x++) {
	    real    theta;
	    PixelInfo *q;

	    if (y == p->pos.y && x == p->pos.x)
		continue;
	    q = PMap(IROUND(x), IROUND(y));
	    if (q == NULL || q->type != P_CORNER)
		continue;
	    theta = AngleFix(atan2(y - p->pos.y, x - p->pos.x));
	    if (theta < traj - SweepAngle
		|| theta > traj + SweepAngle)
		continue;

	    /* verify that the line draws without intercepts */
	    __c = c;
	    __end = end;
	    ApplyToLineWhile(IROUND(p->pos.x), IROUND(p->pos.y),
			     IROUND(q->pos.x), IROUND(q->pos.y),
			     CheckNeighbor,
			     ADDED_NEW_POINT);
	    if (__lastx != IROUND(q->pos.x)
		|| __lasty != IROUND(q->pos.y))
		continue;

	    if (qbest == NULL
		|| distance(p->pos.x, p->pos.y, q->pos.x, q->pos.y)
		< distance(p->pos.x, p->pos.y, qbest->pos.x, qbest->pos.y))
		qbest = q;
	}
    }
    if (qbest != NULL) {
	int     pno = 0;

	/* connect with a straight line! */
	DBUG(3, ("Jump from (%g,%g) to CORNER(%g,%g)\n",
		 p->pos.x, p->pos.y, qbest->pos.x, qbest->pos.y));

	__c = c;
	__end = end;
	__Ppointno = &pno;
	ApplyToLineWhile(IROUND(p->pos.x), IROUND(p->pos.y),
			 IROUND(qbest->pos.x), IROUND(qbest->pos.y),
			 AddNeighbor,
			 ADDED_NEW_POINT);
	return TRUE;
    }
    return FALSE;
}


/*------------- Point lists for detecting corners -----------*/
typedef struct PointList {
    int     n;			/* # points in each array */
    real   *x, *y, *tan, *k, *L;/* L is for cornerness */
} PointList;

real   *
AllocReal(n)
int     n;
{
    real   *r;

    if ((r = (real *) calloc(n, sizeof (real))) == NULL) {
	fprintf(stderr, "Not enough memory for real-valued arrays.\n");
	exit(2);
    }
    return r;
}

PointList *
NewPointList(n)
int     n;
{
    PointList *plist;

    if ((plist = (PointList *) calloc(sizeof (PointList), 1)) == NULL) {
	fprintf(stderr, "Not enough memory for PointList.\n");
	exit(2);
    }
    plist->n = n;
    plist->x = AllocReal(n);
    plist->y = AllocReal(n);
    plist->tan = AllocReal(n);
    plist->k = AllocReal(n);
    plist->L = AllocReal(n);

    return plist;
}

/*
 * BuildPointList(con) allocates and fills a PointList for contour con
 */
PointList *
BuildPointList(con)
Contour *con;
{
    PointList *plist = NewPointList(con->length);

    FillPointListFromContour(plist, con, HEAD, con->length, 0, 1);

    return plist;
}

/*
 * fill plist from end end of contour con with n points of info placing the
 * data starting at index i0 and bumping by di
 */
FillPointListFromContour(plist, con, end, n, i0, di)
PointList *plist;
Contour *con;
int     end;
int     n, i0, di;
{
    int     i, count;
    ContourPoint *conpt;
    PixelInfo *p;
    int     direction = (end == HEAD ? FORWARD : BACKWARD);

    count = 0;
    for (i = i0, GetEnd(con, end, &conpt, &p);
	 count++ < n && conpt != NULL;
	 GetNext(con, direction, &conpt, &p)) {

	if (i < 0 || i > plist->n - 1) {
	    warn("FillPointListFromContour -- out of range");
	    return;
	}
	plist->x[i] = p->pos.x;
	plist->y[i] = p->pos.y;
	plist->tan[i] = conpt->angle;
	plist->k[i] = p->k;
	plist->L[i] = CORNERNESS(IROUND(p->pos.x), IROUND(p->pos.y));
	i += di;
    }
}

void
FreePointList(plist)
PointList *plist;
{
    if (plist != NULL) {
	free(plist->x);
	free(plist->y);
	free(plist->tan);
	free(plist->k);
	free(plist->L);
	free(plist);
    }
}

/*
 * FindCorners(thresh, r) finds local maxima of Cornerness >= thresh along
 * contours. Only one is allowed in a neighborhood of radius r. Ends of
 * contours do not count as corners.
 */
FindCorners(thresh, r)
real    thresh, r;
{
    int     changed;
    Contour *c;
    int     nfound = 0;

    do {

	changed = FALSE;

	for (c = Contours; c != NULL; c = c->R) {

	    PointList *plist = BuildPointList(c);
	    int     n = c->length;
	    int     i;

	    /* any pixel that is best in the nbhd */
	    for (i = 0; i < n; i++) {

		/*
		 * this marks c as not closed if it has a corner near an
		 * endpoint
		 */
		if (CornerLocalMax(plist, i, thresh, r, c->closed && !c->split)) {

		    /* got one */
		    SplitContour(c, NthFromEndP(c, HEAD, i));
		    changed = TRUE;
		    nfound++;
		    break;
		}
	    }
	    FreePointList(plist);
	    if (changed)
		break;
	}

    } while (changed);

    fprintf(stderr, "Found %d corners\n", nfound);
}

/*
 * PDist(plist,i,j) -- distance between pts i and j
 */
real
PDist(plist, i, j)
PointList *plist;
int     i, j;
{
    double  dx, dy;

    dx = plist->x[j] - plist->x[i];
    dy = plist->y[j] - plist->y[i];

    if (dx == 0 && dy == 0)
	return 0;

    /* fabs is in case of lost precision */
    return sqrt(fabs(dx * dx + dy * dy));
}

/*
 * CornerLocalMax()  -- is cornerness a local max at i? Compare back and
 * forth, wrapping if closed. If not, then disqualify if near either end.
 */
CornerLocalMax(plist, i, thresh, r, closed)
PointList *plist;
int     i;
real    thresh, r;
int     closed;
{
    int     j;
    int     n = plist->n;

    /* must be above thresh */
    if (plist->L[i] < thresh) {
	DBUG(8, ("-- (%g,%g) below thresh\n", plist->x[i], plist->y[i]));
	return FALSE;
    }
    if (!closed) {
	if (i == 0 || i == n - 1) {
	    DBUG(8, ("-- (%g,%g) too close to end\n", plist->x[i], plist->y[i]));
	    return FALSE;
	}
	/* within r of either end ? */
	if (PDist(plist, 0, i) < r || PDist(plist, n - 1, i) < r) {
	    DBUG(8, ("-- (%g,%g) too close to end\n", plist->x[i], plist->y[i]));
	    return FALSE;
	}
    }
    /* must be local max within the nbhd */
    for (j = i + 1;; j++) {

	if (closed && j > n - 1)
	    j -= n;

	if (j == i || j > n - 1)
	    break;

	if (PDist(plist, i, j) >= r)
	    break;

	/* finally, we can check */
	if (plist->L[i] < plist->L[j]) {
	    DBUG(8, ("-- (%g,%g) neighbor (%g,%g) better\n",
		     plist->x[i], plist->y[i],
		     plist->x[j], plist->y[j]));
	    return FALSE;
	}
    }

    for (j = i - 1;; j--) {

	if (closed && j < 0)
	    j += n;

	if (j == i || j < 0)
	    break;

	if (PDist(plist, i, j) >= r)
	    break;

	if (plist->L[i] < plist->L[j]) {
	    DBUG(8, ("-- (%g,%g) neighbor (%g,%g) better\n",
		     plist->x[i], plist->y[i],
		     plist->x[j], plist->y[j]));
	    return FALSE;
	}
    }

    return TRUE;
}

DeleteShortContours(length)
int     length;
{
    Contour *c, *next;

    DBUG(1, ("Deleting all contours < %d pixels long\n",
	     length));

    for (c = Contours; c != NULL; c = next) {
	next = c->R;
	if (!c->boundary && c->length < length) {
	    DeleteContour(c);
	}
    }
}

/*
 * delete standalone contours < length
 */
DeleteStandaloneContours(length)
int     length;
{
    Contour *c, *next;
    ContourPoint *conpt;
    PixelInfo *p, *q;

    DBUG(1, ("Deleting standalone contours < %d pixels long\n",
	     length));

    for (c = Contours; c != NULL; c = next) {
	next = c->R;
	if (!c->boundary && c->length < length) {
	    GetEnd(c, HEAD, &conpt, &p);
	    GetEnd(c, TAIL, &conpt, &q);
	    if (p->type == P_POINT || (p->type == P_END && q->type == P_END))
		DeleteContour(c);
	}
    }
}

/*
 * Delete non-closed contours with one end == END or POINT, or which are
 * connected to no more than one other contour and are shorter than min.
 */
DeleteStrayContours(min)
int     min;
{
    Contour *c, *next;
    ContourPoint *conpt;
    PixelInfo *p;
    int     end;
    int     changed;
    int     i, j, nothers;
    int     hitboundary;
    Contour *others[6];

    fprintf(stderr,
    "Deleting non-closed contours with one end of type END or POINT...\n");

    do {
	changed = FALSE;
	DeleteStandaloneContours(min);
	MergeCorners();

	for (c = Contours; c != NULL; c = next) {
	    next = c->R;

	    if (c->boundary)
		continue;

	    if (!c->closed) {

		for (end = 0; end <= 1; end++) {
		    GetEnd(c, end, &conpt, &p);
		    if (p->type == P_END || p->type == P_POINT) {
			DeleteContour(c);
			changed = TRUE;
			break;	/* to outer loop */
		    }
		}
	    }

	    /*
	     * if connected to only 1 other contour, delete ... this applies
	     * to closed contours as well
	     */
	    if (changed)
		break;

	    if (c->length >= min)
		continue;

	    /* count # other contours to which it is connected */
	    /* (count image boundary only once) */
	    nothers = 0;
	    hitboundary = FALSE;
	    for (end = 0; end <= 1; end++) {
		GetEnd(c, end, &conpt, &p);
		for (i = 0; i < 4; i++) {

		    Contour *c2 = p->conpt[i].con;

		    if (c2 != NULL && c2 != c) {

			/* only 1st of image border */
			if (c2->boundary) {
			    if (hitboundary)
				break;
			    hitboundary = TRUE;
			}
			for (j = 0; j < nothers; j++)
			    if (c2 == others[j])
				break;

			/* connected to somebody else */
			if (j >= nothers)
			    others[nothers++] = c2;
		    }
		}
	    }
	    if (nothers <= 1) {
		DeleteContour(c);
		changed = TRUE;
		break;
	    }
	}
    } while (changed);
}

/*
 * exclude and destroy contour.
 */
DeleteContour(c)
Contour *c;
{
    DBUG(7, ("Deleting contour %d(%d,%d)\n",
	     c->length, c->end[HEAD].x, c->end[HEAD].y));
    LopOff(c, HEAD, c->length);
    ExcludeContour(c);
    free(c);
    NContours--;
}

/*
 * lop off n points from contour c starting at end end
 */
LopOff(c, end, npoints)
Contour *c;
int     end;
int     npoints;
{
    ContourPoint *conpt, *nextconpt;
    PixelInfo *p, *nextp;
    PixelType oldtype;
    int     direction = (end == HEAD ? FORWARD : BACKWARD);
    int     lopped;

    if (npoints == 0)
	return;

    /* c better have npoints points to lop off */
    assert(npoints <= c->length);

    DBUG(9, ("Lopping: "));

    /* leave pointers to this contour */
    for (lopped = 0, GetEnd(c, end, &conpt, &p);
	 conpt != NULL && lopped < npoints; lopped++) {

	/* find next ones */
	nextconpt = conpt;
	nextp = p;

	oldtype = p->type;

	GetNext(c, direction, &nextconpt, &nextp);

	switch (p->type) {

	case P_EMPTY:		/* not on any contour */
	    break;		/* nothing to do */

	case P_POINT:
	case P_END:
	case P_MIDDLE:
	    p->type = P_EMPTY;
	    break;

	case P_CORNER:		/* 2 contours end here, different tangents */
	    p->type = P_END;

	    /* shift */
	    if (p->conpt[0].con == c) {
		p->conpt[0] = p->conpt[1];
	    }
	    conpt = &p->conpt[1];	/* to be nulled */
	    break;


	case P_T:		/* triple -> corner, even if it shouldn't */
	case P_Y:
	    p->type = P_CORNER;
	    if (p->conpt[0].con == c) {
		p->conpt[0] = p->conpt[1];
		p->conpt[1] = p->conpt[2];
	    } else if (p->conpt[1].con == c) {
		p->conpt[1] = p->conpt[2];
	    }
	    conpt = &p->conpt[2];	/* to be nulled */

	    break;

	case P_X:		/* 4 contours -> triple */
	    p->type = P_Y;
	    if (p->conpt[0].con == c) {
		p->conpt[0] = p->conpt[1];
		p->conpt[1] = p->conpt[2];
		p->conpt[2] = p->conpt[3];
	    } else if (p->conpt[1].con == c) {
		p->conpt[1] = p->conpt[2];
		p->conpt[2] = p->conpt[3];
	    } else if (p->conpt[2].con == c) {
		p->conpt[2] = p->conpt[3];
	    }
	    conpt = &p->conpt[3];	/* to be nulled */
	    break;

	}

	DBUG(9, ("%s->%s(%g,%g) ",
		 TYPE_STRING(oldtype), TYPE_STRING(p->type),
		 p->pos.x, p->pos.y));

	/* null out the now-unused contour point */
	conpt->n[0] = conpt->n[1] = 0;
	conpt->con = NULL;
	conpt->angle = 0.0;

	/* bump the "end" up to the next point, if there is one */
	if (nextconpt != NULL) {
	    nextconpt->n[direction == FORWARD ? BACKWARD : FORWARD] = 0;
	    c->end[end].x = IROUND(nextp->pos.x);
	    c->end[end].y = IROUND(nextp->pos.y);
	    if (nextp->type == P_MIDDLE)
		nextp->type = P_END;
	    else if (nextp->type == P_END)
		nextp->type = P_POINT;
	}
	c->length--;

	/* set p and conpt to next */
	conpt = nextconpt;
	p = nextp;
    }

    DBUG(9, ("\n"));
    c->closed = FALSE;		/* not closed any more! */

    /* make sure we're not leaving a trivial contour at a multiple-point */
    if (nextconpt != NULL && c->length == 1 && nextp->type != P_POINT) {
	char    buf[512];

	sprintf(buf, "Lopped off all but stub of contour at %s(%g,%g)\n",
		TYPE_STRING(nextp->type), nextp->pos.x, nextp->pos.y);
	warn(buf);
    }
}

/* exclude contour from Contours */
ExcludeContour(c)
Contour *c;
{
    Contour *prev;

    if (Contours == NULL)
	return;

    if (c == Contours) {
	Contours = Contours->R;
	return;
    }
    for (prev = Contours; prev->R != NULL && prev->R != c;)
	prev = prev->R;

    assert(prev->R == c);	/* Pointer bug:  contour not in list */

    prev->R = c->R;
}

/* reverses order of ContourPoints in a contour */
ReverseContour(c)
Contour *c;
{
    ContourPoint *conpt;
    PixelInfo *p;
    PixelPos postemp;
    int     temp;

    GetEnd(c, HEAD, &conpt, &p);

    DBUG(7, ("Reversing contour %d(%d,%d)\n",
	     c->length, c->end[HEAD].x, c->end[HEAD].y));

    /* switch head & tail */
    postemp = c->end[HEAD];
    c->end[HEAD] = c->end[TAIL];
    c->end[TAIL] = postemp;

    /* switch n[FORWARD] with n[BACKWARD] and reverse theta */
    do {
	/* stash the next one */

	temp = conpt->n[FORWARD];
	conpt->n[FORWARD] = conpt->n[BACKWARD];
	conpt->n[BACKWARD] = temp;

	conpt->angle = AngleFix(conpt->angle + M_PI);

	/* bump forward (remember we switched) */
	GetNext(c, BACKWARD, &conpt, &p);

    } while (conpt != NULL);
}

PrintParameters(stream, prefix)
FILE   *stream;
char   *prefix;
{
    fprintf(stream,
    "%scommand: %s -a %g -b %d -c %g -C %g -g %d -m %d %s -s %g -t %g %g %s\n",
	    prefix,
	    PgmName,
	    DEG(SweepAngle),
	    MinBeforeGrowth,
	    CornerPercentile,
	    CornerRadius,
	    MaxJumpGap,
	    MinStandaloneContour,
	    (PruneCrackTips ? "-p" : ""),
	    SDev,
	    LoPercentile, HiPercentile,
	    InputFileName);

    fprintf(stream, "%sEdge strength thresholds at %g%%ile and %g%%ile\n",
	    prefix, LoPercentile, HiPercentile);
    fprintf(stream, "%sCornerness threshold at %g%%ile\n", prefix,
	    CornerPercentile);
    fprintf(stream, "%sCorners must be local max with %g-pixel radius\n",
	    prefix, CornerRadius);
    fprintf(stream,
    "%sStd Dev %g for gaussian averaging of Ix^2, IxIy, and Iy^2\n", prefix,
	    SDev);
    fprintf(stream, "%s... thus length scale for closed contours, etc. is %g\n",
	    prefix, LengthScale);
    fprintf(stream, "%sContours < %d pixels are deleted before gap-jumping\n",
	    prefix, MinBeforeGrowth);
    fprintf(stream, "%sGaps are jumped by at most %d pixels\n", prefix,
	    MaxJumpGap);
    fprintf(stream, "%sGrowing endpoints bend to +/- %g degrees\n", prefix,
	    DEG(SweepAngle));
    if (MinStandaloneContour > 1)
	fprintf(stream, "%sSelf-contained contours of < %d pixels are pruned\n",
		prefix, MinStandaloneContour);
    if (PruneCrackTips)
	fprintf(stream, "%sCrack tips pruned\n", prefix);
}

/*
 * EDGE OUTPUT: print x y theta k  for each ContourPoint; separate edges by
 * blank lines.
 */
PrintContours(explanation, ext)
char   *explanation, *ext;
{
    char   *ctime();
    time_t  now;
    FILE   *f;
    char   *name;

    Contour *c;
    ContourPoint *conpt;
    PixelInfo *p;

    name = FileName(ext);
    f = fopen(name, "w");
    if (f == NULL) {
	fprintf(stderr, "Couldn't save to %s -- writing to standard output\n",
		name);
	f = stdout;
    }
    time(&now);

    fprintf(f, "# Contour list from image file `%s' produced %s\n",
	    InputFileName, ctime(&now));	/* ctime puts in 2nd \n */

    fprintf(f, "# %s\n", explanation);

    PrintParameters(f, "# ");

    fprintf(f, "\n# Contour map size -- width height\n%d %d\n\n", Width, Height);

    fprintf(f, "# x y theta(deg) curvature\n");
    for (c = Contours; c != NULL; c = c->R) {

	fprintf(f, "# CONTOUR %d points", c->length);
	if (c->closed) {
	    fprintf(f, " CLOSED");

	    if (c->split)
		fprintf(f, " WITHCORNER");
	}
	if (c->boundary)
	    fprintf(f, " BOUNDARY");
	fprintf(f, "\n");

	for (GetEnd(c, HEAD, &conpt, &p); conpt != NULL;
	     GetNext(c, FORWARD, &conpt, &p))
	    fprintf(f, "%g %g %g %g\n",
		    p->pos.x, p->pos.y, DEG(conpt->angle), p->k);

	fprintf(f, "\n");
    }
    if (f != stdout) {
	fclose(f);
	fprintf(stderr, "Wrote contour lists `%s' to `%s'\n",
		explanation, name);
    }
}

/*
 * SplitContour(contour, p) splits contour from its HEAD up to and including
 * point at PixelInfo *p, into a new contour structure, which is placed at
 * the beginning of the Contours list. The original contour is truncated to
 * start at that same point, now a P_CORNER, and continues to the original
 * TAIL.
 * 
 * ASSUMES contour points besides head and tail are all of type P_MIDDLE.
 * 
 * Bookkeeping is the real hard part.
 */
SplitContour(c, cutp)
Contour *c;
PixelInfo *cutp;
{
    Contour *newcon;
    ContourPoint *head, *tail, *cutpoint, *startpoint;
    int     length, origlength = c->length;
    PixelPos pos;
    PixelInfo *p, *q;
    int     tailmatch;

    pos = vectopp(cutp->pos);

    cutpoint = &cutp->conpt[0];

    if (c->closed &&
	((pos.x == c->end[HEAD].x && pos.y == c->end[HEAD].y)
    || (tailmatch = (pos.x == c->end[TAIL].x && pos.y == c->end[TAIL].y)))) {

	if (!AlwaysMergeCorners) {
	    DBUG(4, ("Splitting closed contour at split point\n"));
	    c->split = TRUE;
	}
	/* assure that cut point is at head */
	if (tailmatch)
	    ReverseContour(c);
	return;
    }
    /* do some checking */
    assert(cutp->type == P_MIDDLE);

    /* get head & tail */
    GetEnd(c, HEAD, &head, &p);
    GetEnd(c, TAIL, &tail, &q);

    DBUG(4, ("Splitting %scontour %s(%g,%g) to %s(%g,%g) length %d at %s(%g,%g) => ",
	     c->closed ? "CLOSED " : "",
	     TYPE_STRING(p->type),
	     p->pos.x, p->pos.y,
	     TYPE_STRING(q->type),
	     q->pos.x, q->pos.y, c->length,
	     TYPE_STRING(cutp->type),
	     cutp->pos.x, cutp->pos.y));

    if (c->closed && !c->split && p->type == P_END && q->type == P_END) {

	/***
	 * We just split a closed contour in the middle, one of whose ends
	 * connected to other contour(s), or which was marked "split", ie.
	 * had a kink at the implicit connection.  Now make the connection
	 * explicit between the original head & tail of c,
	 * now the TAIL of c and the HEAD of newcon.
	 *   HEAD of _
	 *    newcon  \
	 * x<- x<- x<- p<- --- other contour connects here
	 * |           .
	 * v           . <- make this connection explicit
	 * x   x ->x ->q
	 *   ^          \__TAIL of c
	 *   |____  just "split" here.
	 */
	ContourPoint *cutpoint1;
	PixelInfo *cutp1;

	DBUG(4, ("(just shuffling ends of closed contour)\n"));
	p->type = P_MIDDLE;
	q->type = P_MIDDLE;
	p->conpt[0].n[BACKWARD] = DirectionToNeighbor(c->end[HEAD], c->end[TAIL]);
	q->conpt[0].n[FORWARD] = AntiNeighbor[p->conpt[0].n[BACKWARD]];

	/* make cutp new HEAD and cutp1 (next backward of cutp) new TAIL */
	cutp1 = cutp;
	cutpoint1 = cutpoint;
	GetNext(c, BACKWARD, &cutpoint1, &cutp1);

	cutp->type = P_END;
	cutp->conpt[0].n[BACKWARD] = 0;
	c->end[HEAD] = vectopp(cutp->pos);

	cutp1->type = P_END;
	cutp1->conpt[0].n[FORWARD] = 0;
	c->end[TAIL] = vectopp(cutp1->pos);

	/* make a note that it was deliberately split there */
	if (!AlwaysMergeCorners) {
	    DBUG(4, ("Splitting closed contour at split point\n"));
	    c->split = TRUE;
	}
	return;
    }
    /* passed the test -- make a new contour structure & add to list */
    newcon = NewContour(pos.x, pos.y);
    newcon->end[HEAD] = c->end[HEAD];
    newcon->end[TAIL] = pos;	/* this was already done by NewContour
				 * (clarity?) */
    newcon->boundary = c->boundary;	/* boundary con splits into boundary
					 * con's */

    /* bump head up to cutpoint, changing contour pointers to newcon */
    length = 0;
    for (;;) {
	head->con = newcon;
	length++;
	if (head == cutpoint)
	    break;
	GetNext(c, FORWARD, &head, &p);
	assert(head != NULL);	/* contour ended before cutpoint -- bad n[]
				 * list */
    }

    /* bookkeeping */
    cutp->type = P_CORNER;

    /* take the next contour point slot */
    startpoint = &cutp->conpt[1];
    startpoint->con = c;
    startpoint->angle = cutpoint->angle;	/* so why is it a corner? */
    startpoint->n[FORWARD] = cutpoint->n[FORWARD];
    cutpoint->n[FORWARD] = 0;

    /* change HEAD of the original contour to the cutpoint */
    c->end[HEAD] = vectopp(cutp->pos);

    newcon->length = length;
    c->length = origlength - length + 1;	/* count cutpoint twice */

    if (c->closed) {
	/***
	 * We just split a closed contour in the middle, one of whose ends
	 * connected to other contour(s), or which was marked "split", ie.
	 * had a kink at the implicit connection.  Now make the connection
	 * explicit between the original head & tail of c,
	 * now the TAIL of c and the HEAD of newcon.
	 *   HEAD of _
	 *    newcon  \
	 * x<- x<- x<- p<- --- other contour connects here
	 * |           .
	 * v           . <- make this connection explicit
	 * x   x ->x ->q
	 *   ^          \__TAIL of c
	 *   |____  just "split" here.
	 */
	c->closed = FALSE;
	DBUG(4, ("\nSplit closed %scontour -- rejoining old head & tail\n",
		 c->split ? "split " : ""));

	if (PMap(newcon->end[HEAD].x, newcon->end[HEAD].y)->type == P_END)
	    AddPointToContour(newcon, HEAD,
			      (real) c->end[TAIL].x, (real) c->end[TAIL].y);

	else if (PMap(c->end[TAIL].x, c->end[TAIL].y)->type == P_END)
	    AddPointToContour(c, TAIL,
		    (real) newcon->end[HEAD].x, (real) newcon->end[HEAD].y);
    }
    c->closed = FALSE;
    c->split = FALSE;

    DBUG(4, ("%d + %d\n", newcon->length, c->length));
}

/*
 * JoinContour(c, end, p) joins contour c at end end to point p.
 * 
 * Just a bunch of bookkeeping.
 */
JoinContour(c, end, p)
Contour *c;
int     end;
PixelInfo *p;
{
    PixelPos pos;
    ContourPoint *endconpt, *conpt;
    PixelInfo *endp;
    Contour *ac;
    int     dir;		/* neighbor direction from p to end of c */

    pos.x = IROUND(p->pos.x);
    pos.y = IROUND(p->pos.y);
    dir = DirectionToNeighbor(pos, c->end[end]);

    /* get end pixel and conpt */
    GetEnd(c, end, &endconpt, &endp);

    DBUG(4, ("Joining %s of contour %d(%g,%g) to %s(%g,%g)\n",
	     end == HEAD ? "HEAD" : "TAIL",
	     c->length,
	     endp->pos.x, endp->pos.y,
	     TYPE_STRING(p->type),
	     p->pos.x, p->pos.y));

    /* make sure we're not joining p to p */
    if (p == endp) {
	DBUG(0, ("JoinContour asked to join %s of %d(%d,%d) to itself--not joining\n",
		 end == HEAD ? "HEAD" : "TAIL",
		 c->length,
		 c->end[HEAD].x, c->end[HEAD].y));
	return;
    }
    /* attach to p according to type */
    switch (p->type) {

    case P_EMPTY:		/* not on any contour--just add to head */
	p->type = P_END;
	conpt = &p->conpt[0];
	break;

    case P_POINT:		/* attach to trivial contour */
	conpt = &p->conpt[0];
	DeleteContour(conpt->con);	/* don't leave bits around! */
	/* remember Delete turns p from POINT into EMPTY */
	p->type = P_END;
	break;

    case P_END:		/* another contour ends here--call it a
				 * corner? */

	if (p->conpt[0].con == c) {

	    DBUG(4, ("(self--"));

	    /* just ran into its own tail */
	    if (c->length >= MIN_CLOSED_CONTOUR) {
		c->closed = TRUE;
		DBUG(4, ("joined)\n"));
	    } else
		DBUG(4, ("not joined)\n"));
	    return;
	}
	p->type = P_CORNER;
	conpt = &p->conpt[1];
	break;

    case P_MIDDLE:		/* end of contour runs into middle of
				 * another: becomes P_Y */

	if (p->conpt[0].con == c) {

	    DBUG(4, ("(Joining contour to self)\n"));

	    /*
	     * ran into self!  Split c at p by turning it into a corner
	     */
	    SplitContour(c, p);
	    if (p->type != P_CORNER) {
		fprintf(stderr,
			"?closed? contour ran into its own middle (%g,%g)\n",
			p->pos.x, p->pos.y);
		return;
	    }
	    /* mark the contour that goes from c's old end to p as closed */
	    if (end == HEAD)
		ac = p->conpt[0].con;
	    else
		ac = p->conpt[1].con;

	    if (ac->length >= MIN_CLOSED_CONTOUR)
		ac->closed = TRUE;

	    return;
	}
	/* split into P_CORNER and then turn into Y */
	SplitContour(p->conpt[0].con, p);

	if (p->type == P_END) {

	    /* splitting simple closed contour just shuffles ends around */
	    p->type = P_CORNER;
	    conpt = &p->conpt[1];
	    break;
	}
	p->type = P_Y;
	conpt = &p->conpt[2];
	break;

    case P_CORNER:		/* 2 contours end here, diff angles, becomes
				 * a Y */

	if (p->conpt[0].con == c || p->conpt[1].con == c) {

	    /*
	     * c ran into its own other end
	     */
	    DBUG(4, ("(CORNER is other end of self)\n"));

	    if (c->length >= MIN_CLOSED_CONTOUR)
		c->closed = TRUE;
	    return;
	}
	p->type = P_Y;
	conpt = &p->conpt[2];
	break;

    case P_T:			/* 3 contours end here -- turn into P_X */
    case P_Y:
	if (p->conpt[0].con == c || p->conpt[1].con == c || p->conpt[2].con == c) {

	    /*
	     * highly unlikely:  a three-leaf clover?
	     */
	    if (c->length >= MIN_CLOSED_CONTOUR)
		c->closed = TRUE;
	    return;
	}
	p->type = P_X;
	conpt = &p->conpt[3];
	break;

    case P_X:			/* 4 contours end here, 2 pairs of matching
				 * tangents */
	/* can't join */
	DBUG(0, ("5 contours at (%g,%g)?--not joining.\n",
		 p->pos.x, p->pos.y));
	return;
    }

    /* update the old end point info */
    if (endp->type == P_POINT)
	endp->type = P_END;
    else if (endp->type == P_END)
	endp->type = P_MIDDLE;
    else			/* two neighboring corners or Y's? */
	assert(0);

    /* pointers */
    conpt->con = c;
    conpt->n[end == HEAD ? FORWARD : BACKWARD] = dir;	/* dir goes from p to
							 * old end */
    endconpt->n[end == HEAD ? BACKWARD : FORWARD] = AntiNeighbor[dir];

    c->end[end] = pos;		/* p is c's new end */
    c->length++;

    if (AlwaysMergeCorners && p->type == P_CORNER)
	MergeCorner(p);
}

/***
 * return input file name with its extension replaced by extension.
 *	"foo.im" --> "foo<extension>"
 *	"bar" --> "bar<extension>"
 * (If there was no '.' in the file name, just appends extension.)
 *
 * returns a static buffer
 */
char   *
FileName(extension)
char   *extension;
{
    static char buf[512];
    char   *rindex(), *lastdot;

    strcpy(buf, InputFileName);

    lastdot = rindex(buf, '.');
    if (lastdot)
	*lastdot = '\0';	/* replace any suffix */
    strcat(buf, extension);
    return buf;
}

/*
 * write a file "foo.ebm" with an edge bitmap
 */
WriteEdgeBitMap(ext)
char   *ext;
{
    IMAGE  *EdgeBitMap;
    int     x, y;
    Contour *c;
    ContourPoint *conpt;
    PixelInfo *p;

    EdgeBitMap = hvMakeImage(Height, Width, 1, SEQUENTIAL, ONEBYTE);

    /* make it white, on which we draw black pixels */
    for (y = 0; y < Height; y++)
	for (x = 0; x < Width; x++)
	    BYTE(EdgeBitMap, x, y) = WHITE;

    /* stick one pixel at each ContourPoint in each contour */
    for (c = Contours; c != NULL; c = c->R)
	for (GetEnd(c, HEAD, &conpt, &p); conpt != NULL;
	     GetNext(c, FORWARD, &conpt, &p))
	    BYTE(EdgeBitMap, (int) p->pos.x, (int) p->pos.y) = BLACK;

    WriteImage(EdgeBitMap, "Edge bit map (black on white)", ext);

    hvFreeImage(EdgeBitMap);
}

int
ContourNumber(c)
Contour *c;
{
    int     n;
    Contour *cp;

    if (c == NULL)
	return 0;

    for (n = 1, cp = Contours; cp != NULL; cp = cp->R, n++)
	if (cp == c)
	    return n;
    return 0;
}


/*
 * WriteImage(I, explanation, extension) -- save image I in file
 * foo.<extension> (input image foo.im) and notify user w/explanation
 */
WriteImage(I, explanation, extension)
IMAGE  *I;
char   *explanation, *extension;
{
    hvWriteImage(I, FileName(extension));

    fprintf(stderr, "Saved image `%s' in file `%s'\n",
	    explanation, FileName(extension));
}


/*
 * make angle a between -pi and pi
 */
real
AngleFix(a)
real    a;
{
    while (a > M_PI)
	a -= 2 * M_PI;
    while (a < -M_PI)
	a += 2 * M_PI;

    return a;
}


/*
 * utils to skip around contours
 * 
 * # steps to go from end of con to pixinfo p, returns 10000 if contour doesn't
 * contain p
 */
StepsFromHereToEnd(pcand, end, con)
PixelInfo *pcand;
int     end;
Contour *con;
{
    ContourPoint *conpt;
    PixelInfo *p;
    int     direction = (end == HEAD ? FORWARD : BACKWARD);
    int     nsteps;

    nsteps = 0;
    for (GetEnd(con, end, &conpt, &p); conpt != NULL;
	 GetNext(con, direction, &conpt, &p)) {
	if (p == pcand)
	    break;
	nsteps++;
    }

    if (p != pcand) {
	DBUG(0, ("StepsFromHereToEnd: (%g,%g) not in contour %d(%d,%d)\n",
		 pcand->pos.x, pcand->pos.y,
		 con->length, con->end[HEAD].x, con->end[HEAD].y));
	return 10000;
    }
    return nsteps;
}

/*
 * reflect point p in the line going through nail w/angle nailtheta
 */
vector
Reflect(p, nail, nailtheta)
vector  p, nail;
real    nailtheta;
{
    double  sint, cost;
    vector  result, v, w;

    sincos(2.0 * nailtheta, &sint, &cost);

    /* v = (p-nail) */
    v = v_difference(nail, p);

    /* w = matrix(cost sint / sint -cost).v */
    w.x = cost * v.x + sint * v.y;
    w.y = sint * v.x - cost * v.y;

    /* w + nail is result */
    result = v_sum(w, nail);

    DBUG(10, ("Reflect (%g,%g) in line thru (%g,%g) angle %g gives (%g,%g)\n",
	      p.x, p.y, nail.x, nail.y, DEG(nailtheta), result.x, result.y));

    return result;
}

/* reflect angle theta about an axis of angle axisangle */
real
ReflectAngle(theta, axisangle)
real    theta, axisangle;
{
    return AngleFix(2.0 * (axisangle) - (theta));
}


/*
 * what neighbor direction from p1 to p2 (PixelPos) returns 0 if either
 * they're the same or they're not neighbors
 */
int
DirectionToNeighbor(p1, p2)
PixelPos p1, p2;
{
    /* easy enough */
    PixelPos diff;
    int     n;

    diff.y = p2.y - p1.y;
    diff.x = p2.x - p1.x;

    for (n = 0; n < NEIGHBORS; n++)
	if (diff.x == NeighborPos[n].x
	    && diff.y == NeighborPos[n].y)
	    return n;


    /* non-neighbors */
    return 0;
}

/*
 * pth point along contour c from end (0 means end itself)
 */
PixelInfo *
NthFromEndP(c, end, npix)
Contour *c;
int     end;
int     npix;
{
    PixelInfo *p;
    ContourPoint *conpt;
    int     direction = (end == HEAD ? FORWARD : BACKWARD);

    GetEnd(c, end, &conpt, &p);

    while (npix-- > 0 && conpt != NULL)
	GetNext(c, direction, &conpt, &p);

    if (conpt == NULL)
	return NULL;
    return p;
}

#ifndef NDEBUG

/*
 * find out what kind of mess we made
 */
CheckContourConsistency()
{
    Contour *con, *c;
    ContourPoint *conpt, *conpttail;
    PixelInfo *p, *ptail;
    int     nforward, nbackward;
    PixelPos where;
    static int dontbother;
    int     x, y, i;
    int     dir;

    if (dontbother)
	return;

    /* couple simple checks for consistency */
    for (c = Contours, i = 0; c != NULL; c = c->R, i++) {
	GetEnd(c, HEAD, &conpt, &p);
	if (p->type == P_EMPTY)
	    DBUG(0, ("Contour %d empty head\n", i));

	GetEnd(c, TAIL, &conpttail, &ptail);
	if (ptail->type == P_EMPTY)
	    DBUG(0, ("Contour %d empty tail\n", i));

    }

    for (y = 0; y < Height; y++)
	for (x = 0; x < Width; x++) {
	    p = PMap(x, y);
	    if (p == NULL)
		continue;
	    if (IROUND(p->pos.x) != x || IROUND(p->pos.y) != y) {
		fprintf(stderr,
			"HEY! PMap(%d,%d) thinks it's (%g,%g)! (%s)\n",
			x, y, p->pos.x, p->pos.y, TYPE_STRING(p->type));
		dontbother = TRUE;
	    }
	}

    for (con = Contours; con != NULL; con = con->R) {

	nforward = nbackward = 0;

	where = con->end[HEAD];
	for (GetEnd(con, HEAD, &conpt, &p); conpt != NULL;) {

	    nforward++;

	    if ((dir = conpt->n[FORWARD]) == 0)
		break;
	    where.x += NeighborPos[dir].x;
	    where.y += NeighborPos[dir].y;

	    /*
	     * this is what the next macro would do: p =
	     * PMap(NeighborPos[dir].x + (int) p->pos.x, NeighborPos[dir].y +
	     * (int) p->pos.y);
	     */
	    p = PMap(where.x, where.y);
	    conpt = &p->conpt[CONPT_force(p, con)];
	}

	for (GetEnd(con, TAIL, &conpt, &p); conpt != NULL;
	     GetNext(con, BACKWARD, &conpt, &p))
	    nbackward++;

	if (con->length != nforward || nforward != nbackward)
	    fprintf(stderr, "Contour %d(%d,%d) really has %d points forward & %d back\n",
		    con->length, con->end[HEAD].x, con->end[HEAD].y, nforward, nbackward);
    }
}

/* call dump() from the debugger */
dump()
{
    FILE   *f = fopen("DUMP", "w");

    WritePixelMap(f);
    WriteContourChains(f);
    fclose(f);
    fprintf(stderr, "See file DUMP for PixelInfo and contours.\n");
}

/*
 * Print the basic PixelMap to stream f
 */
WritePixelMap(f)
FILE   *f;
{
    int     x, y, i;
    PixelInfo *p;
    char   *t;

    fprintf(f, "\
\n\
# The Pixel Map.  For each pixel that has a PixInfo structure,\n\
# gives the following information:\n\
#\n\
# <type> (x,y) c:fb c:fb c:fb c:fb\n\
#\n\
# Type is EMPTY, POINT, END, MIDDLE, CORNER, T, Y, or X\n\
# (x,y) is possibly refined pixel location\n\
# and the four c:fb's say that for contour number c,\n\
# f is the forward and b the backward neighbor.  Contours start from 1; 0=error.\n");

    for (y = 0; y < Height; y++) {
	for (x = 0; x < Width; x++) {
	    if ((p = PMap(x, y)) != NULL) {
		/* the type */
		t = TYPE_STRING(p->type);

		fprintf(f, "%s (%g,%g)", t, p->pos.x, p->pos.y);
		for (i = 0; i < 4; i++) {
		    fprintf(f,
			    " %d:%.1d%.1d",
			    ContourNumber(p->conpt[i].con),
			    p->conpt[i].n[FORWARD],
			    p->conpt[i].n[BACKWARD]);
		}
		fprintf(f, "\n");
	    }
	}
    }
}

/*
 * write a contour list in chain codes to stream f
 */
WriteContourChains(f)
FILE   *f;
{
    Contour *c;
    ContourPoint *conpt;
    PixelInfo *p;

    /* the contour chain codes */
    fprintf(f, "\
# Number of contours:\n\
%d\n\
# For each contour, a number, the starting point, and a chain code:\n\
#   4 3 2\n\
#   5 0 1   (neighbor numbers)\n\
#   6 7 8\n\
# <contour-number> (x,y) <one-digit-per-pixel>\n",
	    NContours);

    for (c = Contours; c != NULL; c = c->R) {
	fprintf(f, "%d (%d,%d) ",
		ContourNumber(c),
		c->end[HEAD].x, c->end[HEAD].y);

	for (GetEnd(c, HEAD, &conpt, &p); conpt != NULL;
	     GetNext(c, FORWARD, &conpt, &p))
	    if (conpt->n[FORWARD] != 0)
		fprintf(f, "%d", (int) conpt->n[FORWARD]);

	fprintf(f, "\n");
    }
}

#endif				/* NDEBUG */


/*
 * set *conpt and *pixinfo to end whichend of contour con. NOTE:  the
 * assert's all go away with -DNDEBUG
 */
GetEnd(con, whichend, conpt, pixinfo)
Contour *con;
int     whichend;
ContourPoint **conpt;
PixelInfo **pixinfo;
{
    PixelInfo *p;

    assert(con != NULL);
    p = PMap(con->end[whichend].x, con->end[whichend].y);

    assert(p != NULL);
    *conpt = &p->conpt[CONPT_force(p, con)];
    *pixinfo = p;
}

/*
 * set *conpt and *pixinfo to next point of contour con, in direction
 * direction sets *conpt to NULL if there is no next point
 */
GetNext(con, direction, conpt, pixinfo)
Contour *con;
int     direction;
ContourPoint **conpt;
PixelInfo **pixinfo;
{
    int     dir;
    PixelInfo *p;

    assert(con != NULL);
    assert(conpt != NULL);
    dir = (*conpt)->n[direction];
    if (dir == 0) {
	*conpt = NULL;
	return;
    }
    p = *pixinfo;
    p = PMap(NeighborPos[dir].x + IROUND(p->pos.x),
	     NeighborPos[dir].y + IROUND(p->pos.y));
    assert(p != NULL);
    *conpt = &p->conpt[CONPT_force(p, con)];
    *pixinfo = p;
}

/*
 * ApplyToLineWhile(fromx, fromy, tox, toy, f, val) -- call f(x,y) for grid
 * points x,y on ~line from from to to, *while* f() returns val.  If f()
 * returns anything else, aborts and returns the value that f() returned.
 * Returns val on success.
 */

#ifndef SGN
#define	SGN(x)		((x)<0 ? -1 : 1)
#endif

int
ApplyToLineWhile(fromx, fromy, tox, toy, f, val)
int     fromx, fromy, tox, toy, val;
int     (*f) ();
{
    int     x, y, dx, dy, flipxy, retval;
    int     xinc, yinc, incr1, incr2, d, xmin, xmax;
    int     npoints;

    dx = ABS(fromx - tox);
    dy = ABS(fromy - toy);

    if (dx == 0 && dy == 0) {
	DBUG(7, ("Line drawing no-op start==end (%d,%d)\n", fromx, fromy));
	return val;
    }
    DBUG(7, ("Drawing line (%d,%d)->(%d,%d)\n", fromx, fromy, tox, toy));

    if (dy < dx) {
	/* iterate x, and then sometimes y */
	flipxy = FALSE;

	/* initialize */
	d = (2 * dy) - dx;	/* "divergence" from line */
	incr1 = 2 * dy;		/* increment if d < 0 */
	incr2 = 2 * (dy - dx);	/* increment if d >= 0 */

	/* x moves from from to to, y moves according to slope */
	xinc = SGN(tox - fromx);
	yinc = xinc * SGN((toy - fromy) * (tox - fromx));
	x = fromx;
	y = fromy;
	xmin = MIN(fromx, tox);
	xmax = MAX(fromx, tox);

    } else {
	/* above code with fromx/y, tox/y, and dx/y flipped */
	flipxy = TRUE;

	/* initialize */
	d = (2 * dx) - dy;	/* "divergence" from line */
	incr1 = 2 * dx;		/* increment if d < 0 */
	incr2 = 2 * (dx - dy);	/* increment if d >= 0 */

	/* x moves from from to to, y moves according to slope */
	xinc = SGN(toy - fromy);
	yinc = xinc * SGN((tox - fromx) * (toy - fromy));
	x = fromy;
	y = fromx;
	xmin = MIN(fromy, toy);
	xmax = MAX(fromy, toy);
    }

    /* draw first point */
    retval = (flipxy ? (*f) (y, x) : (*f) (x, y));

    if (retval != val)
	return retval;

    for (npoints = 0;; npoints++) {
	x += xinc;
	if (d < 0)
	    d += incr1;
	else {
	    y += yinc;
	    d += incr2;
	}

	if (x < xmin || x > xmax)
	    break;

	retval = (flipxy ? (*f) (y, x) : (*f) (x, y));

	if (retval != val)
	    return retval;
    }
    DBUG(7, ("Line drew %d points\n", npoints));
    return val;
}

#ifndef NDEBUG
warning(string)
char   *string;
{
    fprintf(stderr, "\"%s\" warning: %s\n", __FILE__, string);
}

#endif
