/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            VIP
* MODULE:		snap.c - Program for image capturing using the
*				 Videopix board on the Suns.
* REVISION:             3.1
* AUTHOR:               DH
* CREATION DATE:        28 Oct 1991
* REVISION DATE:	7/10/92        
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:		3.1
* REVISION DATE:	11 July 1992
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)snap.c	3.1 7/10/92";
#endif


#include <stdio.h>
#include <stdlib.h>
/*
#include <sys/types.h>
*/
#include <sys/time.h>
#include <X11/Xlib.h>

#include <xview/canvas.h>
#include <xview/svrimage.h>
#include <xview/panel.h>
#include <xview/xv_xrect.h>
#include <xview/icon.h>
#include <xview/xview.h>

#include "vfc_lib.h"
#include "vip.h"
#include "snapfn.h"

#define UNDEFINED       1234
#define MAX_WIDTH       768	/* the maximum image size that one can */
#define MAX_HEIGHT      575	/* capture using the Videopix board. */

#define	TIMER_OFF	0	/* Preview timer is off */
#define	TIMER_ON	1	/* Preview timer is on */
#define	WM_OFFSET	16	/* Default window manager offset */

/*
 * Preview mode states
 */
#define PREVIEW_OFF     0	/* Not in preview mode */
#define PREVIEW_FULL    1	/* Full screen grayscale */
#define PREVIEW_HALF    2	/* Half screen grayscale */
#define PREVIEW_QRTR    3	/* Quarter screen grayscale */
#define PREVIEW_COL     4	/* Color preview mode */

#define BUF_LEN		256

/*
 * Global variables
 */
VfcDev *vfc_dev;		/* The VideoPix device pointer */
int     v_format;		/* Which type of video are we getting */
int     wm_offset = 0;		/* The colormap window manager offset */
int     pw_map_length = 0;	/* The current color map length */
int     state = PREVIEW_OFF;	/* The state of the tool */

int     width = MAX_WIDTH;
int     height = MAX_HEIGHT;	/* width and height of the image to be saved */
int     xoffset = 0, yoffset = 0;	/* offset specified by the user */
int     force_draw = 0;		/* force redrawing the window frame that
				 * bounds the image to be saved. */

/*
 * This timer is very short and gives the impression of
 * an infinite loop
 */
static struct itimerval preview_timer;

/*
 * Colrmap arrays
 */
static u_char y_ramp[VFC_GRAY_MAPLENGTH];
static u_char red_map[VFC_CCUBE_MAPLENGTH];
static u_char green_map[VFC_CCUBE_MAPLENGTH];
static u_char blue_map[VFC_CCUBE_MAPLENGTH];

/*
 * Colormaps
 */
Colormap pw_def_cmap, pw_priv_cmap, current_cmap;

/*
 * X11 variables related to the canvas
 */
XImage *pw_ximage = NULL;	/* The X11 image */
XImage *yuv_ximage = NULL;	/* The X11 image for the YUV data */
Xv_window canvas_pw;		/* The canvas paint window */
Display *pw_display;		/* Paint window display */
Drawable drawable;		/* Paint window XID */
Visual *pw_visual;		/* Paint window Visual */
GC      pw_gc;			/* Paint window Graphics context */
int     pw_screen;		/* Paint window screen */


Xv_window base_frame;		/* The XView base frame */
Icon    icon;
Server_image icon_image;

/* define the icon bitmap array */

short   icon_bits[] = {
#include "snapshot.icon" 
};


/*
 * The two panel items on the panel
 */
Panel_item preview_button, preview_choice, grab_button;
Panel_item fname_button;	/* item for entering file name */
Panel_item xoffset_button;	/* item for entering the image's X offset */
Panel_item yoffset_button;	/* item for entering the image's Y offset */
Panel_item width_button;	/* item for entering width of the image */
Panel_item height_button;	/* item for entering height of the image */
Panel_item msg_button;		/* item for displaying error/warning messages */


int     colorimage = 0;		/* image captured is color? (depend on the
				 * mode selected) */



/*- Set_Up_Hware ----------------------------------------------------

Set up the hard ware for using videopix.  The function returns -1
if any error (e.g. camera adaptor power off, unidentified video signal
detected, etc.) encountered in the setup, 0 otherwise.

--------------------------------------------------------------------*/

int     Setup_Hware()
{
    int     rc, format;

    xv_set(msg_button, PANEL_VALUE, (char *) NULL, NULL);
    /*
     * Open the hardware just use the default case "/dev/vfc0" and test for
     * success. Lock the hardware to deny other programs access
     */
    if ((vfc_dev = vfc_open(NULL, VFC_LOCKDEV)) == NULL) {
	xv_set(msg_button, PANEL_VALUE, "Could not open hardware.", NULL);
	return (-1);
    }

    /*
     * default video port is 1.  Use port 2 only if no vfc_set_port returns
     * -1.
     */
    vfc_set_port(vfc_dev, VFC_PORT1);

    /*
     * Determine the video type, and convert to VFC_PAL or VFC_NTSC
     */
    rc = vfc_set_format(vfc_dev, VFC_AUTO, &format);
    if (rc < 0) {
	vfc_set_port(vfc_dev, VFC_PORT2);
	if (vfc_set_format(vfc_dev, VFC_AUTO, &format)) {
	    xv_set(msg_button, PANEL_VALUE,
		   "Could not determine video format.", NULL);
	    return (-1);
	}
    }

    if (format == NO_LOCK) {
	/*
	 * No video signal detected, vfc_video_format defaults to NTSC
	 */
	xv_set(msg_button, PANEL_VALUE,
	       "Warning : no video signal in detected.", NULL);
    }
    v_format = vfc_video_format(format);

    /*
     * If it's NTSC, set the hue offset to 0 This is the default case.
     */
    vfc_adjust_hue(vfc_dev, 0);
    return (0);
}


/*- Create_Windows --------------------------------------------------

A brief description of what the function does.  Please put enough
comments in the function body as well if the code inside the function
body is too complicated.  Please run "indent" to indent the programs
you have written.  Thankyou.

--------------------------------------------------------------------*/

void    Create_Windows(argc, argv)
int     argc;
char  **argv;
{
    Panel   panel;
    Canvas  canvas;
    void    Quit_Proc(), Start_Notify(), Grab_Notify();
    void    Save_Notify(), Canvas_Repaint_Proc(), Canvas_Event_Proc();
    void    Get_Width(), Get_Height(), Get_Xoffset(), Get_Yoffset();

    /*
     * initialise xview environment.
     */
    (void) xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);

    /* create a server image using the icon bit array */
    icon_image = (Server_image) xv_create(XV_NULL, SERVER_IMAGE,
					  XV_WIDTH, 64,
					  XV_HEIGHT, 64,
					  SERVER_IMAGE_BITS, icon_bits,
					  XV_NULL);

    /* create icon using server image & place icon label on it */
    icon = (Icon)
	xv_create(XV_NULL, ICON,
		  ICON_IMAGE, icon_image,
		  ICON_LABEL, "snap",
		  XV_NULL);

    /*
     * Create the base_frame and assign the icon image to it.
     */
    base_frame =  xv_create(XV_NULL, FRAME_BASE,
			   XV_LABEL, "V.I.P. - snap",
			   FRAME_ICON, icon,
			   XV_NULL);
    /*
     * Create a simple control panel with a choice of preview modes and a
     * stop/start button.
     */
    panel = xv_create(base_frame, PANEL,
		      XV_HEIGHT, 100,
		      XV_WIDTH, vfc_format_width(v_format),
		      0);

    preview_choice = xv_create(panel, PANEL_CHOICE_STACK,
			       PANEL_LABEL_STRING, "Preview Mode : ",
			       PANEL_CHOICE_STRINGS,
			       "Full Grayscale",
			       "Half Grayscale",
			       "Qrtr Grayscale",
			       "Color",
			       0,
			       0);
    preview_button = xv_create(panel, PANEL_BUTTON,
			       PANEL_LABEL_STRING, "Start",
			       PANEL_NOTIFY_PROC, Start_Notify,
			       0);
    grab_button = xv_create(panel, PANEL_BUTTON,
			    PANEL_LABEL_STRING, "Grab",
			    PANEL_NOTIFY_PROC, Grab_Notify,
			    0);
    fname_button = xv_create(panel, PANEL_TEXT,
			     PANEL_LABEL_STRING, "File name:",
			     PANEL_NOTIFY_STRING, "\n\t\r\033",
			     PANEL_VALUE_DISPLAY_LENGTH, 20,
			     PANEL_VALUE_STORED_LENGTH, BUF_LEN,
			     0);
    xoffset_button = xv_create(panel, PANEL_TEXT,
			       PANEL_ITEM_X, 5,
			       PANEL_ITEM_Y, 30,
			       PANEL_LABEL_STRING, "X offset:",
			       PANEL_VALUE_DISPLAY_LENGTH, 4,
			       PANEL_VALUE_STORED_LENGTH, 4,
			       PANEL_NOTIFY_STRING, "\n\t\r\033",
			       PANEL_NOTIFY_PROC, Get_Xoffset,
			       0);
    yoffset_button = xv_create(panel, PANEL_TEXT,
			       PANEL_LABEL_STRING, "Y offset:",
			       PANEL_VALUE_DISPLAY_LENGTH, 4,
			       PANEL_VALUE_STORED_LENGTH, 4,
			       PANEL_NOTIFY_STRING, "\n\t\r\033",
			       PANEL_NOTIFY_PROC, Get_Yoffset,
			       0);
    (void) xv_create(panel, PANEL_BUTTON,
		     PANEL_LABEL_STRING, "Save",
		     PANEL_NOTIFY_PROC, Save_Notify,
		     0);
    width_button = xv_create(panel, PANEL_TEXT,
			     PANEL_ITEM_X, 5,
			     PANEL_ITEM_Y, 55,
			     PANEL_LABEL_STRING, "Image width:",
			     PANEL_VALUE_DISPLAY_LENGTH, 4,
			     PANEL_VALUE_STORED_LENGTH, 4,
			     PANEL_NOTIFY_STRING, "\n\t\r\033",
			     PANEL_NOTIFY_PROC, Get_Width,
			     0);
    height_button = xv_create(panel, PANEL_TEXT,
			      PANEL_LABEL_STRING, "Image height:",
			      PANEL_VALUE_DISPLAY_LENGTH, 4,
			      PANEL_VALUE_STORED_LENGTH, 4,
			      PANEL_NOTIFY_STRING, "\n\t\r\033",
			      PANEL_NOTIFY_PROC, Get_Height,
			      0);
    (void) xv_create(panel, PANEL_BUTTON,
		     PANEL_ITEM_X, 550,
		     PANEL_ITEM_Y, 55,
		     PANEL_LABEL_STRING, "Quit",
		     PANEL_NOTIFY_PROC, Quit_Proc,
		     0);
    msg_button = xv_create(panel, PANEL_TEXT,
			   PANEL_ITEM_X, 5,
			   PANEL_ITEM_Y, 85,
			   PANEL_LABEL_STRING, "",
			   PANEL_READ_ONLY, TRUE,
			   PANEL_VALUE_DISPLAY_LENGTH, BUF_LEN,
			   PANEL_VALUE_STORED_LENGTH, BUF_LEN,
			   0);

    window_fit_height(panel);

    /*
     * Create a canvas on which to draw the image.
     */
    canvas = xv_create(base_frame, CANVAS,
		       XV_WIDTH, vfc_format_width(v_format),
		       XV_HEIGHT, vfc_format_height(v_format),
		       XV_X, 0,
		       WIN_BELOW, panel,
		       CANVAS_WIDTH, vfc_format_width(v_format),
		       CANVAS_HEIGHT, vfc_format_height(v_format),
		       WIN_DYNAMIC_VISUAL, TRUE,
		       CANVAS_RETAINED, FALSE,
		       CANVAS_X_PAINT_WINDOW, TRUE,
		       WIN_EVENT_PROC, Canvas_Event_Proc,
		       0);
    xv_set(canvas, CANVAS_REPAINT_PROC, Canvas_Repaint_Proc, 0);

    canvas_pw = canvas_paint_window(canvas);
    xv_set(canvas_pw,
	   WIN_CONSUME_EVENTS,
	   LOC_WINENTER, 0,
	   WIN_EVENT_PROC, Canvas_Event_Proc,
	   WIN_BIT_GRAVITY, NorthWestGravity,
	   0);

    /*
     * Get some X11 properties from the canvas
     */
    pw_display = (Display *) xv_get(canvas_pw, XV_DISPLAY);
    drawable = xv_get(canvas_pw, XV_XID);
    pw_screen = DefaultScreen(pw_display);
    pw_gc = DefaultGC(pw_display, pw_screen);
    pw_visual = DefaultVisual(pw_display, pw_screen);

    /*
     * Get the default colormap and create a private colormap and install it
     * in the canvas.
     */
    pw_def_cmap = DefaultColormap(pw_display, pw_screen);
    pw_priv_cmap = XCreateColormap(pw_display, drawable, pw_visual,
				   AllocNone);
    XSetWindowColormap(pw_display, drawable, pw_priv_cmap);

    current_cmap = pw_def_cmap;

    window_fit(canvas);
    window_fit(base_frame);
    return;
}


/*- Canvas_Repaint_Proc ---------------------------------------------

Repaint proc for the canvas.

--------------------------------------------------------------------*/

void    Canvas_Repaint_Proc(canvas, pw, display, maydraw, xrects)
Canvas  canvas;
Xv_window *pw;
Display *display;
XID     maydraw;
Xv_xrectlist *xrects;
{
    if (pw_ximage)
	/* Just repaint the whole image */
	XPutImage(pw_display, maydraw, pw_gc, pw_ximage, 0, 0, 0, 0,
		  pw_ximage->width, pw_ximage->height);

    force_draw = 1;
    Get_Width();
    force_draw = 0;
    Get_Height();
    Get_Xoffset();
    Get_Yoffset();
}


/*- Canvas_Event_Proc -----------------------------------------------

Catch canvas events.

--------------------------------------------------------------------*/

void    Canvas_Event_Proc(window, event)
Xv_window window;
Event  *event;
{

    switch (event_action(event)) {
    case LOC_WINENTER:
	/*
	 * Re-install the private map if it is being used.
	 */
	if (current_cmap == pw_priv_cmap)
	    XInstallColormap(pw_display, current_cmap);
	break;
    case LOC_DRAG:
	xoffset = event_x(event);
	yoffset = event_y(event);
	xv_set(xoffset_button, PANEL_VALUE, atoi(( char * ) xoffset), NULL);
	xv_set(yoffset_button, PANEL_VALUE, atoi(( char * ) yoffset), NULL);
	Draw_Frame();
	break;
    }
    return;
}


/*- Get_Xoffset -----------------------------------------------------

Function to update the x-offset of the image to be saved.

--------------------------------------------------------------------*/

void    Get_Xoffset()
{
    char   *str;

    str = (char *) xv_get(xoffset_button, PANEL_VALUE);
    xoffset = atoi(str);

    if (str[0] == '\0' || str[0] == ' ' && xoffset == 0)
	/* not specified */
	xoffset = (MAX_WIDTH - width) >> 1;
    else if (xoffset < 0 || xoffset >= MAX_WIDTH) {
	xv_set(msg_button, PANEL_VALUE, "invalid x-offset value.", NULL);
	xoffset = 0;
    }
    Draw_Frame();
}


/*- Get_Yoffset -----------------------------------------------------

Function to update the y-offset of the image to be saved.

--------------------------------------------------------------------*/

void    Get_Yoffset()
{
    char   *str;

    str = (char *) xv_get(yoffset_button, PANEL_VALUE);
    yoffset = atoi(str);

    if (str[0] == '\0' || str[0] == ' ' && yoffset == 0)
	/* not specified */
	yoffset = (MAX_HEIGHT - height) >> 1;
    else if (yoffset < 0 || yoffset >= MAX_HEIGHT) {
	xv_set(msg_button, PANEL_VALUE, "invalid y-offset value.", NULL);
	yoffset = 0;
    }
    Draw_Frame();
}


/*- Get_Width -------------------------------------------------------

Function to update the width of the image to be saved.

--------------------------------------------------------------------*/

void    Get_Width()
{
    char   *str;

    str = (char *) xv_get(width_button, PANEL_VALUE);
    width = atoi(str);

    if (str[0] == '\0' || str[0] == ' ' && width == 0)
	width = MAX_WIDTH;
    else if (width <= 0 || width > MAX_WIDTH) {
	xv_set(msg_button, PANEL_VALUE, "invalid width value.", NULL);
	width = MAX_WIDTH;
    }
    Draw_Frame();
}


/*- Get_Height -------------------------------------------------------

Function to update the height of the image to be saved.

--------------------------------------------------------------------*/

void    Get_Height()
{
    char   *str;

    str = (char *) xv_get(height_button, PANEL_VALUE);
    height = atoi(str);

    if (str[0] == '\0' || str[0] == ' ' && height == 0)
	height = MAX_WIDTH;
    else if (height <= 0 || height > MAX_HEIGHT) {
	xv_set(msg_button, PANEL_VALUE, "invalid height value.", NULL);
	height = MAX_HEIGHT;
    }
    Draw_Frame();
}


/*- Quit_Notify -----------------------------------------------------

Function to quit the entire program.

--------------------------------------------------------------------*/

void    Quit_Proc(item, event)
Panel_item item;
Event  *event;
{
    /* destroy vfc and free the memory */
    vfc_destroy(vfc_dev);
    free(( char * ) yuv_ximage);

    /* now destroy window */
    xv_set(base_frame, FRAME_NO_CONFIRM, TRUE, 0);
    xv_destroy_safe(base_frame);
}


/*- Save_Notify -----------------------------------------------------

Function to get the file name for saving the image captured.

--------------------------------------------------------------------*/

void    Save_Notify(item, event)
Panel_item item;
Event  *event;
{
    char   *filename, *msg[BUF_LEN];
    u_char *grey_ptr;
    u_int  *rgb_ptr, *rgbj;
    IMAGE  *image;
    XImage *rgb_xim, *rgb8, *vfc_create_ximage();
    int     wid, hgt, w, h, offset;
    register int i, j;

    xv_set(msg_button, PANEL_VALUE, (char *) NULL, NULL);
    filename = (char *) xv_get(fname_button, PANEL_VALUE);

    if (pw_ximage == NULL) {
	xv_set(msg_button, PANEL_VALUE,
	       "image is NULL, save operation abort.", NULL);
	return;
    }

    if ((hgt = height) > (MAX_HEIGHT - yoffset))
	hgt = MAX_HEIGHT - yoffset;
    if ((wid = width) > (MAX_WIDTH - xoffset))
	wid = MAX_WIDTH - xoffset;

    /* allocate space for image */
    if (colorimage)
	image = ( IMAGE * ) Allocate_Image(xoffset, yoffset, hgt, wid, RGBTYPE);
    else
	image = ( IMAGE * ) Allocate_Image(xoffset, yoffset, hgt, wid, BYTETYPE);

    if (image == NULL) {
	xv_set(msg_button, PANEL_VALUE,
	       "Fail to Allocate_Image.", NULL);
	return;
    }

    /* copy yuv_ximage to the V.I.P. image allocated */
    if (colorimage) {
	w = vfc_format_width(v_format);
	h = vfc_format_height(v_format);

	if (!(rgb_xim = vfc_create_ximage(pw_display, w, h, 8 * sizeof(u_int)))) {
	    xv_set(msg_button, PANEL_VALUE, "Fail to malloc.", NULL);
	    return;
	}

	if (!(rgb8 = vfc_create_ximage(pw_display, w, h, 8))) {
	    xv_set(msg_button, PANEL_VALUE, "Fail to malloc.", NULL);
	    return;
	}
	vfc_yuv2rgb8_sq(yuv_ximage->data, rgb8->data, v_format);
	vfc_rgb8_to_rgb(rgb8, red_map, green_map, blue_map, rgb_xim, wm_offset);
	for (i = 0,
	     offset = w,
	     rgb_ptr = (u_int *) rgb_xim->data + yoffset * w;
	     i < hgt;
	     i++, rgb_ptr += offset)
	    for (j = 0, rgbj = rgb_ptr + xoffset;
		 j < wid;
		 j++, rgbj++) {
		grey_ptr = image->i.rgb[i][j];
		*grey_ptr++ = *(rgbj) & 0xff;	/* red component */
		*grey_ptr++ = (*(rgbj) >> 8) & 0xff;	/* green component */
		*grey_ptr = (*(rgbj) >> 16) & 0xff;	/* blue component */
	    }
    }
    else {
	for (i = 0,
	     grey_ptr = (u_char *) pw_ximage->data +
	     yoffset * pw_ximage->bytes_per_line;
	     i < hgt;
	     i++, grey_ptr += pw_ximage->bytes_per_line)
	    memcpy(image->i.c[i], grey_ptr + xoffset, wid);
    }
    if (!Write_Image(image, filename))
	xv_set(msg_button, PANEL_VALUE,
	       "Write_Image failed.", NULL);
    Free_Image(image);

    (void) sprintf(( char * ) msg, "Image from (%d,%d) to (%d,%d) saved.",
	    xoffset, yoffset, wid + xoffset, hgt + xoffset);
    xv_set(msg_button, PANEL_VALUE, msg, NULL);
}


/*- Start_Notify ----------------------------------------------------

Function to get the file name for saving the image captured.

--------------------------------------------------------------------*/

void    Start_Notify(item, event)
Panel_item item;
Event  *event;
{
    int     value, w, h;

/*
    static Notify_value Full_Screen(), Half_Screen();
    static Notify_value Qrtr_Screen(), Col_Preview();
    void    Load_Colormap(), Free_Colormap();
    XImage *vfc_create_ximage();
*/

    XClearArea(pw_display, drawable, 0, 0,
	       vfc_format_width(v_format),
	       vfc_format_height(v_format), False);

    if (pw_ximage)
	XDestroyImage(pw_ximage);
    /*
     * Get the preview mode selected
     */
    value = xv_get(preview_choice, PANEL_VALUE);

    /*
     * Create the appropriate X11 image. Install the timer to start preview
     * mode based on the value of preview_choice.
     */
    switch (value) {
    case 0:			/* Full screen grayscale */
	colorimage = 0;
	Free_Colormap();
	Load_Colormap(y_ramp, y_ramp, y_ramp,
		      VFC_GRAY_MAPLENGTH);
	/* Create the XImage memory */
	w = vfc_format_width(v_format);
	h = vfc_format_height(v_format);
	pw_ximage = vfc_create_ximage(pw_display, w, h, 8);
	state = PREVIEW_FULL;
	notify_set_itimer_func(base_frame, Full_Screen,
			       ITIMER_REAL, &preview_timer, 0);
	break;
    case 1:			/* Half Screen Grayscale */
	colorimage = 0;
	Free_Colormap();
	Load_Colormap(y_ramp, y_ramp, y_ramp,
		      VFC_GRAY_MAPLENGTH);
	/* Create the XImage memory */
	w = VFC_YUV_WIDTH / 2;
	h = vfc_format_height(v_format) / 2;
	pw_ximage = vfc_create_ximage(pw_display, w, h, 8);
	state = PREVIEW_QRTR;
	notify_set_itimer_func(base_frame, Half_Screen,
			       ITIMER_REAL, &preview_timer, 0);
	break;
    case 2:			/* Quarter Screen Grayscale */
	colorimage = 0;
	Free_Colormap();
	Load_Colormap(y_ramp, y_ramp, y_ramp,
		      VFC_GRAY_MAPLENGTH);
	/* Create the XImage memory */
	w = VFC_YUV_WIDTH / 4;
	h = vfc_format_height(v_format) / 4;
	pw_ximage = vfc_create_ximage(pw_display, w, h, 8);
	state = PREVIEW_HALF;
	notify_set_itimer_func(base_frame, Qrtr_Screen,
			       ITIMER_REAL, &preview_timer, 0);
	break;
    case 3:			/* Color Preview */
	colorimage = 1;
	Free_Colormap();
	Load_Colormap(red_map, green_map, blue_map,
		      VFC_CCUBE_MAPLENGTH);
	/* Create the XImage memory */
	h = vfc_format_height(v_format);
	w = vfc_format_width(v_format);
	pw_ximage = vfc_create_ximage(pw_display, w, h, 8);
	state = PREVIEW_COL;
	notify_set_itimer_func(base_frame, Col_Preview,
			       ITIMER_REAL, &preview_timer, 0);
	break;
    default:
	xv_set(msg_button, PANEL_VALUE, "No such option.", NULL);
	return;
    }
}


/*- Grab_Notify -----------------------------------------------------

Function for image grabbing.

--------------------------------------------------------------------*/

void    Grab_Notify(item, event)
Panel_item item;
Event  *event;
{
    if (state != PREVIEW_OFF)
	notify_set_itimer_func(base_frame, NOTIFY_FUNC_NULL,
			       ITIMER_REAL, NULL, NULL);

    switch (state) {
    case PREVIEW_OFF:
	/*
	 * Not in preview mode, do a grab then read the data from the
	 * hardware. Also recreate display image memory just to be sure.
	 */
	if (vfc_grab(vfc_dev) < 0) {
	    xv_set(msg_button, PANEL_VALUE, "Grab failed\n", NULL);
	    return;
	}
	vfc_yuvread_sq(vfc_dev, (u_short *) yuv_ximage->data,
		       v_format);
	if (pw_ximage)
	    XDestroyImage(pw_ximage);
	pw_ximage = vfc_create_ximage(pw_display,
				      vfc_format_width(v_format),
				      vfc_format_height(v_format),
				      8);
	break;
    case PREVIEW_FULL:
    case PREVIEW_COL:
	/*
	 * No need to resize display image. Just read the current image in
	 * the hardware. This gets the complete frame, not just the field as
	 * in Preview mode.
	 */
	vfc_yuvread_sq(vfc_dev, (u_short *) yuv_ximage->data,
		       v_format);
	state = PREVIEW_OFF;
	break;
    case PREVIEW_HALF:
    case PREVIEW_QRTR:
	/*
	 * Display image is the wrong size.  Destroy it and recreate it
	 * correctly. Then read the hardware.
	 */
	if (pw_ximage)
	    XDestroyImage(pw_ximage);
	pw_ximage = vfc_create_ximage(pw_display,
				      vfc_format_width(v_format),
				      vfc_format_height(v_format),
				      8);
	vfc_yuvread_sq(vfc_dev, (u_short *) yuv_ximage->data,
		       v_format);
	state = PREVIEW_OFF;
	break;
    default:
	xv_set(msg_button, PANEL_VALUE, "No such state\n", NULL);
	return;
    }
    /* yuv_display_image((int) xv_get(yuv_display_choice, PANEL_VALUE)); */
}


/*- Full_Screen -----------------------------------------------------

Display the image in full screen size (MAX_HEIGHTxMAX_WIDTH).

--------------------------------------------------------------------*/

static Notify_value Full_Screen()
{
    vfc_preview_sq(vfc_dev, (u_char *) pw_ximage->data,
		   wm_offset, v_format);
    XPutImage(pw_display, drawable, pw_gc, pw_ximage, 0, 0, 0, 0,
	      pw_ximage->width, pw_ximage->height);
    return (NOTIFY_DONE);
}


/*- Half_Screen -----------------------------------------------------

Display the image in half screen size (287x384).

--------------------------------------------------------------------*/

static Notify_value Half_Screen()
{
    vfc_preview_half_sq(vfc_dev, (u_char *) pw_ximage->data,
			wm_offset, v_format);
    XPutImage(pw_display, drawable, pw_gc, pw_ximage, 0, 0, 0, 0,
	      pw_ximage->width, pw_ximage->height);
    return (NOTIFY_DONE);
}


/*- Qrtr_Screen -----------------------------------------------------

Display the image in quarter screen size (143x192).

--------------------------------------------------------------------*/

static Notify_value Qrtr_Screen()
{
    vfc_preview_quarter_sq(vfc_dev, (u_char *) pw_ximage->data,
			   wm_offset, v_format);
    XPutImage(pw_display, drawable, pw_gc, pw_ximage, 0, 0, 0, 0,
	      pw_ximage->width, pw_ximage->height);
    return (NOTIFY_DONE);
}


/*- Col_Preview -----------------------------------------------------

Display full size (MAX_HEIGHTxMAX_WIDTH) live image in color mode.

--------------------------------------------------------------------*/

static Notify_value Col_Preview()
{
    vfc_cpreview_sq(vfc_dev, (u_char *) pw_ximage->data, v_format);
    XPutImage(pw_display, drawable, pw_gc, pw_ximage, 0, 0, 0, 0,
	      pw_ximage->width, pw_ximage->height);
    return (NOTIFY_DONE);
}


/*- Draw_Frame ------------------------------------------------------

Draw a rectangle that bounds the region of the image to be saved
to disk.

--------------------------------------------------------------------*/

void Draw_Frame()
{
    static int minx = -1, miny = -1, maxx = -1, maxy = -1;
    int     newminx, newminy, newmaxx, newmaxy;

    newminx = xoffset;
    newminy = yoffset;
    newmaxx = xoffset + width;
    newmaxy = yoffset + height;
    if (newmaxx >= MAX_WIDTH)
	newmaxx = MAX_WIDTH - 1;
    if (newmaxy >= MAX_HEIGHT)
	newmaxy = MAX_HEIGHT - 1;

    if (newminx != minx || newminy != miny ||
	newmaxx != maxx || newmaxy != maxy || force_draw) {
	XSetFunction(pw_display, pw_gc, GXinvert);
	XSetLineAttributes(pw_display, pw_gc, 2, LineSolid, CapButt, JoinBevel);
	/*
	 * XSetForeground(pw_display, pw_gc, BlackPixel(pw_display,
	 * pw_screen));
	 */

	/* redraw the old lines (being wiped off) */
	if (minx != -1 && !force_draw) {
	    XDrawLine(pw_display, drawable, pw_gc, minx, miny, minx, maxy);
	    XDrawLine(pw_display, drawable, pw_gc, minx, maxy, maxx, maxy);
	    XDrawLine(pw_display, drawable, pw_gc, maxx, maxy, maxx, miny);
	    XDrawLine(pw_display, drawable, pw_gc, maxx, miny, minx, miny);
	}
	XDrawLine(pw_display, drawable, pw_gc, newminx, newminy, newminx, newmaxy);
	XDrawLine(pw_display, drawable, pw_gc, newminx, newmaxy, newmaxx, newmaxy);
	XDrawLine(pw_display, drawable, pw_gc, newmaxx, newmaxy, newmaxx, newminy);
	XDrawLine(pw_display, drawable, pw_gc, newmaxx, newminy, newminx, newminy);
	XSetFunction(pw_display, pw_gc, GXcopy);

	minx = newminx;
	miny = newminy;
	maxx = newmaxx;
	maxy = newmaxy;
    }
}


/*- vfc_create_ximage -----------------------------------------------

Create an X11 image given a desired
width, height and depth.

--------------------------------------------------------------------*/

XImage *vfc_create_ximage(display, w, h, depth)
Display *display;
int     w, h, depth;
{
    register int i;
    register long howmuch;
    register u_char *imdata;
    Visual *visual;
    XImage *r_ximage;

    visual = DefaultVisual(display, DefaultScreen(display));
    if (depth == 1)
	r_ximage = XCreateImage(display, visual, 1, XYBitmap,
				0, 0, w, h, 16, 0);
    else
	r_ximage = XCreateImage(display, visual, depth, ZPixmap,
				0, 0, w, h, 16, 0);
    if (r_ximage == NULL) {
	xv_set(msg_button, PANEL_VALUE, "XCreateImage failed.", NULL);
	return (NULL);
    }

    howmuch = r_ximage->height * r_ximage->bytes_per_line;
    r_ximage->data = (char *) malloc(howmuch);
    if (r_ximage->data == NULL) {
	perror("malloc");
	return (NULL);
    }

    /* Clear the memory out */
    imdata = (u_char *) r_ximage->data;
    for (i = 0; i < howmuch; i++)
	*imdata++ = 0;
    return (r_ximage);
}


/*- vfc_load_private_cmap -------------------------------------------

Load private color map.

--------------------------------------------------------------------*/

void    vfc_load_private_cmap(display, cmap, r, g, b, map_length, offset)
Display *display;
Colormap cmap;
u_char *r, *g, *b;
int     map_length, offset;
{
    register int i;
    u_long  pixels[256], plane_masks[256];
    XColor  colors[256];

    /* Allocate 256 cells again */
    XAllocColorCells(display,
		     cmap, True, plane_masks, 0, pixels, 256);

    /* Now set the values of the colour array */
    for (i = 0; i < map_length; i++) {
	colors[i].pixel = i + offset;
	colors[i].red = r[i] << 8;
	colors[i].green = g[i] << 8;
	colors[i].blue = b[i] << 8;
	colors[i].flags = DoRed | DoGreen | DoBlue;
    }

    /* Free the ones we don't need */
    if (offset) {
	/* Free window manager colors */
	for (i = 0; i < offset; i++)
	    pixels[i] = i;
	XFreeColors(display, cmap, pixels, offset, XV_NULL);

	/* Free colors at the top of the map */
	for (i = (offset + map_length); i < 256; i++)
	    pixels[i - (offset + map_length)] = i;
	XFreeColors(display, 
		       cmap, 
		       pixels,
		       ( int ) 256 - (offset + map_length),
 		       XV_NULL);
    }

    /* Store it in the colour map */
    XStoreColors(display, cmap, colors, map_length);

    /* Now install it in the h/ware */
    XInstallColormap(display, cmap);
    return;
}


/*- vfc_load_default_cmap -------------------------------------------

Try to allocate read/write cells from the default color map.
This is the preferred choice as the window manager will install
the map for us. Need to return the value of the offset so
that we can compensate for it elsewhere.

--------------------------------------------------------------------*/

int     vfc_load_default_cmap(display, cmap, r, g, b, map_length, offset)
Display *display;
Colormap cmap;
u_char *r, *g, *b;
int     map_length, *offset;
{
    Status  rc;
    register int i;
    u_long  pixels[256], plane_masks[256];
    XColor  colors[256];

    rc = XAllocColorCells(display,
			  cmap, True, plane_masks, 0, pixels, map_length);

    if (!rc) {
	/* XAllocColorCells failed, return a failure */
	return (-1);
    }

    if ((pixels[0] + map_length - 1) != pixels[map_length - 1]) {
	/*
	 * The map returned wasn't contigous. Free the colors returned in
	 * pixels. Return a failure.
	 */
	XFreeColors(display, cmap, ( unsigned long  * ) pixels, 
		    ( int ) map_length, XV_NULL);
	return (-1);
    }

    /* Now set the values of the colour array */
    for (i = 0; i < map_length; i++) {
	colors[i].pixel = i + pixels[0];
	colors[i].red = r[i] << 8;
	colors[i].green = g[i] << 8;
	colors[i].blue = b[i] << 8;
	colors[i].flags = DoRed | DoGreen | DoBlue;
    }

    /* Return the value of offset to the user */
    *offset = pixels[0];

    /* Store it in the colour map */
    XStoreColors(display, cmap, colors, map_length);
    return (0);
}


/*- Load_Colormap ---------------------------------------------------

Load the desired colormap arrays into the h/ware.

--------------------------------------------------------------------*/

void    Load_Colormap(r, g, b, length)
u_char *r, *g, *b;
int     length;
{
    current_cmap = pw_def_cmap;
    pw_map_length = length;

    if (vfc_load_default_cmap(pw_display, pw_def_cmap,
			      r, g, b, length, &wm_offset) < 0) {

	/*
	 * Can't use the default map, someone else is hogging it use our own
	 * private map.
	 */
	vfc_load_private_cmap(pw_display, pw_priv_cmap,
			      r, g, b, length, WM_OFFSET);
	wm_offset = WM_OFFSET;
	current_cmap = pw_priv_cmap;
    }

    /*
     * Reset the color dither tables
     */
    vfc_init_lut(wm_offset);
    return;
}


/*- Free_Colormap ---------------------------------------------------

Free all the colors we were using in the current color map.

--------------------------------------------------------------------*/

void    Free_Colormap()
{
    register int i;
    long    pixels[256];

    for (i = wm_offset; i < (pw_map_length + wm_offset); i++)
	pixels[i - wm_offset] = i;
    XFreeColors(pw_display, current_cmap, ( unsigned long * ) pixels, pw_map_length, XV_NULL);
    return;
}


/*- Main ------------------------------------------------------------

Main body of the program.

--------------------------------------------------------------------*/

main(argc, argv)
int     argc;
char  **argv;
{
    int     Setup_Hware();
    void    create_windows();

    Create_Windows(argc, argv);

    /* set external variables appropriately for error messages handling */
    VIP_STD_ERR = NULL;
    VIP_LOG_FILE = NULL;
    VIP_LOG = 0;
    VIP_VERBOSE = 0;
    VIP_DISPLAY = 0;

    /*
     * Initialize the hardware
     */
    if (Setup_Hware() < 0)
	exit(1);

    /*
     * Get the colormap tables
     */
    if (vfc_get_colormaps(y_ramp, red_map, green_map, blue_map) < 0) {
	xv_set(msg_button, PANEL_VALUE,
	       "Look up table initialization failed.", NULL);
	vfc_destroy(vfc_dev);
	exit(1);
    }

    /*
     * Set the interval timer values
     */
    preview_timer.it_interval.tv_usec = 50;
    preview_timer.it_interval.tv_sec = 0;
    preview_timer.it_value.tv_usec = 50;
    preview_timer.it_value.tv_sec = 0;


    /*
     * Create the memory for the YUV data
     */
    yuv_ximage = vfc_create_ximage(pw_display, ( int ) VFC_YUV_WIDTH,
				    vfc_format_height(v_format), 16);
    if (yuv_ximage == NULL) {
	xv_set(msg_button, PANEL_VALUE, "Not enough memory available\n", NULL);
	vfc_destroy(vfc_dev);
	exit(1);
    }

    /*
     * Hand control over to the notifier
     */
    xv_main_loop(base_frame);
    exit(0);
}
