/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            VIP
* MODULE:		laser.c - Driver for Macintosh Laser Printer.
*		       		  Co-ordinate system used in the Postscript
*				  Language is: origin at the bottom left corner
*				  of the paper; x axis points tothe right
*				  direction, and y axis points straight up.
*				  The user can specify window in Cm, rather 
*				  than the unit used in	Postscript language.
* REVISION:             3.4
* AUTHOR:               CA/DH
* CREATION DATE:        27 Feb 1991
* REVISION DATE:	4/29/94        
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:		3.4
* REVISION DATE:	29 April 1994
* COMMENT:		Fixed ftime and localtime!
*                       Added conversion for non BYTE images
* BY:			PK
*
* REVISION:		3.3
* REVISION DATE:	24 March 1994
* COMMENT:		Fixed localtime
* BY:			CFF
*
* REVISION:		3.2
* REVISION DATE:	13 Sept 1993
* COMMENT:		Fixed GENERIC build
* BY:			CFF
*
* REVISION:		
* REVISION DATE:	12 Dec 1991
* COMMENT:
* BY:			DH
*
* REVISION:		
* REVISION DATE:	30 Oct 1991
* COMMENT:		NEWVIP
* BY:			PK
*
* REVISION:		
* REVISION DATE:	18 April 1991
* COMMENT:		NEWVIP
* BY:			DH
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)laser.c	3.4 4/29/94";
#endif


#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "vip.h"

#ifdef U_NIX
#include <pwd.h>
#include <sys/types.h>
#include <sys/timeb.h>
#endif

#include <time.h>

#ifdef DOS
#include <fcntl.h>
#include <sys\stat.h>
#endif


#define TOP_LEFT 1
#define TOP_RIGHT 2
#define BOT_LEFT 3
#define BOT_RIGHT 4


#define XMAX 2308.0		/* maximum X value on an A4 page */
#define YMAX 3180.0		/* maximum Y value on an A4 page */
#define OFFX 95.0		/* inserted by Du */
#define OFFY 118.0		/* inserted by Du */
#define DotsPerCM 28.346457	/* 72 dots/inch, 2.54cm/inch */

static int copy = 1;
int     image_row, image_col = 0, raw = 0;
FILE   *ifp, *fopen();
IMAGE  *imp;
float   dx, dy;			/* some offset to avoid the image being
				 * clipped off from the page */
float   width = 16.0, height = 24.0;	/* default width and height (in cm)
					 * of the image on the page */

int     runleng = 0;		/* perform runlength encoding */
int     noshowpage = 0;		/* no show page (for embedding the image into
				 * documents such as Latex). */
unsigned char imval;



void            Usage();
void            Pperror( char * );
unsigned char   Next_Pixel();
void            Prologue( char *, int, int );
void            Epilogue( char *, int );
void            PutHexPix( int );
void            PutRunLeng( int, int );



/*- Usage -----------------------------------------------------------

Print message and exit program.

--------------------------------------------------------------------*/

void Usage()
{
    (void) fprintf(stderr, "usage: laser [file] [-w width height] [-r nr nc] ");
    (void) fprintf(stderr, "[-c n] [-n|-p position] [-e] [-l1|-l2|-l3|-l4 label]\n");
    (void) fprintf(stderr, "       -w width and height of image in Cm\n");
    (void) fprintf(stderr, "       -r image dimension (if image is raw)\n");
    (void) fprintf(stderr, "       -c number of copies to print\n");
    (void) fprintf(stderr, "       -n no show page.  Use this flag if the\n");
    (void) fprintf(stderr, "          image is part of a large document\n");
    (void) fprintf(stderr, "       -p can be ll, lr, tl, tr, ce\n");
    (void) fprintf(stderr, "       -e perform runlength encoding (useful for synthetic images)\n");
    (void) fprintf(stderr, "       -l1,-l2,-l3,-l4 a string of characters to be put at the\n");
    (void) fprintf(stderr, "          top-left, top-right, bottom-left, or bottom-right corner of the picture\n");
    (void) fprintf(stderr, "default: laser [stdin] [-w 16.0 24.0] [-c 1] [-p ul]\n");
    exit(1);
}


/*- Pperror ---------------------------------------------------------

Print message and exit program.

--------------------------------------------------------------------*/

void Pperror(m)
char   *m;
{
    (void) fprintf(stderr, "laser: %s\n", m);
    exit(1);
}


/*- Next_Pixel ------------------------------------------------------

Process the next pixel of the image.

--------------------------------------------------------------------*/

unsigned char Next_Pixel()
{
    unsigned char pix;

    if (raw)
	return ((unsigned char) getc(ifp));
    pix = imp->i.c[image_row][image_col];
    if (image_col == imp->cols - 1) {
	image_col = 0;
	image_row--;
    }
    else
	image_col++;
    return (pix);
}



/*- Prologue --------------------------------------------------------

--------------------------------------------------------------------*/

void Prologue(fname, nr, nc)
char   *fname;
int     nr, nc;
{
    struct tm *tp;
    struct timeb timec;
    char   *timez, host[80];
    struct passwd *pw;
/*
This is commented out as it is not very portable 
    ftime(&timec);
    tp = localtime((time_t *)&timec);
    timez = asctime(tp);
*/
    (void) printf("%%!\n");
    (void) printf("%%%%File name: %s\n", (fname ? fname : "stdin"));

#ifdef U_NIX
    pw = getpwuid(getuid());
    if (gethostname(host, 80) == -1)
	(void) sprintf(host, "");
    (void) printf("%%%%Creator: %s@%s\n", pw->pw_name, host);
#endif

/*    (void) printf("%%%%CreationDate: %s", timez); */
    (void) printf("%%%%Pages: 1\n");
    (void) printf("%%%%BoundingBox: 0 0 %1.2f %1.2f\n",
	   width * DotsPerCM, height * DotsPerCM);
    (void) printf("%%%%EndComments\n");
    (void) printf("%% Prolog for VIP to PostScript converter\n");
    (void) printf("%% Author: Du Huynh\n");
    (void) printf("save 50 dict begin /VIPlaser exch def\n");
    (void) printf("%%%%EndProlog\n");

    (void) printf("0.24 0.24 scale\n");

    if (runleng) {
	(void) printf("/code 2 string def\n");
	(void) printf("/decode {\n");
	(void) printf("    /imline exch 1 get string def\n");
	(void) printf("    0 code 1 get\n");
	(void) printf("    { dup imline exch code 0 get put 1 add } repeat pop\n");
	(void) printf("    imline\n");
	(void) printf("} def\n");
    }
    else if (nr % 2)
	(void) printf("/imline %d string def\n", nc);
    else
	(void) printf("/imline %d string def\n", nc * 2);

    (void) printf("/drawimage {\n");
    (void) printf("    %d %d 8\n", nc, nr);
    (void) printf("    [%d 0 0 %d 0 %d]\n", nc, nr, nr);
    if (runleng)
	(void) printf("    { currentfile code readhexstring pop decode } image\n");
    else
	(void) printf("    { currentfile imline readhexstring pop } image\n");
    (void) printf("} def\n");

    (void) printf("/DotsPerCM %f def\n", DotsPerCM);
    (void) printf("/Cm {DotsPerCM mul} def\n");
    (void) printf("/WidthInCM {%f} def\n", width);
    (void) printf("/HeightInCM {%f} def\n", height);
    (void) printf("/WScale {WidthInCM Cm 0.24 div} def\n");
    (void) printf("/HScale {HeightInCM Cm 0.24 div} def\n");
    if (noshowpage)
	(void) printf("0 HScale translate\n");
    else
	(void) printf("%f %f translate\n", dx + OFFX, dy + OFFY);
    (void) printf("WScale HScale scale\n");

    (void) printf("drawimage\n");
}


/*- Epilogue --------------------------------------------------------

--------------------------------------------------------------------*/

void Epilogue(label, label_pos)
char   *label;
int     label_pos;
{
    int     gap = 3;		/* the gap (in postscript units) between the
				 * label (if specified) and the image
				 * boundary */

    if (label) {
	/* turn the scaling factor back to 1 */
	(void) printf("1.0 WidthInCM Cm div 1.0 HeightInCM Cm div scale\n");
	(void) printf("/Times-Bold findfont 10 scalefont setfont\n");
	(void) printf("/w (%s) stringwidth pop 10 add def\n", label);
	switch (label_pos) {
	case TOP_LEFT:
	    (void) printf("/x %d def ", gap);
	    (void) printf("/y 10 %d add neg def\n", gap);
	    break;
	case TOP_RIGHT:
	    (void) printf("/x WidthInCM Cm w %d add sub def ", gap);
	    (void) printf("/y 10 %d add neg def\n", gap);
	    break;
	case BOT_LEFT:
	    (void) printf("/x %d def ", gap);
	    (void) printf("/y HeightInCM Cm neg %d add def\n", gap);
	    break;
	case BOT_RIGHT:;
	default:
	    (void) printf("/x WidthInCM Cm w %d add sub def ", gap);
	    (void) printf("/y HeightInCM Cm neg %d add def\n", gap);
	}
	(void) printf("newpath\n");
	(void) printf("  x y moveto\n");
	(void) printf("  w 0 rlineto\n");
	(void) printf("  0 10 rlineto\n");
	(void) printf("  w neg 0 rlineto\n");
	(void) printf("closepath\n");
	(void) printf("1 setgray fill\n");

	(void) printf("x 3 add y 3 add moveto\n");
	(void) printf("0 setgray (%s) show\n", label);
    }
    if (copy > 1)
	(void) printf("%d {copypage} repeat\n", copy - 1);

    (void) printf("%%%%Trailer\n");
    if (!noshowpage)
	(void) printf("showpage\n");
    (void) printf("%% trailer for vip to Postscript converter\n");
    (void) printf("VIPlaser end restore\n");
}


/*- PutHexPix -------------------------------------------------------

Output the pixel value at the current position in hexadecimal format
to the standard output.

--------------------------------------------------------------------*/

void PutHexPix(p)
unsigned char p;
{
    static int npixo = 0;

    (void) printf("%02X", p);
    npixo += 1;
    if (npixo >= 32) {
	(void) printf("\n");
	npixo = 0;
    }
}


/*- PutRunLeng ------------------------------------------------------

Put runlength encoding for row r of the image.

--------------------------------------------------------------------*/

void PutRunLeng(r, end)
int     r, end;
{
#define writeln() if (++lineleng >= 16) { \
                      (void) printf("\n"); \
                      lineleng = 0; \
                      }
    register int i;
    static int cnt = 0;
    static int lineleng = 0;

    if (end) {			/* finishing off */
	(void) printf("%02X%02X", imval, cnt);
	return;
    }
    for (i = 0; i < imp->cols; i++) {
	if (imp->i.c[r][i] == imval) {
	    if (cnt == 0xff) {
		(void) printf("%02X%02X", imval, cnt);
		cnt = 0;
		writeln()
	    }
	    cnt++;
	}
	else {
	    (void) printf("%02X%02X", imval, cnt);
	    writeln()
		imval = imp->i.c[r][i];
	    cnt = 1;
	}
    }
}


/*- Main ------------------------------------------------------------

Main body of the program.

--------------------------------------------------------------------*/

main(argc, argv)
int     argc;
char   *argv[];
{
    int     r, c, nr, nc, argn;
    float   wdots, hdots;
    unsigned char pix;
    char   *image_arg = NULL, *shift = NULL, *label = NULL;
    int     label_pos;		/* position of the label */
    IMAGE *timp;

    for (argn = 1; argn < argc; argn++) {
	if (argv[argn][0] == '-') {
	    switch (argv[argn][1]) {
	    case 'n':
		noshowpage = 1;
		break;
	    case 'w':
		if (++argn < argc)
		    width = atof(argv[argn]);
		else
		    Usage();
		if (++argn < argc)
		    height = atof(argv[argn]);
		else
		    Usage();
		break;
	    case 'r':
		raw = 1;
		if (++argn < argc)
		    nr = atoi(argv[argn]);
		else
		    Usage();
		if (++argn < argc)
		    nc = atoi(argv[argn]);
		else
		    Usage();
		break;
	    case 'c':
		if (++argn < argc)
		    copy = atoi(argv[argn]);
		else
		    Usage();
		break;
	    case 'p':
		if (++argn < argc)
		    shift = argv[argn];
		else
		    Usage();
		break;
	    case 'e':
		runleng = 1;	/* runlength encoding */
		break;
	    case 'l':
		if (!strcmp(argv[argn], "-l1"))
		    /* put label at top left corner */
		    label_pos = TOP_LEFT;
		else if (!strcmp(argv[argn], "-l2"))
		    /* put label at top right corner */
		    label_pos = TOP_RIGHT;
		else if (!strcmp(argv[argn], "-l3"))
		    /* put label at bottom left corner */
		    label_pos = BOT_LEFT;
		else if (!strcmp(argv[argn], "-l4"))
		    /* put label at bottom right corner */
		    label_pos = BOT_RIGHT;
		else
		    Usage();
		if (++argn < argc)
		    label = argv[argn];
		else
		    Usage();
		break;
	    default:
		Usage();
	    }
	}
	else if (image_arg)
	    Usage();
	else
	    image_arg = argv[argn];
    }

    if (noshowpage && shift)
	Pperror("The -p and -n flag should not be used together");
    wdots = width * DotsPerCM / 0.24;
    hdots = height * DotsPerCM / 0.24;
    /*
     * check image position specification.  If not given, the 4 numbers
     * follow the -w flag are used.
     */
    if (shift) {
	if (!strcmp(shift, "ce")) {
	    /* want the image in the centre of the page */
	    dx = (XMAX - wdots) / 2.0;
	    dy = (YMAX + hdots) / 2.0;
	}
	else if (!strcmp(shift, "ll")) {
	    /* image at lower left corner of page */
	    dx = 0.0;
	    dy = hdots;
	}
	else if (!strcmp(shift, "lr")) {
	    /* image at lower right corner of page */
	    dx = XMAX - wdots;
	    dy = hdots;
	}
	else if (!strcmp(shift, "ul")) {
	    /* image at upper left corner of page */
	    dx = 0.0;
	    dy = YMAX;
	}
	else if (!strcmp(shift, "ur")) {
	    /* image at upper right corner of page */
	    dx = XMAX - wdots;
	    dy = YMAX;
	}
	else
	    Usage();
    }
    else {
	dx = 0.0;
	dy = YMAX;
    }


    if (!raw) {
	if (!(imp = ( IMAGE * ) Read_Image(image_arg))) {
	    Pperror("vision system error in read_image");
            exit(0);
	  }

        if(imp->type != BYTETYPE) {  /* need to convert to BYTETYPE */
           timp = imp;
           imp = Convert2Byte_Image(timp);
           Free_Image(timp);
	}
	nr = imp->rows;
	nc = imp->cols;
	image_row = imp->rows - 1;
    }
    else {
	if (!image_arg) {
	    ifp = stdin;

#ifdef DOS
	    setmode(fileno(ifp), O_BINARY);
#endif
	}
	else {
	    ifp = fopen(image_arg, "rb");
	    if (!ifp) {
		Pperror("error cannot open image file");
                exit(0);
	    }
	}
    }


    if (image_arg)
	Prologue(image_arg, nr, nc);
    else
	Prologue(label, nr, nc);

    imval = imp->i.c[nr - 1][0];
    for (r = nr - 1; r >= 0; r--) {
	if (runleng)
	    PutRunLeng(r, 0);
	else
	    for (c = nc - 1; c >= 0; c--) {
		pix = Next_Pixel();
		PutHexPix(pix);
	    }
    }

    if (runleng)
	PutRunLeng(0, 1);
    (void) printf("\n");
    Epilogue(label, label_pos);

    if (raw)
	(void) fclose(ifp);

    exit(0);
}
