/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            VIP
* MODULE:		d2gaussian.c - Produces a D2G (second derivative of a
*			        Gaussian) mask
* REVISION:             3.1
* AUTHOR:               CA
* CREATION DATE:        1 July 1986
* REVISION DATE:	3/14/94        
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:		3.1
* REVISION DATE:	14 MArch 1994
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* REVISION:		
* REVISION DATE:	03 Feb 1992
* COMMENT:		NEWVIP
* BY:			DH
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)d2gaussian.c	3.1 3/14/94";
#endif


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "vip.h"

#define TWO_ROOT2   2.828427124
#ifndef PI
#define PI          3.141592654
#endif
#define SQRT_2PI    2.506628274

void            Usage();
void            Pperror( char * );
static          Normalise_Vector( double *, double * );
static  IMAGE  *D2G_1D_Image( int );
static  IMAGE  *D2G_2D_Image( int, double, double );



/*- Usage -----------------------------------------------------------

Print error message and exit the program.

--------------------------------------------------------------------*/

void Usage()
{
    (void) fprintf(stderr, "usage: d2gaussian omega [-m Vx Vy] [-o out_file]\n");
    exit(1);
}


/*- Pperror ---------------------------------------------------------

Print error message and exit the program.

--------------------------------------------------------------------*/

void pperror(e)
char   *e;
{
    (void) fprintf(stderr, "d2gaussian: %s\n", e);
    exit(1);
}


/*- Noramlize_Vector ------------------------------------------------

Print error message and exit the program.

--------------------------------------------------------------------*/

static  Normalise_Vector(xvec, yvec)
double *xvec, *yvec;
{
    double  factor;

    if ((*xvec == 0) || (*yvec == 0))
	pperror("vision system error zero vector");

    factor = sqrt((*xvec * *xvec) + (*yvec * *yvec));
    *xvec /= factor;
    *yvec /= factor;
}


/*- D2G_1D_Image ----------------------------------------------------

Produce a one-dimensional second derivative of a Gaussian image.

--------------------------------------------------------------------*/

static IMAGE  *D2G_1D_Image(omega)
int     omega;
{
    register int x;
    double  sigma, sigma2, C, K;
    int     rad;
    IMAGE  *f;

    rad = 1.8 * omega;
    if (rad < 1)
	return (NULL);

    sigma = omega / 2.0;
    sigma2 = sigma * sigma;
    if (!(f = ( IMAGE * ) Allocate_Image(0, 0, 1, 2 * rad + 1, FLOATTYPE))) {
	VIP_Error_Msg("D2G_1D_Image: out of memory");
	return (NULL);
    }

    C = 1 / (sigma2 * sigma2 * sigma / SQRT_2PI);
    for (x = -rad; x <= rad; x++) {
	K = x * x / (2 * sigma2);
	f->i.f[0][x + rad] = (sigma2 - x * x) * C * exp(-K);
    }

    return (f);
}


/*- D2G_2D_Image ----------------------------------------------------

Produce a two-dimensional second derivative of a Gaussian image.

--------------------------------------------------------------------*/

static IMAGE  *D2G_2D_Image(omega, xvec, yvec)
int     omega;
double  xvec, yvec;
{
    register int y, x;
    double  sigma, sigma2, C, K, y2;
    int     rad;
    IMAGE  *f;

    rad = 1.8 * omega;
    if (rad < 1)
	return (NULL);

    sigma = omega / TWO_ROOT2;
    sigma2 = sigma * sigma;
    if (!(f = ( IMAGE * ) Allocate_Image(0, 0, 2 * rad + 1, 2 * rad + 1, FLOATTYPE))) {
	VIP_Error_Msg("D2D_2D_Image: out of memory");
	return (NULL);
    }

    C = 1 / (2 * PI * sigma2 * sigma2 * sigma2);
    for (y = -rad; y <= rad; y++) {
	y2 = y * y;
	for (x = -rad; x <= rad; x++) {
	    K = (x * x + y2) / (2 * sigma2);
	    f->i.f[rad - y][x + rad] = C * exp(-K) * ((sigma2 - x * x) * xvec +
					      (sigma2 - y2) * yvec);
	}
    }

    return (f);
}


/*- Main ------------------------------------------------------------

Main body of the program.

--------------------------------------------------------------------*/

main(argc, argv)
int     argc;
char   *argv[];
{
    int     argn;
    char   *mask_arg = NULL;
    int     two_dim_arg = 0;
    int     omega_arg = 0;
    double  xvec, yvec;
    IMAGE  *f;

    for (argn = 1; argn < argc; argn++)
	if (argv[argn][0] == '-')
	    switch (argv[argn][1]) {
	    case 'm':
		two_dim_arg = argn;
		if (++argn < argc)
		    xvec = atof(argv[argn]);
		else
		    Usage();
		if (++argn < argc)
		    yvec = atof(argv[argn]);
		else
		    Usage();
		break;
	    case 'o':
		if (++argn < argc)
		    mask_arg = argv[argn];
		else
		    Usage();
		break;
	    default:
		Usage();
	    }
	else if (!omega_arg)
	    omega_arg = atoi(argv[argn]);
	else
	    Usage();


    if (!omega_arg)
	Usage();

    if (two_dim_arg) {
	if (!xvec || !yvec)
	    pperror("error zero direction vector");
	Normalise_Vector(&xvec, &yvec);
	f = ( IMAGE * ) D2G_2D_Image(omega_arg, xvec, yvec);
    }
    else
	f = ( IMAGE * ) D2G_1D_Image(omega_arg);

    if (f)
	Write_Image(f, mask_arg);

    exit(0);
}











