/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		%R%
* SUBSYSTEM:            LIB
* MODULE:		vipio.c - Library of service functions for image
*                                 allocation, reading and writing.
*
* REVISION:             %I%
* AUTHOR:               PK
* CREATION DATE:        September 1986
* REVISION DATE:	%G%
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:             3.19
* REVISION DATE:        16 May 1994
* COMMENT:              Allocate_Image changed to allocate continuous section
*                       of memory for non-DOS compilations.  This seems to fix
*                       problems with OSF/1 version 2 (don't know why)
*                       Free_Image changed accordingly.
*                       New function Copy_Image added.
*                       Copy_Header changed to use bcopy rather than memcpy.
* BY:                   PK
*
* REVISION:             3.18
* REVISION DATE:        7 April 1994
* COMMENT:              Testing SCCS - nothing changed
* BY:                   BJR
*
* REVISION:             3.17
* REVISION DATE:        5 April 1994
* COMMENT:              Added Construct_Complex_Image, Extract_Real, Extract_Imaginary
* BY:                   PK
*
* REVISION:             3.16
* REVISION DATE:        29 March 1994
* COMMENT:              Fixed bug in Conver2Byte_Image
* BY:                   PK
*
* REVISION:		3.15
* REVISION DATE:	03 March 1994
* COMMENT:		Fixed for __vip_h build
* BY:			CFF
*
* REVISION:		3.14
* REVISION DATE:	25 Feb 1994
* COMMENT:		Added new fn : Image_OK
* BY:			CFF
*
* REVISION:		3.13
* REVISION DATE:	24 Feb 1994
* COMMENT:		Added scaling to Convert2Byte and changed Free_Image
* BY:			CFF
*
* REVISION:		3.12
* REVISION DATE:	02 Feb 1994
* COMMENT:		Fixed bug in Convert2Complex
* BY:			CFF
*
* REVISION:		3.11
* REVISION DATE:	18 Nov 1993
* COMMENT:		Fixed bug in Write_Image
* BY:			CFF
*
* REVISION:		3.10
* REVISION DATE:	15 Nov 1993
* COMMENT:		Fixed bug Float2Byte_Image
* BY:			CFF
*
* REVISION:		3.9
* REVISION DATE:	08 Nov 1993
* COMMENT:		Fixed bug in Write_Image
* BY:			CFF
*
* REVISION:		3.8
* REVISION DATE:	29 Sept 1993
* COMMENT:		Fixed up Pixel_In_Image
* BY:			CFF
*
* REVISION:		3.7
* REVISION DATE:	17 Sept 1993
* COMMENT:		Fixed up GENERIC build
* BY:			CFF
*
* REVISION:		3.6
* REVISION DATE:	13 Sept 1993
* COMMENT:		Fixed up DEC build
* BY:			CFF
*
* REVISION:		3.5
* REVISION DATE:	10 July 1993
* COMMENT:		Fixed up DOS build
* BY:			CFF
*
* REVISION:		3.4
* REVISION DATE:	28 June 1993
* COMMENT:		Second set of enhancements -
*			    added new fns :
*			        Complex2MagFloat_Image : makes MAGNITUDE type images
*				Convert2Byte_Image
*				Convert2Float_Image
*				Convert2Complex_Image
*				Convert2Double_Image
* BY:			CFF
*
* REVISION:		3.3
* REVISION DATE:	03 June 1993
* COMMENT:		Second set of enhancements -
*			    COMPLEX image handling - added new fns :
*			        Complex2PhaseFloat_Image : makes PHASE type images
*			        Complex2Float_Image : makes MAGNITUDE type images
* BY:			CFF
*
* REVISION:		3.2
* REVISION DATE:	04 August 1992
* COMMENT:		Separated fprint of rows&cols from maxcolour in
*		        VIP2PPM_Image; lines 1203 to 1204.
* BY:			CFF
*
* REVISION:		3.1
* REVISION DATE:	9 July 1992
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* REVISION:
* REVISION DATE:	05 Feb 1992
* COMMENT:		(insert Allocate_Header)
* BY:			DH
*
* REVISION:
* REVISION DATE:	16 Oct 1991
* COMMENT:
* BY:			DH
*
*******************************************************************************/



#ifdef U_NIX

#ifndef lint
static char *sccs_id = "%W% %G%";

#endif

#endif




#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <memory.h>
#include <sys/types.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <malloc.h>
#include <misc.h>


#include "vip.h"
#include "vipiofn.h"


#ifdef U_NIX			/* UNIX is a resvd gcc macro */
#include "oldvip.h"
#endif

/*-  VIP_Error_Msg  -------------------------------------------------

Function to print error message and exit from program.

--------------------------------------------------------------------*/

void    VIP_Error_Msg(message)
char   *message;
{
    if (VIP_STD_ERR)
	(void) fprintf(VIP_STD_ERR, "%s\n", message);
    if (VIP_LOG_FILE)
	(void) fprintf(VIP_LOG_FILE, "%s\n", message);
    (void) strncat(VIP_MESSAGE, message, VIP_MESSAGE_LENGTH - strlen(VIP_MESSAGE));
    (void) strncat(VIP_MESSAGE, "\n", VIP_MESSAGE_LENGTH - strlen(VIP_MESSAGE));
}


#ifndef DOS
/*-  Allocate_Image  -------------------------------------------------

Function to allocate memory for an image.

Non-DOS version - Memory is allocated as a continuous segment.

--------------------------------------------------------------------*/

IMAGE  *Allocate_Image(umin, vmin, rows, cols, type)
int32   umin, vmin, rows, cols, type;
{
    IMAGE  *image;

    image = (IMAGE *) calloc(1, sizeof(IMAGE));

    if (!image) {
	VIP_Error_Msg("Allocate_Image:	unable to allocate space for image");
	return (NULL);
    }

    if (rows < 0 || cols < 0) {
	VIP_Error_Msg("Allocate_Image:	-ve No of rows or columns requested");
	return (NULL);
    }

    image->idcode = IMAGE_ID;
    image->version[0] = VIP_MAJOR_VERSION;
    image->version[1] = VIP_MINOR_VERSION;

    image->umin = umin;
    image->vmin = vmin;
    image->type = type;
    image->rows = rows;
    image->cols = cols;
    image->resolution = 1;	/* assume full resolution */
    image->info[0] = '\0';
    image->pntr[0] = (void *) NULL;
    image->pntr[1] = (void *) NULL;

    rows = rows + 1;		/* Allocate an extra row on each image for us
				 * to use as a memory sentinel */

    switch (type) {

    case BYTETYPE:
	image->i.c = (c_array *) Allocate_Matrix((int) rows, (int) cols, sizeof(unsigned char));
	if (!image->i.c)
	    VIP_Error_Msg("AllocateImage unable to allocate space");
	break;

    case SHORTTYPE:
	image->i.s = (s_array *) Allocate_Matrix((int) rows, (int) cols, sizeof(short));
	if (!image->i.s)
	    VIP_Error_Msg("AllocateImage unable to allocate space");
	break;

    case LONGTYPE:
	image->i.l = (l_array *) Allocate_Matrix((int) rows, (int) cols, sizeof(long));
	if (!image->i.l)
	    VIP_Error_Msg("AllocateImage unable to allocate space");
	break;

    case FLOATTYPE:
	image->i.f = (f_array *) Allocate_Matrix((int) rows, (int) cols, sizeof(float));
	if (!image->i.f)
	    VIP_Error_Msg("AllocateImage unable to allocate space");
	break;

    case DOUBLETYPE:
	image->i.d = (d_array *) Allocate_Matrix((int) rows, (int) cols, sizeof(double));
	if (!image->i.d)
	    VIP_Error_Msg("AllocateImage unable to allocate space");
	break;

    case COMPLEXTYPE:
	image->i.cx = (cx_array *) Allocate_Matrix((int) rows, (int) cols, sizeof(COMPLEX));
	if (!image->i.cx)
	    VIP_Error_Msg("AllocateImage unable to allocate space");
	break;

    case RGBTYPE:
	image->i.rgb = (rgb_array *) Allocate_Matrix((int) rows, (int) cols, sizeof(RGB));
	if (!image->i.rgb)
	    VIP_Error_Msg("AllocateImage unable to allocate space");
	break;

    case HSITYPE:
	image->i.hsi = (hsi_array *) Allocate_Matrix((int) rows, (int) cols, sizeof(HSI));
	if (!image->i.hsi)
	    VIP_Error_Msg("AllocateImage unable to allocate space");
	break;

    default:
	VIP_Error_Msg("AllocateImage: Attempt to allocate illegal image type");
	return (NULL);

    }

/* set up memory sentinels */

    image->reserved[2] = VIP_START_SENTINEL;

switch(image->type) {

 case BYTETYPE:
    image->i.c[image->rows][0] = VIP_END_SENTINEL;
    break;
 case SHORTTYPE:
    image->i.s[image->rows][0] = VIP_END_SENTINEL;
    break;
 case LONGTYPE:
    image->i.l[image->rows][0] = VIP_END_SENTINEL;
    break;
 case FLOATTYPE:
    image->i.f[image->rows][0] = VIP_END_SENTINEL;
    break;
 case DOUBLETYPE:
    image->i.d[image->rows][0] = VIP_END_SENTINEL;
    break;
 case COMPLEXTYPE:
    image->i.cx[image->rows][0].r = VIP_END_SENTINEL;
    break;
 case RGBTYPE:
    image->i.rgb[image->rows][0][0] = VIP_END_SENTINEL;
    break;
 case HSITYPE:
    image->i.hsi[image->rows][0][0] = VIP_END_SENTINEL;
    break;
  default:
    VIP_Error_Msg("Allocate_Image: illegal type");
  } /* end of switch */

    return (image);

}

#endif


#ifdef DOS
/*-  Allocate_Image  -------------------------------------------------

Function to allocate memory for an image

DOS version - individual rows are allocated so that no individual
section of memory exceeds 64k.

WARNING:  Some functions may expect image data to be a continuous
segment of memory and hence will not work under DOS.

--------------------------------------------------------------------*/

IMAGE  *Allocate_Image(umin, vmin, rows, cols, type)
int32   umin, vmin, rows, cols, type;
{
    IMAGE  *image;
    int     i;

    image = (IMAGE *) malloc(sizeof(struct struct_vip_IMAGE));

    if (!image) {
	VIP_Error_Msg("Allocate_Image:	unable to allocate space for image");
	return (NULL);
    }

    if (rows < 0 || cols < 0) {
	VIP_Error_Msg("Allocate_Image:	-ve No of rows or columns requested");
	return (NULL);
    }

    image->idcode = IMAGE_ID;
    image->version[0] = VIP_MAJOR_VERSION;
    image->version[1] = VIP_MINOR_VERSION;
    image->umin = umin;
    image->vmin = vmin;
    image->type = type;
    image->rows = rows;
    image->cols = cols;
    image->resolution = 1;	/* assume full resolution */
    image->info[0] = '\0';
    image->pntr[0] = (void *) NULL;
    image->pntr[1] = (void *) NULL;

    switch (type) {

    case BYTETYPE:
	image->i.c = (c_array *) malloc((int) rows * sizeof(c_array));
	for (i = 0; i < rows; i++) {
	    image->i.c[i] = (c_array) calloc((int) cols, sizeof(unsigned char));
	    if (!image->i.c[i])
		VIP_Error_Msg("AllocateImage unable to allocate space");
	}
	break;

    case SHORTTYPE:
	image->i.s = (s_array *) malloc((int) rows * sizeof(s_array));
	for (i = 0; i < rows; i++) {
	    image->i.s[i] = (s_array) calloc((int) cols, sizeof(short));
	    if (!image->i.s[i])
		VIP_Error_Msg("AllocateImage unable to allocate space");
	}
	break;

    case LONGTYPE:
	image->i.l = (l_array *) malloc((int) rows * sizeof(l_array));
	for (i = 0; i < rows; i++) {
	    image->i.l[i] = (l_array) calloc((int) cols, sizeof(int32));
	    if (!image->i.l[i])
		VIP_Error_Msg("AllocateImage unable to allocate space");
	}
	break;

    case FLOATTYPE:
	image->i.f = (f_array *) malloc((int) rows * sizeof(f_array));
	for (i = 0; i < rows; i++) {
	    image->i.f[i] = (f_array) calloc((int) cols, sizeof(float));
	    if (!image->i.f[i])
		VIP_Error_Msg("AllocateImage unable to allocate space");
	}
	break;

    case DOUBLETYPE:
	image->i.d = (d_array *) malloc((int) rows * sizeof(d_array));
	for (i = 0; i < rows; i++) {
	    image->i.d[i] = (d_array) calloc((int) cols, sizeof(double));
	    if (!image->i.d[i])
		VIP_Error_Msg("AllocateImage unable to allocate space");
	}
	break;

    case COMPLEXTYPE:
	image->i.cx = (cx_array *) malloc((int) rows * sizeof(cx_array));
	for (i = 0; i < rows; i++) {
	    image->i.cx[i] = (cx_array) calloc((int) cols, sizeof(COMPLEX));
	    if (!image->i.cx[i])
		VIP_Error_Msg("AllocateImage unable to allocate space");
	}
	break;

    case RGBTYPE:
	image->i.rgb = (rgb_array *) malloc((int) rows * sizeof(rgb_array));
	for (i = 0; i < rows; i++) {
	    image->i.rgb[i] = (rgb_array) calloc((int) cols, sizeof(RGB));
	    if (!image->i.rgb[i])
		VIP_Error_Msg("AllocateImage unable to allocate space");
	}
	break;

    case HSITYPE:
	image->i.hsi = (hsi_array *) malloc((int) rows * sizeof(hsi_array));
	for (i = 0; i < rows; i++) {
	    image->i.hsi[i] = (hsi_array) calloc((int) cols, sizeof(HSI));
	    if (!image->i.hsi[i])
		VIP_Error_Msg("AllocateImage unable to allocate space");
	}
	break;

    default:
	VIP_Error_Msg("AllocateImage: Attempt to allocate illegal image type");
	return (NULL);
    }
    return (image);
}

#endif

/*------------------------------------------------------------------*/

float   Complex_Magnitude_2(z)
COMPLEX *z;
{
    float   tmp1 = 0.0, tmp2 = 0.0, tmp3 = 0.0;

    tmp1 = z->r * z->r;
    tmp2 = z->i * z->i;
    tmp3 = (float) sqrt(tmp1 + tmp2);
    return (tmp3);
}

/*-  Complex2Float_Image ---------------------------------------------

Function to convert a COMPLEXTYPE image to a MAG FLOATTYPE image.
For each pixel, the magnitude is calculated and stored in the output image.

--------------------------------------------------------------------*/

IMAGE  *Complex2Float_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    int     i, j;
    float   tempfloat = 0.0;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Complex2Float_Image: received invalid image");
	return (NULL);
    }

    if (image->type != COMPLEXTYPE) {
	VIP_Error_Msg("Complex2Float_Image: Not a COMPLEXTYPE image");
	return (NULL);
    }
    new_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, FLOATTYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = FLOATTYPE;
    for (i = image->rows - 1; i >= 0; i--) {
	for (j = image->cols - 1; j >= 0; j--) {
	    tempfloat = (float) Complex_Magnitude_2(&image->i.cx[i][j]);

	    new_image->i.f[i][j] = tempfloat;
	}
    }
    return (new_image);
}

/*-  Float2Byte_Image ---------------------------------------------

Function to convert a FLOATTYPE image to a BYTETYPE image.

--------------------------------------------------------------------*/
IMAGE  *Float2Byte_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    int     i, j;
    float   min, max;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Float2Byte_Image: received invalid image");
	return (NULL);
    }

    if (image->type != FLOATTYPE) {
	VIP_Error_Msg("Byte2Float: Not a FLOATTYPE image");
	return (NULL);
    }
    new_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, BYTETYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = BYTETYPE;

    /* scan through the image to get the min and max values */
    min = max = image->i.f[0][0];
    for (i = 0; i < image->rows; i++) {
	for (j = 0; j < image->cols; j++) {
	    if (min > image->i.f[i][j])
		min = image->i.f[i][j];
	    if (max < image->i.f[i][j])
		max = image->i.f[i][j];
	}
    }

    /* now variable max contains a scaling factor */

    if (max - min == 0) {
	max = 0;
    }
    else {
	max = 255.0 / (max - min);
    }

    for (i = 0; i < image->rows; i++)
	for (j = 0; j < image->cols; j++)
	    new_image->i.c[i][j] = (unsigned char) ((image->i.f[i][j] - min) * max);

    return (new_image);
}



/*-  Allocate_Header -------------------------------------------------

Function to allocate memory for an image header.

--------------------------------------------------------------------*/

IMAGE  *Allocate_Header(umin, vmin, rows, cols, type)
int32   umin, vmin, rows, cols, type;
{
    IMAGE  *im;

    if (!(im = (IMAGE *) calloc(1, sizeof(IMAGE)))) {
	VIP_Error_Msg("Allocate_Header: out of memory");
	return (NULL);
    }

    if (rows < 0 || cols < 0) {
        VIP_Error_Msg("Allocate_Header:  -ve No of rows or columns requested");
        return (NULL);
      }

    im->idcode = IMAGE_ID;
    im->version[0] = VIP_MAJOR_VERSION;
    im->version[1] = VIP_MINOR_VERSION;
    im->umin = umin;
    im->vmin = vmin;
    im->type = type;
    im->rows = rows;
    im->cols = cols;
    im->resolution = 1;		/* assume full resolution */
    im->info[0] = '\0';
    im->pntr[0] = (void *) NULL;
    im->pntr[1] = (void *) NULL;

/* set up memory sentinel */

    im->reserved[2] = VIP_START_SENTINEL;

    return (im);
}


#ifndef DOS
/*- Free_Image  -----------------------------------------------------

Function to free memory allocated for an image.

Non-DOS version

--------------------------------------------------------------------*/
int     Free_Image(image)
IMAGE  *image;
{
    if (image == NULL) {
	VIP_Error_Msg("Free_Image : warning - attempt to free NULL image structure");
	return (OK);		/* it was already free */
    }

    if (!Image_OK(image)) {
	VIP_Error_Msg("Free_Image: Attempt to free a non-IMAGE structure");
	return (ERROR);
    }

    switch (image->type) {

    case BYTETYPE:
	Free_Matrix((char **) image->i.c);
	break;

    case SHORTTYPE:
	Free_Matrix((char **) image->i.s);
	break;

    case LONGTYPE:
	Free_Matrix((char **) image->i.l);
	break;

    case FLOATTYPE:
	Free_Matrix((char **) image->i.f);
	break;

    case DOUBLETYPE:
	Free_Matrix((char **) image->i.d);
	break;

    case COMPLEXTYPE:
	Free_Matrix((char **) image->i.cx);
	break;

    case RGBTYPE:
	Free_Matrix((char **) image->i.rgb);
	break;

    case HSITYPE:
	Free_Matrix((char **) image->i.hsi);
	break;

    default:
	VIP_Error_Msg("Free_Image: Attempt to free unknown type of image");
	return (ERROR);
    }

    (void) free((char *) image);

    image = (IMAGE *) NULL;

    return (OK);

}

#endif


#ifdef DOS
/*- Free_Image  -----------------------------------------------------

Function to free memory allocated for an image.

DOS version

--------------------------------------------------------------------*/
int     Free_Image(image)
IMAGE  *image;
{
    int     i;

    if (image == NULL) {
	VIP_Error_Msg("Free_Image : warning - attempt to free NULL image structure");
	return (OK);		/* it was already free */
    }

    if (!Image_OK(image)) {
	VIP_Error_Msg("Free_Image: Attempt to free a non-IMAGE structure");
	return (ERROR);
    }

    switch (image->type) {

    case BYTETYPE:
	for (i = 0; i < image->rows; i++)
	    (void) free((char *) image->i.c[i]);

	(void) free((char *) image->i.c);
	break;

    case SHORTTYPE:
	for (i = 0; i < image->rows; i++)
	    (void) free((char *) image->i.s[i]);

	(void) free((char *) image->i.s);
	break;

    case LONGTYPE:
	for (i = 0; i < image->rows; i++)
	    (void) free((char *) image->i.l[i]);

	(void) free((char *) image->i.l);
	break;

    case FLOATTYPE:
	for (i = 0; i < image->rows; i++)
	    (void) free((char *) image->i.f[i]);

	(void) free((char *) image->i.f);
	break;

    case DOUBLETYPE:
	for (i = 0; i < image->rows; i++)
	    (void) free((char *) image->i.d[i]);

	(void) free((char *) image->i.d);
	break;

    case COMPLEXTYPE:
	for (i = 0; i < image->rows; i++)
	    (void) free((char *) image->i.cx[i]);

	(void) free((char *) image->i.cx);
	break;

    case RGBTYPE:
	for (i = 0; i < image->rows; i++)
	    (void) free((char *) image->i.rgb[i]);

	(void) free((char *) image->i.rgb);
	break;

    case HSITYPE:
	for (i = 0; i < image->rows; i++)
	    (void) free((char *) image->i.hsi[i]);

	(void) free((char *) image->i.hsi);
	break;

    default:
	VIP_Error_Msg("Free_Image: Attempt to free unknown type of image");
	return (ERROR);
    }

    (void) free((char *) image);

    image = (IMAGE *) NULL;

    return (OK);

}

#endif

/*- Set_Image  -----------------------------------------------------

Function to set image to a constant value.

--------------------------------------------------------------------*/
int     Set_Image(image, value)
IMAGE  *image;
double  value;
{
    register int r, c;
    int ivalue;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Set_Image: Attempt to set a non-IMAGE structure");
	return (ERROR);
    }


    ivalue = (int)(value);

    switch (image->type) {

    case BYTETYPE:{
	    unsigned char v;

	    if ((ivalue < 0) || (ivalue > 255)) {
		VIP_Error_Msg("Set_Image: Attempt to set BYTETYPE image to value out of range");
		return (ERROR);
	    }
	    v = (unsigned char) ivalue;
	    for (r = 0; r < image->rows; r++)
		for (c = 0; c < image->cols; c++)
		    image->i.c[r][c] = v;
	    break;
	}

    case SHORTTYPE:{
	    short   v;

	    if ((ivalue < nint(pow(2.0, (double) (8 * sizeof(short) - 1)))) ||
		(ivalue > nint(pow(2.0, (double) (8 * sizeof(short)))))) {
		VIP_Error_Msg("Set_Image: Attempt to set SHORTTYPE image to value out of range");
		return (ERROR);
	    }
	    v = (short) ivalue;
	    for (r = 0; r < image->rows; r++)
		for (c = 0; c < image->cols; c++)
		    image->i.s[r][c] = v;
	    break;
	}

    case LONGTYPE:{
	    long    v;

	    if ((ivalue < nint(pow(2.0, (double) (8 * sizeof(long) - 1)))) ||
		(ivalue > nint(pow(2.0, (double) (8 * sizeof(long)))))) {
		VIP_Error_Msg("Set_Image: Attempt to set LONGTYPE image to value out of range");
		return (ERROR);
	    }
	    v = (long) ivalue;
	    for (r = 0; r < image->rows; r++)
		for (c = 0; c < image->cols; c++)
		    image->i.l[r][c] = v;
	    break;
	}

    case FLOATTYPE:{
	    float   v = (float) value;

	    for (r = 0; r < image->rows; r++)
		for (c = 0; c < image->cols; c++)
		    image->i.f[r][c] = v;
	    break;
	}

    case DOUBLETYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++)
		image->i.d[r][c] = value;
	break;

    case COMPLEXTYPE:{
	    float   v = (float) value;

	    for (r = 0; r < image->rows; r++)
		for (c = 0; c < image->cols; c++) {
		    image->i.cx[r][c].r = v;
		    image->i.cx[r][c].i = v;
		}
	    break;
	}

    case RGBTYPE:{
	    unsigned char v;

	    if ((ivalue < 0) || (ivalue > 255)) {
		VIP_Error_Msg("Set_Image: Attempt to set RGBTYPE image to value out of range"
		    );
		return (ERROR);
	    }
	    v = (unsigned char) ivalue;
	    for (r = 0; r < image->rows; r++)
		for (c = 0; c < image->cols; c++) {
		    image->i.rgb[r][c][0] = v;
		    image->i.rgb[r][c][1] = v;
		    image->i.rgb[r][c][2] = v;
		}
	    break;
	}

    case HSITYPE:{
	    unsigned char v;

	    if ((ivalue < 0) || (ivalue > 255)) {
		VIP_Error_Msg("Set_Image: Attempt to set HSITYPE image to value out of range"
		    );
		return (ERROR);
	    }
	    v = (unsigned char) ivalue;
	    for (r = 0; r < image->rows; r++)
		for (c = 0; c < image->cols; c++) {
		    image->i.hsi[r][c][0] = v;
		    image->i.hsi[r][c][1] = v;
		    image->i.hsi[r][c][2] = v;
		}
	    break;
	}

    default:
	VIP_Error_Msg("Set_Image: Attempt to set unknown type of image");
	return (ERROR);
    }

    return (OK);

}

/*---------------------------------------------------------------------*/

#ifdef L_ENDIAN
#	define REVERSE(x) RevByte((char *)&x, sizeof(typeof(x)));
#else
#	define REVERSE(x) {}
#endif

/*---------------------------------------------------------------------*/

static void RevByte(char *s, int n)
{
#ifdef L_ENDIAN
	register b;
	register char * d = s + n - 1;

	n >>= 1; /* Divide by two */
	while (n--) {
		b = *s;
		*(s++) = *d;
		*(d--) = b;
	}
#endif
}

/*---------------------------------------------------------------------*/
static int SizeOfVipType(int type)
{
	switch(type) {
	case BYTETYPE: return sizeof(unsigned char);
	case SHORTTYPE: return sizeof(int16);
	case LONGTYPE: return sizeof(int32);
	case FLOATTYPE: return sizeof(float);
	case DOUBLETYPE: return sizeof(double);
	case COMPLEXTYPE: return sizeof(COMPLEX);
	case RGBTYPE: return sizeof(RGB);
	case HSITYPE: return sizeof(HSI);
	default:
		fprintf(stderr,"SizeOfVipType: invalid type %d.\n", type);
		exit(1);
	}
}

/*---------------------------------------------------------------------*/

void ByteSwapHeader(IMAGE *head)
{
	register int i;

    /* do the byte swappings for little endian word format */
	/* imagedata */
    REVERSE(head->idcode);
    REVERSE(head->version[0]);
    REVERSE(head->version[1]);
    REVERSE(head->type);
    REVERSE(head->umin);
    REVERSE(head->vmin);
    REVERSE(head->rows);
    REVERSE(head->cols);
    REVERSE(head->resolution);

    /* camera structure */
    REVERSE(head->camera.idcode);
    REVERSE(head->camera.channel);
    REVERSE(head->camera.gain);
    REVERSE(head->camera.offset);
    REVERSE(head->camera.uvoffset);
    REVERSE(head->camera.calib_status);

    /* gamma */
	for(i = 0; i < 5; i++)
    	REVERSE(head->camera.gamma);

    /* calibration matrix */
	for(i = 0; i < 3; i++) {
		register int j;
    	for (j = 0; j < 4; j++)
    		REVERSE(head->camera.calib_matrix[i][j]);
	}

    /* Ucomp */
	for(i = 0; i < 18; i++)
    	REVERSE(head->camera.Ucomp[i]);

    /* Vcomp */
	for(i = 0; i < 18; i++)
    	REVERSE(head->camera.Vcomp[i]);

    for (i = 0; i < 3; i++) {
    	REVERSE(head->reserved[i]);
    }

	return;
}

/*---------------------------------------------------------------*/

void ByteSwapRow(char *row, int length, int size)
{
	while(length--) {
		RevByte(row, size);
		row += size;
	}

	return;
} 

/*-  HEADER_SIZE ---------------------------------------------------*/

#ifdef DEC_ALPHA
#   define HEADER_SIZE (sizeof(IMAGE) - 4 - sizeof(imagedata) - 2 * sizeof(void *))
#else
#   define HEADER_SIZE (sizeof(IMAGE) - sizeof(imagedata) - 2 * sizeof(void *))
#endif


/*-  Read_Header  ---------------------------------------------------

Function to read in an image header from a file pointer.

--------------------------------------------------------------------*/

void    Read_Header(head, fp)
IMAGE  *head;
FILE   *fp;
{
    (void) fread((char *)head, (size_t) HEADER_SIZE, (size_t) 1, fp);

#ifdef L_ENDIAN
    ByteSwapHeader(head);
#endif				/* L_ENDIAN */

return;

}


/*-  Write_Header  --------------------------------------------------

Function to write an image header to a file pointer.

--------------------------------------------------------------------*/

void    Write_Header(head, fp)
IMAGE  *head;
FILE   *fp;
{
    IMAGE   revhead;

    if (!Image_OK(head)) {
	VIP_Error_Msg("Write_Header: received invalid image");
	return;
    }

    memcpy(&revhead, head, sizeof(IMAGE));

	/* Set the version to relfect the VIP Version writing the file */
    revhead.version[0] = VIP_MAJOR_VERSION;
    revhead.version[1] = VIP_MINOR_VERSION;

#ifdef L_ENDIAN
    /* do the byte swappings for little endian word format */

    ByteSwapHeader(&revhead);

#endif				/* L_ENDIAN */

    (void) fwrite(&revhead, HEADER_SIZE, 1, fp);
}

/*-  Write_Image  ---------------------------------------------------

Function to write image data structure to a disc file.

--------------------------------------------------------------------*/

int     Write_Image(image, filename)
IMAGE  *image;
char   *filename;
{
    FILE   *outfile;
	int     row_size;
    char   *buf;
    cx_array cxbuf;
	int     r, c;

    if (!Image_OK(image)) {
	    VIP_Error_Msg("Write_Image: Attempt to write a non-IMAGE structure");
	    return (ERROR);
    }

    if (filename) {
	    outfile = fopen(filename, "wb");
	    if (outfile == NULL) {
	        VIP_Error_Msg("Write_Image could not open file");
	        return (ERROR);
	    }
    } else {
	    outfile = stdout;
    }

    /* Write the header, but not the pointer to the image data */

    Write_Header(image, outfile);

    row_size = image->cols * SizeOfVipType(image->type);

    switch(image->type) {

#ifdef L_ENDIAN

        case SHORTTYPE:
        case LONGTYPE:
        case FLOATTYPE:
        case DOUBLETYPE:

            buf = malloc(image->cols * SizeOfVipType(image->type));

	        for (r = 0; r < image->rows; r++) {
                memcpy(buf, image->i.c[r], row_size);
                ByteSwapRow(buf, image->cols, SizeOfVipType(image->type));

	            (void) fwrite(buf, sizeof(unsigned char), row_size, outfile);
	        }

            free(buf);
            break;

        case COMPLEXTYPE:

            cxbuf = (cx_array) malloc(image->cols * SizeOfVipType(image->type));

	        for (r = 0; r < image->rows; r++) {
                memcpy((char *) cxbuf, image->i.c[r], row_size);
                for (c = 0; c < image->cols; c++) {
                    REVERSE(cxbuf[c].r);
                    REVERSE(cxbuf[c].i);
                }
	            (void) fwrite((char *) cxbuf, sizeof(unsigned char),
                    row_size, outfile);
	        }

            free(cxbuf);
            break;

#else
        case SHORTTYPE:
        case LONGTYPE:
        case FLOATTYPE:
        case DOUBLETYPE:
        case COMPLEXTYPE:
#endif 
        case BYTETYPE:
        case RGBTYPE:
        case HSITYPE:   /* no byte swapping needed */

	        for (r = 0; r < image->rows; r++) {
	            (void) fwrite(image->i.c[r], sizeof(unsigned char),
                    row_size, outfile);
	        }
            break;

        default:
	        VIP_Error_Msg("Write_Image: Attempt to write unknown type of image");
	        break;
        }

    (void) fclose(outfile);

    return (OK);

}


/*-  Read_Image  ----------------------------------------------------

Function to read image data from a file into memory

--------------------------------------------------------------------*/

IMAGE  *Read_Image(filename)
char   *filename;
{
    IMAGE  *image;
    IMAGE   header;
    FILE   *infile;
    int     r, c;
    int     row_size;
    int     handle;
    int     magicid;

#ifdef U_NIX			/* UNIX is a resvd gcc macro */
#include "oldvipfn.h"
    old_image old_im;		/* the following variables are */
    old_matrix old_mat;		/* for old V.I.P. images only */
    old_cp_image old_cpim;

#endif

    if (filename) {

	    infile = (FILE *) fopen(filename, "rb");
	    if (!infile) {
	        VIP_Error_Msg("Read_Image could not open file");
	        return (NULL);
	    }

    } else {
	    infile = stdin;
	}

    /* read the header */

    Read_Header(&header, infile);

    if (header.idcode != IMAGE_ID) {	/* check that it is an image */

#ifdef U_NIX			/* UNIX is a resvd gcc macro */

	(void) fclose(infile);

	/* possibly an old VIP image */

	if ((magicid = old_read_id(filename, &handle)) == OLD_IMAGE_ID) {
	    if ((old_im = old_read_image(handle)) == NULL) {
		    VIP_Error_Msg("Read_Image: out of memory");
		    return (NULL);
	    }
	    image = Convert_Image(old_im);
	    return (image);

	} else if (magicid == OLD_MATRIX_ID) {
	    if ((old_mat = old_read_matrix(handle)) == NULL) {
		    VIP_Error_Msg("Read_Image: out of memory");
		    return (NULL);
	    }
	    image = Convert_Matrix(old_mat);
	    return (image);

	} else if (magicid == OLD_COMPLEX_ID) {
	    if ((old_cpim = old_read_cpimage(handle)) == NULL) {
		    VIP_Error_Msg("Read_Image: out of memory");
		    return (NULL);
	    }
	    image = Convert_Complex_Image(old_cpim);
	    return (image);

	} else {
	    VIP_Error_Msg("Read_Image error:  This is not an image file!");
	    return (NULL);

	}
	}
#else

	    VIP_Error_Msg("Read_Image error:  This is not an image file!");
	    return (NULL);
    }
#endif

    /* Allocate space for the image */

    image = (IMAGE *) Allocate_Image(header.umin, header.vmin,
        header.rows, header.cols, header.type);

    if (image == NULL) {
	    (void) fclose(infile);
	    return (NULL);
    }
    (void) Copy_Header(&header, image);	/* Copy the header information */

/*  Set image memory sentinel.  This is only needed for images that
    have been created before May 1994.  After this date images would
    have had the sentinel value written into their headers.
*/
    image->reserved[2] = VIP_START_SENTINEL;

    row_size = image->cols * SizeOfVipType(image->type);

    /* The data is read in by row in several lots of row_size bytes */
	for (r = 0; r < image->rows; r++) {
	    (void) fread((char *) (image->i.c[r]), sizeof(unsigned char),
            row_size, infile);
    }

    switch (image->type) {

#ifdef L_ENDIAN

    case SHORTTYPE:
    case LONGTYPE:
    case FLOATTYPE:
    case DOUBLETYPE:

	    for (r = 0; r < image->rows; r++) {
            ByteSwapRow(image->i.c[r], image->cols, SizeOfVipType(image->type));
	    }

	    break;

    case COMPLEXTYPE:
	    for (r = 0; r < image->rows; r++) {
            for (c = 0; c < image->cols; c++) {
                REVERSE(image->i.cx[r][c].r);
                REVERSE(image->i.cx[r][c].i);
            }
	    }

	    break;

#else
    case SHORTTYPE:
    case LONGTYPE:
    case FLOATTYPE:
    case DOUBLETYPE:
    case COMPLEXTYPE:
#endif
    case BYTETYPE:
    case RGBTYPE:
    case HSITYPE:
		/* No byte swapping necessary */
	    break;

    default:
	    VIP_Error_Msg("Read_Image: Attempt to read unknown type of image");
	    break;
    }

    (void) fclose(infile);
    return (image);

}


/*-  Copy_Header  ----------------------------------------------------

Function to copy header information from one image to another.
The pointer to the image data is not overwritten.
Calling format: Copy_Header(src_header_pointer, dest_header_pointer).

--------------------------------------------------------------------*/
int     Copy_Header(image1, image2)
IMAGE  *image1, *image2;
{

    if (image1->idcode != IMAGE_ID || image2->idcode != IMAGE_ID) {
	VIP_Error_Msg("Copy_Header: Attempt to copy from or to a structure that is not an image");
	return (ERROR);
    }

#ifdef DEC_ALPHA
    memcpy(image2, image1, sizeof(IMAGE) - 4 - sizeof(imagedata) - sizeof(void *) * 2);
#else
    memcpy(image2, image1, sizeof(IMAGE) - sizeof(imagedata) - sizeof(void *) * 2);
#endif

    return (OK);
}


/*- Copy_Image ------------------------------------------------------

Function to copy an image.

--------------------------------------------------------------------*/

IMAGE  *Copy_Image(IMAGE * image)
{
    IMAGE  *newimage;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Copy_Image: received non-image structure to copy");
	return (NULL);
    }

    newimage = Allocate_Image(image->umin, image->vmin, image->rows, image->cols, image->type);

    Copy_Header(image, newimage);	/* get all the other data such as
					 * calibration stuff */

    switch (image->type) {

    case BYTETYPE:
	bcopy((char *) image->i.c[0], (char *) newimage->i.c[0],
	    image->rows * image->cols * sizeof(unsigned char));
	break;

    case SHORTTYPE:
	bcopy((char *) image->i.s[0], (char *) newimage->i.s[0],
	    image->rows * image->cols * sizeof(short));
	break;

    case LONGTYPE:
	bcopy((char *) image->i.l[0], (char *) newimage->i.l[0],
	    image->rows * image->cols * sizeof(long));
	break;

    case FLOATTYPE:
	bcopy((char *) image->i.f[0], (char *) newimage->i.f[0],
	    image->rows * image->cols * sizeof(float));
	break;

    case DOUBLETYPE:
	bcopy((char *) image->i.d[0], (char *) newimage->i.d[0],
	    image->rows * image->cols * sizeof(double));
	break;

    case COMPLEXTYPE:
	bcopy((char *) image->i.cx[0], (char *) newimage->i.cx[0],
	    image->rows * image->cols * sizeof(COMPLEX));
	break;

    case RGBTYPE:
	bcopy((char *) image->i.rgb[0], (char *) newimage->i.rgb[0],
	    image->rows * image->cols * sizeof(RGB));
	break;

    case HSITYPE:
	bcopy((char *) image->i.hsi[0], (char *) newimage->i.hsi[0],
	    image->rows * image->cols * sizeof(HSI));
	break;

    default:
	VIP_Error_Msg("Copy_Image: works only for BYTE SHORT LONG FLOAT DOUBLE COMPLEX RBG and HSI images");
	Free_Image(newimage);
	return (NULL);
    }

    return (newimage);
}


/*-  Byte2Short_Image  ------------------------------------------------

Function to convert an BYTETYPE image to an SHORTTYPE image.

--------------------------------------------------------------------*/

IMAGE  *Byte2Short_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    int     i, j;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Byte2Short_Image: received invalid image");
	return (NULL);
    }



    if (image->type != BYTETYPE) {
	VIP_Error_Msg("Byte2Short: Not a BYTE image");
	return (NULL);
    }
    new_image = Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, SHORTTYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = SHORTTYPE;
    for (i = 0; i < image->rows; i++) {
	for (j = 0; j < image->cols; j++)
	    new_image->i.s[i][j] = image->i.c[i][j];
    }
    return (new_image);
}

/*-  Convert2Float_Image ------------------------------------------------

Function to convert a BYTETYPE, SHORTTYPE, LONGTYPE and DOUBLETYPE image
to a FLOATTYPE image.

--------------------------------------------------------------------*/

IMAGE  *Convert2Float_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    int     i, j;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Convert2Float_Image: received invalid image");
	return (NULL);
    }


    new_image = Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, FLOATTYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = FLOATTYPE;

    switch (image->type) {

    case BYTETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.f[i][j] = image->i.c[i][j];
	}
	break;

    case SHORTTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.f[i][j] = image->i.s[i][j];
	}
	break;

    case LONGTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.f[i][j] = image->i.l[i][j];
	}
	break;

    case FLOATTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.f[i][j] = image->i.f[i][j];
	}
	break;

    case DOUBLETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.f[i][j] = image->i.d[i][j];
	}
	break;

    case COMPLEXTYPE:
	new_image = (IMAGE *) Complex2Float_Image(image);
	break;

    default:
	VIP_Error_Msg("Convert2Float_Image: Can only take BYTE, SHORT, LONG, FLOAT, DOUBLE and COMPLEX type images");
	(void) Free_Image(new_image);
	return (NULL);
    }

    return (new_image);
}

/*-  Convert2Double_Image ------------------------------------------------

Function to convert a BYTETYPE, SHORTTYPE, LONGTYPE and FLOATTYPE
image to a DOUBLETYPE image.

--------------------------------------------------------------------*/

IMAGE  *Convert2Double_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    int     i, j;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Convert2Double_Image: received invalid image");
	return (NULL);
    }

    new_image = Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, DOUBLETYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = DOUBLETYPE;

    switch (image->type) {

    case BYTETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.d[i][j] = image->i.c[i][j];
	}
	break;

    case SHORTTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.d[i][j] = image->i.s[i][j];
	}
	break;

    case LONGTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.d[i][j] = image->i.l[i][j];
	}
	break;

    case FLOATTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.d[i][j] = image->i.f[i][j];
	}
	break;

    case DOUBLETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.d[i][j] = image->i.d[i][j];
	}
	break;

    default:
	VIP_Error_Msg("Convert2Double_Image: Can only take BYTE, SHORT, LONG, FLOAT and DOUBLE type images");
	(void) Free_Image(new_image);
	return (NULL);
    }

    return (new_image);
}


/*-- Convert2Complex_Image -----------------------------

Function to convert a BYTETYPE, SHORTTYPE, LONGTYPE, DOUBLETYPE and
FLOATTYPE image to a COMPLEX image.
The real components of the output image are a copy of the input image
and the imaginary components are set to zeros.
--------------------------------------------------------------------*/

IMAGE  *Convert2Complex_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    int     i, j;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Convert2Complex_Image: received invalid image");
	return (NULL);
    }


    new_image = Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, COMPLEXTYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = COMPLEXTYPE;

    switch (image->type) {

    case BYTETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++) {
		new_image->i.cx[i][j].r = (float) image->i.c[i][j];
		new_image->i.cx[i][j].i = 0.0;
	    }
	}
	break;

    case SHORTTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++) {
		new_image->i.cx[i][j].r = (float) image->i.s[i][j];
		new_image->i.cx[i][j].i = 0.0;
	    }
	}
	break;

    case LONGTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++) {
		new_image->i.cx[i][j].r = (float) image->i.l[i][j];
		new_image->i.cx[i][j].i = 0.0;
	    }
	}
	break;

    case FLOATTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++) {
		new_image->i.cx[i][j].r = (float) image->i.f[i][j];
		new_image->i.cx[i][j].i = 0.0;
	    }
	}
	break;

    case DOUBLETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++) {
		new_image->i.cx[i][j].r = (float) image->i.d[i][j];
		new_image->i.cx[i][j].i = 0.0;
	    }
	}
	break;

    case COMPLEXTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++) {
		new_image->i.cx[i][j].r = image->i.cx[i][j].r;
		new_image->i.cx[i][j].i = image->i.cx[i][j].i;
	    }
	}
	break;

    default:
	VIP_Error_Msg("Convert2Complex_Image: Can only take BYTE, SHORT, LONG, DOUBLE, FLOAT and COMPLEX type images");
	(void) Free_Image(new_image);
	return (NULL);
    }

    return (new_image);
}



/*-  Byte2Float_Image  ------------------------------------------------

Function to convert an BYTETYPE image to a FLOATTYPE image.

--------------------------------------------------------------------*/

IMAGE  *Byte2Float_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    int     i, j;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Byte2Float_Image: received invalid image");
	return (NULL);
    }


    if (image->type != BYTETYPE) {
	VIP_Error_Msg("Byte2Float: Not a BYTE image");
	return (NULL);
    }
    new_image = Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, FLOATTYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = FLOATTYPE;
    for (i = 0; i < image->rows; i++)
	for (j = 0; j < image->cols; j++)
	    new_image->i.f[i][j] = image->i.c[i][j];

    return (new_image);
}


/*-  Byte2Complex_Image  ---------------------------------------------

Function to convert an BYTETYPE image to a COMPLEXTYPE image.
The real components of the output image is a copy of the input image
and the imaginary components are set to zeros.

--------------------------------------------------------------------*/

IMAGE  *Byte2Complex_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image = (IMAGE *) NULL;
    int     i, j;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Byte2Complex_Image: received invalid image");
	return (NULL);
    }



    if (image->type != BYTETYPE) {
	VIP_Error_Msg("Byte2Complex: Not a BYTE image");
	return (NULL);
    }
    new_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, COMPLEXTYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = COMPLEXTYPE;
    for (i = 0; i < image->rows; i++)
	for (j = 0; j < image->cols; j++) {
	    new_image->i.cx[i][j].r = (float) image->i.c[i][j];
	    new_image->i.cx[i][j].i = (float) 0.0;
	}

    return (new_image);
}


/*-  Convert2Byte_Image ------------------------------------------------

Function to convert a FLOATTYPE, SHORTTYPE, LONGTYPE and DOUBLETYPE
image to a BYTETYPE image.

--------------------------------------------------------------------*/

IMAGE  *Convert2Byte_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    IMAGE  *temp_image;
    int     i, j;
    double  min, max, min2, max2, min3, max3, rescale;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Convert2Byte_Image: received invalid image");
	return (NULL);
    }



    new_image = Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, BYTETYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = BYTETYPE;

    Image_Min_Max(image, &min, &max, &min2, &max2, &min3, &max3);
    rescale = 255.0 / (max - min);

    switch (image->type) {

    case BYTETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.c[i][j] = image->i.c[i][j];
	}
	break;

    case SHORTTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.c[i][j] = ((double) image->i.s[i][j] - min) * rescale;
	}
	break;

    case LONGTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.c[i][j] = ((double) image->i.l[i][j] - min) * rescale;
	}
	break;

    case FLOATTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.c[i][j] = ((double) image->i.f[i][j] - min) * rescale;
	}
	break;

    case DOUBLETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.c[i][j] = (image->i.d[i][j] - min) * rescale;
	}
	break;

    case COMPLEXTYPE:
	temp_image = (IMAGE *) Complex2Float_Image(image);
	new_image = (IMAGE *) Float2Byte_Image(temp_image);
	(void) Free_Image(temp_image);
	break;

    default:
	VIP_Error_Msg("Convert2Byte_Image: Can only take BYTE, SHORT, LONG, FLOAT, COMPLEX and DOUBLE type images");
	(void) Free_Image(new_image);
	return (NULL);
    }

    return (new_image);
}



/*-  Float2Short_Image ---------------------------------------------

Function to convert a FLOATTYPE image to a BYTETYPE image.

--------------------------------------------------------------------*/
IMAGE  *Float2Short_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    int     i, j;
    float   min, max;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Float2Short_Image: received invalid image");
	return (NULL);
    }

    if (image->type != FLOATTYPE) {
	VIP_Error_Msg("Byte2Float: Not a FLOATTYPE image");
	return (NULL);
    }
    new_image = Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, SHORTTYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = SHORTTYPE;

    /* scan through the image to get the min and max values */
    min = max = image->i.f[0][0];
    for (i = 0; i < image->rows; i++)
	for (j = 0; j < image->cols; j++) {
	    if (min > image->i.f[i][j])
		min = image->i.f[i][j];
	    if (max < image->i.f[i][j])
		max = image->i.f[i][j];
	}
    /* now variable max contains a scaling factor */
    max = 255.0 / (max - min);
    for (i = 0; i < image->rows; i++)
	for (j = 0; j < image->cols; j++)
	    new_image->i.s[i][j] = (short) ((image->i.f[i][j] - min) * max);

    return (new_image);
}



/*-  Convert2Short_Image ------------------------------------------------

Function to convert a BYTETYPE, FLOATTYPE, LONGTYPE and DOUBLETYPE
image to a SHORTTYPE image.

--------------------------------------------------------------------*/

IMAGE  *Convert2Short_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    IMAGE  *temp_image;
    int     i, j;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Convert2Short_Image: received invalid image");
	return (NULL);
    }

    new_image = Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, SHORTTYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = SHORTTYPE;

    switch (image->type) {

    case BYTETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.s[i][j] = image->i.c[i][j];
	}
	break;

    case SHORTTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.s[i][j] = image->i.s[i][j];
	}
	break;

    case LONGTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.s[i][j] = image->i.l[i][j];
	}
	break;

    case FLOATTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.s[i][j] = image->i.f[i][j];
	}
	break;

    case DOUBLETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.s[i][j] = image->i.d[i][j];
	}
	break;

    case COMPLEXTYPE:
	temp_image = Allocate_Image(image->umin, image->vmin,
	    image->rows, image->cols, FLOATTYPE);
	(void) Copy_Header(image, temp_image);
	temp_image->type = FLOATTYPE;

	temp_image = (IMAGE *) Complex2Float_Image(image);
	new_image = (IMAGE *) Float2Short_Image(temp_image);
	(void) Free_Image(temp_image);
	break;

    default:
	VIP_Error_Msg("Convert2Short_Image: Can only take BYTE, SHORT, LONG, FLOAT, COMPLEX and DOUBLE type images");
	(void) Free_Image(new_image);
	return (NULL);
    }

    return (new_image);
}


/*-  Convert2Long_Image ------------------------------------------------

Function to convert a BYTETYPE, FLOATTYPE, SHORTTYPE and DOUBLETYPE
image to a LONGTYPE image.

--------------------------------------------------------------------*/

IMAGE  *Convert2Long_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    IMAGE  *temp_image;
    int     i, j;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Convert2Long_Image: received invalid image");
	return (NULL);
    }

    new_image = Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, LONGTYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = LONGTYPE;

    switch (image->type) {

    case BYTETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.l[i][j] = image->i.c[i][j];
	}
	break;

    case SHORTTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.l[i][j] = image->i.s[i][j];
	}
	break;

    case LONGTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.l[i][j] = image->i.l[i][j];
	}
	break;

    case FLOATTYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.l[i][j] = image->i.f[i][j];
	}
	break;

    case DOUBLETYPE:
	for (i = 0; i < image->rows; i++) {
	    for (j = 0; j < image->cols; j++)
		new_image->i.l[i][j] = image->i.d[i][j];
	}
	break;

    case COMPLEXTYPE:
	temp_image = Allocate_Image(image->umin, image->vmin,
	    image->rows, image->cols, FLOATTYPE);
	(void) Copy_Header(image, temp_image);
	temp_image->type = FLOATTYPE;

	temp_image = (IMAGE *) Complex2Float_Image(image);

	for (i = 0; i < temp_image->rows; i++) {
	    for (j = 0; j < temp_image->cols; j++) {
		new_image->i.l[i][j] = temp_image->i.f[i][j];
	    }
	}

	(void) Free_Image(temp_image);
	break;
    default:
	VIP_Error_Msg("Convert2Long_Image: Can only take BYTE, SHORT, FLOAT, COMPLEX and DOUBLE type images");
	(void) Free_Image(new_image);
	return (NULL);
    }

    return (new_image);
}


/*-  VIP2PPM_Image  --------------------------------------------------

Converts a V.I.P. image to a PPM image.  The output PPM image will be
stored to "outfile".  This function only works for BYTETYPE and
RGBTYPE images.  The function returns ERROR if any error is encountered
in the function body and returns OK otherwise.

--------------------------------------------------------------------*/

int     VIP2PPM_Image(im, outfile)
IMAGE  *im;
char   *outfile;
{
    FILE   *f;
    register int r, c;
    unsigned char *buf, *tmp, *bptr;
    unsigned char *rgbptr;
    int     n;

    if (!Image_OK(im)) {
	VIP_Error_Msg("VIP2PPM: received invalid image");
	return (ERROR);
    }



    if (outfile) {
	f = fopen(outfile, "w");
	if (f == NULL) {
	    VIP_Error_Msg("write_output: fail to open output file");
	    return (ERROR);
	}
    }
    else
	f = stdout;

    n = im->cols * 3;
    if ((buf = (unsigned char *) malloc(n)) == NULL) {
	VIP_Error_Msg("VIP_2_PPM: out of memory");
	(void) fclose(f);
	return (ERROR);
    }

    /* write magic number */
    (void) fprintf(f, "P6\n");
    (void) fprintf(f, "%d %d\n", im->cols, im->rows);
    (void) fprintf(f, "%d\n", 255);
    switch (im->type) {
    case BYTETYPE:
	for (r = 0; r < im->rows; r++) {
	    for (c = 0, tmp = buf, bptr = im->i.c[r];
		c < im->cols; c++, tmp += 3, bptr++)
		(void) sprintf((char *) tmp, "%c%c%c", *bptr, *bptr, *bptr);
	    if (fwrite((char *) buf, n, 1, f) != 1) {
		VIP_Error_Msg("VIP_2_PPM: short write");
		(void) free(buf);
		(void) fclose(f);
		return (ERROR);
	    }
	}
	break;
    case RGBTYPE:
	for (r = 0; r < im->rows; r++) {
	    for (c = 0, tmp = buf;
		c < im->cols; c++, tmp += 3) {
		rgbptr = im->i.rgb[r][c];
		(void) sprintf((char *) tmp, "%c%c%c", rgbptr[0], rgbptr[1], rgbptr[2]);
	    }
	    if (fwrite((char *) buf, n, 1, f) != 1) {
		VIP_Error_Msg("VIP_2_PPM: short write");
		(void) free(buf);
		(void) fclose(f);
		return (ERROR);
	    }
	}
	break;
    default:
	VIP_Error_Msg("VIP_2_PPM: can only process BYTETYPE and RGBTYPE images");
	(void) free(buf);
	(void) fclose(f);
	return (ERROR);
    }

    (void) free(buf);
    (void) fclose(f);
    return (OK);
}


/*-  Write_VFFImage   ------------------------------------------------

Function to convert an image into vff format and write it to a file
for later display.

--------------------------------------------------------------------*/

int     Write_VFFImage(image, filename)
IMAGE  *image;
char   *filename;
{
    FILE   *output;
    int     i;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Write_VFFImage: received invalid image");
	return (ERROR);
    }

    output = fopen(filename, "w");
    (void) fprintf(output, "ncaa\n");
    (void) fprintf(output, "rank=2;\n");
    (void) fprintf(output, "type=raster;\n");
    (void) fprintf(output, "size=%d %d;\n", image->cols, image->rows);
    (void) fprintf(output, "format=base;\n");

    switch (image->type) {

    case BYTETYPE:
	(void) fprintf(output, "rawsize=%d;\n", (int) (image->rows * image->cols * sizeof(unsigned char)));
	(void) fprintf(output, "bands=1;\n");
	(void) fprintf(output, "bits=8;\n");
	(void) fprintf(output, "%c", 0xc);	/* octal for control L */
	for (i = 0; i < image->rows; i++)
	    (void) fwrite((char *) image->i.c[i], sizeof(unsigned char), image->cols, output);

	break;

    default:
	VIP_Error_Msg("awc2vff: Can only convert BYTE images");
	return (ERROR);

    }

    (void) fclose(output);
    return (OK);
}


/*-  Not_An_Image  ---------------------------------------------------

Function to check whether an address is the address of an IMAGE

--------------------------------------------------------------------*/

int     Not_An_Image(image)
IMAGE  *image;
{
    if (image->idcode != IMAGE_ID)
	return (TRUE);
    else
	return (FALSE);
}


/*-  Image_OK  ---------------------------------------------------

Function to check whether an address is the address of an IMAGE

--------------------------------------------------------------------*/

int     Image_OK(image)
IMAGE  *image;
{
int corrupt = FALSE;

    if (image == NULL) {
	return (FALSE);
    }

    if (image->idcode != IMAGE_ID) {
	VIP_Error_Msg("Corrupt image ID");
	return (FALSE);
    }

    if (image->reserved[2] != VIP_START_SENTINEL) {
	VIP_Error_Msg("Beginning of image data has been corrupted");
	return (FALSE);
    }


switch(image->type) {

 case BYTETYPE:
    if (image->i.c[image->rows][0] != VIP_END_SENTINEL)
      corrupt = TRUE;
    break;
 case SHORTTYPE:
    if (image->i.s[image->rows][0] != VIP_END_SENTINEL)
      corrupt = TRUE;
    break;
 case LONGTYPE:
    if (image->i.l[image->rows][0] != VIP_END_SENTINEL)
      corrupt = TRUE;
    break;
 case FLOATTYPE:
    if (image->i.f[image->rows][0] != VIP_END_SENTINEL)
      corrupt = TRUE;
    break;
 case DOUBLETYPE:
    if (image->i.d[image->rows][0] != VIP_END_SENTINEL)
      corrupt = TRUE;
    break;
 case COMPLEXTYPE:
    if (image->i.cx[image->rows][0].r != VIP_END_SENTINEL)
      corrupt = TRUE;
    break;
 case RGBTYPE:
    if (image->i.rgb[image->rows][0][0] != VIP_END_SENTINEL)
      corrupt = TRUE;
    break;
 case HSITYPE:
    if (image->i.hsi[image->rows][0][0] != VIP_END_SENTINEL)
      corrupt = TRUE;
    break;

  default:
    VIP_Error_Msg("Image_OK: illegal image type");
} /* end of switch */

   if(corrupt) {
	VIP_Error_Msg("End of image has been corrupted");
	return (FALSE);
    }

    return (TRUE);

}


/*-  Common_Window  -------------------------------------------------

Function to determine common area between two images.

--------------------------------------------------------------------*/

int     Common_Window(image1, image2, umin, vmin, rows, cols)
IMAGE  *image1, *image2;
int32  *umin, *vmin, *rows, *cols;
{

    if (!Image_OK(image1) || !Image_OK(image2)) {
	VIP_Error_Msg("Common_Window: NULL image pointer received");
	return (ERROR);
    }

    *umin = MAX(image1->umin, image2->umin);
    *vmin = MAX(image1->vmin, image2->vmin);

    *cols = MIN((image1->umin + image1->cols),
	(image2->umin + image2->cols)) - *umin;
    *rows = MIN((image1->vmin + image1->rows),
	(image2->vmin + image2->rows)) - *vmin;

    if (*rows == 0 OR * cols == 0)
	return (ERROR);
    else
	return (OK);
}

/*-  Pixel_In_Image  --------------------------------------------------

Function to test whether pixel coordinates are within image bounds

--------------------------------------------------------------------*/

int     Pixel_In_Image(image, u, v)
IMAGE  *image;
int32   u, v;
{

    if (!Image_OK(image)) {
	VIP_Error_Msg("Pixel_In_Image: NULL image pointer received");
	return (FALSE);
    }


    if (u < image->umin OR v < image->vmin OR
	u > (image->umin + image->cols - 1) OR v > (image->vmin + image->rows - 1)) {
	return (FALSE);
    }
    else
	return (TRUE);
}



float   ComplexPhase(z)
COMPLEX *z;
{
    float   tmp1 = 0.0;


    if (z->i == 0.0 && z->r == 0.0) {
	tmp1 = 0.0;
    }
    else {
	tmp1 = (float) atan2((double) z->i, (double) z->r);
    }

    return (tmp1);
}




/*-  Complex2PhaseFloat_Image ---------------------------------------------

Function to convert a COMPLEXTYPE image to a PHASE FLOATTYPE image.
For each pixel, the phase is calculated and stored in the output image.

--------------------------------------------------------------------*/

IMAGE  *Complex2PhaseFloat_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    int     i, j;
    float   tempfloat = 0.0;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Complex2PhaseFloat_Image: received invalid image");
	return (NULL);
    }


    if (image->type != COMPLEXTYPE) {
	VIP_Error_Msg("Complex2PhaseFloat_Image: Not a COMPLEXTYPE image");
	return (NULL);
    }
    new_image = Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, FLOATTYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = FLOATTYPE;
    for (i = image->rows - 1; i >= 0; i--) {
	for (j = image->cols - 1; j >= 0; j--) {
	    tempfloat = (float) ComplexPhase(&image->i.cx[i][j]);
	    new_image->i.f[i][j] = tempfloat;
	}
    }
    return (new_image);
}



/*-  Complex2MagFloat_Image ------------------

Function to convert a COMPLEXTYPE image to a MAG FLOATTYPE image.
For each pixel, the magnitude is calculated and stored in the output image.

--------------------------------------------------------------------*/

IMAGE  *Complex2MagFloat_Image(image)
IMAGE  *image;
{
    IMAGE  *new_image;
    int     i, j;
    float   tempfloat = 0.0;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Complex2MagFloat_Image: received invalid image");
	return (NULL);
    }

    if (image->type != COMPLEXTYPE) {
	VIP_Error_Msg("Complex2Float_Image: Not a COMPLEXTYPE image");
	return (NULL);
    }
    new_image = Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, FLOATTYPE);
    (void) Copy_Header(image, new_image);
    new_image->type = FLOATTYPE;
    for (i = image->rows - 1; i >= 0; i--) {
	for (j = image->cols - 1; j >= 0; j--) {
	    tempfloat = (float) Complex_Magnitude_2(&image->i.cx[i][j]);

	    new_image->i.f[i][j] = tempfloat;
	}
    }
    return (new_image);
}


/*- Construct_Complex_Image ------------------------------------------

Function to construct a COMPLEX TYPE image from two FLOAT TYPE images,
one being the real and the other the imaginary component of the new
image.

---------------------------------------------------------------------*/

IMAGE  *Construct_Complex_Image(IMAGE * Re, IMAGE * Im)
{
    register int row, col;
    IMAGE  *image;


    if (!Image_OK(Re) || !Image_OK(Im)) {
	VIP_Error_Msg("Construct_Complex_Image: received invalid image");
	return (NULL);
    }


    if (Re->rows != Im->rows ||
	Re->cols != Im->cols) {
	VIP_Error_Msg("Construct_Complex_Image: component images are of different sizes");
	return (NULL);
    }

    if (Re->type != FLOATTYPE ||
	Im->type != FLOATTYPE) {
	VIP_Error_Msg("Construct_Complex_Image: component image types are not FLOAT TYPE");
	return (NULL);
    }

    image = (IMAGE *) Allocate_Image(0, 0, Re->rows, Re->cols, COMPLEXTYPE);

    for (row = 0; row < Re->rows; row++)
	for (col = 0; col < Re->cols; col++) {
	    image->i.cx[row][col].r = Re->i.f[row][col];
	    image->i.cx[row][col].i = Im->i.f[row][col];
	}
    return (image);

}

/*- Extract_Real -------------------------------------------------------

Function to extract the real part of a complex image.  This
is returned as a FLOAT image.

-----------------------------------------------------------------------*/

IMAGE  *Extract_Real(IMAGE * image)
{
    int     row, col;
    IMAGE  *outimage;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Extract_Real: Received an invalid image");
	return (NULL);
    }

    if (image->type != COMPLEXTYPE) {
	VIP_Error_Msg("Extract_Real: Image must be of COMPLEXTYPE");
	return (NULL);
    }

    outimage = Allocate_Image(0, 0, image->rows, image->cols, FLOATTYPE);

    for (row = 0; row < image->rows; row++)
	for (col = 0; col < image->cols; col++)
	    outimage->i.f[row][col] = image->i.cx[row][col].r;

    return (outimage);
}


/*- Extract_Imaginary -------------------------------------------------------

Function to extract the imaginary part of a complex image.  This
is returned as a FLOAT image.

-----------------------------------------------------------------------*/

IMAGE  *Extract_Imaginary(IMAGE * image)
{
    int     row, col;
    IMAGE  *outimage;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Extract_Imaginary: Received an invalid image");
	return (NULL);
    }
    if (image->type != COMPLEXTYPE) {
	VIP_Error_Msg("Extract_Imaginary: Image must be of COMPLEXTYPE");
	return (NULL);
    }

    outimage = Allocate_Image(0, 0, image->rows, image->cols, FLOATTYPE);

    for (row = 0; row < image->rows; row++)
	for (col = 0; col < image->cols; col++)
	    outimage->i.f[row][col] = image->i.cx[row][col].i;

    return (outimage);
}
