/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            LIB
* MODULE:		rast.c -  Functions to create and manipulate
*                                 raster images.		
* REVISION:             3.3
* AUTHOR:               DH/GH
* CREATION DATE:	29 Nov 1988        
* REVISION DATE:	9/15/93        
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:		3.3
* REVISION DATE:	15 Sept 1993
* COMMENT:		Fixed up for GENERIC build
* BY:			CFF
*
* REVISION:		3.2
* REVISION DATE:	16 August 1993
* COMMENT:		Fixed up for DEC build
* BY:			CFF
*
* REVISION:		3.1
* REVISION DATE:	9 July 1992
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* REVISION:
* REVISION DATE:	14 Jan 1992
* COMMENT:		NEWVIP
* BY:			DH
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)rast.c	3.3 9/15/93";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>

#include "vip.h"

#ifdef SUN 
#include <rasterfile.h>
#include <pixrect/pixrect.h>
#endif

#ifdef DEC_ALPHA
#include "rasterfile.h"
#include "pixrect.h"
#endif

/* above 2 includes temporarily suppress compiler errors! */

#include "vipiofn.h"
#include "rast.h"
#include "rastfn.h"
#include "misc.h"
#include "miscfn.h"

/*- Ropen -----------------------------------------------------------

Read/write a raster image from/to a file (only part of the header
information will be read/write).
The function returns the pointer to the raster image that has been
read/written for a successful read/write, NULL otherwise.

--------------------------------------------------------------------*/

struct STRUCT_RASTER *Ropen(f, m)
char   *f;
int     m;
{
/*
    RASTER *im, *Ropen_read(), *Ropen_write();
*/
    RASTER *im;

    im = NULL;
    if (m == R)
	im = (RASTER *) Ropen_read(f);
    if (m == W)
	im = (RASTER *) Ropen_write(f);
    return ((struct STRUCT_RASTER *) im);
}


/*- Rdopen ----------------------------------------------------------

Same as Ropen but the read/write stream is the standard-input/
standard-output.

--------------------------------------------------------------------*/

struct STRUCT_RASTER *Rdopen(f, m)
int     f, m;
{
/*
    RASTER *im, *Rdopen_stdin(), *Rdopen_stdout();
*/
    RASTER *im;

    im = NULL;
    if (f == 0 && m == R)
	im = (RASTER *) Rdopen_stdin();
    if (f == 1 && m == W)
	im = (RASTER *) Rdopen_stdout();
    return ((struct STRUCT_RASTER *) im);
}


/*- Ropen_read ------------------------------------------------------

Read a raster image from a file (only part of the header information
will be read).
The function returns the pointer to the raster image that has been
read, NULL otherwise.

--------------------------------------------------------------------*/

struct STRUCT_RASTER *Ropen_read(f)
char   *f;
{
    RASTER *im;

    im = (RASTER *) malloc(sizeof(RASTER));
    im->fp = fopen(f, "r");
    im->mode = R;
    if (im->fp == NULL) {
	(void) fprintf(stderr, "read open fails for file \"%s\"!\n", f);
	im = NULL;
	return ((struct STRUCT_RASTER *) im);
    }
    return ((struct STRUCT_RASTER *) im);
}


/*- Rdopen_stdin ----------------------------------------------------

Same as Ropen_read but the raster image to be read is from the standard
input stream.

--------------------------------------------------------------------*/

struct STRUCT_RASTER *Rdopen_stdin()
{
    RASTER *im;

    im = (RASTER *) malloc(sizeof(RASTER));
    im->fp = stdin;
    im->mode = R;
    if (im->fp == NULL) {
	(void) fprintf(stderr, "read open fails for stdin!\n");
	im = NULL;
	return ((struct STRUCT_RASTER *) im);
    }
    return ((struct STRUCT_RASTER *) im);
}


/*- Ropen_Write -----------------------------------------------------

Write a raster image to a file (only part of the header information
will be read/write).
The function returns the pointer to the raster image that has been
written for a successful write, NULL otherwise.

--------------------------------------------------------------------*/

struct STRUCT_RASTER *Ropen_write(f)
char   *f;
{
    RASTER *im;

    im = (RASTER *) malloc(sizeof(RASTER));
    im->fp = fopen(f, "w");
    if (im->fp == NULL) {
	(void) fprintf(stderr, "write open fails for file \"%s\"!\n", f);
	im = NULL;
	return ((struct STRUCT_RASTER *) im);
    }

    im->mode = W;
    im->width = 0;
    im->height = 0;
    im->depth = 0;
    im->maplength = 0;
    im->maptype = RMT_NONE;

    if (im->fp == NULL)
	im = NULL;
    return ((struct STRUCT_RASTER * ) im);
}


/*- Rdopen_stdout ---------------------------------------------------

Same as Ropen_write but the raster image to be written is to the standard
output stream.

--------------------------------------------------------------------*/

struct STRUCT_RASTER *Rdopen_stdout()
{
    RASTER *im;

    im = (RASTER *) malloc(sizeof(RASTER));
    im->fp = stdout;
    if (im->fp == NULL) {
	(void) fprintf(stderr, "write open fails for stdout!\n");
	im = NULL;
	return ((struct STRUCT_RASTER *) im);
    }

    im->mode = W;
    im->width = 0;
    im->height = 0;
    im->depth = 0;
    im->maplength = 0;
    im->maptype = RMT_NONE;

    return ((struct STRUCT_RASTER * ) im);
}


/*- Rinitheader -----------------------------------------------------

Initialize image header.

--------------------------------------------------------------------*/

void    Rinitheader(im, height, width, depth)
RASTER *im;
int     height, width, depth;

/* Allocate space and initialize the given fields in the raster image header. */
{
    im->height = height;
    im->width = width;
    im->linelen = width;
    im->length = width*height;
    im->depth = depth;
}


/*- Rgetheader ------------------------------------------------------

Return header of a raster image.

--------------------------------------------------------------------*/

int     Rgetheader(im)
RASTER *im;
{
    int     i, rc, ll, extra;
    struct rasterfile rhead;

    rc = fread(&rhead, (sizeof(struct rasterfile)), 1, im->fp);

    if (rc <= 0) {
	(void) fprintf(stderr, "Rgetheader failed!\n");
	return (0);
    }
    if (rhead.ras_magic != RAS_MAGIC)
	return (0);

    im->height = rhead.ras_height;
    im->width = rhead.ras_width;
    im->depth = rhead.ras_depth;
    im->length = rhead.ras_length;
    im->type = rhead.ras_type;
    im->maptype = rhead.ras_maptype;
    im->maplength = rhead.ras_maplength;

    extra = (16 - ((im->width * im->depth) % 16)) % 16;
    ll = ((im->width * im->depth) + extra) / 8;

    im->linelen = ll;
    im->cache = (char *) malloc(ll);
    im->c_pixel = im->width + 1;
    im->c_pr = (struct pixrect *)  mem_point(im->width, 1, im->depth, im->cache);
    if (im->maplength > 0) {
	im->map = (char *)malloc(im->maplength);
	for (i = 0; i < im->maplength; i++)
	    im->map[i] = getc(im->fp);
    }
    return (1);
}


/*- Rputheader ------------------------------------------------------

Write header information stored in a raster image to file (the file
descriptor is stored in one of the fields in the raster image structure).

--------------------------------------------------------------------*/

int     Rputheader(im)
RASTER *im;
{
    int    rc, ll, extra;
    struct rasterfile rhead;

    extra = (16 - ((im->width * im->depth) % 16)) % 16;
    ll = ((im->width * im->depth) + extra) / 8;

    if (im->width == 0) {
	(void) fprintf(stderr, "rasterfile width is zero!\n");
	return (0);
    }
    if (im->height == 0) {
	(void) fprintf(stderr, "rasterfile height is zero!\n");
	return (0);
    }
    if (im->depth == 0) {
	(void) fprintf(stderr, "rasterfile depth is zero!\n");
	return (0);
    }

    rhead.ras_magic = RAS_MAGIC;
    rhead.ras_width = im->width;
    rhead.ras_height = im->height;
    rhead.ras_depth = im->depth;
    rhead.ras_length = ll * im->height;
    rhead.ras_type = RT_STANDARD;
    rhead.ras_maptype = im->maptype;
    rhead.ras_maplength = im->maplength;

    im->linelen = ll;
    im->cache = (char *)malloc(ll);
    im->c_pixel = 0;
    im->c_pr = (struct pixrect * ) mem_point(im->width, 1, im->depth, im->cache);

    rc = fwrite(&rhead, (sizeof(struct rasterfile)), 1, im->fp);

    if (rc <= 0) {
	(void) fprintf(stderr, "Rputheader failed!\n");
	return (0);
    }

    if (im->maptype != RMT_NONE && im->maplength > 0)
	(void) fwrite(im->map, im->maplength, 1, im->fp);
    return (1);
}


/*- Rgetpix ---------------------------------------------------------

Return the current pixel value (or colour index) of a raster image
(the current pixel position is stored in one of the fields of the
image structure).

--------------------------------------------------------------------*/

int     Rgetpix(im)
RASTER *im;
{
    int     pix;

    if (im->mode != R) {
	(void) fprintf(stderr, "attempt to read rasterfile opened for write!\n");
	return (-1);
    }

    if (im->c_pixel >= im->width) {
	(void) fread(im->cache, im->linelen, 1, im->fp);
	im->c_pixel = 0;
    }

    pix = pr_get(im->c_pr, im->c_pixel, 0);
    im->c_pixel++;
    return (pix);
}


/*- Rputpix ---------------------------------------------------------

Store a given pixel value (or colour index) to the current position
of a raster image (the current pixel position is stored in one of
the fields of the image structure).

--------------------------------------------------------------------*/

void    Rputpix(im, pix)
RASTER *im;
int     pix;
{
    if (im->mode != W) {
	(void) fprintf(stderr, "attempt to write rasterfile opened for read!\n");
	return;
    }

    pr_put(im->c_pr, im->c_pixel, 0, pix);
    im->c_pixel++;

    if (im->c_pixel >= im->width) {
	(void) fwrite(im->cache, im->linelen, 1, im->fp);
	im->c_pixel = 0;
    }
}


/*- Rclose ----------------------------------------------------------

Close the file descriptor pointed by a field of a raster image.

--------------------------------------------------------------------*/

void    Rclose(im)
RASTER *im;
{
    if (im->mode == W)
	(void) fflush(im->fp);
    (void) fclose(im->fp);
}


/*- Rinfo -----------------------------------------------------------

Output header information of a raster image to standard error.

--------------------------------------------------------------------*/

void    Rinfo(im)
RASTER *im;
{
    (void) fprintf(stderr, "  height    = %d\n", im->height);
    (void) fprintf(stderr, "  width     = %d\n", im->width);
    (void) fprintf(stderr, "  depth     = %d\n", im->depth);
    (void) fprintf(stderr, "  length    = %d\n", im->length);
    (void) fprintf(stderr, "  type      = %d\n", im->type);
    (void) fprintf(stderr, "  maptype   = %d\n", im->maptype);
    (void) fprintf(stderr, "  maplength = %d\n", im->maplength);
}


/*- Rinitmap --------------------------------------------------------

Initialize the colour map field of a raster image.

--------------------------------------------------------------------*/

void    Rinitmap(im, t, l)
RASTER *im;
int     t, l;
{
    im->maptype = t;
    im->maplength = l;
    im->map = (char *) malloc(l);
}


/*- Rputmap ---------------------------------------------------------

Store a given colour (r,g,b) to the i-th location in the colour map
of a raster image.

--------------------------------------------------------------------*/

void    Rputmap(im, i, r, g, b)
RASTER *im;
int     i, r, g, b;
{
    int     x;

    switch (im->maptype) {
    case RMT_NONE:
	(void) fprintf(stderr, "can't put in a null map!\n");
	break;
    case RMT_RAW:
	im->map[i] = r;
	break;
    case RMT_EQUAL_RGB:
	x = i * 3;
	im->map[x++] = r;
	im->map[x++] = g;
	im->map[x] = b;
	break;
    default:
	(void) fprintf(stderr, "unknown map type in rasterfile!\n");
    }
}


/*- Rgetmap ---------------------------------------------------------

Return the i-th colour stored in the colour map of a raster image.

--------------------------------------------------------------------*/

void    Rgetmap(im, i, r, g, b)
RASTER *im;
int     i, *r, *g, *b;
{
    int     x;

    switch (im->maptype) {
    case RMT_NONE:
	(void) fprintf(stderr, "can't get from a null map!\n");
	break;
    case RMT_RAW:
	*r = im->map[i];
	break;
    case RMT_EQUAL_RGB:
	x = i * 3;
	*r = im->map[x++];
	*g = im->map[x++];
	*b = im->map[x];
	break;
    default:
	(void) fprintf(stderr, "unknown map type in rasterfile!\n");
    }
}


/*- Rgetmappedpix ---------------------------------------------------

Return the red, green, and blue components at the current pixel position
in a raster image.

--------------------------------------------------------------------*/

void    Rgetmappedpix(im, r, g, b)
RASTER *im;
int    *r, *g, *b;
{
    int     pix, x;

    if (im->mode != R) {
	(void) fprintf(stderr, "attempt to read rasterfile opened for write!\n");
	return;
    }

    if (im->c_pixel >= im->width) {
	(void) fread(im->cache, im->linelen, 1, im->fp);
	im->c_pixel = 0;
    }

    pix = pr_get(im->c_pr, im->c_pixel, 0);
    im->c_pixel++;

    switch (im->maptype) {
    case RMT_RAW:
	*r = im->map[pix];
	break;
    case RMT_EQUAL_RGB:
	x = pix * 3;
	*r = im->map[x++];
	*g = im->map[x++];
	*b = im->map[x];
	break;
    default:
	*r = pix;
	break;
    }
}


/*- Is_Colour_Raster ------------------------------------------------

Return 1 if the given raster image is coloured, 0 otherwise.
Note that this function must be called after function Rgetheader.

--------------------------------------------------------------------*/

int     Is_Colour_Raster(ras)
RASTER *ras;
{
    register int i;
    int     length, red;

    (void) fprintf(stderr, "ras->maplength=%d\n", ras->maplength);
    length = ras->maplength / 3;
    for (i = length - 1; i >= 0; i--) {
	red = ras->map[i];
	/* check if red = green = blue */
	if (red != ras->map[i + length] ||
	    red != ras->map[i + length + length])
	    break;
    }
    if (i >= 0)
	return (1);
    else
	return (0);
}


/*- Read_Raster_Image -----------------------------------------------

Read a raster image from disk.

--------------------------------------------------------------------*/

struct STRUCT_RASTER *Read_Raster_Image(fname)
char   *fname;
{
    RASTER *ras;

    if (fname)			/* input from disk */
	ras = (RASTER *) Ropen(fname, R);
    else
	ras = (RASTER *) Rdopen(0, R);	/* input from terminal */

    if (ras == NULL)
	/* Fail to open the file */
	VIP_Error_Msg("Fail to open input file for a raster image");
    /* Get raster image header */
    else if (!Rgetheader(ras)) {
	/* error on ras_magic number */
	Rclose(ras);
	VIP_Error_Msg("Error in ras_magic number in input file");
    }
    else if (ras->height <= 0 || ras->width <= 0) {
	/* image size = 0 */
	Rclose(ras);
	VIP_Error_Msg("Error in input raster image size");
    }
    return ((struct STRUCT_RASTER *) ras);
}


/*- Raster_To_VIP ---------------------------------------------------

Return a VIP image of either BYTETYPE or RGBTYPE from a raster image.
Note that the file containing the raster image must still be open
when this function is called.

--------------------------------------------------------------------*/

IMAGE  *Raster_To_VIP(ras)
RASTER *ras;
{
    IMAGE  *im;
    unsigned char *rp;
    register int rr, cc, nr, nc, i;
    int     mapleng, iscolour, colindex, intensity;

    if (iscolour = Is_Colour_Raster(ras)) {
	if (!(im = ( IMAGE * ) Allocate_Image(0, 0, ras->height, ras->width, RGBTYPE))) {
	    VIP_Error_Msg("Raster_To_VIP: out of memory");
	    return (NULL);
	}
    }
    else {
	if (!(im = ( IMAGE * ) Allocate_Image(0, 0, ras->height, ras->width, BYTETYPE))) {
	    VIP_Error_Msg("Raster_To_VIP: out of memory");
	    return (NULL);
	}
    }
    nr = ras->height;
    nc = ras->width;
    mapleng = ras->maplength / 3;
    for (rr = 0; rr < nr; rr++) {
	(void) fread(ras->cache, ras->linelen, 1, ras->fp);
	rp = im->i.c[rr];
	for (cc = 0; cc < nc; cc++, rp++) {
	    colindex = (unsigned char) (pr_get(ras->c_pr, cc, 0));
	    if (iscolour) {
		/* a colour image */
		rp = im->i.rgb[rr][cc];
		for (i = 0; i < 3; i++, colindex += mapleng)
		    *rp++ = ras->map[colindex];
	    }
	    else {
		/* a grey scale image */
		/* calculate average intensity value */
		for (i = intensity = 0; i < 3; i++, colindex += mapleng)
		    intensity += ras->map[colindex];
/*crashes above as ras->map == nil ptr*/
		intensity /= 3;
		im->i.c[rr][cc] = intensity;
	    }
	}
    }
    return (im);
}


/*- VIP_Raster_Colour_Map -------------------------------------------

From a VIP image, compute the colour map (very computational intensive
process) to be used for a raster image.
Notes: 1. The colour map should be of length 256*3.
       2. im is assumed to be of RGBTYPE (no checking is done in this
          function).

--------------------------------------------------------------------*/

int     VIP_Raster_Colour_Map(im, map)
IMAGE  *im;
unsigned char map[];
{
    register int rr, cc, indx, i;
    unsigned char r, g, b, *t, *red_map, *green_map, *blue_map;

    red_map = map;
    green_map = map + 256;
    blue_map = map + 256 + 256;
    indx = 0;
    for (rr = im->rows - 1; rr >= 0; rr--)
	for (cc = im->cols - 1; cc >= 0; cc--) {
	    t = im->i.rgb[rr][cc];
	    r = *t++;
	    g = *t++;
	    b = *t;
	    for (i = indx - 1; i >= 0; i--)
		if (red_map[i] == r && green_map[i] == g &&
		    blue_map[i] == b)
		    break;
	    if (i < 0) {
		if (indx == 256) {
		    VIP_Error_Msg("VIP_Raster_Colour_Map: too many colours");
		    return (0);
		}
		/* OK, assign new colours found to the colour map */
		red_map[indx] = r;
		green_map[indx] = g;
		blue_map[indx] = b;
		indx++;
	    }
	}
    return(1);
}


/*- Raster_Colour_Index ---------------------------------------------

Given a colour C=(r, g, b) and a colour map, this function returns
the index number of the colour in the colour map that matches C.
If no matching is found, the function returns -1.
Note: The colour map should be of length 256*3.

--------------------------------------------------------------------*/

int Raster_Colour_Index(r, g, b, map)
unsigned char r, g, b, map[];
{
	register int i;

	for (i=0; i < 256; i++)
		if (map[i] == r &&
		map[i+256] == g &&
		map[i+512] == b)
		break;
	return (i);
}


/*- VIP_To_Raster_File ----------------------------------------------

Given a VIP image, this function returns a raster image saved
to a file.

--------------------------------------------------------------------*/

int     VIP_To_Raster_File(im, fname)
IMAGE  *im;
char   *fname;
{
    register int rr, cc, nr, nc;
    int     indx;
    RASTER *ras;
    unsigned char *rp;
    unsigned char *t;

    if (fname)
	ras = (RASTER *) Ropen(fname, W);
    else
	ras = (RASTER *) Rdopen_stdout();

    if (ras == NULL) {
	/* Fail to open the file */
	VIP_Error_Msg("Fail to open output raster file");
	return (0);
    }
    else if (im->rows <= 0 || im->cols <= 0) {
	VIP_Error_Msg("Error in input image size");
	return (0);
    }

    /* colour map is piggyback at the tail of the raster image */
    nr = im->rows;
    nc = im->cols;
    switch (im->type) {
    case BYTETYPE:
	/* Initialize and put header */
	Rinitheader(ras, im->rows, im->cols, 8);
	Rinitmap(ras, RMT_EQUAL_RGB, 256 * 3);
	ras->type = RT_STANDARD;

	for (rr = 0, t = (unsigned char *) ras->map; rr < 3; rr++)
	    for (cc = 0; cc < 256; cc++) {
		*t = (char) cc;
		t++;
	    }
	ras->maplength = 256 * 3;
	/* write raster image header */
	(void) Rputheader(ras);
	for (rr = 0; rr < nr; rr++) {
	    rp = im->i.c[rr];
	    for (cc = 0; cc < nc; cc++, rp++)
		pr_put(ras->c_pr, cc, 0, (int) (*rp));
	    (void) fwrite(ras->cache, ras->linelen, 1, ras->fp);
	}
	break;
    case RGBTYPE:
	/* Initialize and put header */
	Rinitheader(ras, im->rows, im->cols, 8);
	Rinitmap(ras, RMT_EQUAL_RGB, 256 * 3);
	ras->type = RT_STANDARD;

	if (VIP_Raster_Colour_Map(im, (unsigned char * )ras->map) == OK) {
	    ras->maplength = 256 * 3;
	    (void) Rputheader(ras);
	    for (rr = 0; rr < nr; rr++) {
		for (cc = 0; cc < nc; cc++) {
		    rp = im->i.rgb[rr][cc];
		    indx = Raster_Colour_Index(*rp, *(rp + 1), *(rp + 2), ( unsigned char * )ras->map);
		    /*
		     * no need to check return value of function
		     * Raster_Colour_Index as we have already gone
		     * past the test at function
		     * VIP_Raster_Colour_Map.
		     */
		    pr_put(ras->c_pr, cc, 0, (int) indx);
		}
	        (void) fwrite(ras->cache, ras->linelen, 1, ras->fp);
            }
	}
	break;
    default:
	VIP_Error_Msg("VIP_To_Raster_File: can only process BYTETYPE and RGBTYPE images");
    }

    Rclose(ras);
}

