/* control.c      Distribution 1.2   91/1/28   Scry */

/*   The Scry system is copyright (C) 1988-1991 Regents  of  the
University  of  California.   Anyone may reproduce ``Scry'',
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  Scry  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  Scry  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     Scry is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
Scry, or any part thereof. */


/* This file contains routines to establish and close a connection
   with the server */

/* scry_connect:	establishes client side and initializes server
   check_host_type:	check whether symbolic name or Internet
			address of server has been provided
   scry_close:  closes client and relinquishes control of server
   abn_close_scry:  on interrupt, close client and exit
   xdr_filename:	XDR encodes name of Anima file to be created
			on server
   sc_send_file:	communicates name of Anima file to be
			created on server    */

#include <stdio.h>
#include <math.h>
#include <scry_anima.h>
#include <scry_image.h>
#include <scry_client.h>
#include <scry_limits.h>
#include <signal.h>
#include <ctype.h>

#define PCPORT 2000	/* hard-wired PC port */
#define TESTVERS 1	/* version */

    /* client handle */
CLIENT *S_client ;

    /* give up after certain amount of time */
struct timeval S_total_timeout ;

    /* closing routine called if program interrupted */
void abn_close_scry() ;

int S_during_rpc = 0 ;
int S_make_closing_call = 0 ;


/* establishes client side and initializes server */

int
scry_connect (host,prognum)

char *host ;		/* symbolic name of server machine */
int prognum ;		/* remote program number */

{
    struct hostent *hp ;		/* server machine information */
    struct sockaddr_in server_addr ;	/* server communication information */
    int socket = RPC_ANYSOCK ;		/* don't care what socket number is */
    enum clnt_stat err ;		/* error status of RPC call */
    bool_t unused ;			/* if true, PC not in use */
    int is_int_addr ;	/* is in Internet address form */
    unsigned long internet_addr ;	/* Internet address */


	/* check whether symbolic name or Internet address */
    check_host_type (host,&is_int_addr,&internet_addr) ;
	/* if "dotted" Internet address */
    if (is_int_addr)
	server_addr.sin_addr.s_addr = internet_addr ;
    else	/* symbolic name */
    {
        if ((hp = gethostbyname(host)) == NULL)
        {
            fprintf (stderr, "can't find server address '%s'\n",host) ;
            return (0) ;
        }
        memcpy((caddr_t)&server_addr.sin_addr, hp->h_addr, hp->h_length) ;
    }
	/* use Internet protocol */
    server_addr.sin_family = AF_INET ;
    S_total_timeout.tv_sec = 240 ;	/* give up after 240 seconds */
    S_total_timeout.tv_usec = 0 ;
    if (prognum < S_MIN_RPC_PROGNUM)
    {
	fprintf (stderr,"%d is an illegal RPC program number\n",prognum) ;
	return(0) ;
    }

	/* if PC server program number */
    if (prognum == S_PCPROG)
	    /* hard-wire port number on PC */
        server_addr.sin_port = htons (PCPORT) ;
    else
	server_addr.sin_port = 0 ;
        if ((S_client = clnttcp_create(&server_addr, prognum, TESTVERS, 
	    &socket,0,0)) == NULL)
        {
	    fprintf (stderr,"connection refused: check that server is running\n") ;
	    fprintf (stderr,"or that the client and server program numbers are the same\n") ;
	    return (0) ;
        }
       /* arm signals */
    signal(SIGHUP, abn_close_scry);
    signal(SIGQUIT, abn_close_scry);
	/* arm signal with routine to call if program ^C'd */
    signal(SIGINT, abn_close_scry);
    signal(SIGBUS, abn_close_scry);
	/* arm signal with routine to call if segmentation fault */
    signal(SIGSEGV, abn_close_scry);
	/* initialize Anima footer variables (needed if saving Anima
	   file locally */
    S_anima_file = NULL ;
    S_anima_filename[0] = '\0' ;
    S_anima_bytes = 0 ;
    S_anima_count = 0 ;
    S_a_index = NULL ;

    return (1) ;
}



/* check whether symbolic name or Internet address of server */

check_host_type (host, inter_addr, coded)

char *host ;		/* reference to server */
int *inter_addr ;	/* Internet address or not */
unsigned long *coded ;	/* Internet address */

{
    int i ;
    int j = 0 ;
    int parse_ptr = 0 ;
    char parse_addr[4][5] ;
    int temp_int ;

    *coded = 0 ;
    *inter_addr = 0 ;
	/* see if contains 4 numbers and 3 periods */
    for (i = 0 ; i < strlen(host) ; i++)
    {
	if (host[i] == '.')
	{
	    if (j == 0)
	    {
		*inter_addr = 0 ;
		break ;
	    }
	    else
	    {
		parse_addr[parse_ptr][j] = '\0' ;
		j = 0 ;
		++parse_ptr ;
		if (parse_ptr > 3)
		{
		    *inter_addr = 0 ;
		    break ;
		}
	    }
	}
	else if (j < 3)
	{
	    if (!isdigit(host[i]))
	    {
		*inter_addr = 0 ;
		break ;
	    }
	    parse_addr[parse_ptr][j++] = host[i] ;
	    *inter_addr = 1 ;
	}
	else
	{
	    *inter_addr = 0 ;
	    break ;
	}
    }
    if (*inter_addr && (parse_ptr == 3))
    {
        parse_addr[3][j] = '\0' ;
        if ((temp_int = atoi(parse_addr[0])) > 255)
            *inter_addr = 0 ;
        else
        {
            *coded = (unsigned char) temp_int ;
		/* convert to long form */
            for (i = 1 ; i < 4 ; i++)
            {
                *coded <<= 8 ;
                if ((temp_int = atoi(parse_addr[i])) > 255)
                {
                    *inter_addr = 0 ;
                    break ;
                }
                else
                    *coded |= (unsigned char) temp_int ;
            }
        }
    }
}




/* closes client and relinquishes control of server */

scry_close ()

{
    enum clnt_stat err ;	/* error status of RPC call */
    int normal = 1 ;
    unsigned char footer[7] ;
    int i ;

	/* if not saving an Anima file locally */
    if (S_client != NULL)
    {
	S_during_rpc = 2 ;
	    /* RPC call to relinquish control of server */
        if ((int) (err = clnt_call(S_client,S_CLOSEPROC,xdr_int,&normal,
			 xdr_void,(char *) 0,S_total_timeout)) != 0)
        {
	    clnt_perrno ((int) err) ;
	    fprintf (stderr,"can't make S_CLOSEPROC call\n") ;
	    clnt_destroy (S_client) ;
	    exit (0) ;
        }
	    /* destroy client handle */
        clnt_destroy (S_client) ;
    }
    return (1) ;
}




/* upon interrupt, closes client and exits program */

void
abn_close_scry (sig)

int sig ;

{
    enum clnt_stat err ;	/* error status of RPC call */
    int normal = 0 ;

    if (S_during_rpc == 0)
    {
	S_during_rpc = 2 ;
	    /* RPC call to relinquish control of server */
        if ((int) (err = clnt_call(S_client,S_CLOSEPROC,xdr_int,&normal,
			     xdr_void,(char *) 0,S_total_timeout)) != 0)
        {
	    clnt_perrno ((int) err) ;
	    fprintf (stderr,"can't make S_CLOSEPROC call\n") ;
        }
	    /* destroy client handle */
        clnt_destroy (S_client) ;
        fprintf(stderr,"Exiting on signal %d.\n",sig) ;
 
	    /* exit program */
        exit (-1) ;
    }
    else if (S_during_rpc != 2)
    {
	fprintf (stderr,"signal caught during RPC: will exit on its completion\n") ;
	/* CHANGE: need to set signal proc to NULL here */
       /* arm signals */
    signal(SIGHUP, NULL);
    signal(SIGQUIT, NULL);
	/* arm signal with routine to call if program ^C'd */
    signal(SIGINT, NULL);
    signal(SIGBUS, NULL);
	/* arm signal with routine to call if segmentation fault */
    signal(SIGSEGV, NULL);
	S_make_closing_call = 1 ;
    }
}




/* after interrupt and current RPC finishes, closes client and exits program */

void
scry_abn_exit()

{
    enum clnt_stat err ;	/* error status of RPC call */
    int normal = 0 ;

        /* RPC call to relinquish control of server */
    if ((int) (err = clnt_call(S_client,S_CLOSEPROC,xdr_int,&normal,
                         xdr_void,(char *) 0,S_total_timeout)) != 0)
    {
        clnt_perrno ((int) err) ;
        fprintf (stderr,"can't make S_CLOSEPROC call\n") ;
    }
        /* destroy client handle */
    clnt_destroy (S_client) ;
    fprintf (stderr,"exiting\n") ;
        /* exit program */
    exit (-1) ;
}



struct fileinfo
{
    int total_frames ;
    char filename[80] ;
} ;


/* XDR encode Anima file name to be used on server */

bool_t 
xdr_filesave (xdrsp,file_stuff)

XDR *xdrsp ;		/* XDR handle */
struct fileinfo *file_stuff ;

{
    if (!xdr_int(xdrsp,&(file_stuff->total_frames)))
	return (0) ;
    if (!xdr_opaque(xdrsp,file_stuff->filename,80))
	return (0) ;
    return (1) ;
}




/* inform server of name to save Anima file under */

sc_send_file(filename,total_frames)

char *filename ;
int total_frames ;

{
    enum clnt_stat err ;
    bool_t xdr_filesave() ;
    struct fileinfo file_stuff ;

    strcpy(file_stuff.filename,filename) ;
    file_stuff.total_frames = total_frames ;
    S_image_info.current_frame_to_save = 0 ;
    S_image_info.total_frames = total_frames ;
    S_during_rpc = 1 ;
    if ((int) (err = clnt_call(S_client,S_FILENAMEPROC,xdr_filesave,&file_stuff,
			 xdr_void,(char *) 0,S_total_timeout)) != 0)
    {
	    clnt_perrno ((int) err) ;
	    fprintf (stderr,"can't make S_FILENAMEPROC call\n") ;
	    clnt_destroy (S_client) ;
	    return ;
    }
    S_during_rpc = 0 ;
    if (S_make_closing_call)
	scry_abn_exit() ;
}
