/* cccalg.c      Distribution 1.2   91/1/28   Scry */

/*   The Scry system is copyright (C) 1988-1991 Regents  of  the
University  of  California.   Anyone may reproduce ``Scry'',
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  Scry  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  Scry  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     Scry is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
Scry, or any part thereof. */


#include 	 <stdio.h> 
#include	"imglib.h"
#include        <scry_image.h>


/* ccc_compress:	CCC and run length compress an image
   reassign_ccc:	find nearest colors in color map to the ones
			associated with the bitmaps
			and do run length encoding for entire image
   init_run_length:	initialize run length encoding information
   btc_alg:		generate BTC 4x4 bitmaps and replace pixels in
			the image with the associated colors in the bitmaps
   ccc_run_length:	find nearest colors in color map to the ones
			associated with the bitmaps and
			do run length encoding for block of scan lines */

#define		SIZBLOC		4 	/* pixels per side of block */
#define		NB_BLOC		8	/* group of blocks to take together */

    /* number of pixels per block */
#define		BLOC		SIZBLOC*SIZBLOC
#define		DEPTH		3	/* color resolution in bytes */

unsigned char *bufout ;		/* storage for 4 compressed scan lines */
unsigned char *precod ;		/* intermediate storage for result of
				   BTC compression */

int block_width_num ;		/* number of blocks in scan line */
int block_height_num ;		/* number of blocks in image height */
int decal ;			/* skips to proper place in next line */

static int ccc_prev_color ;	/* color in a run */
static unsigned short ccc_run_ctr ;	/* run ctr */
static int ccc_finished ;	/* finished with a run */

static int ccc_total ;		/* total bytes in CCC and run length
				   compressed image */


/* CCC and run length compress an image (IFF compression format) */

ccc_compress (image,compimage,datasize,height,width)

unsigned char *image ;
unsigned char *compimage ;
int *datasize ;
int height ;
int width; 

{
    int i ;

    ccc_total = *datasize = 0 ;
    block_width_num  = width  / SIZBLOC ;
    block_height_num = height / SIZBLOC ;
    decal = (width - SIZBLOC) * DEPTH ;
    bufout = (unsigned char *) calloc(block_width_num*8,sizeof(char)) ;
	/* stores result of first step:  BTC encoding */
    precod = (unsigned char *) calloc (block_width_num*block_height_num*8,
				       sizeof(char)) ;
	/* performs first step:  BTC encoding */
    btc_alg (image,height,width) ;					
	/* quantizes colors resulting from BTC encoding to at most S_maxcol
	   entries using Paul Raveling's modified code */
    S_mapnum = quantize(image,compimage,S_map,S_maxcol,height,width,1) ;
	/* for TAAC compatibility:  first color must be black */
    for (i = (S_mapnum - 1) ; i >= 0 ; i--)
    {
	S_map[i+1][S_RED] = S_map[i][S_RED] ;
	S_map[i+1][S_GREEN] = S_map[i][S_GREEN] ;
	S_map[i+1][S_BLUE] = S_map[i][S_BLUE] ;
    }
    S_mapnum++ ;
    S_map[0][S_RED] = 0 ;
    S_map[0][S_GREEN] = 0 ;
    S_map[0][S_BLUE] = 0 ;
	/* total bytes in compressed image */
    *datasize = ccc_total ;
}



/* quantization has been performed on the colors resulting from
   generating the 4x4 blocks; this routine assigns quantized
   colors to their associated bitmaps and does run length
   encoding for the entire image */

reassign_ccc (compdata,height,width,cpa,root)

unsigned char *compdata ;	/* compressed image */
int height ;
int width; 
    /* following two variables are globals used in Paul Raveling's
       modified quantization */
Ctnode **cpa ;
Ctnode *root ;

{
       /* pointer to group of 4 BTC compressed scan lines */
    unsigned char *pt ;
	/* number of bytes in compressed group */
    short to_send ;
	/* do run length encoding */
    unsigned char *ccc_run_length () ;
	/* current group of 4 scan lines */
    short line ;
	/* pointer into resulting compressed image */
    unsigned char *part ;
	/* number of bytes in 4 compressed scan lines */
    int total_sent = 0 ;
	/* last group of 4 scan lines to process */
    int last ;
    int i ;

        /* initialize run length encoding variables */
    init_run_length () ;
	/* precod stores result of previous BTC encoding */
    pt = precod ;
	/* compdata stores resulting compressed image */
    part = compdata ;
       /* for each group of 4 scan lines to be compressed */
    for ( line=0 ; line<height ; line+=SIZBLOC )
    {
	if (line == (height - SIZBLOC))
	    last = 1 ;
	else

	    last = 0 ;
		/* assign quantized colors and do run length encoding */
        pt = ccc_run_length ( pt , &to_send, last , cpa, root) ;
	    /* copy into outgoing compressed image */
        memcpy (part,bufout,(int) to_send) ;
	    /* update current number of bytes in compressed image */
        total_sent += to_send ;
        part = &(compdata[total_sent]) ;
	    /* update byte total */
        ccc_total += (int) to_send ;
    }
    free(bufout) ;
    free(precod) ;
}



/* initialize run length encoding information */

init_run_length()

{
    ccc_prev_color = -1 ;
    ccc_run_ctr = 1 ;
    ccc_finished = 1 ;
}



/* CCC is a combination of BTC (Block Truncation Coding) and
   color quantization.  This routine performs the BTC step,
   saves the results in the "precod" data structure, and
   replaces pixel colors in the image with those generated
   by BTC. */

btc_alg(image,height,width)

unsigned char *image ;
int height ;
int width ;

{
        /* Blocs built from the pixels blocs */
    int green[BLOC], red[BLOC], blue[BLOC], lumi[BLOC];
        /* Pointers into the previous arrays */
    register int *ptgreen, *ptblue, *ptred, *ptlumi;
        /* Means of the colors */
    int	red1, green1, blue1, red0, green0, blue0;
    register unsigned char *ptc;	/* input pointer */
    register unsigned char *ptout;	/* output pointer */
    register unsigned short bitmap;	/* Bitmap of bloc according to luminance */
    register unsigned char *ptbloc;	/* Pointer to beginning of blocs */ 
    unsigned short shifter ;		/* steps through the bit map */
    unsigned short on_bit ;		/* determines which color */
    int mean ;			/* Mean of luminance in the bloc */
    int nb;			/* Number of pixel with lumi > mean */
    int lin, k, l, m ;
    
    ptout = precod;
	/* for all groups of 4 scan lines */
    for(lin = 0; lin < height; lin += SIZBLOC)
    {
	ptbloc = &(image[lin*width*DEPTH]);
	    /* for all blocks in row */
	for(k = 0; k < block_width_num; k++, ptbloc += (SIZBLOC*DEPTH))
	{
	    ptc = ptbloc; 
	    ptred = red; ptgreen = green; ptblue = blue;
	        /* find all colors in 4x4 block */
	    for(l = 0; l < SIZBLOC; l++, ptc += decal)
		for(m = 0; m < SIZBLOC; m++, ptred++, ptgreen++, ptblue++)
		{
		    *ptred = (int)*ptc++ ;
		    *ptgreen = (int)*ptc++ ;
		    *ptblue = (int)*ptc++ ;
		} 
	            /* find the luminance and its mean */
	    ptlumi = lumi;
	    ptred = red; ptgreen = green; ptblue = blue;
	    mean = 0;
	    for(m = 0; m < BLOC; m++, ptlumi++, ptred++, ptgreen++, ptblue++)
	    {
		*ptlumi = 30*(*ptred) + 59*(*ptgreen) + 11*(*ptblue);
		mean += *ptlumi;
	    }
	    mean /= BLOC;
	        /* build the bitmap */
	    ptlumi = lumi;
	    ptred = red; ptgreen = green; ptblue = blue;
	    red1 = 0; green1 = 0; blue1 = 0;
	    red0 = 0; green0 = 0; blue0 = 0;
	    nb = 0; bitmap = 0;
	    for(m = 0; m < BLOC; m++, ptlumi++, ptred++, ptgreen++, ptblue++)
	    {
		    /* if pixel luminance above average luminance */
		if(*ptlumi > mean)
		{
		    bitmap |= (1 << m);
		    red1 += *ptred; green1 += *ptgreen; blue1 += *ptblue;
		    nb++;
		}
		else
		{
		    red0 += *ptred; green0 += *ptgreen; blue0 += *ptblue;
		}
	    }
		/* if not all one color */
	    if(nb)
	    {
		red1 /= nb; green1 /= nb; blue1 /= nb;
		nb = BLOC - nb;
		red0 /= nb; green0 /= nb; blue0 /= nb; 
		    /* intermediate storage bitmap and the two colors */
		    /* colors will likely be reassigned based on
		       color quantization in ccc_run_length */
		*ptout = (bitmap >> 8) & 0xff ; ++ptout ;
		*ptout = bitmap & 0xff ; ++ptout ;
		*ptout++ = (unsigned char) red0 ;
		*ptout++ = (unsigned char) green0 ;
		*ptout++ = (unsigned char) blue0 ;
		*ptout++ = (unsigned char) red1 ;
		*ptout++ = (unsigned char) green1 ;
		*ptout++ = (unsigned char) blue1 ;
		ptc = ptbloc ;
		shifter = 0 ;
		    /* replace pixels in the image with their BTC
		       colors */
	        for(l = 0; l < SIZBLOC; l++, ptc += decal)
		    for(m = 0; m < SIZBLOC; m++)
		    {
		        if ((bitmap >> shifter) & 0x1)
			    on_bit = 1 ;
		        else
			    on_bit = 0 ;
		        *ptc++ = (unsigned char) (on_bit ? red1 : red0) ;
		        *ptc++ = (unsigned char) (on_bit ? green1 : green0) ;
		        *ptc++ = (unsigned char) (on_bit ? blue1 : blue0) ;
		        ++shifter ;
		    } 
	    }
	    else
	    {		/* only one color in bit map */
		red0 /= BLOC; green0 /= BLOC; blue0 /= BLOC; 
		*ptout++ = 0;
		*ptout++ = 0;
		*ptout++ = (unsigned char) red0 ;
		*ptout++ = (unsigned char) green0 ;
		*ptout++ = (unsigned char) blue0 ;
		ptc = ptbloc ;
		    /* replace colors in the block in the image
		       with BTC colors */
	        for(l = 0; l < SIZBLOC; l++, ptc += decal)
		for(m = 0; m < SIZBLOC; m++)
		{
		    *ptc++ = (unsigned char) red0 ;
		    *ptc++ = (unsigned char) green0 ;
		    *ptc++ = (unsigned char) blue0 ;
		}
	    }			/* End processing the bloc */
	}			/* End loop in line group */
    }				/* End inside the group of lines */
}



/* Using the "precod" data structure generated by the routine btc_alg,
   assigns quantized colors to bitmaps and does run length
   encoding for block of (4) scan lines. */

unsigned char *ccc_run_length(pt, to_send, last, cpa, root) 

short *to_send;		/* total bytes in 4 compressed lines */
unsigned char *pt;	/* pointer to intermediate storage */
int last ;		/* whether last group of 4 lines to process */

Ctnode **cpa;		/* global variables for use with */
Ctnode *root ;		/* Paul Raveling's code */

{     
    register unsigned short bitmap;	/* bit map for 4x4 block */
    register unsigned char *ptout;	/* output ptr */
    int	i;
    unsigned char tempind1, tempind2;
    unsigned char r0,g0,b0,r1,g1,b1 ;
    unsigned char find_isi_color() ;	/* look up closest color */
    
    init_run_length() ;		/* initialize run length variables */
    ptout = bufout;
    *to_send = 0;
	/* for all blocks in group of 4 scan lines */
    for(i = 0; i < block_width_num; i++)
    {
	bitmap = *pt++ ;
	bitmap = (bitmap << 8) | *pt++ ;
	if(bitmap)	/* if 2 colors */
	{
		/* if at end of a run */
	    if ((!ccc_finished) && (ccc_prev_color >= 0))
	    {
		*ptout = (unsigned char) (ccc_prev_color+1) ;
	        ++ptout;
	        *ptout = (unsigned char) (ccc_prev_color+1) ;
	        ++ptout;
	        *ptout = (ccc_run_ctr >> 8) & 0xff; ++ptout;
	        *ptout = ccc_run_ctr & 0xff; ++ptout;
	        *to_send += 4;
	    }
	    ccc_prev_color = -1 ;
	    r0 = *pt++ ;
	    g0 = *pt++ ;
	    b0 = *pt++ ;
	    r1 = *pt++ ;
	    g1 = *pt++ ;
	    b1 = *pt++ ;
		/* look up closest colors as result of
		   quantization */
	    tempind1 = find_isi_color (r0,g0,b0,cpa,root) ;
	    tempind2 = find_isi_color (r1,g1,b1,cpa,root) ;
	    ccc_run_ctr = 1 ;
	    ccc_finished = 1 ;
		/* if colors are now the same */
	    if (tempind1 == tempind2)
	    {
		*ptout = (unsigned char) (tempind1+1) ;
	        ++ptout;
	        *ptout = (unsigned char) (tempind2+1) ;
	        ++ptout;
	        *ptout = (ccc_run_ctr >> 8) & 0xff; ++ptout;
	        *ptout = ccc_run_ctr & 0xff; ++ptout;
	        *to_send += 4;
	    }
	    else
	    {
                *ptout = (unsigned char)(tempind1+1);
                ++ptout;
                *ptout = (unsigned char)(tempind2+1);
                ++ptout;
	        *ptout = (bitmap >> 8) & 0xff; ++ptout;
	        *ptout = bitmap & 0xff; ++ptout;
	        *to_send += 4;
	    }
	}
	else 
	{
	        /* try run-length encoding */
	    r0 = *pt++ ;
	    g0 = *pt++ ;
	    b0 = *pt++ ;
		/* find color as result of quantization */
	    tempind1 = find_isi_color(r0,g0,b0,cpa,root) ;
	    ccc_finished = 0 ;
		/* if run is continuing */
	    if (tempind1 == ccc_prev_color)
	    {
		++ccc_run_ctr ;
	    }
	    else if (ccc_prev_color >= 0) 	/* finish run */
	    {
		*ptout = (unsigned char) (ccc_prev_color+1) ;
	        ++ptout;
	        *ptout = (unsigned char) (ccc_prev_color+1) ;
	        ++ptout;
	        *ptout = (ccc_run_ctr >> 8) & 0xff; ++ptout;
	        *ptout = ccc_run_ctr & 0xff; ++ptout;
	        *to_send += 4;
		ccc_run_ctr = 1 ;
		ccc_prev_color = tempind1 ;
	    }
	    else	/* at start of run */
                ccc_prev_color = tempind1 ;
	}
    }
	/* finish up if necessary at end of row */
    if (!ccc_finished)
    {
        *ptout = (unsigned char) (ccc_prev_color+1) ;
        ++ptout;
        *ptout = (unsigned char) (ccc_prev_color+1) ;
        ++ptout;
        *ptout = (ccc_run_ctr >> 8) & 0xff; ++ptout;
        *ptout = ccc_run_ctr & 0xff; ++ptout;
        *to_send += 4;
    }
    return(pt);
}
