
/*
 * segal.c        Segemtation Analysis Tool      -Brian Tierney  LBL
 *
 *  This in the main routine for segal, a binary image mask building/editing
 *          analysis tool.
 *
 *  This program required X-windows to running, and the xview toolkit to
 *   complile.  Much of the user interface was created using 'guide' from
 *   Sun Microsystems. All files ending with _ui.c and _ui.h were created
 *   with 'gxv', the code generator for 'guide'.
 */

#include "segal.h"

segal_win_objects *segal_win;
segal_stats_win_objects *segal_stats_win;
extern view_win_objects *view_win;
extern pixedit_win_objects *edit_win;

Attr_attribute INSTANCE;

char     *Progname;
int       ac;
char    **av;

/* global stuff used in all routines (see segal.h) */
IMAGE_INFO himage, hmask;
ZOOM_INFO zoom_info;
SEGAL_INFO segal;
EDIT_INFO edit_info;
Display  *display;
u_long   *colors, standout;
GC        gc;
int       screen;
Xv_cmsdata cms_data;

XImage   *image;
XImage   *mask_image;
XImage   *blend_image;
XImage   *zoom_image;
XImage   *zoom_mask_image;
XImage   *zoom_blend_image;

u_char  **work_buf;		/* used in editting */

Cursor    watch_cursor;

XID       view_xid, edit_xid, edit_control_xid, segal_control_xid;

/* command line args */
char     *in_image, *in_mask;
int       overlay_hue;

/**********************************************************/
void
main(argc, argv)
    int       argc;
    char    **argv;
{
    void      parse_args(), view_win_init(), edit_win_init(), load_proc();

    xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);
    INSTANCE = xv_unique_key();

    /* save in globals */
    Progname = strsave(*argv);	/* for hips error routines */
    ac = argc;			/* save argc and argv in globals to be used
				 * later */
    av = argv;			/* by the hips header routines */

    parse_args(argc, argv);

    /*
     * Initialize XView.
     */

    image = mask_image = blend_image = NULL;
    zoom_image = zoom_mask_image = zoom_blend_image = NULL;
    work_buf = NULL;
    himage.data = hmask.data = NULL;
    himage.fp = hmask.fp = NULL;

    /*
     * Initialize user interface components.
     */
    segal_win = segal_win_objects_initialize(NULL, NULL);
    segal_stats_win = segal_stats_win_objects_initialize(NULL, segal_win->win);

    /* set panel default items */
    (void) xv_set(segal_win->image1, PANEL_VALUE, in_image, NULL);
    (void) xv_set(segal_win->image2, PANEL_VALUE, in_mask, NULL);

    /* set text input fields to call load_proc on carraige-return */
    (void) xv_set(segal_win->image1, PANEL_NOTIFY_LEVEL, PANEL_SPECIFIED);
    (void) xv_set(segal_win->image1, PANEL_NOTIFY_PROC, load_proc, NULL);

    /* default gamma set to 1.5 */
    (void) xv_set(segal_win->gamma_val, PANEL_VALUE, 15, NULL);

    /* default slider values */
    (void) xv_set(segal_win->bg_slider, PANEL_VALUE, 100, NULL);
    (void) xv_set(segal_win->fg_slider, PANEL_VALUE, 0, NULL);

    /* get global window stuff */
    display = (Display *) xv_get(segal_win->win, XV_DISPLAY, NULL);
    screen = DefaultScreen(display);

    if (DisplayPlanes(display, screen) == 1) {
	fprintf(stderr, "This program requires an 8-bit color monitor \n");
	exit(0);
    }
    gc = DefaultGC(display, screen);
    XSetForeground(display, gc, XWhitePixel(display, screen));
    watch_cursor = XCreateFontCursor(display, XC_watch);

    view_win_init(segal_win->win);
    edit_win_init(segal_win->win);

    /* windows start out closed */
    (void) xv_set(edit_win->win, XV_SHOW, FALSE, NULL);
    (void) xv_set(view_win->win, XV_SHOW, FALSE, NULL);

    /* set edit defaults */
    (void) xv_set(edit_win->mag_item, PANEL_VALUE, 1, NULL);
    (void) xv_set(edit_win->brush_type, PANEL_VALUE, 3, NULL);

    /* polygon fill defaults */
    (void) xv_set(segal_win->poly_mesg, PANEL_LABEL_STRING, " ", NULL);
    (void) xv_set(segal_win->show_poly_item, PANEL_INACTIVE, TRUE, NULL);
    (void) xv_set(segal_win->poly_fill_item, PANEL_INACTIVE, TRUE, NULL);
    (void) xv_set(segal_win->clear_item, PANEL_INACTIVE, TRUE, NULL);

    /* set XID's */
    view_xid = (XID) xv_get(canvas_paint_window(view_win->canvas), XV_XID);
    edit_xid = (XID) xv_get(canvas_paint_window(edit_win->canvas), XV_XID);
    edit_control_xid =
	(XID) xv_get(canvas_paint_window(edit_win->control), XV_XID);
    segal_control_xid =
	(XID) xv_get(canvas_paint_window(segal_win->controls), XV_XID);

    if (in_image != NULL || in_mask != NULL)
	load_proc(NULL, NULL);

    /*
     * Turn control over to XView.
     */
    window_main_loop(segal_win->win);
    exit(0);
}

/**************************************************************/
void
load_proc(item, event)		/* procedure for the 'load' button */
    Panel_item item;
    Event    *event;
{
    void      build_ximages(), view_setup(), edit_setup();
    int       rval1, rval2;

    set_watch_cursor();

    if ((rval1 = load_image()) < 0) {	/* continue of no image */
	himage.fp = NULL;
    }
    if ((rval2 = load_mask(item)) < 0) {	/* continue if no mask */
	hmask.fp = NULL;
    }
    if (hmask.fp == NULL && himage.fp == NULL) {
	unset_watch_cursor();
	return;
    }
    if (rval1 > 0 || rval2 > 0) {	/* one of the files not loaded
					 * already */
	build_ximages();
	if (himage.fp == NULL) {
	    segal.display_type = 1;
	    xv_set(segal_win->display_type, PANEL_VALUE, 1, NULL);
	} else {
	    if (segal.display_type != 0) {
		segal.display_type = 0;
		xv_set(segal_win->display_type, PANEL_VALUE, 0, NULL);
	    }
	}
    }

    if ((int) xv_get(edit_win->win, XV_SHOW) == TRUE) {
	edit_setup();
	edit_repaint_proc();
    }
    if (himage.data != NULL || hmask.data != NULL) {
	view_setup();
	image_repaint_proc();
    }
    unset_watch_cursor();
}

/**************************************************************/
void
mode_proc(item, event)		/* procedure for 'mode' setting */
    Panel_item item;
    Event    *event;
{
    void      edit_setup();
    int       mode;

    if (himage.fp == NULL && hmask.fp == NULL)
	return;			/* do nothing until an image is loaded */

    mode = (int) xv_get(segal_win->mode_item, PANEL_VALUE, NULL);
    segal.mode = mode;

    if (mode == 0) {
	(void) xv_set(edit_win->win, XV_SHOW, FALSE, NULL);
	zoom_info.x = zoom_info.y = -1;
	image_repaint_proc();	/* re-draw without zoom box */
    } else {
	if ((int) xv_get(edit_win->win, XV_SHOW, NULL) == FALSE) {
	    set_watch_cursor();
	    edit_setup();
	    edit_repaint_proc();
	    unset_watch_cursor();
	}
    }

    if ((int) xv_get(edit_win->win, XV_SHOW, NULL) == FALSE) {
	/*
	 * if window still not open, then something did not work, and should
	 * set mode back to 0
	 */
	segal.mode = 0;
	xv_set(segal_win->mode_item, PANEL_VALUE, 0, NULL);
    }
}

/*************************************************************/
void
quit_proc(item, event)
    Panel_item item;
    Event    *event;
{
    void      save_first();
    Panel     panel = (Panel) xv_get(item, PANEL_PARENT_PANEL);
    int       result;

    if (segal.changed == 1) {
	result = notice_prompt(panel, NULL,
			       NOTICE_MESSAGE_STRINGS,
			       "Mask image have not been saved!", NULL,
			       NOTICE_BUTTON_YES, "Save mask image",
			       NOTICE_BUTTON_NO, "Exit without saving",
			       NULL);

	if (result == NOTICE_YES) {
	    mask_save_proc(item, event);
	    return;
	}
    }
    /* reset mouse speed */
    XChangePointerControl(display, True, True, 1, 1, 1);

    (void) fclose(himage.fp);
    (void) fclose(hmask.fp);

    xv_set(segal_win->win, FRAME_NO_CONFIRM, TRUE, NULL);
    (void) xv_destroy_safe(segal_win->win);
    /* this will also kill all child windows */
}


/**********************************************************/
void
build_ximages()
{				/* allocates and maps all ximages */
    u_char   *dbuf;
    XVisualInfo *winv;
    void      map_images(), blend();

    if (verbose)
	fprintf(stderr, "creating X images ... \n");

    winv = (XVisualInfo *) malloc(sizeof(XVisualInfo));
    if ((int) XMatchVisualInfo(display, screen,
			       XDisplayPlanes(display, screen),
			       PseudoColor, winv) == 0) {
	fprintf(stderr, "unable to find correct visual \n");
	exit(0);
    }
    if (image == NULL) {
	dbuf = Calloc(segal.rows * segal.cols, u_char);
	image = XCreateImage(display, winv->visual, 8, ZPixmap, 0,
			     (char *) dbuf, segal.cols, segal.rows, 8, 0);
	if (image == NULL) {
	    fprintf(stderr, " Error creating image! \n");
	    return;
	}
    }
    if (mask_image == NULL) {
	dbuf = Calloc(segal.rows * segal.cols, u_char);
	mask_image = XCreateImage(display, winv->visual, 8, ZPixmap, 0,
			       (char *) dbuf, segal.cols, segal.rows, 8, 0);
	if (mask_image == NULL) {
	    fprintf(stderr, " Error creating mask_image! \n");
	    return;
	}
    }
    if (blend_image == NULL) {
	dbuf = Calloc(segal.rows * segal.cols, u_char);
	blend_image = XCreateImage(display, winv->visual, 8, ZPixmap, 0,
				   (char *) dbuf,
				   segal.cols, segal.rows, 8, 0);
	if (blend_image == NULL) {
	    fprintf(stderr, " Error creating blend_image! \n");
	    return;
	}
    }
    map_images();
}

/****************************************************/
void
map_images()
{				/* called by build_ximages, step_up, and
				 * step_down routines */
    if (himage.fp != NULL)
	map_image_to_lut(himage.data[0], image->data,
			 segal.rows * segal.cols);

    if (hmask.fp != NULL)
	map_image_to_lut(hmask.data[0], mask_image->data,
			 segal.rows * segal.cols);

    if (himage.fp != NULL && hmask.fp != NULL)
	blend(himage.data[0], hmask.data[0], (u_char *) blend_image->data,
	      segal.rows * segal.cols);
}

/***********************************************************/
void
blend_type_proc(item, value, event)	/* procedure for 'blend type' setting */
    Panel_item item;
    int       value;
    Event    *event;
{
    static int old_val = -1;
    if (verbose)
	fprintf(stderr, "segal: blend_type_proc: value: %u\n", value);

    if (value == old_val)
	return;

    set_watch_cursor();
    segal.blend_type = value;

    if (himage.fp != NULL) {
	blend(himage.data[0], work_buf[0], (u_char *) blend_image->data,
	      segal.rows * segal.cols);
    }
    image_repaint_proc();

    if ((int) xv_get(edit_win->win, XV_SHOW, NULL) == TRUE) {
	zoom();
	edit_repaint_proc();
    }
    old_val = value;
    unset_watch_cursor();
}

/***********************************************************/
void
mask_type_proc(item, value, event)	/* procedure for 'mask type' setting */
    Panel_item item;
    int       value;
    Event    *event;
{
    static int old_val = -1;
    if (verbose)
	fprintf(stderr, "segal: mask_type_proc: value: %u\n", value);

    if (value == old_val)
	return;

    set_watch_cursor();
    segal.mask_type = value;

    if (himage.fp != NULL) {
	blend(himage.data[0], work_buf[0], (u_char *) blend_image->data,
	      segal.rows * segal.cols);
    }
    image_repaint_proc();

    if ((int) xv_get(edit_win->win, XV_SHOW, NULL) == TRUE) {
	zoom();
	edit_repaint_proc();
    }
    old_val = value;

    if (value == 1)
	xv_set(edit_win->edit_mode, PANEL_CHOICE_STRINGS,
	       "paint", "erase", 0, NULL);
    else
	xv_set(edit_win->edit_mode, PANEL_CHOICE_STRINGS,
	       " cut ", "fill ", 0, NULL);

    panel_paint(edit_win->edit_mode, PANEL_CLEAR);
    unset_watch_cursor();
}

/**************************************************************/
void
change_image_proc()
{				/* selects which image to display */
    segal.display_type = (int) xv_get(segal_win->display_type,
				      PANEL_VALUE, 0);

    image_repaint_proc();
    edit_repaint_proc();
}

/****************************************************************/
void
parse_args(argc, argv)
    int       argc;
    char     *argv[];
{
    void      usageterm();

    in_image = in_mask = NULL;
    verbose = 0;
    overlay_hue = 170;		/* default  (dark greenish) */

    /* Interpret options  */
    while (--argc > 0 && (*++argv)[0] == '-') {
	char     *s;
	for (s = argv[0] + 1; *s; s++)
	    switch (*s) {
	    case 'i':
		if (argc < 2)
		    usageterm();
		in_image = *++argv;
		fprintf(stderr, " using image file: %s\n", in_image);
		argc--;
		break;
	    case 'm':
		if (argc < 2)
		    usageterm();
		in_mask = *++argv;
		fprintf(stderr, " using mask file: %s\n", in_mask);
		argc--;
		break;
	    case 'c':
		if (argc < 2)
		    usageterm();
		sscanf(*++argv, "%d", &overlay_hue);
		fprintf(stderr, " overlay hue set to %d \n", overlay_hue);
		argc--;
		break;
	    case 'v':
		verbose++;
		break;
	    case 'h':
		usageterm();
		break;
	    default:
		usageterm();
		break;
	    }
    }				/* while */
    if (overlay_hue < 0 || overlay_hue > 360) {
	fprintf(stderr, " Error: color value (hue) must be between 0 and 360. \n");
	exit(0);
    }
}

/******************************************************/
void
usageterm()
{
    fprintf(stderr, "Usage: segal [-i image][-m mask] [-v] [-c NN][-h][-help] \n");
    fprintf(stderr, "        [-i HIPS file] load specified image file \n");
    fprintf(stderr, "        [-m HIPS file] load specified mask file \n");
    fprintf(stderr, "        [-v] verbose mode \n");
    fprintf(stderr, "        [-c NN] hue (0 to 360) for overlay blend color (default = 170) \n");
    fprintf(stderr, "        [-h] this list \n");
    fprintf(stderr, "        [-help] list of window attribute help \n\n");
    exit(0);
}
