/* $Id: setutils.c,v 1.26 90/05/09 22:11:32 pturner Exp Locker: pturner $
 *
 * routines to allocate, manipulate, and return
 * information about sets. Autoscaling stuff.
 *
 */

#include <stdio.h>
#include <math.h>
#include "defines.h"
#include "objdefs.h"
#include "globals.h"

#include "setdef.h"
/*
 * sets - for default graph
 */
plotarr *plots[MAXPLOT];

#ifdef EX
/*
 * for plot files
 */
file_plotarr fplots[MAXPLOT];

#endif

static int default_color[MAXPLOT] = {1, 1, 1, 1, 1, 1, 1, 1,
1, 1, 1, 1, 1, 1, 1};
static int default_symbol[MAXPLOT] = {0, 0, 0, 0, 0, 0, 0, 0,
0, 0, 0, 0, 0, 0, 0};
static int default_linestyle[MAXPLOT] = {1, 1, 1, 1, 1, 1, 1,
1, 1, 1, 1, 1, 1, 1, 1};

int nsets = 0;

extern plotstr pstr[];

char *calloc();

/*
 * allocate a single set
 */
plotarr *allocplot()
{
    return ((plotarr *) calloc(1, sizeof(plotarr)));
}

/*
 * allocate arrays for set i of length len.
 */
allocxy(i, len)
    int i, len;
{
    if (plots[i]->x == NULL) {
	if ((plots[i]->x = (double *) calloc(len, sizeof(double))) == NULL) {
	    fprintf(stderr, "Insufficient memory to allocate for plots\n");
	    exit(1);
	}
    }
    if (plots[i]->y == NULL) {
	if ((plots[i]->y = (double *) calloc(len, sizeof(double))) == NULL) {
	    fprintf(stderr, "Insufficient memory to allocate for plots\n");
	    exit(1);
	}
    }
}

/*
 * initialize sets for the default graph
 */
initplots()
{
    int i;
    extern double *ax, *bx, *cx, *dx;	/* used in compute.c */

    if ((ax = (double *) calloc(maxarr, sizeof(double))) == NULL) {
	fprintf(stderr, "Insufficient memory to allocate for ax\n");
	exit(1);
    }
    if ((bx = (double *) calloc(maxarr, sizeof(double))) == NULL) {
	fprintf(stderr, "Insufficient memory to allocate for bx\n");
	exit(1);
    }
    if ((cx = (double *) calloc(maxarr, sizeof(double))) == NULL) {
	fprintf(stderr, "Insufficient memory to allocate for cx\n");
	exit(1);
    }
    if ((dx = (double *) calloc(maxarr, sizeof(double))) == NULL) {
	fprintf(stderr, "Insufficient memory to allocate for dx\n");
	exit(1);
    }
    for (i = 0; i < maxplot; i++) {
	if ((plots[i] = allocplot()) == NULL) {
	    fprintf(stderr, "Insufficient memory to allocate for plots\n");
	    exit(1);
	}
	plots[i]->x = NULL;
	plots[i]->y = NULL;
	plots[i]->comments[0] = 0;
	plots[i]->xmin = 0.0;
	plots[i]->ymin = 0.0;
	plots[i]->xmax = 0.0;
	plots[i]->ymax = 0.0;
	plots[i]->plotsym = default_symbol[i];
	plots[i]->linesym = default_linestyle[i];
	plots[i]->color = default_color[i];
	plots[i]->len = 0;
	plots[i]->active = FALSE;
	plots[i]->errbar = -1;
	plots[i]->errbarxy = 0;
#ifdef EX
	set_plotfile_parms(i, "", 0, 0, 0);
#endif
    }
}

/*
 * see if set setno is an error bar set
 */
int iserrbar(setno)
    int setno;
{
    return (plots[setno]->errbar);
}

/*
 * set the type of error bar for error bars in setno
 */
seterrbarxy(setno, etype)
    int setno, etype;
{
    plots[setno]->errbarxy = etype;
}

/*
 * return the type of error bar set setno is.
 */
int getseterrbarxy(setno)
    int setno;
{
    return (plots[setno]->errbarxy);
}

/*
 * make setno error bars on set setto
 */
makeseterrbar(setno, setto)
    int setno, setto;
{
    plots[setno]->errbar = setto;
    if (setto >= 0) {
	sprintf(buf, "Error bar on set %d", setto);
	setcomment(setno, buf);
	update_status(setno);
    }
}

/*
 * make setno a regular set
 */
clearseterrbar(setno)
    int setno;
{
    if (plots[setno]->errbar > -1) {
	sprintf(buf, "Was error bar on set %d", plots[setno]->errbar);
	setcomment(setno, buf);
	update_status(setno);
	plots[setno]->errbar = -1;
    }
}

/*
 * set the min/max fields of a set
 */
setminmax(setno, x1, x2, y1, y2)
    int setno;
    double x1, x2, y1, y2;
{
    plots[setno]->xmin = x1;
    plots[setno]->xmax = x2;
    plots[setno]->ymin = y1;
    plots[setno]->ymax = y2;
}

/*
 * get the min/max fields of a set
 */
getsetminmax(setno, x1, x2, y1, y2)
    int setno;
    double *x1, *x2, *y1, *y2;
{
    *x1 = plots[setno]->xmin;
    *x2 = plots[setno]->xmax;
    *y1 = plots[setno]->ymin;
    *y2 = plots[setno]->ymax;
}

/*
 * compute the mins and maxes of both x & y for plot array p
 */
fminmax(p, xmi, xma, ymi, yma)
    plotarr *p;
    double *xmi, *xma, *ymi, *yma;
{
    int i;

    *xmi = p->x[0];
    *xma = p->x[0];
    *ymi = p->y[0];
    *yma = p->y[0];
    for (i = 1; i < (p->len); i++) {
	if (p->x[i] < *xmi)
	    *xmi = p->x[i];
	if (p->x[i] > *xma)
	    *xma = p->x[i];
	if (p->y[i] < *ymi)
	    *ymi = p->y[i];
	if (p->y[i] > *yma)
	    *yma = p->y[i];
    }
}

/*
 * compute the mins/maxes and update the apporiate fields of
 * set i.
 */
updatesetminmax(i)
    int i;
{
    double x1, x2, y1, y2;

    fminmax(plots[i], &x1, &x2, &y1, &y2);
    plots[i]->xmin = x1;
    plots[i]->xmax = x2;
    plots[i]->ymin = y1;
    plots[i]->ymax = y2;
}

setxy(x, y, setno, len)
    double *x, *y;
    int setno, len;
{
    plots[setno]->x = x;
    plots[setno]->y = y;
    plots[setno]->len = len;
}

double *getx(i)
    int i;
{
    return plots[i]->x;
}

double *gety(i)
    int i;
{
    return plots[i]->y;
}

int getsetplotsym(i)
    int i;
{
    return plots[i]->plotsym;
}

int getsetlinesym(i)
    int i;
{
    return plots[i]->linesym;
}

int getsetcolor(i)
    int i;
{
    return plots[i]->color;
}

int getsetlength(setno)
    int setno;
{
    return plots[setno]->len;
}

char *getcomment(setno)
    int setno;
{
    return (char *) (plots[setno]->comments);
}

setplotsym(i, sym)
    int i, sym;
{
    plots[i]->plotsym = sym;
}

setlinesym(i, sym)
    int i, sym;
{
    plots[i]->linesym = sym;
}

setplotcolor(i, sym)
    int i, sym;
{
    plots[i]->color = sym;
}

setlength(i, length)
    int i, length;
{
    cxfree(plots[i]->x);
    cxfree(plots[i]->y);
    plots[i]->x = NULL;
    plots[i]->y = NULL;
    allocxy(i, length);
    plots[i]->len = length;
}

setcomment(i, s)
    int i;
    char *s;
{
    strcpy(plots[i]->comments, s);
}

copyx(setfrom, setto)
    int setfrom, setto;
{
    int i, n;

    n = getsetlength(setfrom);
    for (i = 0; i < n; i++)
	plots[setto]->x[i] = plots[setfrom]->x[i];
}

copyy(setfrom, setto)
    int setfrom, setto;
{
    int i, n;

    n = getsetlength(setfrom);
    for (i = 0; i < n; i++)
	plots[setto]->y[i] = plots[setfrom]->y[i];
}

/*
 * copyall assumes both sets exist, have their length
 * properly set, and that they are both active
 * this also does the legend string
 */
copyall(setfrom, setto)
    int setfrom, setto;
{
    extern plotstr legstr[];

    copyx(setfrom, setto);
    copyy(setfrom, setto);
    strcpy(plots[setto]->comments, plots[setfrom]->comments);
    plots[setto]->plotsym = plots[setfrom]->plotsym;
    plots[setto]->linesym = plots[setfrom]->linesym;
    plots[setto]->color = plots[setfrom]->color;
    plots[setto]->errbar = plots[setfrom]->errbar;
    plots[setto]->errbarxy = plots[setfrom]->errbarxy;
    strcpy(legstr[setto].s, legstr[setfrom].s);
    legstr[setfrom].s[0] = 0;
    plots[setfrom]->plotsym = default_symbol[setfrom];
    plots[setfrom]->linesym = default_linestyle[setfrom];
    plots[setfrom]->color = default_color[setfrom];
}

int nextset()
{
    int i;

    i = 0;
    for (i = 0; i < maxplot; i++) {
	if (!isactive(i)) {
	    return (i);
	}
    }
    errwin("Error - no sets available");
    return (-1);
}

killset(setno)
    int setno;
{
    if (isactive(setno)) {
	plots[setno]->active = FALSE;
	plots[setno]->len = 0;
	plots[setno]->xmin = 0.0;
	plots[setno]->xmax = 0.0;
	plots[setno]->ymin = 0.0;
	plots[setno]->ymax = 0.0;
	plots[setno]->comments[0] = 0;
	cxfree(plots[setno]->x);
	cxfree(plots[setno]->y);
	plots[setno]->x = NULL;
	plots[setno]->y = NULL;
	nsets--;
    }
}

activateset(setno)
    int setno;
{
    if (!(plots[setno]->active)) {
	plots[setno]->active = TRUE;
	nsets++;
    }
}

int activeset()
{
    int i;

    for (i = 0; i < maxplot; i++) {
	if (plots[i]->active)
	    return (1);
    }
    return (0);
}

int isactive(setno)
    int setno;
{
    if (0 <= setno && setno < maxplot)
	return (plots[setno]->active);
    return (0);
}

int legalset(setno)
    int setno;
{
    if (0 <= setno && setno < maxplot)
	return (1);
    return (0);
}

#ifdef EX

int isactive_plotfile(i)
    int i;
{
    return fplots[i].active;
}

int plotfile_type(i)
    int i;
{
    return fplots[i].filetype;
}

int plotfile_data(i)
    int i;
{
    return fplots[i].datatype;
}

plotfile_filename(i, s)
    int i;
    char *s;
{
    strcpy(s, fplots[i].filen);
}

set_plotfile_parms(i, buf, itype, iformat, iact)
    int i, itype, iformat, iact;
    char buf[];

{
    fplots[i].color = 1;
    fplots[i].linesym = 1;
    fplots[i].plotsym = 0;
    fplots[i].active = iact;
    fplots[i].filetype = itype;
    fplots[i].datatype = iformat;
    if (strlen(buf) > 0 && strlen(buf) < 128) {
	strcpy(fplots[i].filen, buf);
    } else
	fplots[i].filen[0] = 0;
}

#endif

/*
	 the following routines determine default scaling and ticmarks
*/

defaultgraph()
{
    double xmax, xmin, ymax, ymin;
    double xgmax, xgmin, ygmax, ygmin;
    int i;

    xgmax = (MBIG);
    xgmin = BIG;
    ygmax = (MBIG);
    ygmin = BIG;
    for (i = 0; i < maxplot; i++) {
	if (isactive(i) && !(iserrbar(i) >= 0)) {
	    getsetminmax(i, &xmin, &xmax, &ymin, &ymax);
	    if (xmin < xgmin)
		xgmin = xmin;
	    if (xmax > xgmax)
		xgmax = xmax;
	    if (ymin < ygmin)
		ygmin = ymin;
	    if (ymax > ygmax)
		ygmax = ymax;
	}
    }
    if (xgmin != xgmax) {
	xg2 = xgmax;
	xg1 = xgmin;
    } else {
	if ((xgmin == 0.0) && (xgmax == 0.0)) {
	    xgmin = 1.0;
	}
	xg1 = xgmin - 0.1 * fabs(xgmin);
	xg2 = xgmin + 0.1 * fabs(xgmin);
    }
    if (ygmin != ygmax) {
	yg2 = ygmax;
	yg1 = ygmin;
    } else {
	if ((ygmin == 0.0) && (ygmax == 0.0)) {
	    ygmin = 1.0;
	}
	yg1 = ygmin - 0.1 * fabs(ygmin);
	yg2 = ygmin + 0.1 * fabs(ygmin);
    }
}

defaultx()
{
    int i;
    double xgmin, xgmax, xmax, xmin, tmp;

    xgmax = (MBIG);
    xgmin = BIG;
    for (i = 0; i < maxplot; i++) {
	if (isactive(i) && !(iserrbar(i) >= 0)) {
	    getsetminmax(i, &xmin, &xmax, &tmp, &tmp);
	    if (xmin < xgmin)
		xgmin = xmin;
	    if (xmax > xgmax)
		xgmax = xmax;
	}
    }
    if (xgmin != xgmax) {
	xg2 = xgmax;
	xg1 = xgmin;
    } else {
	if ((xgmin == 0.0) && (xgmax == 0.0)) {
	    xgmin = 1.0;
	}
	xg1 = xgmin - 0.1 * fabs(xgmin);
	xg2 = xgmin + 0.1 * fabs(xgmin);
    }
}

defaulty()
{
    int i;
    double ygmax, ygmin, ymin, ymax, tmp;

    ygmax = (MBIG);
    ygmin = BIG;
    for (i = 0; i < maxplot; i++) {
	if (isactive(i) && !(iserrbar(i) >= 0)) {
	    getsetminmax(i, &tmp, &tmp, &ymin, &ymax);
	    if (ymin < ygmin)
		ygmin = ymin;
	    if (ymax > ygmax)
		ygmax = ymax;
	}
    }
    if (ygmin != ygmax) {
	yg2 = ygmax;
	yg1 = ygmin;
    } else {
	if ((ygmin == 0.0) && (ygmax == 0.0)) {
	    ygmin = 1.0;
	}
	yg1 = ygmin - 0.1 * fabs(ygmin);
	yg2 = ygmin + 0.1 * fabs(ygmin);
    }
}

/*
 * label: test program to demonstrate nice graph axis labeling
 *
 * Paul Heckbert, 2 Dec 88
 */

double expt(), tic(), nicenum();

ylabeldef(ymin, ymax)
    double ymin, ymax;
{
    int exp;
    double graphymin, graphymax, range, d;

    /* we expect ymin!=ymax */
    range = nicenum(ymax - ymin, 0);
    d = nicenum(range / (yticnum - 1), 1);	/* tic mark spacing */
    graphymin = floor(ymin / d) * d;
    graphymax = ceil(ymax / d) * d;
    yg1 = graphymin;
    yg2 = graphymax;
    yt1 = d;
    yt2 = d * 0.5;
}

xlabeldef(xmin, xmax)
    double xmin, xmax;
{
    int exp;
    double graphxmin, graphxmax, range, d;

    /* we expect xmin!=xmax */
    range = nicenum(xmax - xmin, 0);
    d = nicenum(range / (xticnum - 1), 1);	/* tic mark spacing */
    graphxmin = floor(xmin / d) * d;
    graphxmax = ceil(xmax / d) * d;
    xg1 = graphxmin;
    xg2 = graphxmax;
    xt1 = d;
    xt2 = d * 0.5;
}

/*
 * nicenum: find a "nice" number approximately equal to x
 * round if round=1, ceil if round=0
 */

static double nicenum(x, round)
    double x;
    int round;
{
    int exp;
    double f, y;

    exp = floor(log10(x));
    f = x / expt(10., exp);	/* fraction between 1 and 10 */
    if (round)
	if (f < 1.5)
	    y = 1.;
	else if (f < 3.)
	    y = 2.;
	else if (f < 7.)
	    y = 5.;
	else
	    y = 10.;
    else if (f <= 1.)
	y = 1.;
    else if (f <= 2.)
	y = 2.;
    else if (f <= 5.)
	y = 5.;
    else
	y = 10.;
    return y * expt(10., exp);
}

/*
 * expt(a,n)=a^n for integer n
 * roundoff errors in pow were causing problems, so I wrote my own
 */

double expt(a, n)
    double a;
    register int n;
{
    double x;

    x = 1.;
    if (n > 0)
	for (; n > 0; n--)
	    x *= a;
    else
	for (; n < 0; n++)
	    x /= a;
    return x;
}

defaultsetgraph(setno)
    int setno;
{
    double xmax, xmin, ymax, ymin;
    double xgmax, xgmin, ygmax, ygmin;

    xgmax = (MBIG);
    xgmin = BIG;
    ygmax = (MBIG);
    ygmin = BIG;
    getsetminmax(setno, &xmin, &xmax, &ymin, &ymax);
    if (xmin < xgmin)
	xgmin = xmin;
    if (xmax > xgmax)
	xgmax = xmax;
    if (ymin < ygmin)
	ygmin = ymin;
    if (ymax > ygmax)
	ygmax = ymax;
    if (xgmin != xgmax) {
	xg2 = xgmax;
	xg1 = xgmin;
    } else {
	if ((xgmin == 0.0) && (xgmax == 0.0)) {
	    xgmin = 1.0;
	}
	xg1 = xgmin - 0.1 * fabs(xgmin);
	xg2 = xgmin + 0.1 * fabs(xgmin);
    }
    if (ygmin != ygmax) {
	yg2 = ygmax;
	yg1 = ygmin;
    } else {
	if ((ygmin == 0.0) && (ygmax == 0.0)) {
	    ygmin = 1.0;
	}
	yg1 = ygmin - 0.1 * fabs(ygmin);
	yg2 = ygmin + 0.1 * fabs(ygmin);
    }
}

defaulttics(method)
    int method;
{
    double log10();
    int cx, cy;

    if (xg1 >= xg2 || yg1 >= yg2) {
	errwin("World coordinates not properly set.");
	return;
    }
    xt1 = (xg2 - xg1) / xticnum;
    xt2 = xt1 / 2.0;
    yt1 = (yg2 - yg1) / yticnum;
    yt2 = yt1 / 2.0;
    cx = (int) log10(xt1);
    cy = (int) log10(yt1);
    xform = ((cx < 0) ? -cx + 1 : 1);
    yform = ((cy < 0) ? -cy + 1 : 1);
    if (xform > 9) {
	xform = 2;
	fformx = 0;
    }
    if (yform > 9) {
	yform = 2;
	fformy = 0;
    }
    if (!method) {
	ylabeldef(yg1, yg2);
	xlabeldef(xg1, xg2);
    }
    if (xticslog) {
	xt2 = 0.0;
    }
    if (yticslog) {
	yt2 = 0.0;
    }
}

defaultxtics(method)
    int method;
{
    double log10();
    int cx;

    if (xg1 >= xg2) {
	errwin("World coordinates not properly set.");
	return;
    }
    xt1 = (xg2 - xg1) / xticnum;
    xt2 = xt1 / 2.0;
    cx = (int) log10(xt1);
    xform = ((cx < 0) ? -cx + 1 : 1);
    if (xform > 9) {
	xform = 2;
	fformx = 0;
    }
    if (!method)
	xlabeldef(xg1, xg2);
    if (xticslog) {
	xt2 = 0.0;
    }
}

defaultytics(method)
    int method;
{
    double log10();
    int cy;

    if (yg1 >= yg2) {
	errwin("World coordinates not properly set.");
	return;
    }
    yt1 = (yg2 - yg1) / yticnum;
    yt2 = yt1 / 2.0;
    cy = (int) log10(yt1);
    yform = ((cy < 0) ? -cy + 1 : 1);
    if (yform > 9) {
	yform = 2;
	fformy = 0;
    }
    if (!method)
	ylabeldef(yg1, yg2);
    if (yticslog) {
	yt2 = 0.0;
    }
}

logtrans()
{
    int i, j;
    char stmp[80];
    double *x, *y;

    if (inwin) {
	if (!(yesno("Take logs of all sets?", "Press YES or NO", "YES", "NO")))
	    return;
    }
    for (i = 0; i < maxplot; i++) {
	if (isactive(i)) {
	    x = getx(i);
	    y = gety(i);
	    for (j = 0; j < getsetlength(i); j++) {
		if (xticslog) {
		    if (x[j] <= 0.0) {
			sprintf(stmp, "x of set %d loc %d out of domain", i, j + 1);
			errwin(stmp);
			xticslog = 0;
		    }
		}
		if (yticslog) {
		    if (y[j] <= 0.0) {
			sprintf(stmp, "y of set %d loc %d out of domain", i, j + 1);
			errwin(stmp);
			yticslog = 0;
		    }
		}
	    }
	}
    }
    if (!(xticslog || yticslog)) {
	logtransflag = 0;
	return;
    }
    for (i = 0; i < maxplot; i++) {
	if (isactive(i)) {
	    x = getx(i);
	    y = gety(i);
	    for (j = 0; j < getsetlength(i); j++) {
		if (xticslog) {
		    x[j] = log10(x[j]);
		}
		if (yticslog) {
		    y[j] = log10(y[j]);
		}
	    }
	    if (xticslog)
		updatesetminmax(i);
	    if (yticslog)
		updatesetminmax(i);
	}
    }
    logtransflag = 1;
}

invlogtrans(askflag)
    int askflag;
{
    int i, j;
    char stmp[80];
    double *x, *y;

    if (inwin && askflag) {
	if (!(yesno("Compute pow10 of all sets?", "Press YES or NO", "YES", "NO")))
	    return;
    }
    for (i = 0; i < maxplot; i++) {
	if (isactive(i)) {
	    x = getx(i);
	    y = gety(i);
	    for (j = 0; j < getsetlength(i); j++) {
		if (xticslog) {
		    if (x[j] > 30.0) {
			sprintf(stmp, "x of set %d loc %d out of domain", i, j + 1);
			errwin(stmp);
			logtransflag = 1;
			return;
		    }
		}
		if (yticslog) {
		    if (y[j] > 30.0) {
			sprintf(stmp, "y of set %d loc %d out of domain", i, j + 1);
			errwin(stmp);
			logtransflag = 1;
			return;
		    }
		}
	    }
	}
    }
    for (i = 0; i < maxplot; i++) {
	if (isactive(i)) {
	    x = getx(i);
	    y = gety(i);
	    for (j = 0; j < getsetlength(i); j++) {
		if (xticslog) {
		    x[j] = pow(10.0, x[j]);
		}
		if (yticslog) {
		    y[j] = pow(10.0, y[j]);
		}
	    }
	    if (xticslog)
		updatesetminmax(i);
	    if (yticslog)
		updatesetminmax(i);
	}
    }
    xticslog = 0;
    yticslog = 0;
    logtransflag = 0;
}
