#include <RpSGIReader.h>
#include <fstream.h>
#include <assert.h>

// ************************************************************************

signed
min( signed iA, signed iB )
{
   return (( iA < iB ) ? ( iA ) : ( iB ));
}

signed
max( signed iA, signed iB )
{
   return (( iA > iB ) ? ( iA ) : ( iB )); 
}

// ------------------------------------------------------------------------

inline
unsigned short
RpSGIReader::castToShort( unsigned char *cp )
{
   return ( ( cp[0] << 8 ) | cp[1] );
}


inline
unsigned long
RpSGIReader::castToLong( unsigned char *cp )
{
   return( 
      ( ( unsigned long )cp[0] << 24 ) |
      ( ( unsigned long )cp[1] << 16 ) |
      ( ( unsigned long )cp[2] <<  8 ) |
      ( cp[3] )
   );      
}

// ------------------------------------------------------------------------

RpSGIReader::RpSGIReader( const char *cpFilename )
   : _fileStream( cpFilename )
{
   RpImageArea		area;
   unsigned char	cp[512];
   
   _isRLE = 0;
   _status = OK;
   
   if ( ( !cpFilename ) ||  _fileStream.fail() )
   {
      _status = BadFilename;
      return;
   }
   
   // read header:
   _fileStream.read( cp, 512 );
   
   //
   // check magic cookie
   //
   if ( castToShort( cp ) != 474 )
   {
      _status = BadSGIFormat;
      return;
   }
   
   //
   // RLE or not RLE?
   //
   switch( cp[2] )
   {
      case( 0 ):
      case( 1 ):
         _isRLE = (int) cp[2];
         break;
         
      default:
         _status = BadSGIFormat;
   };
   
   //
   // Bytes per pixel:
   //
   switch( cp[3] )
   {
      case( 1 ):
         setType( RpUChar );
         break;
         
      case(2):
         setType( RpUShort );
         break;
         
      default:
         _status = BadSGIFormat;
   };
   
   //
   // Dimensions, area and channels:
   //
   area.x = 0;
   area.y = 0;
   switch( castToShort( cp + 4 ) )
   {
      case( 1 ):
         area.c = 1;
         area.z = 1;
         area.height = 1;
         area.width = castToShort( cp + 6 );
         break;
         
      case( 2 ):
         area.z = 1;
         area.c = 1;
         area.width = castToShort( cp + 6 );
         area.height = castToShort( cp + 8 );
         break;         
               
      case( 3 ):
         area.z = 1;
         area.width = castToShort( cp + 6 );
         area.height = castToShort( cp + 8 );
         area.c = castToShort( cp + 10 );
         break;
         
      default:
         _status = BadSGIFormat;         
   };
   
   _lMin = castToLong( cp + 12 );
   _lMax = castToLong( cp + 16 );
   
   if ( castToLong( cp + 104 ) )
      _status = UnsupportedFeature;
    
   //
   // set up RLE seek tables:
   //  
   if (( _status == OK ) && ( _isRLE ))
   {
      if ( 
         _startTable.useBytes((area.c * area.height * sizeof(unsigned long)),0)
         &&
         _lengthTable.useBytes((area.c * area.height * sizeof(unsigned long)),0)
      )
      {
         unsigned long lLoop;
         unsigned long *lpStartTable = (unsigned long *) _startTable.getBuffer();
         unsigned long *lpLengthTable = (unsigned long *) _lengthTable.getBuffer();
         
         for ( lLoop=0; lLoop <(area.c * area.height); lLoop++ )
         {
            _fileStream.read( cp, 4 );
            lpStartTable[ lLoop ] = castToLong( cp );
         }
         
         for ( lLoop=0; lLoop <(area.c * area.height); lLoop++ )
         {
            _fileStream.read( cp, 4 );
            lpLengthTable[ lLoop ] = castToLong( cp );
         }
      
         if ( _fileStream.fail() )
            _status = BadSGIFormat;
         
      }
      else
         _status = RAMError;
   }
   
   setArea( area );
   return;
}


RpSGIReader::~RpSGIReader( void )
{
   return;
}


int
RpSGIReader::seekRow( int iRow, int iChannel )
{
   int 			iRet = -1;
   long			lOffset = 0;
   unsigned long	*lpStartTable = (unsigned long *) _startTable.getBuffer();
   
   assert( iRow < (int) getArea().height );
   assert( iChannel < getArea().c );
   
   if ( _status == OK )
   {
     if( _isRLE )
        lOffset = lpStartTable[ iRow + (iChannel * getArea().height)];
     else
        lOffset = 512 + 
           ( iChannel * getArea().width * getArea().height * bytesize( getType() )) +
           ( iRow * getArea().width * bytesize(getType()) );
        
     _fileStream.seekg( lOffset );
     if ( _fileStream.fail() )
     {
        _status = FileError;
        iRet = 0;
     }
        
   } else
      iRet = 0;
      
   return( iRet );
}



int
RpSGIReader::fillTile( RpImageTile *pWriteHere )
{
   int			iRet = 0;
   RpBinaryBuffer	buffer;

   RpImageArea		area;
   signed		iLoop;
      
   unsigned char 	*cpDest;
   unsigned char	*cpSource;
   unsigned		uCopyBytes;
   
   if ( pWriteHere )
   {
      //
      // allocate buffer:
      //
      if ( buffer.useBytes( bytesize( getType() ) * getArea().width, 0 ) )
      {
	 assert( getType() == pWriteHere->getType() ); // or safer&slower: pWriteHere->typecast( getType() );

	 area = getArea();
	 area.c = pWriteHere->getArea().c;
	 area.z = pWriteHere->getArea().z;
	 area = intersect( area, pWriteHere->getArea());

	 if ( (area.width > 0 ) && ( area.height > 0 ) )
	 {
	    uCopyBytes = area.width * bytesize( getType() );

	    cpSource = (unsigned char *) buffer.getBuffer();
	    cpSource += ( area.x - getArea().x ) * (signed) bytesize( getType() );

	    //
	    // for each scanline
	    //
	    iLoop = area.y + area.height;
	    while ( iLoop > area.y )	    
	    {  
	       iLoop--;
	       assert( iLoop >= getArea().y );
	       assert( iLoop < (int)(getArea().y + getArea().height ));

	       cpDest = (unsigned char*) pWriteHere->getBufferAt( area.x, iLoop );

	       //
	       // write scanline into buffer
	       //
	       readRow( buffer.getBuffer(), iLoop, area.c );

               //
               // copy data from buffer into tile:
               //
               memcpy( cpDest, cpSource, uCopyBytes );
            }
         }
      }
	 
      //
      // fill out-of-frame area with fill color:
      //
      applyFillStrategy( pWriteHere );      
   }
   
   return( iRet );
}


void
RpSGIReader::readRow( void *pDest, int iRow, int iChannel )
{
   unsigned short		*spLoop;
   
   if ( seekRow( iRow, iChannel ) )
   {
      if ( _isRLE )
      {
         switch( getType() )
         {
            case( RpUChar ):
               _decode8( (unsigned char *) pDest );
               break;
            case( RpUShort ):
               _decode16( (unsigned short *) pDest );
               break;
            default:
               break;
         };
         
      } else {      
         _fileStream.read( (unsigned char *) pDest, bytesize( getType() ) * getArea().width );
         if ( getType() == RpUShort )
         {
            spLoop = (unsigned short *) pDest;
            spLoop += getArea().width;
            while ( (void *) spLoop != pDest )
            {
               spLoop--;
               *spLoop = castToShort( (unsigned char *) spLoop );
            }
         }
      }

   }  
   return;
}


void
RpSGIReader::_decode16( unsigned short *spDest )
{
   unsigned char		cp[2];
   register unsigned short	uPixel, uCount;

   while( -1 ) 
   {
      _fileStream.read( cp, 2 );
      uPixel = castToShort( cp );
      
      if ( !( uCount = (uPixel & 0x7F )) )
         break; // infinite while
         
      if ( uPixel & 0x80 )
      {
         while( uCount-- )
         {
            _fileStream.read( cp, 2 );
            *spDest = castToShort( cp );
            spDest++;
         }
      } else {
         _fileStream.read( cp, 2 );
         while( uCount-- )
         {
            *spDest = uPixel;
            spDest++;
         }
      }
   }
   
   return;
}



void
RpSGIReader::_decode8( unsigned char *cpDest )
{
   unsigned char 	cPixel, cCount;
   
   while( -1 ) 
   {
      _fileStream.get( cPixel );
      if ( !( cCount = (cPixel & 0x7F )) )
         break; // infinite while
         
      if ( cPixel & 0x80 )
      {
         _fileStream.read( cpDest, (int) cCount );
         cpDest += (int) cCount;
      } else {
         _fileStream.get( cPixel );
         while( cCount-- )
         {
            *cpDest = cPixel;
            cpDest++;
         }
      }
   }
   
   return;
}
