#ifndef RpBinaryOp_CLASS
#define RpBinaryOp_CLASS

#include <RpInputImage.h>
#include <RpType.h>
#include <assert.h>

// ************************************************************************
// TEMPLATE CLASS : RpBinaryOp< BinaryFunctoid >
// DESC:
//    Provides a fast way of creating simple image processing nodes.
//    This template uses a functoid to perform the same operation
//    for every pixel of every channel.
// SEE ALSO:
//    References on component programming or generic programming.
//    RpUnaryOp.h
// 
template< class BinaryFunctoid >
class RpBinaryOp : public RpInputImage
{
   public:
      RpBinaryOp( RpInputImage *pOperand1, RpInputImage *pOperand2, const BinaryFunctoid& setF = BinaryFunctoid() );
      ~RpBinaryOp( void );
      
      int
      fillTile( RpImageTile *pWriteHere );
      
   protected:   
      RpInputImage	*_pHost1;
      RpInputImage	*_pHost2;      
      BinaryFunctoid	_func;
};


template< class BinaryFunctoid >
RpBinaryOp< BinaryFunctoid >::RpBinaryOp( RpInputImage *pOp1, RpInputImage *pOp2, const BinaryFunctoid& setF )
:   _func( setF )
{
   RpType	thisType = RpFilm;
   
   _pHost1 = pOp1;
   if ( _pHost1 )
   {
      _pHost1->registerReference();
      thisType = _pHost1->getType();
   }
   
   _pHost2 = pOp2;
   if ( _pHost2 )
   {
      _pHost2->registerReference();
      thisType = 
         (precedence( _pHost2->getType() ) > precedence( thisType )) ?
         ( _pHost2->getType() ) : ( thisType );
   }
   
   setType( thisType );
   return;
}


template< class BinaryFunctoid >
RpBinaryOp< BinaryFunctoid >::~RpBinaryOp( void )
{
   if ( _pHost1 )
      _pHost1->unregisterReference();      
   if ( _pHost2 )
      _pHost2->unregisterReference();      
   return;
}



template< class BinaryFunctoid >
int
RpBinaryOp< BinaryFunctoid >::fillTile( RpImageTile *pWriteHere )
{
   int			iRet = 0;
   
   register float	fResult;
   register float	fOp1;
   register float	fOp2;
   
   unsigned long	ulLoop;
   
   RpImageTile		*pTile1;
   RpImageTile		*pTile2;
   
#define RpTYPEMACRO( label, type, min, max, black, white, precedence, func ) \
   type *			label##pSrc1; \
   type *			label##pSrc2; \
   type *			label##pDest
#include <RpType.m>

   if ( _pHost1 && _pHost2 && pWriteHere )
   {
      pTile1 = _pHost1->newTile( pWriteHere->getArea() );
      pTile2 = _pHost2->newTile( pWriteHere->getArea() );
      if ( pTile1 && pTile2 )
      {
         assert( getType() == pWriteHere->getType() );
         pTile1->typecast( getType() );
         pTile2->typecast( getType() );
         
         ulLoop = pWriteHere->getArea().height * pWriteHere->getArea().width;
         switch( getType() )
         {
#define RpTYPEMACRO( label, type, low, high, black, white, prec, func ) \
      case( label ): 							\
	 label##pSrc1 = ( type * ) pTile1->getBuffer();	 		\
	 label##pSrc2 = ( type * ) pTile2->getBuffer();	 		\
	 label##pDest = ( type * ) pWriteHere->getBuffer(); 		\
	 while( ulLoop ) { 						\
	 ulLoop--; 							\
	 fOp1 = ((float) * label##pSrc1 );	\
	 fOp1 -= black;				\
	 fOp1 *= 1.0 / ( float )( white - black );	\
	 fOp2 = ((float) * label##pSrc2 );	\
	 fOp2 -= black;				\
	 fOp2 *= 1.0 / ( float )( white - black );	\
	 fResult = _func( fOp1, fOp2 );			\
	 fResult *= (float) ( white - black );		\
	 fResult += black;				\
	 fResult = ( fResult > high ) ? ( high ) : ( fResult ); \
	 fResult = ( fResult < low ) ? ( low ) : ( fResult );   \
	 * label##pDest = ( type ) fResult ; 			\
	 label##pDest ++; label##pSrc1 ++;  label##pSrc2 ++;	\
	 } break
#include <RpType.m>            
         };
         
      } else
         iRet = -1;
         
      if ( pTile1 )
         pTile1->deleteTile();
      
      if ( pTile2 )
         pTile2->deleteTile();
         
   } else
      iRet = -1;
   
   return ( iRet );
}

#endif
