#ifndef	lint
static       char    rcsid[] = "$Header: circle.c,v 1.2 90/12/12 15:30:21 zhang Exp $";
#endif

/*
 * $Log:	circle.c,v $
 * 
 * Revision 1.1  90/06/17  03:34:51  zhang
 * Initial revision
 * 
 * Revision 1.2  90/12/12  15:30:21  zhang
 * Add backend for RADIANCE output
 */

#include "defs.h"

/*
 * a CIRCLE is treated like this
 *
 * 1. if its has thickness, then it defines a closed cylinder
 * 2. if its has not, the it defines a circle disk
 * 3. if it defines a cylinder, and the reference point of the layer
 *       is not defined and the layer name has an "_I" suffix,
 *    then the orientation of the cylinder has to be reversed
 */

typedef	struct	circle	{
	FLOAT	center[3];		/* center point of the circle */
	FLOAT	radius;			/* radius of the circle */
} CIRCLE;

/*
 * parse a CIRCLE
 */

ENTITY	*CircleDxfParse(layerlist)
LAYER	**layerlist;
{
	ENTITY	*entity;
	CIRCLE	*circle;
	INT	centerset = 0;
	INT	radiusset = 0;
	INT	defaultflag = 0;

	entity = Malloc(ENTITY, 1);
	circle = Malloc(CIRCLE, 1);
	entity->type = ENTITY_CIRCLE;
	entity->data = (VOID *) circle;

	entity->layer = LayerDxfParse(layerlist);

	do {
		GetNextGroup();
		switch(Group->code) {
		case 10:
			/*
			 * center point
			 * 10, 20, 30
			 */

			if (centerset != 0)
				DXFERR("duplicated center for a CIRCLE %s", "\n");
			centerset = 1;
			CoordDxfParse(0, circle->center);
			break;

		case 40:
			/*
			 * radius
			 */

			if (radiusset != 0)
				DXFERR("duplicated radius for a CIRCLE %s", "\n");

			radiusset = 1;
			circle->radius = Group->fltnum;
			break;

		default:
			if (OptionsDxfParse(&entity->options) == 0)
				defaultflag = 1;
			break;
		}
	} while (defaultflag == 0);

	/*
	 * check if center point and radius are all defined
	 */

	if (centerset == 0)
		DXFERR("center of a CIRCLE undefined %s", "\n");

	if (radiusset == 0)
		DXFERR("radius of a CIRCLE undefined %s", "\n");

	return(entity);
}

/*
 * tesselate a triangle of a circle and output faces into DEF/NFF/DXF file
 */

VOID	CircleTriangleTesselation(n, center, start, end, entity)
INT	n;
FLOAT	center[3];
FLOAT	start[3];
FLOAT	end[3];
ENTITY	*entity;
{
	INT	i;
	FLOAT	sdelta[3];
	FLOAT	edelta[3];
	FLOAT	polygon[4][3];

	if (n <= 0)
		n = 1;

	VecSub(start, center, sdelta);
	VecSub(end  , center, edelta);
	VecScale(1.0 / (FLOAT) n, sdelta, sdelta);
	VecScale(1.0 / (FLOAT) n, edelta, edelta);

	for (i = n; i > 1; i--) {
		VecScale((FLOAT) i, sdelta, polygon[0]);
		VecScale((FLOAT) i, edelta, polygon[1]);
		VecScale((FLOAT) (i - 1), edelta, polygon[2]);
		VecScale((FLOAT) (i - 1), sdelta, polygon[3]);
		VecAdd(polygon[0], center, polygon[0]);
		VecAdd(polygon[1], center, polygon[1]);
		VecAdd(polygon[2], center, polygon[2]);
		VecAdd(polygon[3], center, polygon[3]);
		OutputPolygon(4, polygon, entity);
	}

	VecAdd(edelta, center, polygon[0]);
	VecCopy(center, polygon[1]);
	VecAdd(sdelta, center, polygon[2]);
	OutputPolygon(3, polygon, entity);
}

/*
 * tesselation the side faces (rectangles) of a cylinder
 * and output faces into DEF/NFF/DXF file
 */

/*ARGSUSED*/
VOID	CircleRectangleTesselation(m, n, p0, p1, p2, p3, entity)
INT	m, n;
FLOAT	p0[3], p1[3], p2[2], p3[3];
ENTITY	*entity;
{
	INT	i, j;
	FLOAT	mdelta[3];
	FLOAT	ndelta[3];
	FLOAT	polygon[4][3];

	if (m <= 0)
		m = 1;
	if (n <= 0)
		n = 1;

	VecSub(p1, p0, mdelta);
	VecSub(p3, p0, ndelta);
	VecScale(1.0 / (FLOAT) m, mdelta, mdelta);
	VecScale(1.0 / (FLOAT) n, ndelta, ndelta);

	for (i = 0; i < m; i++)
		for (j = 0; j < n; j++) {
		VecScale((FLOAT) i, mdelta, polygon[0]);
		VecScale((FLOAT) j, ndelta, polygon[1]);
		VecAdd(polygon[0], polygon[1], polygon[0]);
		VecAdd(polygon[0], p0, polygon[0]);
		VecAdd(polygon[0], mdelta, polygon[1]);
		VecAdd(polygon[1], ndelta, polygon[2]);
		VecAdd(polygon[0], ndelta, polygon[3]);
		OutputPolygon(4, polygon, entity);
	}
}

/*
 * calculate tesselation number for a circle
 */

INT	CircleTesselationNumber(radius)
FLOAT	radius;
{
	INT	n;

	if (radius < MinCircleRadius)
		return(MinCircleTesselation);

	n = (INT) ((radius / MinCircleRadius) * (FLOAT) MinCircleTesselation);

	return(min(n, MaxCircleTesselation - 1));
}

/*
 * output a disk defined by the circle into DEF/NFF/DXF file
 */

VOID	CircleDiskOutput(entity, circle)
ENTITY	*entity;
CIRCLE	*circle;
{
	INT	i;
	INT	n;
	FLOAT	deltaangle;
	FLOAT	*extrusion;
	MATRIX	matrix[2];
	FLOAT	delta[3];
	INT	ntesselation;
	FLOAT	p[3];
	FLOAT	polygon[MaxCircleTesselation][3];

	extrusion = OptionsExtrusion(entity->options);

	MatrixExtrusion(matrix, extrusion);

	n = CircleTesselationNumber(circle->radius);
	deltaangle = 2.0 * M_PI / (FLOAT) n;

	for (i = 0; i <= n; i++) {
		p[0] = circle->radius * cos(deltaangle * (FLOAT) i);
		p[1] = circle->radius * sin(deltaangle * (FLOAT) i);
		p[2] = 0.0;
		XformPoint(matrix[0], p, polygon[i]);
		VecAdd(polygon[i], circle->center, polygon[i]);
	}

	if (OutFileType == FILE_NFF) {
		OutputPolygon(n, polygon, entity);
		return;
	}

	/*
	 * tesselation triangles of the circle
	 */

	VecSub(polygon[0], polygon[1], delta);
	ntesselation = (INT) (circle->radius / VecLength(delta) + 0.5);
	if (ntesselation <= 0)
		ntesselation = 1;

	for (i = 0; i < n; i++)
		CircleTriangleTesselation(ntesselation, circle->center,
			polygon[i], polygon[i + 1], entity);
}

/*
 * output faces defined by a circle into DEF/DXF file
 */

VOID	CircleDefOutput(entity)
ENTITY	*entity;
{
	CIRCLE	*circle;
	FLOAT	thickness;
	FLOAT	*extrusion;
	FLOAT	delta[3];
	FLOAT	center[3];
	INT	i;
	INT	n;
	FLOAT	deltaangle;
	FLOAT	width;
	FLOAT	height;
	INT	ntesselation;
	INT	heighttesselation;
	MATRIX	matrix[2];
	FLOAT	p[3];
	FLOAT	polygon[2][MaxCircleTesselation][3];
	FLOAT	a, b, c, d;

	circle = (CIRCLE *) entity->data;

	if (circle->radius < TOE)
		return;

	NameOutput(entity->layer);

	if (OptionsThickness(entity->options, &thickness) == 0) {
		CircleDiskOutput(entity, circle);
		return;
	}

	extrusion = OptionsExtrusion(entity->options);

	VecScale(thickness, extrusion, delta);
	VecAdd(delta, circle->center, center);

	height = VecLength(delta);

	MatrixExtrusion(matrix, extrusion);

	n = CircleTesselationNumber(circle->radius);
	deltaangle = 2.0 * M_PI / (FLOAT) n;

	for (i = 0; i <= n; i++) {
		p[0] = circle->radius * cos(deltaangle * (FLOAT) i);
		p[1] = circle->radius * sin(deltaangle * (FLOAT) i);
		p[2] = 0.0;
		XformPoint(matrix[0], p, polygon[0][i]);
		VecAdd(polygon[0][i], circle->center, polygon[0][i]);
		VecAdd(polygon[0][i], delta, polygon[1][i]);
	}

	/*
	 * check the orientation of the circle against the
	 * extrusion direction and make the normals of disks point outside
	 */

	PlaneEquation(n, polygon[0], &a, &b, &c, &d);
	d = a * extrusion[0] + b * extrusion[1] + c * extrusion[2];
	if (d > TOE)
		ReversePolygon(n + 1, polygon[0]);
	else
	if (d < - TOE)
		ReversePolygon(n + 1, polygon[1]);
	else
		GENERR("cannot extrude a CIRCLE %s", "\n");

	/*
	 * check the orientation against the reference layer name
	 * or layer reference point
	 * the orientations of the two circle should be changed
	 * simulatously
	 */

	if (CheckOrientation(n, polygon[0], entity->layer) !=
	    CheckOrientation(n, polygon[1], entity->layer))
		GENERR("two disks of a cylinder in layer %s have the same orientation after checking\n",
			entity->layer->name);

	/*
	 * if there is no reference point for the layer
	 * check if the layer name has an "_I" suffix
	 * if it has, then reverse the orientations of the circle
	 */

	if (entity->layer->point == NULL)
		if (CheckLayerName(entity->layer) != 0) {
			ReversePolygon(n + 1, polygon[0]);
			ReversePolygon(n + 1, polygon[1]);
		}

	/*
	 * tesselation the two circles
	 */

	/*
	 * tesselation triangles of the circle
	 */

	VecSub(polygon[0][0], polygon[0][1], delta);
	ntesselation = (INT) (circle->radius / VecLength(delta) + 0.5);
	if (ntesselation <= 0)
		ntesselation = 1;

	for (i = 0; i < n; i++)
		CircleTriangleTesselation(ntesselation, circle->center,
			polygon[0][i], polygon[0][i + 1], entity);

	for (i = 0; i < n; i++)
		CircleTriangleTesselation(ntesselation, center,
			polygon[1][i], polygon[1][i + 1], entity);

	/*
	 * tesselation side faces
	 * note: the direction in width is not tesselated
	 */

	VecSub(polygon[0][0], polygon[0][1], delta);
	width = VecLength(delta);
	heighttesselation = (INT) (height / width + 0.5);
	if (heighttesselation <= 0)
		heighttesselation = 1;

	for (i = 0; i < n; i++)
		CircleRectangleTesselation(1, heighttesselation,
			polygon[0][i + 1], polygon[0][i],
			polygon[1][n - i], polygon[1][n - i - 1],
			entity);
}

/*
 * output faces defined by a circle into NFF file
 */

VOID	CircleNffOutput(entity)
ENTITY	*entity;
{
	CIRCLE	*circle;
	FLOAT	thickness;
	FLOAT	*extrusion;
	FLOAT	delta[3];
	FLOAT	center[3];

	circle = (CIRCLE *) entity->data;

	if (circle->radius < TOE)
		return;

	NameOutput(entity->layer);

	if (OptionsThickness(entity->options, &thickness) == 0) {
		CircleDiskOutput(entity, circle);
		return;
	}

	extrusion = OptionsExtrusion(entity->options);

	VecScale(thickness, extrusion, delta);

	fprintf(OutFile, "cone %g %g %g %g",
			circle->center[0], circle->center[1], circle->center[2],
			circle->radius);

	VecAdd(delta, circle->center, center);

	fprintf(OutFile, " %g %g %g %g\n",
			center[0], center[1], center[2], circle->radius);
}

/*
 * output faces defined by a circle into RAD file
 */

VOID	CircleRadOutput(entity)
ENTITY	*entity;
{
	CIRCLE	*circle;
	FLOAT	thickness;
	FLOAT	*extrusion;
	FLOAT	delta[3];
	FLOAT	center[3];

	circle = (CIRCLE *) entity->data;

	if (circle->radius < TOE)
		return;


	if (OptionsThickness(entity->options, &thickness) == 0) {
		NameOutput(entity->layer);
		CircleDiskOutput(entity, circle);
		return;
	}

	NameRadOutput(entity->layer);

	extrusion = OptionsExtrusion(entity->options);

	VecScale(thickness, extrusion, delta);

	fprintf(OutFile, "cylinder %d\n0\n0\n7%g %g %g\n", RadId++,
			circle->center[0], circle->center[1], circle->center[2]);
	VecAdd(delta, circle->center, center);

	fprintf(OutFile, "%g %g %g\n%g\n",
			center[0], center[1], center[2], circle->radius);
}
