/**********************************************************************/
/* rtime.c :                                                          */
/*                                                                    */
/* Program timing routines                                            */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#include <stdio.h>
#include <sys/time.h>
#include <sys/resource.h>
#include "rtime.h"
#include "io.h"

extern OptionType Option;
extern FILE *rlogfile;
Time_Stats tstats;

/**********************************************************************/
/* Get current user and system time                                   */
/**********************************************************************/
float Cpu_Time(utime, stime)
     float *utime, *stime;
{
  struct rusage usage;
  
  if (!getrusage(RUSAGE_SELF, &usage)) {
    *utime = (float) usage.ru_utime.tv_sec + 
      (float) usage.ru_utime.tv_usec / 1000000.0;
    *stime = (float) usage.ru_stime.tv_sec + 
      (float) usage.ru_stime.tv_usec / 1000000.0;
  } else {
    fprintf(stderr,"GetTime: can't get time using getrusage\n");
    exit(1);
  }
  return (*utime + *stime);
}  

/**********************************************************************/
/* Return time elapsed since last check                               */
/**********************************************************************/
float Elapsed_Time()
{
  float elapsed;

  Cpu_Time(&tstats.utime, &tstats.stime);
  tstats.cur_time = tstats.utime + tstats.stime;
  elapsed = tstats.cur_time - tstats.last_time;

  /* printf("last = %g, cur = %g\n", tstats.last_time, tstats.cur_time); */
  tstats.last_time = tstats.cur_time;

  return elapsed;
}

/**********************************************************************/
/* Initialize time record                                             */
/**********************************************************************/
void Init_Time()
{
  tstats.tot_time = 0.0; 
  tstats.last_time = 0.0;
  tstats.cur_time = 0.0;
  tstats.avg_ray = 0.0;
  tstats.avg_shaft = 0.0;
  tstats.avg_ff = 0.0;
  tstats.avg_iter = 0.0;
  tstats.prep_oHBV = 0.0;
  tstats.prep_pHBV = 0.0;
  tstats.prep_Model = 0.0;
}

/**********************************************************************/
/* Print times */
/**********************************************************************/
void Print_Times(fptr, use_shaft)
     FILE *fptr;
     int use_shaft;
{
  FILE *fp;

  if (Option.device == PRINT)
    fp = stdout;
  else 
    fp = fptr;

  fprintf(fp,"\n\tCPU times (sec):\n");
  fprintf(fp,  "\t----------------\n");
  fprintf(fp,"\tProg. Refine: %g; Prep: (pHBV): %g; (oHBV): %g; (Mod): %g\n",
	  tstats.tot_time, tstats.prep_pHBV, tstats.prep_oHBV, 
	  tstats.prep_Model);
  fprintf(fp,"\tAverage: ");
  fprintf(fp,"iter(%g); ", tstats.avg_iter);
  fprintf(fp,"ff(%g); ", tstats.avg_ff);
  if (use_shaft) fprintf(fp,"shaft(%g); ",
				     tstats.avg_shaft);
  fprintf(fp,"ray(%g)\n\n", tstats.avg_ray);

  if (Option.tablelog) {
    fp = rlogfile;
    fprintf(fp,"%g \\\\\n%g \\\\\n%g \\\\\n%g \\\\\n",
	    tstats.tot_time, tstats.prep_Model,
	    tstats.prep_pHBV, tstats.prep_oHBV);
    fprintf(fp,"%g \\\\\n", tstats.avg_iter);
    fprintf(fp,"%g \\\\\n", tstats.avg_ff);
    if (use_shaft) fprintf(fp,"%g \\\\\n",
			   tstats.avg_shaft);
    fprintf(fp,"%g \\\\\n", tstats.avg_ray);
  }
}
