/**********************************************************************/
/* raysp.c                                                            */
/*                                                                    */
/* Routines to reduce the number of rays to shoot                     */
/* - Culling of planes against planes                                 */
/* - Culling of bounding volumes against planes                       */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "misc.h"
#include "geo.h"
#include "struct.h"

/**********************************************************************/
/* Test if polygon vertices are completely, partially, or totally     */ 
/* behind the plane of another poly. Includes test if two polys are   */
/* coplanar, and face the same direction (normals point in same       */
/* direction).                                                        */
/**********************************************************************/
int Poly_Behind_Poly(testp, againstp)
     Polygon *testp, *againstp;
{
  int i, behind, front, on, is_behind;
  double plane_val;

  /* If 2 polys belong to same patch then by definition of a patch
     cannot "see" each other */
  if ((testp->Pfather == againstp->Pfather) && (testp->Pfather != 0))
    is_behind = 1;
    
  /* Count number of vertices in front, behind plane, and on the plane */
  else {
    behind = front = on = 0;
    for(i=0;i<testp->numVert;i++) {
      plane_val = dot(&testp->vert[i]->pos, &againstp->normal[0]) +
	againstp->d;
      if (plane_val < 0.0)
	behind++;
      else if (plane_val > 0.0)
	front++;
      else
	on++;
    }
    if (front == testp->numVert)
      is_behind = 0;
    else if ((behind == testp->numVert) || (on == testp->numVert))
      is_behind = 1;
    else
      is_behind = 0;
  }
  return is_behind;
}

/**********************************************************************/
/* Backface culling of a vector */
/**********************************************************************/
int Vertex_Behind_Plane(P, N, d) 
     Vector P, N;
     double d;
{ return ((dot(&P,&N) + d) <= 0.0 ? 1 : 0); }

/**********************************************************************/
/* Find nearest point on box from plane */
/**********************************************************************/
Vector Nearest_Point(box,N,d)
     BoundingBoxType *box;        /* Bounding box to test */
     Vector N;                    /* Plane normal */
     double d;                    /* Plane d */
{
  Vector P;
  
  P.x = (N.x < 0.0 ? box->max.x : box->min.x);
  P.y = (N.y < 0.0 ? box->max.y : box->min.y);
  P.z = (N.z < 0.0 ? box->max.z : box->min.z);

  return P;
}

/**********************************************************************/
/* Find farthest point on box from plane */
/**********************************************************************/
Vector Farthest_Point(box,N,d)
     BoundingBoxType *box;        /* Bounding box to test */
     Vector N;                    /* Plane normal */
     double d;                    /* Plane d */
{
  Vector P;
  
  P.x = (N.x > 0.0 ? box->max.x : box->min.x);
  P.y = (N.y > 0.0 ? box->max.y : box->min.y);
  P.z = (N.z > 0.0 ? box->max.z : box->min.z);

  return P;
}

/**********************************************************************/
/* Backface culling of axis-aligned bounding box */
/**********************************************************************/
int Bounds_Behind_Plane(box, N, d)
     BoundingBoxType *box;        /* Bounding box to test */
     Vector N;                    /* Plane normal */
     double d;                    /* Plane d */
{
  Vector P; /* Farthest point on box from plane */

  P = Farthest_Point(box,N,d);
  return(Vertex_Behind_Plane(P, N, d));
}
