/**********************************************************************/
/* io.c                                                               */
/*                                                                    */
/* Set input and output options.                                      */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "geo.h"
#include "struct.h"
#include "misc.h"
#include "io.h"
#include "rad.h"
#include "ff.h"
#include "scull.h"

extern OptionType Option;
extern FF_OptionType FF_Options;
extern RadParams ReadLog;
extern ShaftStatstype ShaftStats;
extern void ParseArgs();
void SetFilenames();

/**********************************************************************/
OptionType Option;            /* Program options */
extern void Usage();

/**********************************************************************/
/* Switch stats log on / off */
/**********************************************************************/
void LogStats(logflag)
     int logflag;
{
  if (logflag) {
    if (!(Option.StatFile = fopen(Option.StatFilename, "w"))) {
      fprintf(stderr,"%s: cannot open statistics file %s\n", 
	      ProgName, Option.StatFilename);
      exit(1);
    } 
    printf("\n\t*** Logging statistics to %s ***\n", Option.StatFilename);
  } else 
    fclose(Option.StatFile);
}

/**********************************************************************/
/* Set radiosity parameters */
/**********************************************************************/
void SetRadParams()
{
  char *env_var = " ";

  ReadLog.log = TRUE;           /* log reading of objects */ 
  ReadLog.objcount = 0;         /* objects read */
  ReadLog.meshcount = 0;        /* meshes read */
  ReadLog.polycount = 0;        /* polygons per mesh */
  ReadLog.totpoly = 0;          /* total polys */
  ReadLog.totmesh = 0;          /* total meshes */
  ReadLog.num_objects = 0;      /* total objects */
  ReadLog.num_textures = 0;     /* total textures */
  ReadLog.num_elements = 0;     /* no elements */
  ReadLog.num_receivers = 0;    /* no receivers */
  ReadLog.elements =            /* empty element list, and tail */
    ReadLog.eltail = (Elist *)NULL; 
  ReadLog.threshold =           /* Stopping criteria threshold */
    RAD_THRESHOLD;
  ReadLog.max_iterations =      /* Stopping # of iterations */
    RAD_ITERATIONS;
  ReadLog.intensityScale = 1.0; /* Scale intensity for display */
  ReadLog.hemicubeRes = 100;    /* 100 pixels for hc resolution */
  ReadLog.writerad = 0;         /* Do not write out rad solution */
  ReadLog.worldSize = 0.0;      /* World not created yet */
  ReadLog.totalEnergy = 0.0;
  ReadLog.totalEnergyLeft = 0.0;
  ReadLog.totalArea = 0.0;       
  ReadLog.ptotalArea.samples[0] = -1.0;

  if ((env_var = getenv("PR_maxiter")) != NULL) 
    ReadLog.max_iterations = atoi(env_var);  
}

#define FF_AREA_RATIO 0.01     /* Ratio of area of minimum sized
				  patch to area of the world */
/**********************************************************************/
/* Set default form-factor options */
/**********************************************************************/
void SetFFOptions(argv)
     char *argv[];
{
  int temp;
  char *env_var = " ";
  
  /* Numerical disk approx, without adaptve ray sampling, nor
     analytic form factor usage (default) */
  FF_Options.fftype = NUMERICAL_FF;   
  FF_Options.sample_shape = DISC_FF;
  FF_Options.varying_numsamps = 0;
  FF_Options.use_analytic = 0;
  FF_Options.src_rec_cull = 0;      /* Use src/rec object BV culling */
  FF_Options.quadtri_ray = 0;       /* Use general ray/poly intersect
				       routine if ray-casting form-factors */
  FF_Options.shaft_cull = 0;

  if ((env_var = getenv("PR_shaft")) != NULL) 
    FF_Options.shaft_cull = atoi(env_var);  
  if (FF_Options.shaft_cull) {  
    ShaftStats.strategy = RATIO_OPEN; /* Use ratio open = 40% default */
    ShaftStats.ratio_open = 0.4;
    if ((env_var = getenv("PR_cstrag")) != NULL) 
      ShaftStats.strategy = atoi(env_var); 
  }
  if ((env_var = getenv("PR_quadtri")) != NULL) 
    FF_Options.quadtri_ray = atoi(env_var);  
  if ((env_var = getenv("PR_rcull")) != NULL) 
    FF_Options.src_rec_cull = atoi(env_var);  
  if ((env_var = getenv("PR_maxsub")) != NULL) 
    FF_Options.max_levels = atoi(env_var);
  if ((env_var = getenv("PR_adjsamp")) != NULL) 
    FF_Options.varying_numsamps = atoi(env_var);
  if ((env_var = getenv("PR_analyt")) != NULL) 
    FF_Options.use_analytic = atoi(env_var);
  if ((env_var = getenv("FF_mindiff")) != NULL) 
    FF_Options.F_diff_edge = atof(env_var);
  if ((env_var = getenv("FF_perarea")) != NULL) 
    FF_Options.min_element_area = atof(env_var);
  
  /* Settings only for linkage forming [Hanrahan91] (not used) */
  if (ReadLog.forming_links) {
    FF_Options.min_element_area = FF_AREA_RATIO * ReadLog.totalArea;
    FF_Options.num_samples = DEFAULT_LSAMPLES;
  } 
  /* Settings only for progressive refinement */
  else {
    FF_Options.num_samples = DEFAULT_SAMPLES;
  }
}

/**********************************************************************/
/* Set default options                                                */
/**********************************************************************/
void SetOptions(argc, argv)
     int argc;
     char *argv[];
{
  char *env_var = " ";

  Option.ff_raytrace = TRUE;          /* Use ray casting for FF */
  Option.device = PRINT;              /* Default output stats to stdout */
  Option.debug = FALSE;               /* No debug */
  Option.statistics = FALSE;          /* Don't print statistics */
  Option.visibility = 0;              /* Using ray casting for FF */
  Option.grid = TRUE;                 /* Use bounding volumes */
  Option.poly_grid = TRUE;            /* Use object / polygon 
					 hierarchical bounding volumes */
  Option.show_pr_steps = FALSE;       /* Show refinement steps */
  Option.print_scene = FALSE;         /* Log input/output scene */
  Option.ambient = 0;                 /* Use an ambient term for display
					 purposes during PR */
  Option.write_result = 0;            /* Don't write results to file */

  if ((env_var = getenv("PR_prcs")) != NULL) 
    Option.print_scene = atoi(env_var);
  if ((env_var = getenv("PR_amb")) != NULL) 
    Option.ambient = atoi(env_var);
  if ((env_var = getenv("PR_grid")) != NULL) 
    Option.grid = atoi(env_var);
  if ((env_var = getenv("PR_pgrid")) != NULL) 
    Option.poly_grid = atoi(env_var);
  if (Option.poly_grid) Option.grid = TRUE;
  if ((env_var = getenv("PR_debug")) != NULL) 
    Option.debug = atoi(env_var);
  if ((env_var = getenv("PR_walk")) != NULL) 
    Option.write_result = atoi(env_var);

  /* Set radiosity options */
  SetRadParams();
  SetFFOptions(argv);
  ParseArgs(argc,argv);
  
  /* Set display options */
  if ((env_var = getenv("PR_display")) != NULL) 
    Option.show_pr_steps=atoi(env_var);
  if (Option.ff_raytrace) {
    Option.rad_interp_type = INTERP_VTX_FROM_VTX;
  } else 
    Option.rad_interp_type = INTERP_VTX_FROM_PATCH;    

  SetFilenames(Option.meshfilename);
  if (Option.statistics)
    if (Option.device == FILES)
      LogStats(1);
}

/**********************************************************************/
/* Set output filenames                                               */
/**********************************************************************/
void SetFilenames(fname)
     char *fname;
{
  Option.StatFilename = "                                        ";
  Option.InLogFilename = "                                        ";
  Option.OutLogFilename = "                                        ";
  Option.DebugFilename = "                                        ";
  Option.OutSceneFilename = "                                        ";

  sprintf(Option.StatFilename,"%s.stat",fname);   /* Statistics log */
  sprintf(Option.InLogFilename,"%s.ilog",fname);  /* Input log */
  sprintf(Option.OutLogFilename,"%s.olog",fname); /* Output log */
  sprintf(Option.OutSceneFilename,"%s.scene",fname); /* Output scene */
  sprintf(Option.DebugFilename, "%s.dbg", fname); /* Log of debug */
}
