
/*  @(#)libpbm.c 1.2 91/12/30
 *
 *  PBM/PGM/PPM routines used by the popi program.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

/*  Based on routines from the PBMPLUS utility libraries.
 *
 *  Copyright (C) 1989 by Jef Poskanzer.
 *
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation.  This software is provided "as is" without express or
 *  implied warranty.
 */

#include <stdio.h>
#include "popi.h"
#include "libpbm.h"

static bit   pbm_getbit P(( FILE * )) ;
static char  pbm_getc   P(( FILE * )) ;

static unsigned char pbm_getrawbyte P(( FILE * )) ;

static int pbm_getint           P(( FILE * )) ;
static int pbm_readmagicnumber  P(( FILE * )) ;
static int putus                P(( unsigned int, FILE * )) ;

static void pbm_readpbminitrest P(( FILE *, int *, int *)) ;
static void pgm_readpgminitrest P(( FILE *, int *, int *, gray * )) ;
static void ppm_readppminitrest P(( FILE *, int *, int *, pixval * )) ; 

static void pm_perror  P(( char * )) ;

static void pbm_readpbmrow       P(( FILE *, bit *,   int, int)) ;

static void pgm_writepgmrowraw   P(( FILE *, gray *,  int, int)) ;
static void pgm_writepgmrowplain P(( FILE *, gray *,  int, int)) ;
static void ppm_writeppmrowraw   P(( FILE *, pixel *, int, int)) ;
static void ppm_writeppmrowplain P(( FILE *, pixel *, int, int)) ;

static char *pm_progname ;

static bit  *bitrow ;
static gray *grayrow ;

int pm_iserror = FALSE ;


static bit
pbm_getbit(file)
FILE *file ;
{
  register char ch ;

  do
    {
      ch = pbm_getc(file) ;
    }
  while (ch == ' ' || ch == '\t' || ch == '\n') ;

  if (ch != '0' && ch != '1')
    pm_error("junk in file where bits should be",
             (char *) 0, (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;

  return((ch == '1') ? 1 : 0) ;
}


static char
pbm_getc(file)
FILE *file ;
{
  register int ich ;
  register char ch ;

  ich = getc(file) ;
  if (ich == EOF)
    pm_error("premature EOF",
             (char *) 0, (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;
  ch = (char) ich ;

  if (ch == '#')
    {
      do
        {
          ich = getc(file) ;
          if (ich == EOF)
            pm_error("premature EOF", (char *) 0, (char *) 0,
                                      (char *) 0, (char *) 0, (char *) 0) ;
          ch = (char) ich ;
        }
      while ( ch != '\n' );
    }
  return(ch) ;
}


static unsigned char
pbm_getrawbyte(file)
FILE *file ;
{
  register int iby ;
     
  iby = getc(file) ;
  if (iby == EOF)
    pm_error("premature EOF",
             (char *) 0, (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;
  return((unsigned char) iby) ;
}


static int
pbm_getint(file)
FILE *file ;
{
  register char ch ;
  register int i ;
 
  do
    {
      ch = pbm_getc(file) ;
      if (pm_iserror == TRUE) return(0) ;
    }
  while (ch == ' ' || ch == '\t' || ch == '\n') ;

  if (ch < '0' || ch > '9')
    {
      pm_error("junk in file where an integer should be",
               (char *) 0, (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;
      return(0) ;
    }

  i = 0 ;
  do
    {
      i = i * 10 + ch - '0' ;
      ch = pbm_getc(file) ;
      if (pm_iserror == TRUE) return(0) ;
    }
  while (ch >= '0' && ch <= '9') ;

  return(i) ;
}


static int
pbm_readmagicnumber(file)
FILE *file ;
{
  int ich1, ich2 ;
     
  ich1 = getc(file) ;
  if (ich1 == EOF)
    pm_error("premature EOF reading magic number",
             (char *) 0, (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;
  ich2 = getc(file) ;
  if (ich2 == EOF)
    pm_error("premature EOF reading magic number",
              (char *) 0, (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;
  return(ich1 * 256 + ich2) ;
}


static void
pbm_readpbminitrest(file, colsP, rowsP)
FILE *file ;
int *colsP, *rowsP ;
{
  *colsP = pbm_getint(file) ;      /* Read size. */
  *rowsP = pbm_getint(file) ;
}


static void
pbm_readpbmrow(file, bitrow, cols, format)
FILE *file ;
bit *bitrow ;
int cols, format ;
{
  register int col, bitshift ;
  register unsigned char item ;
  register bit *bP ;

  switch (format)
    {
      case PBM_FORMAT  : for (col = 0, bP = bitrow; col < cols; col++, bP++)
                           *bP = pbm_getbit(file) ;
                         break ;

      case RPBM_FORMAT : bitshift = -1 ;
                         for (col = 0, bP = bitrow; col < cols; col++, bP++)
                           {
                             if (bitshift == -1)
                               {
                                 item = pbm_getrawbyte(file) ;
                                 bitshift = 7 ;
                               }
                             *bP = (item >> bitshift) & 1 ;
                             bitshift-- ;
                           }
                         break ;
 
      default          : pm_error("can't happen", (char *) 0, (char *) 0,
                                      (char *) 0, (char *) 0, (char *) 0) ;
    }
}


static void
pgm_readpgminitrest(file, colsP, rowsP, maxvalP)
FILE *file ;
int *colsP, *rowsP ;
gray *maxvalP ;
{
  *colsP = pbm_getint(file) ;         /* Read size. */
  *rowsP = pbm_getint(file) ;

  *maxvalP = pbm_getint(file) ;       /* Read maxval. */
  if (*maxvalP > PGM_MAXMAXVAL)
    pm_error("maxval too large - %d > %d\n",
             (char *) *maxvalP, (char *) PGM_MAXMAXVAL,
             (char *) 0, (char *) 0, (char *) 0) ;
}


/*ARGSUSED*/
void
pgm_readpgmrow(file, grayrow, cols, maxval, format)
FILE *file ;
gray *grayrow ;
int cols, format ;
gray maxval ;
{
  register int col ;
  register gray *gP ;
  register bit *bP ;

  switch (format)
    {
      case PGM_FORMAT  : for (col = 0, gP = grayrow; col < cols; col++, gP++)
                           {
                             *gP = pbm_getint(file) ;
                             if (pm_iserror == TRUE) return ;
                           }
                         break ;

      case RPGM_FORMAT : for (col = 0, gP = grayrow; col < cols; col++, gP++)
                           {
                             *gP = pbm_getrawbyte(file) ;
                             if (pm_iserror == TRUE) return ;
                           }
                         break ;

      case PBM_FORMAT  :
      case RPBM_FORMAT : pbm_readpbmrow(file, bitrow, cols, format) ;
                         gP = grayrow ;
                         bP = bitrow ;
                         for (col = 0; col < cols; col++, gP++, bP++)
                           *gP = (*bP == PBM_WHITE) ? maxval : 0 ;
                         break ;

      default          : pm_error("can't happen", (char *) 0, (char *) 0,
                                      (char *) 0, (char *) 0, (char *) 0) ;
    }
}


void
pgm_writepgminit(file, cols, rows, maxval)
FILE *file ;
int cols, rows ;
gray maxval ;
{
  if (maxval <= 255)
    FPRINTF(file, "%c%c\n%d %d\n%d\n", PGM_MAGIC1, RPGM_MAGIC2,
            cols, rows, maxval) ;
  else
    FPRINTF(file, "%c%c\n%d %d\n%d\n", PGM_MAGIC1, PGM_MAGIC2,
            cols, rows, maxval) ;
}


void
pgm_writepgmrow(file, grayrow, cols, maxval)
FILE *file ;
gray *grayrow ;
int cols ;
gray maxval ;
{
  if (maxval <= 255) pgm_writepgmrowraw(file,   grayrow, cols, maxval) ;
  else               pgm_writepgmrowplain(file, grayrow, cols, maxval) ;
}


/*ARGSUSED*/
static void
pgm_writepgmrowraw(file, grayrow, cols, maxval)
FILE *file ;
gray *grayrow ;
int cols ;
gray maxval ;
{
  register int col ;
  register gray *gP ;

  for (col = 0, gP = grayrow; col < cols; col++, gP++)
    if (putc((char) *gP, file) == EOF) pm_perror((char *) 0) ;
}


/*ARGSUSED*/
static void
pgm_writepgmrowplain(file, grayrow, cols, maxval)
FILE *file ;
gray *grayrow ;
int cols ;
gray maxval ;
{
  register int col, charcount ;
  register gray *gP ;
 
  charcount = 0 ;
  for (col = 0, gP = grayrow; col < cols; col++, gP++)
    {
      if (charcount >= 70)
        {
          if (putc('\n', file) == EOF) pm_perror((char *) 0) ;
          charcount = 0 ;
        }
      if (putus((unsigned int) *gP, file) == EOF) pm_perror((char *) 0) ;
      if (putc(' ', file) == EOF) pm_perror((char *) 0) ;
      charcount += 4 ;
    }    
  if (putc('\n', file) == EOF) pm_perror((char *) 0) ;
}


char *
pm_allocrow(cols, size)
int cols ;
{    
  register char *itrow ;

  itrow = (char *) malloc((unsigned int) (cols * size)) ;
  if (itrow == (char *) 0)
    pm_error("out of memory allocating a row",
             (char *) 0, (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;
  return(itrow) ;
}


void
pm_error(fmt, v1, v2, v3, v4, v5)
char *fmt, *v1, *v2, *v3, *v4, *v5 ;
{
  pm_message(fmt, v1, v2, v3, v4, v5) ;
  pm_iserror = TRUE ;
}


void
pm_freerow(itrow)
char *itrow ;
{
  FREE(itrow) ;
}


int
pm_maxvaltobits(maxval)         /* Log base two hacks. */
int maxval ;
{
       if (maxval <= 1)     return(1) ;
  else if (maxval <= 3)     return(2) ;
  else if (maxval <= 7)     return(3) ;
  else if (maxval <= 15)    return(4) ;
  else if (maxval <= 31)    return(5) ;
  else if (maxval <= 63)    return(6) ;
  else if (maxval <= 127)   return(7) ;
  else if (maxval <= 255)   return(8) ;
  else if (maxval <= 511)   return(9) ;
  else if (maxval <= 1023)  return(10) ;
  else if (maxval <= 2047)  return(11) ;
  else if (maxval <= 4095)  return(12) ;
  else if (maxval <= 8191)  return(13) ;
  else if (maxval <= 16383) return(14) ;
  else if (maxval <= 32767) return(15) ;
  else if ((long) maxval <= 65535L ) return(16) ;
  else pm_error("maxval of %d is too large!", maxval,
                (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;
/*NOTREACHED*/
}


void
pm_message(fmt, v1, v2, v3, v4, v5)
char *fmt, *v1, *v2, *v3, *v4, *v5 ;
{
  FPRINTF(stderr, "%s: ", pm_progname) ;
  FPRINTF(stderr, fmt, v1, v2, v3, v4, v5) ;
  FPUTC('\n', stderr) ;
}


static void
pm_perror(reason)
char *reason ;
{
  extern char *sys_errlist[] ;
  extern int errno ;
  char *e ;

  if (errno == 0) e = "end of file" ;        /* Not particularly kosher. */
  else            e = sys_errlist[errno] ;

  if (reason != 0 && reason[0] != '\0')
    pm_error("%s - %s", reason, e, (char *) 0, (char *) 0, (char *) 0) ;
  else
    pm_error("%s", e, (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;
}


void
ppm_readppminit(file, colsP, rowsP, maxvalP, formatP)
FILE *file ;
int *colsP, *rowsP, *formatP ;
pixval *maxvalP ;
{
  *formatP = pbm_readmagicnumber(file) ;      /* Check magic number. */
  switch (*formatP)
    {
      case PPM_FORMAT  :
      case RPPM_FORMAT : ppm_readppminitrest(file, colsP, rowsP, maxvalP) ;
                         break ;

      case PGM_FORMAT  :
      case RPGM_FORMAT : pgm_readpgminitrest(file, colsP, rowsP, maxvalP) ;
                         grayrow = pgm_allocrow(*colsP) ;
                         break ;

      case PBM_FORMAT  :
      case RPBM_FORMAT : pbm_readpbminitrest(file, colsP, rowsP) ;
                         *maxvalP = PPM_PBMMAXVAL ;
                         bitrow = pbm_allocrow(*colsP) ;
                         break ;

      default          : *formatP = OLD_POPI_FORMAT ;  /* Let's hope it is. */
                         rewind(file) ;
    }
}


static void
ppm_readppminitrest(file, colsP, rowsP, maxvalP)
FILE *file ;
int *colsP, *rowsP ;
pixval *maxvalP ;
{
  *colsP = pbm_getint(file) ;      /* Read size. */
  *rowsP = pbm_getint(file) ;

  *maxvalP = pbm_getint(file) ;    /* Read maxval. */
  if (*maxvalP > PPM_MAXMAXVAL)
    pm_error("maxval too large - %d > %d",
             (char *) *maxvalP, (char *) PPM_MAXMAXVAL,
             (char *) 0, (char *) 0, (char *) 0) ;
}


void
ppm_readppmrow(file, pixelrow, cols, maxval, format)
FILE *file ;
pixel *pixelrow ;
int cols, format ;
pixval maxval ;
{
  register int col ;
  register pixel *pP ;
  register pixval r, g, b ;
  register gray *gP ;

  switch (format)
    {
      case PPM_FORMAT  : for (col = 0, pP = pixelrow; col < cols; col++, pP++)
                           {
                             r = pbm_getint(file) ;
                             g = pbm_getint(file) ;
                             b = pbm_getint(file) ;
                             if (pm_iserror == TRUE) return ;
                             PPM_ASSIGN(*pP, r, g, b) ;
                           }
                         break ;

      case RPPM_FORMAT : for (col = 0, pP = pixelrow; col < cols; col++, pP++)
                           {
                             r = pbm_getrawbyte(file) ;
                             g = pbm_getrawbyte(file) ;
                             b = pbm_getrawbyte(file) ;
                             if (pm_iserror == TRUE) return ;
                             PPM_ASSIGN(*pP, r, g, b) ;
                           }
                         break ;

      case PGM_FORMAT  :
      case RPGM_FORMAT : pgm_readpgmrow(file, grayrow, cols, maxval, format) ;
                         gP = grayrow ;
                         pP = pixelrow ;
                         for (col = 0; col < cols; col++, gP++, pP++)
                           {
                             r = *gP ;
                             if (pm_iserror == TRUE) return ;
                             PPM_ASSIGN(*pP, r, r, r) ;
                           }
                         break ;

      default          : pm_error("can't happen", (char *) 0, (char *) 0,
                                      (char *) 0, (char *) 0, (char *) 0) ;
    }
}


void
ppm_writeppminit(file, cols, rows, maxval)
FILE *file ;
int cols, rows ;
pixval maxval ;
{
  if (maxval <= 255)
    FPRINTF(file, "%c%c\n%d %d\n%d\n", PPM_MAGIC1, RPPM_MAGIC2,
            cols, rows, maxval) ;
  else
    FPRINTF(file, "%c%c\n%d %d\n%d\n", PPM_MAGIC1, PPM_MAGIC2,
            cols, rows, maxval) ;
}


void
ppm_writeppmrow(file, pixelrow, cols, maxval)
FILE *file ;
pixel *pixelrow ;
int cols ;
pixval maxval ;
{
  if (maxval <= 255) ppm_writeppmrowraw(file, pixelrow, cols, maxval) ;
  else               ppm_writeppmrowplain(file, pixelrow, cols, maxval) ;
}


/*ARGSUSED*/
static void
ppm_writeppmrowraw(file, pixelrow, cols, maxval)
FILE *file ;
pixel *pixelrow ;
int cols ;
pixval maxval ;
{
  register int col ;
  register pixel *pP ;
  register pixval val ;

  for (col = 0, pP = pixelrow; col < cols; col++, pP++)
    {
      val = PPM_GETR(*pP) ;
      if (putc((char) val, file) == EOF) pm_perror((char *) 0) ;
      val = PPM_GETG(*pP) ;
      if (putc((char) val, file) == EOF) pm_perror((char *) 0) ;
      val = PPM_GETB(*pP) ;
      if (putc((char) val, file) == EOF) pm_perror((char *) 0) ;
    }
}


/*ARGSUSED*/
static void
ppm_writeppmrowplain(file, pixelrow, cols, maxval)
FILE *file ;
pixel *pixelrow ;
int cols;
pixval maxval ;
{
  register int col, charcount ;
  register pixel *pP ;
  register pixval val ;

  charcount = 0 ;
  for (col = 0, pP = pixelrow; col < cols; col++, pP++)
    {
      if (charcount >= 70)
        {
          if (putc('\n', file) == EOF) pm_perror((char *) 0) ;
          charcount = 0 ;
        }
      val = PPM_GETR(*pP) ;
      if (putus(val, file) == EOF) pm_perror((char *) 0) ;
      if (putc(' ', file)  == EOF) pm_perror((char *) 0) ;
      val = PPM_GETG(*pP) ;
      if (putus(val, file) == EOF) pm_perror((char *) 0) ;
      if (putc(' ', file)  == EOF) pm_perror((char *) 0) ;
      val = PPM_GETB(*pP) ;
      if (putus(val, file) == EOF) pm_perror((char *) 0) ;
      if (putc(' ', file)  == EOF) pm_perror((char *) 0) ;
      if (putc(' ', file)  == EOF) pm_perror((char *) 0) ;
      charcount += 13 ;
    }
  if (putc('\n', file) == EOF) pm_perror((char *) 0) ;
}


static int
putus(n, file)
unsigned int n ;
FILE *file ;
{
  if (n >= 10)
    if (putus(n / 10, file) == EOF) return(EOF) ;
  return(putc((char) (n % 10 + '0'), file)) ;
}
