
/*  @(#)defgen.c 1.2 91/12/30
 *
 *  Default (slow) code generation routines for the current popi expression.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc.
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#include "popi.h"
#include "expr.h"

static int isnew ;     /* Set if expresion starts: new[x,y] = ... */

static void binop    P((Tree *, int)) ;
static void emit     P((int)) ;
static void gen      P((Tree *)) ;
static void unop     P((Tree *, int)) ;


void
compile(t)
Tree *t ;
{
  isnew = prs = 0 ;
  gen(t) ;
  emit((isnew) ? OP_ASGN : OP_AT) ;
  if (noerr) run() ;
}


static void
binop(t, op)
Tree *t ;
int op ;
{
  gen(t->kids[0]) ;
  gen(t->kids[1]) ;
  emit(op) ;
}


static void
unop(t, op)
Tree *t ;
int op ;
{
  gen(t->kids[0]) ;
  emit(op) ;
}


static void
emit(what)
int what ;
{
  DEBUG((Debug, "emit(%d)\n", what)) ;

  if (prs >= MAXTOKENS)
    {
      SPRINTF(ErrBuf, "expression too long") ;
      run_error(ERR_PARSE) ;
    }
  parsed[prs++] = what ;
}


static void
gen(t)
Tree *t ;
{
  int remem1, remem2 ;

  if (!t) return ;
  switch (t->t)
    {
      case T_Abs    : unop(t, OP_ABS) ;
                      break ;
      case T_Add    : binop(t, OP_PLUS) ;
                      break ;
      case T_And    : binop(t, OP_AND) ;
                      break ;
      case T_Atan   : binop(t, OP_ATAN) ;
                      break ;
      case T_Bang   : emit(OP_NOT) ;
                      break ;
      case T_Cond   : gen(t->kids[0]) ;
                      emit(OP_CONDIT) ;
                      remem1 = prs ;
                      emit(0) ;
                      gen(t->kids[1]) ;
                      emit(OP_COLON) ;
                      remem2 = prs ;
                      emit(0) ;
                      parsed[remem1] = prs-1 ;
                      gen(t->kids[2]) ;
                      parsed[remem2] = prs-1 ;
                      break ;
      case T_Coord  : gen(t->kids[0]) ;
                      gen(t->kids[1]) ;
                      break ;
      case T_Cos    : unop(t, OP_COS) ;
                      break ;
      case T_Div    : binop(t, OP_DIV) ;
                      break ;
      case T_Eq     : binop(t, OP_EQ) ;
                      break ;
      case T_File   : if (((Tree *) (t->kids[0]))->i == -1)
                        {
                          STRCPY(ErrBuf, "no such image") ;
                          run_error(ERR_SNARK) ;
                        }
                      gen(t->kids[0]) ;
                      gen(t->kids[1]) ;
                      if (((Tree *) (t->kids[1]))->t == T_Polar)
                        emit(OP_PRVAL) ;
                      else emit(OP_CRVAL) ;
                      break ;
      case T_Fnum   : emit(OP_VALUE) ;
                      emit((int) t->d) ;
                      break ;
      case T_Ge     : binop(t, OP_GE) ;
                      break ;
      case T_Gt     : binop(t, OP_GT) ;
                      break ;
      case T_Hypot  : binop(t, OP_HYPOT) ;
                      break ;
      case T_Inum   : emit(OP_VALUE) ;
                      emit(t->i) ;
                      break ;
      case T_Land   : binop(t, OP_CAND) ;
                      break ;
      case T_Le     : binop(t, OP_LE) ;
                      break ;
      case T_Log    : unop(t, OP_LOG) ;
                      break ;
      case T_Lor    : binop(t, OP_COR) ;
                      break ;
      case T_Lshift : binop(t, OP_LSHIFT) ;
                      break ;
      case T_Lt     : binop(t, OP_LT) ;
                      break ;
      case T_Mod    : binop(t, OP_MOD) ;
                      break ;
      case T_Mul    : binop(t, OP_MUL) ;
                      break ;
      case T_Ne     : binop(t, OP_NE) ;
                      break ;
      case T_Neg    : emit(OP_UMIN) ;
                      break ;
      case T_New    : emit(OP_VALUE) ;
                      emit(getImageNo("new")) ;
                      gen(t->kids[0]) ;
                      if (((Tree *) (t->kids[0]))->t == T_Polar)
                        emit(OP_PLVAL) ;
                      else emit(OP_CLVAL) ;
                      isnew = 1 ;
                      gen(t->kids[1]) ;
                      break ;
      case T_Not    : emit(OP_NOT) ;
                      break ;
      case T_Or     : binop(t, OP_OR) ;
                      break ;
      case T_Pang   : MakePolar() ;     /* Create polar lookup tables */
                      emit(OP_A) ;
                      break ;
      case T_Polar  : gen(t->kids[0]) ;
                      gen(t->kids[1]) ;
                      break ;
      case T_Pow    : binop(t, OP_POW) ;
                      break ;
      case T_Prad   : MakePolar() ;     /* Create polar lookup tables */
                      emit(OP_R) ;
                      break ;
      case T_Rand   : emit(OP_RAND) ;
                      break ;
      case T_Rshift : binop(t, OP_RSHIFT) ;
                      break ;
      case T_Sin    : unop(t, OP_SIN) ;
                      break ;
      case T_Sqrt   : unop(t, OP_SQRT) ;
                      break ;
      case T_Sub    : binop(t, OP_MINUS) ;
                      break ;
      case T_Xcoord : emit(OP_X) ;
                      break ;
      case T_Xor    : binop(t, OP_XOR) ;
                      break ;
      case T_Ycoord : emit(OP_Y) ;
    }
}
