
/*  @(#)trig.c 1.2 93/06/01
 *
 *  Various trig and builtin functions used by popi.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc.
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#include "popi.h"

#define  PI              (3.141592653589793238)
#define  PI_OVER_MAX2    (PI / (MAX_ANG / 2.0))
#define  MA_0            (0)
#define  MA_90           (MAX_ANG / 4)
#define  MA_180          (MAX_ANG / 2)
#define  MA_270          (MAX_ANG * 3 / 4)
#define  MA_360          (MAX_ANG)

#include <math.h>

int sin_cache[MAX_ANG+5], cos_cache[MAX_ANG+5] ;
int tan_cache[MAX_ANG+5], atan_cache[1030] ;

void
cache_trig()
{
  int i ;

  for (i = 0; i < MAX_ANG+5; i++)
    {
      sin_cache[i] = MAX_TRIG * sin(i * PI_OVER_MAX2) ;
      cos_cache[i] = MAX_TRIG * cos(i * PI_OVER_MAX2) ;
      tan_cache[i] = MAX_TRIG * tan(i * PI_OVER_MAX2) ;
    }
  for (i = 0; i < 1030; i++)
    {
      atan_cache[i] = atan(i / 1024.0) * MA_180 / PI ;
    }
#ifdef SYSV
  srand(getpid()) ;
#else
  srandom(getpid());
#endif /*SYSV*/
}


int
isqrt(i)
int i ;
{
#ifdef ISQRT

/* Newton's method sqrt */

  int j = i << 2 ;
  int x, x1 ;
 
  for (x = j/2, x1 = 0; (x1 >> 1) != (x >> 1); )
    {
      x1 = x ;
      x = (x + j / x) / 2 ;
    }
 
  return(x / 2) ;
#else

/* On a SS1, this is faster than the routine above */

  return((int) sqrt(1.0 * i)) ;
#endif
}


#define  abs(x)  (x < 0 ? -(x) : x)


int
dist(x, y, x2, y2)
int x, y, x2, y2 ;
{
  float dx = abs(x-x2) ;
  float dy = abs(y-y2) ;

#ifdef MMDIST

/*  Moler-Morrison algorithm.  Ref: "Programming Pearls" p156
 *  this is slower than the normal sqrt(dx^2 + dy^2) on a sparc, despite it's
 *  algorithmic superiority.  This is probably due to compiler sillyness.
 */
  int i ;

  if (dx < dy)
    {
      float t ;

      t = dx ;
      dx = dy ;
      dy = t ;
    }
         
  if (dx == 0.0) return(dy) ;
         
  for (i = 2; i; i--)
    {
      float r = dy / dx ;
         
      r *= r ;
      r /= (4 + r) ;
      dx += 2 * r * dx ;
      dy *= r ;
    }
  return(dx) ;
#else

/*  On a SS1, with the sun compiler and /usr/lib/libm.il, this is
 *  faster (yep, dx and dy as floats is faster than int).
 */

  return(sqrt(dx*dx + dy*dy)) ;
#endif
}


static struct {
  int adj, scale ;
} octs[8] = {
  { MA_0,    1 },
  { MA_90,  -1 },
  { MA_180, -1 },
  { MA_90,   1 },
  { MA_360, -1 },
  { MA_270,  1 },
  { MA_180,  1 },
  { MA_270, -1 }
} ;

 
int
angle(x, y, x2, y2)
int x, y, x2, y2 ;
{
  int idx = 0, ang ;
  int dx = x - x2 ;
  int dy = y2 - y ;
         
  if (dx == 0 && dy == 0) return(0) ;
         
  if (dx < 0)
    {
      dx = -dx ;
      idx += 2 ;
    }
  if (dy < 0)
    {
      dy = -dy ;
      idx += 4 ;
    }
  if (dx < dy)
    {
      int tmp = dx ;
      dx = dy ;
      dy = tmp ;
      idx++ ;
    }
  ang = atan_cache[(dy * 1024) / dx] ;
  return(octs[idx].adj + (octs[idx].scale == -1 ? -ang : ang)) ;
}


#define  clamp(v, l, u)  ((v) < (l) ? (l) : (v) > (u) ? (u) : (v))


int
polar(a, r)
int a, r ;
{
  int x, y;

  if (abs(a) >= MAX_ANG) a %= MAX_ANG ;
  if (a < 0)             a += MAX_ANG ;

  x = cos_cache[a] * r / MAX_TRIG + Xsize / 2 ;
  y = Ysize - (sin_cache[a] * r / MAX_TRIG + Ysize / 2) ;

  x = clamp(x, 0, Xsize-1);
  y = clamp(y, 0, Ysize-1);
  return(y * Ysize + x) ;
}

int
dorand()
{
#ifdef SYSV
  return(rand() % MAX_TRIG) ;
#else
  return(random() % MAX_TRIG) ;
#endif /*SYSV*/
}


int
doatan(x, y)
int x, y ;
{
  FPRINTF(stderr, "doatan entered.\n") ;
}


int
dohypot(x, y)
int x, y ;
{
  FPRINTF(stderr, "dohypot entered.\n") ;
}
