/*
$Header: undo.c,v 1.5 93/08/30 11:01:50 hmallat Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include "misc.h"
#include "pixel.h"
#include "block.h"
#include "undo.h"
#include "list.h"
#include "X_misc.h"
#include "tools1.h"
#include "error.h"
#include "draw.h"

struct picture *clipboard = NULL;
struct undo_buffer *ud = NULL;
int undid = 0;

/* FUNCTION:       memorize(*pic_p, x1, y1, x2, y2, *other_p, *name)
   DESCRIPTION:    Put data to undo buffer
   PARAMETERS:
   *pic_p          picture where to take from
   x1, y1, x2, y2  rectangle coords
   *other_p        The picture to swap with, if not current one
                   (used when size of the picture changes due to operation)
   *name           name of the operation
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void memorize(struct picture *pic_p, int x1, int y1, int x2, int y2,
	      struct picture *other_p, char *name)
{
  extern struct undo_buffer *ud;
  extern int undid;

  x1 = MAX(0, x1);
  y1 = MAX(0, y1);
  x2 = MIN(pic_p->width-1, x2);
  y2 = MIN(pic_p->height-1, y2);
  
  clear_undo();
  
  ud = (struct undo_buffer *)malloc(sizeof(struct undo_buffer));

  if(ud != NULL)
    {
      if(other_p != NULL)
	{
	  ud->resized = TRUE;
	  ud->pic_p = other_p;
	}
      else
	{
	  ud->resized = FALSE;
	  ud->pic_p = pic_p;
	}
      ud->x = MIN(x1, x2);
      ud->y = MIN(y1, y2);
      ud->name_p = malloc((strlen("Undo ")+strlen(name)+1)*sizeof(char));
      if(ud->name_p == NULL)
	{
	  free(ud);
	  ud = NULL;
	}
      else
	{
	  strcpy(ud->name_p, "Undo ");
	  strcat(ud->name_p, name);
	  ud->data_p = extract_block1(pic_p, MIN(x1, x2), MIN(y1, y2),
				      MAX(x1, x2), MAX(y1, y2), 
				      MASK_BIT | CMASK);
	  if(ud->data_p == NULL)
	    {
	      free(ud->name_p);
	      free(ud);
	      ud = NULL;
	    }
	  else
	    {
	      ud->data_p->min_x = pic_p->min_x;
	      ud->data_p->min_y = pic_p->min_y;
	      ud->data_p->max_x = pic_p->max_x;
	      ud->data_p->max_y = pic_p->max_y;
	    }
	}
    }
  else
    panic("Cannot save undo buffer!", FALSE);
  if(ud != NULL)
    undid = 0;
  update_menu();
}

/* FUNCTION:       clear_undo()
   DESCRIPTION:    Scrap undo buffer
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 30th July 1993.
*/   
void clear_undo()
{
  extern struct undo_buffer *ud;

  if(ud != NULL)
    {
      free(ud->name_p);
      delete_block(ud->data_p);
      free(ud);
      ud = NULL;
    }
}

/* FUNCTION:       undo()
   DESCRIPTION:    swap data between undo buffer and picture
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th July 1993.
*/   
void undo()
{
  extern PLOT xor_data;
  extern struct view *head_p;
  extern struct picture *cur_pic;
  extern struct undo_buffer *ud;
  extern int show_edges;
  struct picture *help;
  struct view *v_p = head_p;
  int yelp;

  all_busy();
  if(ud != NULL)
    if(ud->pic_p == cur_pic)
      if(ud->resized == FALSE)
	{
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  yelp = cur_pic->min_x; cur_pic->min_x = ud->data_p->min_x; 
	  ud->data_p->min_x = yelp;
	  yelp = cur_pic->min_y; cur_pic->min_y = ud->data_p->min_y;
	  ud->data_p->min_y = yelp;
	  yelp = cur_pic->max_x; cur_pic->max_x = ud->data_p->max_x;
	  ud->data_p->max_x = yelp;
	  yelp = cur_pic->max_y; cur_pic->max_y = ud->data_p->max_y;
	  ud->data_p->max_y = yelp;
	  swap_blocks(cur_pic, ud->data_p, ud->x, ud->y);
	  
	  minimize_mask_edge();
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  blockade(MIN(ud->x, cur_pic->min_x), MIN(ud->y, cur_pic->min_y),
		   MAX(ud->x+ud->data_p->width-1, cur_pic->max_x),
		   MAX(ud->y+ud->data_p->height-1, cur_pic->max_y), TRUE);
	  undid = !undid;
	}
      else
	{
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  help = cur_pic;
	  cur_pic = ud->data_p;
	  ud->pic_p = cur_pic;
	  ud->data_p = help;
	  
	  minimize_mask_edge();
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  while(v_p != NULL)
	    {
	      if(v_p->pic_p == ud->data_p)
		{
		  v_p->pic_p = cur_pic;
		  handle_resize(v_p);
		}
	      v_p = v_p->next;
	    }
	  undid = !undid;
	}
  all_busy_nomore();
  update_menu();
}

/* FUNCTION:       delete()
   DESCRIPTION:    Delete masked area from picture and fill the hole
                   with secondary color
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void delete()
{
  extern PLOT xor_data;
  extern struct picture *cur_pic;
  extern struct view *cur_view;
  extern int show_edges;
  extern color cur_color;
  color help_color = cur_color;

  if(cur_pic->mask_edge != NULL)
    {
      all_busy();
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      memorize(cur_pic, cur_pic->min_x, cur_pic->min_y, 
	       cur_pic->max_x, cur_pic->max_y, NULL, "delete");
      cur_color = cur_view->color2;
      fill_box(cur_pic->min_x, cur_pic->min_y,
	       cur_pic->max_x, cur_pic->max_y, replace_pixel2);
      cur_color = help_color;
      fill_box(cur_pic->min_x, cur_pic->min_y,
	       cur_pic->max_x, cur_pic->max_y, clear_mask);
      update_edge();
      delete_mask_edge();
      touch(cur_pic);
      all_busy_nomore();
      update_menu();
    }
}

/* FUNCTION:       cut()
   DESCRIPTION:    cut data to clipboard, then delete it from picture
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void cut()
{
  extern PLOT xor_data;
  extern struct picture *cur_pic, *clipboard;
  extern struct view *cur_view;
  extern int show_edges;
  extern pixel copy_bits;
  extern color cur_color;
  color help_color = cur_color;

  if(cur_pic->mask_edge != NULL)
    {
      all_busy();
      if(clipboard != NULL)
	delete_block(clipboard);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      memorize(cur_pic, cur_pic->min_x, cur_pic->min_y, 
	       cur_pic->max_x, cur_pic->max_y, NULL, "cut");
      clipboard = extract_block2(cur_pic, cur_pic->min_x, cur_pic->min_y,
				 cur_pic->max_x, cur_pic->max_y, 
				 copy_bits);
      if(clipboard != NULL)
	{
	  cur_color = cur_view->color2;
	  fill_box(cur_pic->min_x, cur_pic->min_y,
		   cur_pic->max_x, cur_pic->max_y, replace_pixel2);
	  cur_color = help_color;
	  fill_box(cur_pic->min_x, cur_pic->min_y,
		   cur_pic->max_x, cur_pic->max_y, clear_mask);
	  update_edge();
	  delete_mask_edge(); 
	  touch(cur_pic);
	  all_busy_nomore();
	  update_menu();
	}
      else
	{
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  all_busy_nomore();
	  update_menu();
	  panic("Not enough memory for clipboard!", FALSE);
	}
    }
}

/* FUNCTION:       copy()
   DESCRIPTION:    copy data to clipboard
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void copy()
{
  extern PLOT xor_data;
  extern struct picture *cur_pic, *clipboard;
  extern int show_edges;
  extern pixel copy_bits;

  if(cur_pic->mask_edge != NULL)
    {
      all_busy();
      if(clipboard != NULL)
	delete_block(clipboard);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      clipboard = extract_block2(cur_pic, cur_pic->min_x, cur_pic->min_y,
				 cur_pic->max_x, cur_pic->max_y, 
				 copy_bits);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      all_busy_nomore();
      update_menu();
      if(clipboard == NULL)
	{
	  panic("Not enough memory for clipboard!", FALSE);
	}
    }
}

/* FUNCTION:       paste()
   DESCRIPTION:    Paste clipboard to brush and fill
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void paste()
{
  extern struct picture *clipboard;
  extern int fill_tolerance;

  if(clipboard != NULL)
    {
      all_busy();
      create_brush(CLIP, 0);
      create_fill(CLIP_F, 0, fill_tolerance);
      all_busy_nomore();
    }
  else 
    show_errormsg("The clipboard is empty!", "Bummer!", NULL);
}

/* FUNCTION:       paste to toolbox()
   DESCRIPTION:    Paste brush to brush list
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void paste_to_toolbox()
{
  extern struct picture *clipboard, *user1_b, *user2_b, *user3_b;
  
  if(clipboard != NULL)
    {
      if(user1_b == NULL)
	paste_this(USER1_B, B_OK);
      else if(user2_b == NULL)
	paste_this(USER2_B, B_OK);
      else if(user3_b == NULL)
	paste_this(USER3_B, B_OK);
      else
	ask_paste(paste_this);
    }
}

/* FUNCTION:       paste_this(int which, button)
   DESCRIPTION:    The X module tells which brush to replace, if
                   the brush list is full. This subr actually does the
		   pasting.
   PARAMETERS:
   which
   button          Dialogue button pressed
   RETURNS:        

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void paste_this(int which, int button)
{
  extern struct picture *clipboard, *user1_b, *user2_b, *user3_b;

  all_busy();
  if(button == B_OK)
    if(which == USER1_B)
      {
	user1_b = duplicate_block(clipboard);
	all_busy_nomore();
	if(user1_b == NULL)
	  panic("Unable to paste as user block #1!", FALSE);
      }
    else if(which == USER2_B)
      {
	user2_b = duplicate_block(clipboard);
	all_busy_nomore();
	if(user2_b == NULL)
	  panic("Unable to paste as user block #2!", FALSE);
      }
    else if(which == USER3_B)
      {
	user3_b = duplicate_block(clipboard);
	all_busy_nomore();
	if(user3_b == NULL)
	  panic("Unable to paste as user block #3!", FALSE);
      }
}

/* FUNCTION:       duplicate()
   DESCRIPTION:    Duplicate the current picture or a part of it to a
                   new one. Add the picture to the view list.
   PARAMETERS:
   none
   RETURNS:        View pointer

   Written by Hannu Mallat, last modifications 4th August 1993.
*/   
struct view *duplicate()
{
  extern PLOT xor_data;
  extern struct view *cur_view;
  extern struct picture *cur_pic;
  extern int show_edges, scope;
  extern pixel copy_bits;
  struct picture *pic_p = NULL, *help_p = cur_pic;
  struct view *v_p = NULL;

  if(cur_view != NULL)
    {
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      if(cur_view->pic_p->mask_edge == NULL || scope == SCOPE_EVERYWHERE)
	{
	  pic_p = extract_block1(cur_pic, 0, 0, 
				 cur_pic->width-1, cur_pic->height-1,
				 copy_bits);
	  if(pic_p != NULL)
	    {
	      pic_p->min_x = cur_pic->min_x;
	      pic_p->min_y = cur_pic->min_y;
	      pic_p->max_x = cur_pic->max_x;
	      pic_p->max_y = cur_pic->max_y;
	    }
	}
      else if(scope == SCOPE_INSIDE && cur_view->pic_p->mask_edge != NULL)
	{
	  pic_p = extract_block2(cur_pic, 
				 cur_pic->min_x, cur_pic->min_y, 
				 cur_pic->max_x, cur_pic->max_y,
				 copy_bits);
	  if(pic_p != NULL)
	    {
	      pic_p->min_x = 0;
	      pic_p->min_y = 0;
	      pic_p->max_x = pic_p->width-1;
	      pic_p->max_y = pic_p->height-1;
	    }
	}
      else if(scope == SCOPE_OUTSIDE)
	{
	  pic_p = extract_block3(cur_pic, 0, 0, 
				 cur_pic->width-1, cur_pic->height-1,
				 copy_bits);
	  if(pic_p != NULL)
	    {
	      pic_p->min_x = cur_pic->min_x;
	      pic_p->min_y = cur_pic->min_y;
	      pic_p->max_x = cur_pic->max_x;
	      pic_p->max_y = cur_pic->max_y;
	    }
	}

      if(pic_p != NULL && cur_pic->n_edges != 0)
	{
	  cur_pic = pic_p;
	  minimize_mask_edge();
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	}
      cur_pic = help_p;
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      if(pic_p != NULL)
	{
	  v_p = add_view(pic_p, 0, 0,
			 pic_p->width-1, pic_p->height-1, cur_view->scale);
	  touch(pic_p);
	}
      else
	{
	  all_busy_nomore();
	  panic("Not enough memory to duplicate a view!", FALSE);
	}
    }
  return v_p;
}

