/* FILE: pixel.h

   Prototypes for pixel, block and pattern drawing functions, definitions
   for data structures and macros used in them, external global variable
   declarations. Include this header file, where ever you want to use basic
   drawing functions or need these data structures.

   Written by Petri Kuittinen, last modifications 16th July 1993.
*/ 

#ifndef PIXEL

typedef int pixel; /* Must be 32-bit in size */
typedef int color; /* -""- */

/* Pixels are stored as 32-bit words. Bits 0-7 are for blue, bits 8-15 are
   for green and 15-23 are for red and bit 31 (sign bit) is used for mask
   (1=mask set). Bits 24-30 are not used at the moment and should be left
   zero.

   31      23      15      7       0 bits
   |-------|-------|-------|-------|
   Mask     Red     Green   Blue     
*/

#define CMASK 0xffffff              /* mask for lowest 24 (=color) bits */
#define BMASK 0xff                  /* mask for lowest 8 bits */
#define MASK_BIT (1<<31)            /* sign bit is used as mask bit */
#define NO_MASK ~(MASK_BIT)         /* all bits set, except mask bit */
#define IS_MASK(A) ((A)<0)          /* test for mask bit */
#define SET_MASK(A) ((A)|MASK_BIT)  /* set mask bit */ 
#define CLEAR_MASK(A) ((A)&NO_MASK) /* clear mask bit */
#define XOR_MASK(A) ((A)^MASK_BIT)  /* reverse mask bit */
#define GET_MASK(A) ((A)&MASK_BIT)  /* get mask bit */
#define GET_UP_BITS(A) ((A)&0xff000000) /* get upper eight bits */
#define FB (1<<30)                  /* fill bit, used in some fill routines */
#define NO_FB (~(FB))               /* all bits set, except fill bit */
#define IS_FB(A) (((A)&FB)!=0)      /* test for fill bit */
#define SET_FB(A) ((A)|FB)          /* set fill bit */ 
#define CLEAR_FB(A) ((A)&NO_MASK)   /* clear fill bit */
#define GET_FB(A) ((A)&FB)          /* get fill bit */

#define GET_RED(A) (((A)>>16)&BMASK)  /* get corresponding color value from */
#define GET_GREEN(A) (((A)>>8)&BMASK) /* pixel */
#define GET_BLUE(A) ((A)&BMASK)

#define PUT_RED(A) (((A)&BMASK)<<16)  /* put corresponding color value to */
#define PUT_GREEN(A) (((A)&BMASK)<<8) /* pixel */
#define PUT_BLUE(A) ((A)&BMASK)

#define LIMIT_255(X) X |= ((X)>>8)*255   /* Limit X (down) to value 255 
					    same as: if (X>255) X=255 */
#define LIMIT_15(X) X |= ((X)>>4)*15     /* Limit X (down) to value 15 
					    same as: if (X>15) X=15 */
#define LIMIT_0(X) X &= (((X)>>31)^1)*-1 /* Limit X to value 0
					    same as: if (X<0) X=0 */
/* Limit X to range [0,255] */
#define LIMIT(X) if ((X)<0) (X)=0; else if ((X)>255) (X)=255;

/* Data types */

struct coord {int x, y;}; /* Element of coordinate array */

struct picture {pixel *buf;              /* pointer to picture data */
		char *name;              /* name of picture */
		int width, height;       /* dimensions of the picture */
		struct coord *mask_edge; /* pointer to a list of mask edge
					    points */
		int n_edges;             /* number of mask edge points */
		int min_x;               /* smallest x value in mask edge */
		int min_y;               /* smallest y value in mask edge */
		int max_x;               /* largestt x value in mask edge */
		int max_y;               /* largest y value in mask edge */
		int flags;               /* space for flag bits */
	      };

struct segment {int y, xl, xr, dy;};     /* Filled horizontal segment of
					    scanline y for xl<=x<=xr.
					    Parent segment was on line y-dy.
					    dy=1 or -1 */

/* External global variables */

extern struct picture *cur_pic, *cur_block;
extern color cur_color;
extern int x_off, y_off;

/* Prototypes */

void replace_pixel1 (int x, int y);
void replace_pixel2 (int x, int y);
void replace_pixel3 (int x, int y);
void add_pixel1 (int x, int y);
void add_pixel2 (int x, int y);
void add_pixel3 (int x, int y);
void sub_pixel1 (int x, int y);
void sub_pixel2 (int x, int y);
void sub_pixel3 (int x, int y);
void xor_pixel1 (int x, int y);
void xor_pixel2 (int x, int y);
void xor_pixel3 (int x, int y);

void replace_block1 (int x, int y);
void replace_block2 (int x, int y);
void replace_block3 (int x, int y);
void add_block1 (int x, int y);
void add_block2 (int x, int y);
void add_block3 (int x, int y);
void sub_block1 (int x, int y);
void sub_block2 (int x, int y);
void sub_block3 (int x, int y);
void xor_block1 (int x, int y);
void xor_block2 (int x, int y);
void xor_block3 (int x, int y);

void replace_pattern1 (int x, int y);
void replace_pattern2 (int x, int y);
void replace_pattern3 (int x, int y);
void add_pattern1 (int x, int y);
void add_pattern2 (int x, int y);
void add_pattern3 (int x, int y);
void sub_pattern1 (int x, int y);
void sub_pattern2 (int x, int y);
void sub_pattern3 (int x, int y);
void xor_pattern1 (int x, int y);
void xor_pattern2 (int x, int y);
void xor_pattern3 (int x, int y);

void set_mask (int x, int y);
void clear_mask (int x, int y);
int get_mask (int x, int y);
void set_fill_bit (int x, int y);
void clear_fill_bit (int x, int y);
int get_fill_bit (int x, int y);
color get_color (int x, int y);
void set_pixel_and_mask (int x, int y);

void paste_block (struct picture *pic_p, int x, int y);

void add_mask_edge (int x, int y);
void delete_mask_edge();
void minimize_mask_edge();
void plot_mask_edge (void (*put_pixel) (int x, int y));
int fill_this(int x, int y, int val, int range);
void flood_fill (int xc, int yc, int range, void (*put_pixel) (int x, int y));
void fill_by_color (color col, int range, void (*put_pixel) (int x, int y));

#define PIXEL
#endif
