/*
 * This file is part of
 * Movietool: display a succession of rasterfile frames in real-time
 *
 * Copyright 1989 by Ole H. Nielsen
 * This software may be distributed freely provided that:
 * 1) this copyright notice is kept intact,
 * 2) you do not try to sell this software for a profit,
 * 3) you provide the full source code.
 *
 * Author: Ole H. Nielsen
 *         Lab of Applied Physics, Bygn. 307
 *         Technical University of Denmark, DK-2800 Lyngby
 *         E-mail: ohnielse@ltf.dth.dk
 */

#include "movietool.h"
#include <stdio.h> 
#include <suntool/canvas.h>
#include <suntool/panel.h>
#include <suntool/scrollbar.h>

#define ICONIMAGE	"movietool.icon"
#define WIDTH	500
#define HEIGHT	500
#define PANEL_FONT	"/usr/lib/fonts/fixedwidthfonts/screen.b.12"

/* global declarations */

extern char	*title;
Frame		window_frame;	
Window 		panel; /* Declaring as Window to get colormap capability */
Canvas		canvas;
static Cursor	cursor;
Pixwin		*canvaspw, *panelpw, *windowpw;
Pixfont		*panelfont;
extern int	collect_images();
int		noframes = 0;
extern void	create_panel_items(), canvas_repaint_proc();
void		put_color ();
void		usage();
char		colormapname[20];
int		identical_colormaps	= FALSE;
int		encoded_flag	= FALSE;
int		canvas_width	= WIDTH;	
int		canvas_height	= HEIGHT;
int		canvas_depth	= 1;
unsigned	max_frame_height;
int		xor_flag	= FALSE;

/* local declarations */

static int	panel_exists	= FALSE;
static short	icon_image[]={
#include ICONIMAGE
};
mpr_static(icon_pixrect, 64, 64, 1, icon_image);

main(argc, argv)
int argc; char **argv;
{
	extern char *optarg;
	extern int optind, opterr;
	Icon icon;
	static unsigned char red[CMS_SIZE], green[CMS_SIZE], blue[CMS_SIZE];
	static colormap_t window_cmap = { RMT_NONE, 0, { red, green, blue } };
	int i, j, zoom=0, argcoffset=1, width, height;
	int dx=0, dy=0, dw=0, dh=0;
	int c, errflag = 0;

	if (getenv ("WINDOW_PARENT") == NULL) {
		fprintf (stderr, "%s can only be run under SunView.\n",argv[0]);
		exit (1);
	}

	/* Open the base frame.  Note that window_create consumes SunView args*/
	icon = icon_create (ICON_IMAGE, &icon_pixrect, 0);
	window_frame = window_create (0, FRAME, 
		FRAME_ARGC_PTR_ARGV, &argc, argv,
		FRAME_LABEL, title,
		FRAME_ICON, icon,
		FRAME_CMDLINE_HELP_PROC, usage,
		FRAME_INHERIT_COLORS, TRUE,
		0);

	/* Get command line arguments */

	while ((c = getopt(argc, argv, "iz:s:Xe")) != -1)
		switch (c) {
		case 'i':
			identical_colormaps = TRUE;
			printf ("Colormap of 1st frame used throughout.\n");
			break;
		case 'z':
			(void) sscanf (optarg, "%d", &zoom);
			if (zoom > 4 || zoom < 2)  {
				fprintf (stderr, "Bad arg=%d to -zoom.\n",zoom);
				errflag++;
			}
			(void) printf ("Zoom by factor %d.\n", zoom);
			break;
		case 's':
			j =	sscanf (optarg,		"%d", &dx) +
				sscanf (argv[optind],	"%d", &dy) +
				sscanf (argv[optind+1],	"%d", &dw) +
				sscanf (argv[optind+2],	"%d", &dh);
			if (j != 4 || dx < 0 || dy < 0 || dw < 2 || dh < 2) {
				(void) fprintf (stderr, "Bad args to -sub.\n");
				errflag++;
			} else {
				(void) printf (
				"Displaying subregion at %d,%d of size %d,%d.\n"
					, dx, dy, dw, dh);
				optind += 3;	/* Warning in "man 3 getopt" */
			}
			break;
		case 'X':
			xor_flag = TRUE;
			printf ("XOR operations will be used when playing.\n");
			break;
		case 'e':
			encoded_flag = TRUE;
			printf ("RT_BYTE_ENCODED images will be used.\n");
			break;
		case '?':
			errflag++;
			break;
		}

	if (encoded_flag && (zoom || dx || dy || dw || dh)) {
		printf ("-e switch is incompatible with zoom and subregion.\n");
		errflag++;
	}

	if (errflag) {
		(void) usage (argv[0]);
		exit (1);
	}

	noframes = collect_images (argc - optind, argv + optind,
		zoom, window_cmap, dx, dy, dw, dh);
	max_frame_height = canvas_height;

	(void) printf ("Number of frames loaded in: %d.\n", noframes);
	if (noframes <= 0) {
		(void) usage (argv[0]);
		exit (1);
	}

	/* create a panel for the tool to hold control panels */
	if ((panelfont = pf_open(PANEL_FONT)) == NULL) {
		fprintf(stderr, "Cannot open font %s.\n", PANEL_FONT);
		exit(1);
	}
	if (noframes > 1) {
		panel_exists = TRUE;
		panel = window_create (window_frame, PANEL,
			WIN_FONT, panelfont, 
			0);
		(void) create_panel_items ();
		(void) window_fit_height (panel);
	}

	/* create non-retained canvas for display purposes */
	canvas = window_create (window_frame, CANVAS,
		CANVAS_RETAINED, FALSE,
		CANVAS_FIXED_IMAGE, FALSE,
		CANVAS_AUTO_SHRINK, FALSE,
		CANVAS_AUTO_EXPAND, TRUE,
		WIN_WIDTH, WIDTH,
		WIN_HEIGHT, HEIGHT,
		CANVAS_WIDTH, canvas_width,
		CANVAS_HEIGHT, canvas_height,
		CANVAS_DEPTH, canvas_depth,
		CANVAS_REPAINT_PROC, canvas_repaint_proc,
		0);

	/* Initialize canvas properties */

	/* Get pixwin for displaying movie */
	canvaspw = (Pixwin *)window_get (canvas, CANVAS_PIXWIN);

	if (canvas_depth == 1) {		/* MONOCHROME */
		(void) window_set (canvas,
			CANVAS_FAST_MONO, TRUE,
			0);
		if (window_get (canvas, CANVAS_FAST_MONO))
			fprintf (stderr, "Using frame-buffer's fast overlay plane\n");
		else
			fprintf (stderr, "No frame-buffer overlay plane available\n");
	} else if (canvas_depth == 8) {		/* COLOR */
		windowpw = (Pixwin *)window_get (window_frame, WIN_PIXWIN);
		if (panel_exists)
			panelpw = (Pixwin *)window_get (panel, WIN_PIXWIN);
		(void) sprintf (colormapname, "movietool%d", getpid () );
		(void) pw_setcmsname (windowpw, colormapname);
		if (panel_exists)
			(void) pw_setcmsname (panelpw, colormapname);
		(void) pw_setcmsname (canvaspw, colormapname);
		(void) put_color (&window_cmap);
		if (windowpw->pw_pixrect->pr_depth < 8)
			fprintf (stderr, "%s - WARNING: color frames will not make sense on a monochrome display.\n", argv[0]);
	} else {
		(void) fprintf (stderr, "%s: I don't know canvas depth=%d.\n",
			argv[0], canvas_depth);
		exit(1);
	}

	/* Scrollbars must be set after CANVAS_FAST_MONO (SunView, p. 125) */
	if (canvas_height > HEIGHT)
		(void) window_set (canvas,
		WIN_VERTICAL_SCROLLBAR,
			scrollbar_create (SCROLL_LINE_HEIGHT, 20, 0),
		0);
	if (canvas_width > WIDTH)
		(void) window_set (canvas,
		WIN_HORIZONTAL_SCROLLBAR, 
			scrollbar_create (SCROLL_LINE_HEIGHT, 20, 0),
		0);

	/* Disable mouse inside canvas */
	cursor = window_get (canvas, WIN_CURSOR);
	(void) cursor_set (cursor,
		CURSOR_SHOW_CURSOR, FALSE,
		0);
	(void) window_set (canvas,
		WIN_CURSOR, cursor,
		0);

	(void) window_fit (canvas);
	(void) window_fit (window_frame);
	(void) step_proc ();

#ifdef PANELEDIT
	if (panel_exists)	/* Panel editing: paneledit.c */
		(void) paneledit_init (window_frame);
#endif

	(void) window_main_loop (window_frame);
	exit (0);
}

#define FLAG(string) (void) fprintf (stderr, (string))

static void usage(progname)
char *progname;
{
	(void) fprintf (stderr, "Usage: %s\n", progname);
	FLAG("       -z (or -zoom) factor - Zoom picture by factor\n");
	FLAG("       -s (or -sub) x y w h - Pick out subregion of raster\n");
	FLAG("       -i (or -ident)       - Identical colormaps used\n");
	FLAG("       -X (or -XOR)         - Play with XOR operations\n");
	FLAG("       -e (or -encoded)     - Use encoded rasters (saves RAM)\n");
	FLAG("       file [file...]       - Sun raster files to be displayed\n");
}

/* Set the colormap for the current frame */
void put_color (cmap)
colormap_t *cmap;
{
	if ( ! cmap) return;
	(void) pw_putcolormap (windowpw, 0, CMS_SIZE,
		cmap->map[0], cmap->map[1], cmap->map[2]);
	if (panel_exists)
		(void) pw_putcolormap (panelpw, 0, CMS_SIZE,
			cmap->map[0], cmap->map[1], cmap->map[2]);
	(void) pw_putcolormap (canvaspw, 0, CMS_SIZE,
		cmap->map[0], cmap->map[1], cmap->map[2]);
}
