/* $Id: xmesa3.c,v 1.7 1995/11/30 00:21:36 brianp Exp $ */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Log: xmesa3.c,v $
 * Revision 1.7  1995/11/30  00:21:36  brianp
 * added PF_GRAYSCALE support
 *
 * Revision 1.6  1995/11/14  21:49:09  brianp
 * optimized polygon rendering setup
 *
 * Revision 1.5  1995/11/08  22:08:22  brianp
 * fixed OFFSET4/1 bug in smooth_rgba_z_polygon_ximage()
 *
 * Revision 1.4  1995/11/04  20:09:32  brianp
 * added F suffix to floating point constants
 * replaced incorrect 0.05F with 0.5F
 *
 * Revision 1.3  1995/11/03  17:41:48  brianp
 * removed unused vars, fixed code for C++ compilation
 *
 * Revision 1.2  1995/10/30  15:50:41  brianp
 * make sure CC.ClipSpans is FALSE before using smooth_rgba_z_polygon_ximage
 *
 * Revision 1.1  1995/10/30  15:15:15  brianp
 * Initial revision
 *
 */


/*
 * Mesa/X11 interface, part 3.
 *
 * This file contains "accelerated" point, line, and polygon functions.
 * It should be fairly easy to write new special-purpose point, line or
 * polygon functions and hook them into this module.
 */



#include <stdlib.h>
#include <stdio.h>
#include "X11/Xlib.h"
#include "context.h"
#include "dd.h"
#include "interp.h"
#include "macros.h"
#include "polygons.h"
#include "vb.h"
#include "xmesaP.h"



/*
 * Given a vertex number return an X pixel value.
 */
static unsigned long encode_color( GLuint i )
{
   switch (XMesa->pixelformat) {
      case PF_INDEX:
         return (unsigned long) VB.Index[i];
      case PF_TRUECOLOR:
         {
            register int r, g, b;
            r = VB.Color[i][0] * CC.RedScale;
            g = VB.Color[i][1] * CC.GreenScale;
            b = VB.Color[i][2] * CC.BlueScale;
            return PACK_RGB( r, g, b );
         }
      case PF_8A8B8G8R:
         {
            register int r, g, b, a;
            r = VB.Color[i][0] * 255.0F;
            g = VB.Color[i][1] * 255.0F;
            b = VB.Color[i][2] * 255.0F;
            a = VB.Color[i][3] * 255.0F;
            return (a << 24) | (b << 16) | (g << 8) | r;
         }
      case PF_DITHER:
         {
            register int r, g, b;
            r = VB.Color[i][0] * 255.0F;
            g = VB.Color[i][1] * 255.0F;
            b = VB.Color[i][2] * 255.0F;
            return DITHER_8BIT( 0, 0, r, g, b );
         }
      case PF_1BIT:
         {
            register int r, g, b;
            r = VB.Color[i][0] * 255.0F;
            g = VB.Color[i][1] * 255.0F;
            b = VB.Color[i][2] * 255.0F;
            return (r+g+b) > 382;
         }
      case PF_HPCR:
         {
            register int r, g, b;
            r = VB.Color[i][0] * 255.0F;
            g = VB.Color[i][1] * 255.0F;
            b = VB.Color[i][2] * 255.0F;
            return DITHER_HPCR( 1, 1, r, g, b );
         }
      case PF_LOOKUP:
         {
            register int r, g, b;
            r = VB.Color[i][0] * 255.0F;
            g = VB.Color[i][1] * 255.0F;
            b = VB.Color[i][2] * 255.0F;
            return LOOKUP( r, g, b );
         }
      case PF_GRAYSCALE:
         {
            register int r, g, b;
            r = VB.Color[i][0] * CC.RedScale;
            g = VB.Color[i][1] * CC.GreenScale;
            b = VB.Color[i][2] * CC.BlueScale;
            return GRAY_RGB( r, g, b );
         }
      default:
         abort();
   }
   return 0;
}



/**********************************************************************/
/***                    Point rendering                             ***/
/**********************************************************************/


/*
 * Render an array of points into a pixmap, any pixel format.
 */
static void draw_points_ANY_pixmap( GLuint first, GLuint last )
{
   register GLuint i;
   if (VB.MonoColor) {
      /* all same color */
      XPoint p[VB_SIZE];
      int n = 0;
      for (i=first;i<=last;i++) {
         if (VB.Unclipped[i]) {
            p[n].x =       (GLint) (VB.Win[i][0] + 0.5F);
            p[n].y = FLIP( (GLint) (VB.Win[i][1] + 0.5F) );
            n++;
         }
      }
      XDrawPoints( XMesa->display, XMesa->buffer, XMesa->gc1, p, n,
                   CoordModeOrigin );
   }
   else {
      /* all different colors */
      for (i=first;i<=last;i++) {
         if (VB.Unclipped[i]) {
            register int x, y;
            XSetForeground( XMesa->display, XMesa->gc2, encode_color(i) );
            x =       (GLint) (VB.Win[i][0] + 0.5F);
            y = FLIP( (GLint) (VB.Win[i][1] + 0.5F) );
            XDrawPoint( XMesa->display, XMesa->buffer, XMesa->gc2, x, y);
         }
      }
   }
}



/*
 * Analyze current CC state to see if we can provide a fast points drawing
 * function, like those in points.c.  Otherwise, return NULL.
 */
points_func xmesa_get_points_func( void )
{
   if (CC.Point.Size==1.0F && !CC.Point.SmoothFlag && CC.RasterMask==0
       && !CC.Texture.Enabled) {
      if (XMesa->buffer==XIMAGE) {
         return NULL; /*draw_points_ximage;*/
      }
      else {
         return draw_points_ANY_pixmap;
      }
   }
   else {
      return NULL;
   }
}



/**********************************************************************/
/***                      Line rendering                            ***/
/**********************************************************************/

/*
 * Render a line into a pixmap, any pixel format.
 */
static void draw_line_ANY_pixmap( GLuint v0, GLuint v1, GLuint pv )
{
   register int x0, y0, x1, y1;
   GC gc;
   if (VB.MonoColor) {
      gc = XMesa->gc1;  /* use current color */
   }
   else {
      gc = XMesa->gc2;
      XSetForeground( XMesa->display, XMesa->gc2, encode_color(pv) );
   }
   x0 =       (GLint) (VB.Win[v0][0] + 0.5F);
   y0 = FLIP( (GLint) (VB.Win[v0][1] + 0.5F) );
   x1 =       (GLint) (VB.Win[v1][0] + 0.5F);
   y1 = FLIP( (GLint) (VB.Win[v1][1] + 0.5F) );
   XDrawLine( XMesa->display, XMesa->buffer, gc, x0, y0, x1, y1 );
}



/*
 * Analyze current CC state to see if we can provide a fast line drawing
 * function, like those in lines.c.  Otherwise, return NULL.
 */
line_func xmesa_get_line_func( void )
{
   if (CC.Line.Width==1.0F && !CC.Line.SmoothFlag && !CC.Line.StippleFlag
       && CC.Light.ShadeModel==GL_FLAT && CC.RasterMask==0
       && !CC.Texture.Enabled) {
      if (XMesa->buffer==XIMAGE) {
         /* no Xlib func for drawing lines into ximages */
         return NULL;
      }
      else {
         return draw_line_ANY_pixmap;
      }
   }
   else {
      return NULL;
   }
}




/**********************************************************************/
/***                   Polygon rendering                            ***/
/**********************************************************************/

/*
 * Render a polygon into a pixmap, any pixel format.
 */
void draw_polygon_ANY_pixmap( GLuint n, GLuint vlist[], GLuint pv )
{
   GLuint i;
   XPoint p[VB_SIZE];
   GC gc;
   if (VB.MonoColor) {
      gc = XMesa->gc1;  /* use current color */
   }
   else {
      gc = XMesa->gc2;
      XSetForeground( XMesa->display, XMesa->gc2, encode_color(pv) );
   }
   for (i=0;i<n;i++) {
      GLuint j = vlist[i];
      p[i].x =       (GLint) (VB.Win[j][0] + 0.5F);
      p[i].y = FLIP( (GLint) (VB.Win[j][1] + 0.5F) ) + 1;
   }
   XFillPolygon( XMesa->display, XMesa->buffer, gc,
		 p, n, Convex, CoordModeOrigin );
}



static GLfloat flx[MAX_HEIGHT], frx[MAX_HEIGHT];/* X bounds */
static GLint lr[MAX_HEIGHT], rr[MAX_HEIGHT];	/* Red */
static GLint lg[MAX_HEIGHT], rg[MAX_HEIGHT];	/* Green */
static GLint lb[MAX_HEIGHT], rb[MAX_HEIGHT];	/* Blue */
static GLint la[MAX_HEIGHT], ra[MAX_HEIGHT];	/* Alpha */

#define EDGEMAX MAX_WIDTH



/*
 * Render a smooth shaded, RGBA, depth-bufferd polygon into an XImage.
 */
void smooth_rgba_z_polygon_ximage( GLuint n, GLuint vlist[], GLuint pv )
{
   GLint i, j, y;
   GLint ymin, ymax;

   /* find min and max of window coordinate Y values */
   {
      GLfloat min = 1.0e10F;
      GLfloat max = -1.0e10F;
      if (n==3) {
         GLfloat winy;
         winy = VB.Win[vlist[0]][1];
         if (winy > max)  max = winy;
         if (winy < min)  min = winy;
         winy = VB.Win[vlist[1]][1];
         if (winy > max)  max = winy;
         if (winy < min)  min = winy;
         winy = VB.Win[vlist[2]][1];
         if (winy > max)  max = winy;
         if (winy < min)  min = winy;
      }
      else {
         for (i=0;i<n;i++) {
            GLfloat winy = VB.Win[vlist[i]][1];
            if (winy > max)  max = winy;
            if (winy < min)  min = winy;
         }
      }
      ymin = (GLint) min;
      ymax = (GLint) max;
      ymin = CLAMP( ymin, 0, MAX_HEIGHT-1 );
      ymax = CLAMP( ymax, 0, MAX_HEIGHT-1 );
   }

   /* init edge bounds */
   for (y=ymin;y<=ymax;y++) {
      flx[y] = (GLfloat) (MAX_WIDTH+1);
      frx[y] = -1.0F;
   }

   /* process edges to compute bounds */
   for (i=0;i<n;i++) {
      GLuint j0, j1, len;
      GLfloat ex[EDGEMAX];
      GLint ey[EDGEMAX];
      GLint r0, g0, b0, a0, r1, g1, b1, a1, dr, dg, db, da;

      j0 = (i==0) ? vlist[n-1] : vlist[i-1];
      j1 = vlist[i];

      /* compute edge pixels */
      len = gl_polygon_edge( VB.Win[j0][0], VB.Win[j0][1],
                             VB.Win[j1][0], VB.Win[j1][1],
                             ex, ey );

      /* interpolate colors along edge */
      r0 = (GLint) (VB.Color[j0][0] * CC.RedScale) << 8;
      r1 = (GLint) (VB.Color[j1][0] * CC.RedScale) << 8;
      g0 = (GLint) (VB.Color[j0][1] * CC.GreenScale) << 8;
      g1 = (GLint) (VB.Color[j1][1] * CC.GreenScale) << 8;
      b0 = (GLint) (VB.Color[j0][2] * CC.BlueScale) << 8;
      b1 = (GLint) (VB.Color[j1][2] * CC.BlueScale) << 8;
      a0 = (GLint) (VB.Color[j0][3] * CC.AlphaScale) << 8;
      a1 = (GLint) (VB.Color[j1][3] * CC.AlphaScale) << 8;
      if (len>1) {
         GLint n = len-1;
         dr = (r1-r0) / n;
         dg = (g1-g0) / n;
         db = (b1-b0) / n;
         da = (a1-a0) / n;
      }
      else {
         dr = dg = db = da = 0;
      }

      /* update span bounds */
      for (j=0;j<len;j++) {
	 GLfloat x = ex[j];
	 GLint y = ey[j];
	 if (y>=0 && y<MAX_HEIGHT) {
	    if (x < flx[y]) {
	       flx[y] = x;
	       lr[y] = r0 >> 8;
	       lg[y] = g0 >> 8;
	       lb[y] = b0 >> 8;
	       la[y] = a0 >> 8;
	    }
	    if (x > frx[y]) {
	       frx[y] = x;
	       rr[y] = r0 >> 8;
	       rg[y] = g0 >> 8;
	       rb[y] = b0 >> 8;
	       ra[y] = a0 >> 8;
	    }
	 }
         r0 += dr;   g0 += dg;   b0 += db;   a0 += da;
      }
   }

   /* process spans */
   for (y=ymin;y<=ymax;y++) {
      GLint xmin = (GLint) (flx[y] + 0.5F);
      GLint xmax = (GLint) (frx[y] - 0.5F);
      GLint len = xmax-xmin+1;
      if (len>0) {
	 GLint z0, z1, zspan[MAX_WIDTH];
         GLint *zptr = CC.DepthBuffer + y * CC.BufferWidth + xmin;
         GLint r, g, b, a, dr, dg, db, da;

	 /* interpolate z */
	 z0 = gl_compute_z( flx[y]+0.5F, (GLfloat) y + 0.5F );
	 z1 = gl_compute_z( frx[y]-0.5F, (GLfloat) y + 0.5F );
	 GL_INTERPOLATE_I( len, z0, z1, zspan );

         /* setup for color interpolation */
         r = lr[y] << 8;
         g = lg[y] << 8;
         b = lb[y] << 8;
         a = la[y] << 8;
         if (len>1) {
            dr = ((rr[y] << 8) - r) / (len-1);
            dg = ((rg[y] << 8) - g) / (len-1);
            db = ((rb[y] << 8) - b) / (len-1);
            da = ((ra[y] << 8) - a) / (len-1);
         }
         else {
            dr = dg = db = da = 0;
         }

	 /* do depth test and write interpolated pixels */
         switch (XMesa->pixelformat) {
            case PF_TRUECOLOR:
               {
                  int yy = FLIP(y);
                  for (i=0;i<len;i++) {
                     if (zspan[i]<zptr[i]) {
                        unsigned long p;
                        zptr[i] = zspan[i];
                        p = PACK_RGBA( r>>8, g>>8, b>>8, a>>8 );
                        XPutPixel( XMesa->backimage, xmin+i, yy, p );
                     }
                     r += dr;  g += dg;  b += db;  a += da;
                  }
               }
               break;
            case PF_8A8B8G8R:
               {
                  GLuint *img = (GLuint *) XMesa->backimage->data
                                + OFFSET4(xmin,y);
                  for (i=0;i<len;i++) {
                     if (zspan[i]<zptr[i]) {
                        zptr[i] = zspan[i];
                        img[i] = ((a << 16) & 0xff000000) |
                                 ((b << 8 ) & 0x00ff0000) |
                                 ((g      ) & 0x0000ff00) |
                                 ((r >> 8 ) & 0x000000ff);
                     }
                     r += dr;  g += dg;  b += db;  a += da;
                  }
               }
               break;
            case PF_DITHER:
               if (XMesa->depth==8) {
                  GLubyte *img = (GLubyte *) XMesa->backimage->data
                              + OFFSET1(xmin,y);
                  for (i=0;i<len;i++) {
                     if (zspan[i]<zptr[i]) {
                        zptr[i] = zspan[i];
                        img[i] = DITHER_8BIT( xmin+i, y, r>>8, g>>8, b>>8 );
                     }
                     r += dr;  g += dg;  b += db;
                  }
               }
               else {
                  int yy = FLIP(y);
                  for (i=0;i<len;i++) {
                     if (zspan[i]<zptr[i]) {
                        unsigned long p;
                        zptr[i] = zspan[i];
                        p = DITHER_8BIT( xmin+i, y, r>>8, g>>8, b>>8 );
                        XPutPixel( XMesa->backimage, xmin+i, yy, p );
                     }
                     r += dr;  g += dg;  b += db;
                  }
               }
               break;
            case PF_HPCR:
               {
                  GLubyte *img = (GLubyte *) XMesa->backimage->data
                              + OFFSET1(xmin,y);
                  for (i=0;i<len;i++) {
                     if (zspan[i]<zptr[i]) {
                        zptr[i] = zspan[i];
                        img[i] = DITHER_HPCR( xmin+i, y, r>>8, g>>8, b>>8 );
                     }
                     r += dr;  g += dg;  b += db;
                  }
               }
               break;
            case PF_LOOKUP:
               {
                  int yy = FLIP(y);
                  for (i=0;i<len;i++) {
                     if (zspan[i]<zptr[i]) {
                        unsigned long p;
                        zptr[i] = zspan[i];
                        p = LOOKUP( r>>8, g>>8, b>>8 );
                        XPutPixel( XMesa->backimage, xmin+i, yy, p );
                     }
                     r += dr;  g += dg;  b += db;
                  }
               }
            case PF_GRAYSCALE:
               {
                  int yy = FLIP(y);
                  for (i=0;i<len;i++) {
                     if (zspan[i]<zptr[i]) {
                        unsigned long p;
                        zptr[i] = zspan[i];
                        p = GRAY_RGB( r>>8, g>>8, b>>8 );
                        XPutPixel( XMesa->backimage, xmin+i, yy, p );
                     }
                     r += dr;  g += dg;  b += db;
                  }
               }
               break;
            default:
               abort();
         } /*switch*/
      }
   }
}



/*
 * Analyze current CC and device driver state to see if we can provide a
 * fast polygon drawing function, like those in polygons.c.  Otherwise,
 * return NULL.
 */
polygon_func xmesa_get_polygon_func( void )
{
   if (!CC.Polygon.SmoothFlag && !CC.Polygon.StippleFlag
       && CC.Light.ShadeModel==GL_SMOOTH && CC.RasterMask==DEPTH_BIT
       && CC.Depth.Func==GL_LESS && !CC.Texture.Enabled
       && !CC.ClipSpans
       && XMesa->buffer==XIMAGE) {
      switch (XMesa->pixelformat) {
         case PF_TRUECOLOR:
         case PF_8A8B8G8R:
         case PF_DITHER:
         case PF_HPCR:
         case PF_LOOKUP:
         case PF_GRAYSCALE:
            return smooth_rgba_z_polygon_ximage;
         default:
            return NULL;
      }
   }

   if (!CC.Polygon.SmoothFlag && !CC.Polygon.StippleFlag
       && CC.Light.ShadeModel==GL_FLAT && CC.RasterMask==0
       && !CC.Texture.Enabled && XMesa->buffer!=XIMAGE) {
      return draw_polygon_ANY_pixmap;
   }

   return NULL;
}
