/* nurbscrv.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: nurbscrv.c,v 1.2 1995/07/28 21:37:04 brianp Exp $

$Log: nurbscrv.c,v $
 * Revision 1.2  1995/07/28  21:37:04  brianp
 * updates from Bogdan on July 28
 *
 * Revision 1.1  1995/07/28  14:44:43  brianp
 * Initial revision
 *
 */


/*
 * NURBS implementation written by Bogdan Sikorski (gstbs@io.coi.pw.edu.pl)
 * See README-nurbs for more info.
 */


#include "nurbs.h"
#include <stdlib.h>
#include <math.h>

static GLenum
test_nurbs_curve(GLUnurbsObj *nobj, GLint nknots, GLfloat *knot, GLint stride,
	GLfloat *ctrlarray, GLint order, GLenum type)
{
	GLint tmp_int;

	if(order < 0)
	{
		call_user_error(nobj,GLU_INVALID_VALUE);
		return GLU_ERROR;
	}
	glGetIntegerv(GL_MAX_EVAL_ORDER,&tmp_int);
	if(order > tmp_int || order < 2)
	{
		call_user_error(nobj,GLU_NURBS_ERROR1);
		return GLU_ERROR;
	}
	if(nknots < order +2)
	{
		call_user_error(nobj,GLU_NURBS_ERROR2);
		return GLU_ERROR;
	}
	if(stride < 0)
	{
		call_user_error(nobj,GLU_NURBS_ERROR34);
		return GLU_ERROR;
	}
	if(type!=GL_MAP1_VERTEX_3 && type!=GL_MAP1_VERTEX_4)
	{
		call_user_error(nobj,GLU_INVALID_ENUM);
		return GLU_ERROR;
	}
	if(knot==NULL || ctrlarray==NULL)
	{
		call_user_error(nobj,GLU_NURBS_ERROR8);
		return GLU_ERROR;
	}
	return GLU_NO_ERROR;
}

static GLenum
convert_curve(GLint nknots, GLfloat *knot,GLint stride,GLfloat *ctrl,
	GLint order,GLint dim,GLfloat **new_ctrl,GLint *ncontrol)
{
	knot_str_type the_knot;
	GLenum err;

	the_knot.knot=knot;
	the_knot.order=order;
	the_knot.nknots=nknots;
	if((err=explode_knot(&the_knot))!=GLU_NO_ERROR)
	{
		return err;
	}
	if((err=calc_alphas(&the_knot))!=GLU_NO_ERROR)
	{
		free(the_knot.new_knot);
		return err;
	}
	free(the_knot.new_knot);
	if((err=calc_new_ctrl_pts(ctrl,stride,&the_knot,dim,new_ctrl,ncontrol))
		!=GLU_NO_ERROR)
	{
		free(the_knot.alpha);
		return err;
	}
	free(the_knot.alpha);
	return GLU_NO_ERROR;
}

void do_nurbs_curve( GLUnurbsObj *nobj, GLint nknots, GLfloat *knot,
    GLint stride, GLfloat *ctlarray, GLint order, GLenum type )
{
	GLint n_ctrl;
	GLfloat *new_ctrl;
	GLint *factors;
	GLint i,j;
	GLenum err;
	GLint dim;


	if(test_nurbs_curve(nobj,nknots,knot,stride,ctlarray,order,type)
		!=GLU_NO_ERROR)
		return;

	switch(type)
	{
		case GL_MAP1_VERTEX_3:
			dim=3;
			break;
		case GL_MAP1_VERTEX_4:
			dim=4;
			break;
		default: /* can't be here, but seems gcc needs this */
			dim=0;
			break;
	}
	if((err=convert_curve(nknots,knot,stride,ctlarray,order,dim,
		&new_ctrl,&n_ctrl))!=GLU_NO_ERROR)
	{
		call_user_error(nobj,err);
		return;
	}
	if(glu_do_sampling_2D(nobj,new_ctrl,n_ctrl,order,dim,&factors)
		!=GLU_NO_ERROR)
	{
		free(new_ctrl);
		return;
	}
	glEnable(type);
	for(i=0 , j=0;i<n_ctrl;i+=order , j++)
	{
		if(fine_culling_test_2D(nobj,new_ctrl+i*dim,order,dim,dim))
			continue;
		glMap1f(type, 0.0, 1.0, dim, order, new_ctrl+i*dim);
		glMapGrid1f(factors[j],0.0,1.0);
		glEvalMesh1(GL_LINE,0,factors[j]);
	}
	free(new_ctrl);
	free(factors);
}


