/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/vcrEdit/RCS/vcrEdit.c,v 0.36 92/09/29 01:16:39 drapeau Exp $ */
/* $Log:	vcrEdit.c,v $
 * Revision 0.36  92/09/29  01:16:39  drapeau
 * Removed commented, obsolete code that is no longer used or necessary.
 * 
 * Revision 0.35  92/09/08  14:24:55  drapeau
 * Made several changes:
 * * Replaced NTSC-specific references to frame rates with newly-defined
 *   constant "FrameRate", as was done with the VideoObject library.
 *   This variable is defined at compile time from the Makefile.
 * * Modified CheckSelection() and Search() to remove code that was
 *   based on assumptions for the NEC PC-VCR.  In particular, the NEC
 *   does not allow access to the 1st 3 seconds on the tape, so the
 *   vcrEdit application encoded this NEC-specific limitation into the
 *   code.  The limitations are now coded in the NEC driver in the
 *   VideoObject, where they should be.  The vcrEdit application no
 *   longer has arbitrary restrictions on the length of an edit or
 *   the 1st valid frame on the medium.
 * 
 * Revision 0.34  92/09/04  17:06:50  drapeau
 * Modified ForwardStep() and ReverseStep() functions so that they are more
 * robust.  They now check that the deck is either in Pause or Still mode.
 * This is probably a semantic error in the definition of status codes in the
 * VideoObject, but the difference may prove useful at some point, so the
 * correction was made here in vcrEdit instead of in the VideoObject library.
 * 
 * Revision 0.33  92/09/01  16:51:47  drapeau
 * Fixed logic errors in the function "SpeedChange()" so that it does what was
 * intended (the intention was to call "DevPlayAtSpeedDir()" only when the
 * function "DevCalcSpeed()" indicated that the slider had really moved far
 * enough to warrant a change in playback speed).  The error occurred because
 * of two reasons:
 * 1) the static variable "oldSpeed" was reset everytime the function was
 *    entered, instead of being initialized to zero in the function
 *    declaration;
 * 2) the old value for speed was not being updated correctly; it should have
 *    been updated with the "framesPerSecond" variable returned by
 *    DevCalcSpeed().
 * 
 * Revision 0.32  92/05/13  23:58:24  drapeau
 * * Cosmetic changes to the code (formatting, mostly) to
 *   conform with coding standards.
 * * No longer try to identify tapes by the header id at the beginning
 *   of the tape.  This was a feature that turned out to be too
 *   cumbersome, slow, and unreliable.
 * * Changed the name of untitled documents from "untitled" to
 *   "Untitled".
 * 
 * Revision 0.31  92/01/03  17:58:17  drapeau
 * Changed all calls to Browse() to use "0" instead of "NULL", to take
 * into account the ANSI-C definition of NULL as (void*)0.
 * 
 * Revision 0.30  91/09/29  16:09:25  lim
 * Moved DisplayChoice to videoObjects.c
 * Removed mySerialPort.
 * The first item on Player setting list is "None".
 * 
 * Revision 0.29  91/09/26  12:31:05  lim
 * Did casting for xv_get's.
 * bzero'ed "chosenHost"
 * 
 * Revision 0.28  91/09/02  14:29:20  lim
 * Video Mute on options panel reversed.
 * 
 * Revision 0.27  91/08/30  15:14:56  lim
 * Check for sender != NULL in Quit().
 * 
 * Revision 0.26  91/08/28  17:02:11  lim
 * Eject now clears edit list too.
 * 
 * Revision 0.25  91/08/28  15:53:09  lim
 * Forward and reverse step implement still iff the player isn't
 * already in still.
 * 
 * Revision 0.24  91/08/28  14:26:42  lim
 * *** empty log message ***
 * 
 * Revision 0.23  91/08/28  14:06:02  lim
 * Check for error when querying frame.
 * 
 * Revision 0.22  91/08/24  17:59:33  lim
 * Implemented PrintDiagnostics. With command line flag '-d', the 
 * app will launch such that it prints all diag messages.
 * 
 * Revision 0.21  91/08/23  17:22:51  lim
 * 1. Step implements Still as well.
 * 2. Document format has changed.
 * 
 * Revision 0.20  91/08/21  10:43:37  lim
 * Added new variable, mySerialPort, to denote serial port setting on
 * options panel. Initialized to "NOTHING".
 * 
 * Revision 0.19  91/08/20  10:01:18  lim
 * Play() is revised to always set the current audio setting before play.
 * 
 * Revision 0.18  91/08/17  20:49:50  lim
 * OpenPanel now called Browse.
 * 
 * Revision 0.17  91/08/16  13:02:10  lim
 * > Always display 2-digit numbers for hr, sec, min, frame.
 * 
 * Revision 0.16  91/08/16  12:30:22  lim
 * 1. Rename & redo interface. 
 * 2. editFilename textfield has been removed and filename is put on the title bar.
 * 3. Info button and function placed in options panel.
 * 4. Call Browse() with OPENPANEL_CHECK for command line option.
 * 5. If no default player, options panel is set up.
 * 
 * Revision 0.15  91/08/15  12:58:28  lim
 * Changed FileLoad() calls in callback function for Edit button to
 * OpenHandler().
 * 
 * Revision 0.14  91/08/11  18:43:19  lim
 * Replaced XOpenDisplay() call with xv_get(obj, XV_DISPLAY)
 * 
 * Revision 0.13  91/08/09  17:01:23  lim
 * Included OpenPanel.
 * 
 * Revision 0.12  91/08/08  16:23:48  lim
 * 1. Removed speed popup and now uses speedSlider.
 * 2. speedStg is now an array of integers.
 * 3. Removed index table building due to hardware problems.
 * 4. currTape is now an array 'MaxHeaderLength' long. It is passed as a 
 *    second argument to DevQueryMedium().
 * 5. DisplayError() now has 2 arguments. It has been moved to 'videoObjects.c'.
 * 6. DevInit() has been moved to 'videoObjects.c'
 * 7. Added instance pointer to each videoObject call. 'myVideo'
 * 8. Able to choose device to use vcrEdit with.
 * 9. Able to set default device to be installed on startup by setting "vcrEdit.defaultPlayer"
 *    in ~/.Xdefaults.
 * 
 * Revision 0.11  91/07/20  11:42:16  lim
 * Added support for audio and speed settings for each edit.
 * Changed DevPlayFromTo calls to include speed field.
 * 
 * Revision 0.10  91/07/16  16:15:12  lim
 * Initial revision.
 *  */

#include "vcrEdit.h"

static char rcsid[] = "$Header: /Source/Media/collab/vcrEdit/RCS/vcrEdit.c,v 0.36 92/09/29 01:16:39 drapeau Exp $";

/* Variables for command line arguments */
int receiverPort;						    
char* startFilename;				                    
char hostname[MAX_HOSTNAME];					    

/* Edit list variables */
int lines;							    /* number of lines in the edit list */
int editnum;							    /* the current selection in the edit list */
int change;							    /* indicates if unsaved changes exist in the edit list */
int clearframe;							    /* determines if the frame/chapter textfield is to be cleared*/
int search;							    /* flag to indicate if a notify procedure is to be executed.
								       This is necessary because a textfield notify procedure is
								       executed twice for each key depressed (possibly a XView 
								       bug */

/* Arrays to store edit list info */
char startframe[MAX_LISTSIZE][12]; 
char endframe[MAX_LISTSIZE][12];				    /* arrays to store the start/end frames of
								       each entry in the edit list */
char label[MAX_LISTSIZE][MAX_LABELSIZE+1];			    /* array to store the label of each 
								       entry in the edit list */
enum AudioType audioStg[MAX_LISTSIZE];				    /* array to store the audio setting for each edit */
int speedStg[MAX_LISTSIZE];					    /* array to store the speed setting for each edit */

/* Tape position variables */
int currHr = 0;							    /* current hour */
int currMin = 0;						    /* current minute */
int currSec = 0;						    /* current second */
int currFrame = 0;						    /* current frame */

/* Other variables */
Xv_font *font;							    /* Font used in the panel list */
char deviceName[MaxNameLength];					    /* Name of device currently used */
enum Direction dir = Reverse;					    /* flag to indicate which direction to play */
static int input = 0;						    /* which time field has new input */
int plus = 0;							    /* whether TimePlus or TimeMinus is active */
int hitQuit = 0;						    /* Set when Quit button is hit. */
enum Boolean diagMode = No;					    /* Flag : whether diagnostic messages should be printed */

/*
 * Instance XV_KEY_DATA key.  An instance is a set of related
 * user interface objects.  A pointer to an object's instance
 * is stored under this key in every object.  This must be a
 * global variable.
 */

Attr_attribute	INSTANCE;


/* GENERAL PROCEDURES USED BY THE FUNCTIONS */

  
/* check if a selection is valid */
int
  CheckSelection(int start,
		 int end)
{
  char buf[50];
  if (start > end) 
  {
    DisplayError("Invalid current selection:", "Start greater than End.");
    return -1;
  }
  if ((start < 0) || (end < 0))
  {
    DisplayError("Addresses may not be negative numbers.", " ");
    return -1;
  }
  if ((start > MAX_FRAME) || (end > MAX_FRAME)) 
  {
    sprintf (buf, "Maximum frame number allowed is %d.\n", MAX_FRAME);
    DisplayError(buf, " ");
    return -1;
  }
  return 0;
}								    /* end function CheckSelection */

/*
 * This function parses the command line and retrieves all the known options and their arguments.
 * Currently, the two options are hostname and portnumber.
 * After parsing the options, the variable optind will point to the start of those non-option arguments.  
 * In this case, it will be the filename to be loaded.  At present, only one filename will be handled.  
 * So if there are multiple filenames typed, the last one will be loaded.
 */
void CheckOptions(argc, argv)
     int 	argc;
     char 	**argv;
{
  int optionChar;  
  int option_index = 0;
  static struct option long_options[] =
  {
    {"hostname", 1, 0, 'h'},
    {"portnumber", 1, 0, 'p'},
    {"diagnostics", 0, 0, 'd'},
    {0, 0, 0, 0}
  };

  while (1)							    /* Start parsing all known options */
  {
    optionChar = getopt_long_only (argc, argv, "h:p:d:",
			      long_options, &option_index);
    if (optionChar == EOF)					    /* Done with all known options */
    {
      break;
    }
    switch (optionChar)
    {
     case 'h':
      if (optarg) 
      {
	strcpy (hostname, optarg);
      }
      break;
     case 'p':
      if (optarg) 
      {
	receiverPort = atoi(optarg);
      }
      break;
     case 'd':
      diagMode = Yes;
      break;
     default:
      break;
    }
  }
  if (optind < argc)						    /* Check if a filename has been specified */
  {
    startFilename = (char *) malloc (256);
    strcpy (startFilename, argv[optind]);
  }
}


/* Convert time to address in frames */
int
ConvertToAddress(int hr, int min, int sec, int frame)
{
  int address;
  
  address = FrameRate * (hr * 3600 + min * 60 + sec) + frame;
  return address;
}								    /* end function ConvertToAddress */


/* Convert address in frames to time */
void
ConvertToTime(int* hr, int* min, int* sec, int* frame, int  address)
{
  int	framesPerHour;
  int	framesPerMinute;
  int	framesPerSecond;
  
  framesPerHour = 3600 * FrameRate;				    /* 3600 = seconds per hour */
  framesPerMinute = 60 * FrameRate;
  framesPerSecond = FrameRate;
  
  *hr = address / framesPerHour;
  *min = (address % framesPerHour) / framesPerMinute;
  *sec = ((address % framesPerHour) % framesPerMinute) / framesPerSecond;
  *frame = ((address % framesPerHour) % framesPerMinute) % framesPerSecond;
}								    /* end function ConvertToTime */



/*      REMOTE CONTROLLER FUNCTIONS	*/

/*
    Button notify proc to open the options popup window (optionsButton)
*/
void
Options(item, event)
        Panel_item      item;
        Event           *event;
{
  xv_set(vcrEdit_optionsPopup->optionsPopup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
  xv_set(vcrEdit_optionsPopup->optionsPopup, XV_SHOW, TRUE, NULL);
}

/*
    Button notify proc to open the edit popup window (editButton)
*/
void
Edit(item, event)
        Panel_item      item;
        Event           *event;
{
  xv_set(vcrEdit_editPopup->editPopup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
  xv_set(vcrEdit_editPopup->editPopup, XV_SHOW, TRUE, NULL);
}


/*
 * Notify callback function for `stopButton'.
 * Stop function
 */
void
Stop(item, event)
	Panel_item	item;
	Event		*event;
{
  DevStop(myVideo);
}

/*
 * Notify callback function for `ejectButton'.
 * Eject function.
 * Clears edit list, with option to save changed edit lists.
 */
void
Eject(item, event)
     Panel_item         item;
     Event              *event;
{
  int choice;
  
  if (change && lines)
  {
    choice = DisplayChoice("Ejecting tape will erase edit list", "Save edit list first?",
			   "Yes", "No");
    if (choice == NOTICE_YES)
    {
      Browse(NULL, BrowseSave, 0, "#VCR Edit Document#", "vcrEdit");
      change = 0;
    }
  }
  DevEject(myVideo);
  EditDeleteAll(NULL, NULL);					    
}


/*
 * Notify callback function for `searchButton'.
 * Search frame/chapter
 */
void
  Search(Panel_item	item,
	 Event*		event)
{
  int address;
  
  address = ConvertToAddress(currHr, currMin, currSec, currFrame);
  
  if (CheckSelection (address, MAX_FRAME) == -1) 
  {
    clearframe = 1;
    return;
  }
  DevPlayFromTo(myVideo, address, address, FrameRate);
  clearframe = 1;
}								    /* end function Search */



/* Hide Current Slider */
void
HideCurrSlider() 
{
  switch (input) 
  {
   case 0:
    xv_set(vcrEdit_window1->absSlider, XV_SHOW, FALSE, NULL);
    xv_set(vcrEdit_window1->absSlider, PANEL_INACTIVE, TRUE, NULL);
    break;
   case 1:
    xv_set(vcrEdit_window1->hrSlider, XV_SHOW, FALSE, NULL);
    xv_set(vcrEdit_window1->hrSlider, PANEL_INACTIVE, TRUE, NULL);
    break;
   case 2:
    xv_set(vcrEdit_window1->minSlider, XV_SHOW, FALSE, NULL);
    xv_set(vcrEdit_window1->minSlider, PANEL_INACTIVE, TRUE, NULL);
    break;
   case 3:
    xv_set(vcrEdit_window1->secSlider, XV_SHOW, FALSE, NULL);
    xv_set(vcrEdit_window1->secSlider, PANEL_INACTIVE, TRUE, NULL);
    break;
   case 4:
    xv_set(vcrEdit_window1->frameSlider, XV_SHOW, FALSE, NULL);
    xv_set(vcrEdit_window1->frameSlider, PANEL_INACTIVE, TRUE, NULL);
    break;
  }
}

/* 
 * Show new slider.
 * Places caret in corresponding text field.
 */
void
ShowSlider(value) 
     int value;
{
  switch (value) 
  {
   case 0:
    xv_set(vcrEdit_window1->absSlider, XV_SHOW, TRUE, NULL);
    xv_set(vcrEdit_window1->absSlider, PANEL_INACTIVE, FALSE, NULL);
    xv_set(vcrEdit_window1->remoteCtrlControls, PANEL_CARET_ITEM, vcrEdit_window1->frameTxt1, NULL);
    break;
   case 1:
    xv_set(vcrEdit_window1->hrSlider, XV_SHOW, TRUE, NULL);
    xv_set(vcrEdit_window1->hrSlider, PANEL_INACTIVE, FALSE, NULL);
    xv_set(vcrEdit_window1->remoteCtrlControls, PANEL_CARET_ITEM, vcrEdit_window1->frameTxt1, NULL);
    break;
   case 2:
    xv_set(vcrEdit_window1->minSlider, XV_SHOW, TRUE, NULL);
    xv_set(vcrEdit_window1->minSlider, PANEL_INACTIVE, FALSE, NULL);
    xv_set(vcrEdit_window1->remoteCtrlControls, PANEL_CARET_ITEM, vcrEdit_window1->frameTxt2, NULL);
    break;
   case 3:
    xv_set(vcrEdit_window1->secSlider, XV_SHOW, TRUE, NULL);
    xv_set(vcrEdit_window1->secSlider, PANEL_INACTIVE, FALSE, NULL);
    xv_set(vcrEdit_window1->remoteCtrlControls, PANEL_CARET_ITEM, vcrEdit_window1->frameTxt3, NULL);
    break;
   case 4:
    xv_set(vcrEdit_window1->frameSlider, XV_SHOW, TRUE, NULL);
    xv_set(vcrEdit_window1->frameSlider, PANEL_INACTIVE, FALSE, NULL);
    xv_set(vcrEdit_window1->remoteCtrlControls, PANEL_CARET_ITEM, vcrEdit_window1->frameTxt4, NULL);
    break;
  }
}

/* 
 * Get absolute time. 
 * Show time in textfield.
 * Show time in message. 
 */
void 
ShowAbsTime()
{
  char hr[2];
  char min[2];
  char sec[2];
  char frame[2];
  char time[11];
  
  sprintf(hr, "%.2d", currHr);
  sprintf(min, "%.2d", currMin);
  sprintf(sec, "%.2d", currSec);
  sprintf(frame, "%.2d", currFrame);
  
  xv_set(vcrEdit_window1->frameTxt1, PANEL_VALUE, hr, NULL);
  xv_set(vcrEdit_window1->frameTxt2, PANEL_VALUE, min, NULL);
  xv_set(vcrEdit_window1->frameTxt3, PANEL_VALUE, sec, NULL);
  xv_set(vcrEdit_window1->frameTxt4, PANEL_VALUE, frame, NULL);

  sprintf(time, "%.2d:%.2d:%.2d:%.2d", currHr, currMin, currSec, currFrame);
  
  xv_set(vcrEdit_window1->timeMsg, PANEL_LABEL_STRING, time, NULL);
  
}

/* 
 * Change absolute time.
 */
void
ChangeAbsTime(field, value)
     int field, value;
{

  switch (field) 
  {
   case 1:
    currHr = value;
    break;
   case 2:
    currMin = value;
    break;
   case 3:
    currSec = value;
    break;
   case 4:
    currFrame = value;
    break;
  }
  
  xv_set(vcrEdit_window1->absSlider, PANEL_VALUE, 
	 ConvertToAddress(currHr, currMin, currSec, currFrame), NULL);
  
}

  
  
/*
 * Notify callback function for `timeStg'.
 * Hides current slider.
 * Shows new slider and places caret in corresponding text field.
 * Shows new time.
 */
void
TimeInput(item, value, event)
	Panel_item	item;
	int		value;
	Event		*event;
{
  if (input != value)						    /* The current slider is not new slider */
  {
    HideCurrSlider();	
    input = value;
    ShowSlider(value);
  }
  
  ShowAbsTime();
	
}


/*
 * Notify callback function for `minSlider'.
 */
void
MinInput(item, value, event)
	Panel_item	item;
	int		value;
	Event		*event;
{
  ChangeAbsTime(2, value);
  ShowAbsTime();
  
}

/*
 * Notify callback function for `secSlider'.
 */
void
SecInput(item, value, event)
	Panel_item	item;
	int		value;
	Event		*event;
{
  ChangeAbsTime(3, value);
  ShowAbsTime();
}

/*
 * Notify callback function for `hrSlider'.
 */
void
HrInput(item, value, event)
	Panel_item	item;
	int		value;
	Event		*event;
{
  ChangeAbsTime(1, value);
  ShowAbsTime();
}

/*
 * Notify callback function for `frameSlider'.
 */
void
FrameInput(item, value, event)
	Panel_item	item;
	int		value;
	Event		*event;
{
  ChangeAbsTime(4, value);
  ShowAbsTime();
}

/*
 * Notify callback function for `absSlider'.
 */
void
AbsInput(item, value, event)
	Panel_item	item;
	int		value;
	Event		*event;
{
  int hr;
  int min;
  int sec;
  int frame;
  
  ConvertToTime(&hr, &min, &sec, &frame, value);
  currHr = hr;
  currMin = min;
  currSec = sec;
  currFrame = frame;

  xv_set(vcrEdit_window1->hrSlider, PANEL_VALUE, currHr, NULL);
  xv_set(vcrEdit_window1->minSlider, PANEL_VALUE, currMin, NULL);
  xv_set(vcrEdit_window1->secSlider, PANEL_VALUE, currSec, NULL);
  xv_set(vcrEdit_window1->frameSlider, PANEL_VALUE, currFrame, NULL);

  ShowAbsTime();

}

/*
 * Notify callback function for `timeMinusButton'.
 * Get current text item.
 * Decrement or increment item value.
 * Update corresponding slider value.
 * Display appropriate slider.
 * Display new text item value.
 */
void
TimeMinus(item, event)
	Panel_item	item;
	Event		*event;
{
  Panel_item currItem;
  
  currItem = (Panel_item) xv_get(vcrEdit_window1->remoteCtrlControls, PANEL_CARET_ITEM);
  
  if (currItem == vcrEdit_window1->frameTxt1) {
    if (plus) 
      currHr ++;
    else if (currHr)
      currHr --;
    xv_set(vcrEdit_window1->hrSlider, PANEL_VALUE, currHr, NULL);
    xv_set(vcrEdit_window1->timeStg, PANEL_VALUE, 1, NULL);
    TimeInput(item, 1, event);
  }
  else if (currItem == vcrEdit_window1->frameTxt2) {
    if (plus) 
      currMin ++;
    else if (currMin)
      currMin --;
    xv_set(vcrEdit_window1->minSlider, PANEL_VALUE, currMin, NULL);
    xv_set(vcrEdit_window1->timeStg, PANEL_VALUE, 2, NULL);
    TimeInput(item, 2, event);
  }
  else if (currItem == vcrEdit_window1->frameTxt3) {
    if (plus) 
      currSec ++;
    else if (currSec)
      currSec --;
    xv_set(vcrEdit_window1->secSlider, PANEL_VALUE,currSec, NULL);
    xv_set(vcrEdit_window1->timeStg, PANEL_VALUE, 3, NULL);
    TimeInput(item, 3, event);
  }
  else if (currItem == vcrEdit_window1->frameTxt4) {
    if (plus) 
      currFrame ++;
    else if (currFrame)
      currFrame --;
    xv_set(vcrEdit_window1->frameSlider, PANEL_VALUE, currFrame, NULL);
    xv_set(vcrEdit_window1->timeStg, PANEL_VALUE, 4, NULL);
    TimeInput(item, 4, event);
  }
  
  plus = 0;
}

/*
 * Notify callback function for `timePlusButton'.
 */
void
TimePlus(item, event)
	Panel_item	item;
	Event		*event;
{
  plus = 1;
  TimeMinus(item, event);
}


       

/*
 * Notify callback function for `frameTxt1'.
 */
Panel_setting
HrTxtInput(item, event)
	Panel_item	item;
	Event		*event;
{
  currHr = atoi((char*) xv_get(vcrEdit_window1->frameTxt1, PANEL_VALUE));
  xv_set(vcrEdit_window1->hrSlider, PANEL_VALUE, currHr, NULL);
  xv_set(vcrEdit_window1->timeStg, PANEL_VALUE, 1, NULL);
  TimeInput(item, 1, event);
  return panel_text_notify(item, event);       
}	

/*
 * Notify callback function for `frameTxt2'.
 */
Panel_setting
MinTxtInput(item, event)
	Panel_item	item;
	Event		*event;
{
  currMin = atoi((char*) xv_get(vcrEdit_window1->frameTxt2, PANEL_VALUE));
  xv_set(vcrEdit_window1->minSlider, PANEL_VALUE, currMin, NULL);
  xv_set(vcrEdit_window1->timeStg, PANEL_VALUE, 2, NULL);
  TimeInput(item, 2, event);
  return panel_text_notify(item, event);
}

/*
 * Notify callback function for `frameTxt3'.
 */
Panel_setting
SecTxtInput(item, event)
	Panel_item	item;
	Event		*event;
{
  currSec = atoi((char*) xv_get(vcrEdit_window1->frameTxt3, PANEL_VALUE));
  xv_set(vcrEdit_window1->secSlider, PANEL_VALUE, currSec, NULL);
  xv_set(vcrEdit_window1->timeStg, PANEL_VALUE, 3, NULL);
  TimeInput(item, 3, event);
  return panel_text_notify(item, event);
}

/*
 * Notify callback function for `frameTxt4'.
 */
Panel_setting
FrameTxtInput(item, event)
	Panel_item	item;
	Event		*event;
{
  currFrame = atoi((char*) xv_get(vcrEdit_window1->frameTxt4, PANEL_VALUE));
  xv_set(vcrEdit_window1->frameSlider, PANEL_VALUE, currFrame, NULL);
  xv_set(vcrEdit_window1->timeStg, PANEL_VALUE, 4, NULL);
  TimeInput(item, 4, event);
  return panel_text_notify(item, event);
}

/*
 * Notify callback function for `indexScanFwdButton'.
 */
void
IndexScanFwd(item, event)
	Panel_item	item;
	Event		*event;
{
  DevSetAddMode(myVideo, 1);
  DevFastForward(myVideo);
}

/*
 * Notify callback function for `indexScanRevButton'.
 */
void
IndexScanRev(item, event)
	Panel_item	item;
	Event		*event;
{
  DevSetAddMode(myVideo, 1);
  DevReverse(myVideo);
}

	

/* Displays current time address in text fields
 * Sets the sliders to the appropriate values
 * Displays time message above slider
 */
void 
DisplayCurrentFrame(address)
     int address;
{
  int hr;
  int min;
  int sec;
  int fr;
  char hour[2];
  char minute[2];
  char second[2];
  char frame[2];
  char time[11];
  
  ConvertToTime(&hr, &min, &sec, &fr, address);
  
  currHr = hr;
  currMin = min;
  currSec = sec;
  currFrame = fr;
  
  sprintf(hour, "%.2d", hr);
  sprintf(minute, "%.2d", min);
  sprintf(second, "%.2d", sec);
  sprintf(frame, "%.2d", fr);
  sprintf(time, "%.2d:%.2d:%.2d:%.2d", hr, min, sec, fr);
  
  xv_set(vcrEdit_window1->frameTxt1, PANEL_VALUE, hour, NULL);
  xv_set(vcrEdit_window1->frameTxt2, PANEL_VALUE, minute, NULL);
  xv_set(vcrEdit_window1->frameTxt3, PANEL_VALUE, second, NULL);
  xv_set(vcrEdit_window1->frameTxt4, PANEL_VALUE, frame, NULL);
  
  xv_set(vcrEdit_window1->timeMsg, PANEL_LABEL_STRING, time, NULL);
  
  xv_set(vcrEdit_window1->absSlider, PANEL_VALUE, address, NULL);
  xv_set(vcrEdit_window1->hrSlider, PANEL_VALUE, hr, NULL);
  xv_set(vcrEdit_window1->minSlider, PANEL_VALUE, min, NULL);
  xv_set(vcrEdit_window1->secSlider, PANEL_VALUE, sec, NULL);
  xv_set(vcrEdit_window1->frameSlider, PANEL_VALUE, fr, NULL);
}

/*
 * Notify callback function for `currFrameButton'.
 * Gives the current frame/chapter number.
 */
void
CurrentFrame(item, event)
	Panel_item	item;
	Event		*event;
{
  int address;
  address = DevQueryFrame(myVideo);
  if (address != PlayerReturnError)
    DisplayCurrentFrame(address);
  clearframe = 1;
}


/*
 * Notify callback function for `scanRevButton'.
 */
void
ScanReverse(item, event)
	Panel_item      item;
	Event           *event;
{
  DevReverse(myVideo);
}

/*
 * Notify callback function for `scanFwdButton'.
 */
void
ScanForward(item, event)
	Panel_item      item;
	Event           *event;
{
  DevFastForward(myVideo);
}


/*
 * Notify callback function for `playButton'.
 * Play
 */
void
Play(item, event)
	Panel_item	item;
	Event		*event;
{
  int audioSetting;
  audioSetting = xv_get(vcrEdit_optionsPopup->optionsAudioStg, PANEL_VALUE);
  DevSetAudio(myVideo, audioSetting);
  DevPlay(myVideo);
}

/*
 * Notify callback function for `indexNextButton'.
 */
void
NextIndexSearch(item, event)
	Panel_item	item;
	Event		*event;
{
    DevSetAddMode(myVideo, 1);
    DevPlayFromTo(myVideo, 1, 0, FrameRate);			    /* Special semantics : toAdd is direction */
								    /* 0 = Forward */
}


/*
 * Notify callback function for `indexPrevButton'.
 */
void
PrevIndexSearch(item, event)
	Panel_item	item;
	Event		*event;
{
    DevSetAddMode(myVideo, 1);					    /* Set to index */
    DevPlayFromTo(myVideo, 1, 1, FrameRate);			    /* Special semantics : toAdd is direction */
								    /* 1 = Reverse */
}


/*
 * Notify callback function for `pauseButton'.
 * Pause mode (video does not appear on screen)
 */
void
Pause(item, event)
     Panel_item	item;
     Event		*event;
{  
  DevStill(myVideo);
}



/*
 * Notify callback function for `fwdStepButton'.
 */
void
ForwardStep(item, event)
     Panel_item	item;
     Event		*event;
{
  int status;
  
  status = DevQueryStatus(myVideo);
  if ((status != PlayerPause) && (status != PlayerStill))
    DevStill(myVideo);
  DevStep(myVideo, Forward);
}


/*
 * Notify callback function for `revStepButton'.
 */
void
ReverseStep(item, event)
	Panel_item	item;
	Event		*event;
{
  int status;
  
  status = DevQueryStatus(myVideo);
  if ((status != PlayerPause) && (status != PlayerStill))
    DevStill(myVideo);
  DevStep(myVideo, Reverse);
}



/*
 * Notify callback function for `speedSlider'.
 */
void
  SpeedChange(item, value, event)
Panel_item	item;
int		value;
Event		*event;
{
  char			s[5];
  static int		oldspeed = 0;
  enum Direction	direction;
  int			framesPerSecond;
  
  direction = Forward;
  if (event_is_up(event))					    /* Put in still mode if mouse button is up */
  {
    oldspeed = 0;
    xv_set(item, PANEL_VALUE, 0, NULL);
    xv_set(vcrEdit_window1->speedTxt, PANEL_VALUE, " ", NULL);
    DevStill(myVideo);
    return;
  }
  if (value < 0)						    /* Check if forward or reverse play */
  {
    s[0] = '-';
    value = -value;
    direction = Reverse;
  }
  else 
    s[0] = ' ';
  
  framesPerSecond = DevCalcSpeed(myVideo, value, 0);		    /* Calculate the speed to that requested */
  sprintf (&s[1], "%d", framesPerSecond);
  
  if ((framesPerSecond != oldspeed) && (framesPerSecond != 0))	    /* Was a speed change really requested? */
  {
    oldspeed = framesPerSecond;
    xv_set(vcrEdit_window1->speedTxt, 
	   PANEL_VALUE, s, 
	   NULL);
    DevPlayAtSpeedDir(myVideo, framesPerSecond, direction);
  }
}								    /* end function SpeedChange */

/*
 * Notify callback function for `powerButton'.
 */
void
Power(item, event)
     Panel_item	item;
     Event		*event;
{
  DevPower(myVideo, -1);
}


/*
 * Notify callback function for `quitButton'.
 * Quit from the whole program
 */
void
Quit(item, event)
     Panel_item	item;
     Event		*event;
{
  int result;
  
  hitQuit = 1;
  
								    /* check if unsaved changes exist in edit list */
  if ((change) && (lines > 0)) {
    result = DisplayChoice("Unsaved changes exist in the edit list.",
			   "Go ahead and quit? ",
			   "No", "Yes");			    /* Labels are inverted so that defaults to "No" */
    if (result == NOTICE_YES)
    {
      hitQuit = 0;
      return;
    }
    
  }
  indexing = 0;
  
  if (sender)
    SenderDisconnectFromPortMgr(sender,&(receiver->receivePort));
  xv_destroy_safe(vcrEdit_window1->window1);
}


/* FUNCTIONS FOR INFO POPUP WINDOW */
/*
 * Notify callback function for `infoOKButton'.
 * Close the info popup window
 */
void
InfoDone(item, event)
        Panel_item      item;
        Event           *event;
{
  xv_set(vcrEdit_infoPopup->infoPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(vcrEdit_infoPopup->infoPopup, XV_SHOW, FALSE, NULL);
}
 


#ifdef MAIN

void
  main(argc, argv)
int	argc;
char	**argv;
{
  int i; 
  int value;
  Display* display;
  char* defaultPlayer;  
  Rect  tempRect;
  
  xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);		    /* Initialize XView. */
  INSTANCE = xv_unique_key();
  strcpy(hostname, "localhost");				    /* Check parameters passed in */
  startFilename = NULL;
  sprintf(currentFilename, "Untitled");
  receiverPort = AnyPort;
  CheckOptions(argc, argv);
  
  vcrEdit_window1 =						    /* Initialize user interface components. */
    vcrEdit_window1_objects_initialize(NULL, NULL);
  vcrEdit_optionsPopup =
    vcrEdit_optionsPopup_objects_initialize(NULL,vcrEdit_window1->window1);
  vcrEdit_editPopup =
    vcrEdit_editPopup_objects_initialize(NULL,vcrEdit_window1->window1);
  vcrEdit_previewPopup =
    vcrEdit_previewPopup_objects_initialize(NULL,vcrEdit_window1->window1);
  vcrEdit_infoPopup =
    vcrEdit_infoPopup_objects_initialize(NULL,vcrEdit_window1->window1);
  
  frame_get_rect(vcrEdit_window1->window1,&tempRect);		    /* Get the size of the top-level window */
  tempRect.r_top = 250;
  tempRect.r_left = 200;
  frame_set_rect(vcrEdit_window1->window1,&tempRect);		    /* Set the position of the top-level window */
  
  font = (Xv_font *) xv_find(vcrEdit_editPopup->editPopup, FONT,    /* Set fonts */
			     FONT_FAMILY, FONT_FAMILY_LUCIDA_FIXEDWIDTH,
			     FONT_STYLE,  FONT_STYLE_NORMAL,
			     FONT_SIZE, 12,
			     NULL);
  xv_set(vcrEdit_window1->window1,				    /* Initialize some window attributes */
	 FRAME_LEFT_FOOTER, "Display : Off",
	 FRAME_RIGHT_FOOTER, "Audio : Stereo",
	 NULL);
  xv_set(vcrEdit_optionsPopup->optionsPopup, 
	 XV_X, 200,
	 XV_Y, 0,
	 NULL);
  xv_set(vcrEdit_editPopup->editPopup, 
	 XV_X, 450,
	 XV_Y, 250,
	 FRAME_DONE_PROC, EditDone,
	 NULL);
  xv_set(vcrEdit_previewPopup->previewPopup, 
	 XV_X, 910,
	 XV_Y, 560,
	 NULL);
  xv_set(vcrEdit_infoPopup->infoPopup, 
	 XV_X, 200,
	 XV_Y, 250,
	 NULL);
  
  xv_set(vcrEdit_optionsPopup->optionsAudioStg, PANEL_VALUE, 3, NULL);
  xv_set(vcrEdit_optionsPopup->optionsVideoStg, PANEL_VALUE, 1, NULL);
  xv_set(vcrEdit_optionsPopup->optionsMutedScreen, PANEL_VALUE, 2, NULL);
  xv_set(vcrEdit_optionsPopup->optionsBrightStg, PANEL_VALUE, 0, NULL);
  xv_set(vcrEdit_previewPopup->previewPlayFirstTxt, PANEL_VALUE, 4, NULL);
  xv_set(vcrEdit_previewPopup->previewPlayLastTxt, PANEL_VALUE, 4, NULL);
  xv_set(vcrEdit_window1->frameTxt1, PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_window1->frameTxt2, PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_window1->frameTxt3, PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_window1->frameTxt4, PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_window1->frameTxt1, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(vcrEdit_window1->frameTxt2, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(vcrEdit_window1->frameTxt3, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(vcrEdit_window1->frameTxt4, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(vcrEdit_window1->hrSlider, XV_SHOW, FALSE, NULL);
  xv_set(vcrEdit_window1->minSlider, XV_SHOW, FALSE, NULL);
  xv_set(vcrEdit_window1->secSlider, XV_SHOW, FALSE, NULL);
  xv_set(vcrEdit_window1->frameSlider, XV_SHOW, FALSE, NULL);
  xv_set(vcrEdit_window1->hrSlider, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrEdit_window1->minSlider, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrEdit_window1->secSlider, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrEdit_window1->frameSlider, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrEdit_window1->absSlider, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(vcrEdit_window1->hrSlider, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(vcrEdit_window1->minSlider, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(vcrEdit_window1->secSlider, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(vcrEdit_window1->frameSlider, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(vcrEdit_window1->speedSlider, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(vcrEdit_editPopup->editSpeedTxt, PANEL_VALUE, FrameRate, NULL);
  if (startFilename)
    Browse(startFilename, BrowseCheckOpen, 0,		    
	   "#VCR Edit Document#", "vcrEdit");				
  xv_set(vcrEdit_editPopup->editModButton,			    /* Set modify, delete buttons to inactive */
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrEdit_editPopup->editDelButton,
	 PANEL_INACTIVE, TRUE, NULL);
  InitNetwork(hostname);					    /* Initialize network stuff */
  bzero(chosenHost, MAX_HOSTNAME);				    /* Initialize chosen hostname array */
  
  display =  (Display*) xv_get(vcrEdit_window1->window1, XV_DISPLAY);
  defaultPlayer = XGetDefault(display, "vcrEdit", "defaultPlayer");
  value = 0;
  
  xv_set(vcrEdit_optionsPopup->optionsPlayerStg, 
	 PANEL_NCHOICES, numDevices, NULL);
  for (i=1; i < numDevices+1; i++)
  {
    xv_set(vcrEdit_optionsPopup->optionsPlayerStg,
	   PANEL_CHOICE_STRING, i, allDevices[i-1], 
	   NULL);
    if (defaultPlayer)
      if (strcmp(defaultPlayer, allDevices[i-1]) == 0)
	value = i+1;
  }
  
  if (value)
    xv_set(vcrEdit_optionsPopup->optionsPlayerStg,		    /* Set player setting to default player */
	   PANEL_VALUE, value-1,
	   NULL);
  
  if (defaultPlayer)
    SetPlayer(NULL, value-1, NULL);				    /* Set up default player */
  else
    Options(NULL, NULL);
  
  CreateBrowse(OpenHandler, SaveHandler, vcrEdit_window1->window1);
  
  xv_main_loop(vcrEdit_window1->window1);
  
  if (!hitQuit)
    Quit(NULL, NULL);
  
  exit(0);
}								    /* end function main */

#endif
