/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/cdEdit/RCS/timer.c,v 2.0 91/10/06 21:01:59 chua Exp $ */
/* $Log:	timer.c,v $
 * Revision 2.0  91/10/06  21:01:59  chua
 * Update to version 2.0
 * 
 * Revision 1.13  91/09/10  15:56:02  chua
 * Changed some code in UpdateTime so that currentTrack will not be
 * set to track if the player is in stop mode.  This was causing an error when
 * we search to the last track from rest, and the track duration field
 * would not be updated.
 * 
 * Revision 1.12  91/09/05  17:52:44  chua
 * In TimerNotify, check to see if there is a need to highlight either the play or
 * pause button in green, depending on the mode that the player is in.
 * 
 * Revision 1.11  91/09/03  15:29:04  chua
 * Added the copyright header.
 * 
 * Changed the constants from all capitalized to initial capitalized only.
 * 
 * In TimerNotify, if there is currently no disc (playerState == Nothing), check to see
 * if a disc has been inserted by calling InitializeCD.
 * 
 * Added the menu handlers for the relative time and absolute time menu items (found under
 * the options menu).
 * 
 * Revision 1.1  91/07/09  16:49:00  chua
 * 
 * 
 * Revision 1.0  91/07/08  13:46:43  chua
 * Initial revision
 *  */

static char timerrcsid[] = "$Header: /Source/Media/collab/cdEdit/RCS/timer.c,v 2.0 91/10/06 21:01:59 chua Exp $";

#include "main.h"

static int timetype = Relative;
static currentTrack = -1;
static flag = 0;

/* 
 * This function will update the time display field on the remote control, depending on the value of the parameter, time, passed to it.
 * If time = Current, the current playing time on the CD is displayed.
 * If time = Whole, the total length of the CD is displayed.
 * If time = Clear, the time display field is set to all zeros.
 * The duration slider will also be updated, if the user is not currently dragging the slider (when playerState != Duration).
 */
void UpdateTime(time)
     int time;
{
  int track, min, sec, frame;
  int durationMin, durationSec;
  char t[17], m[12], d[5], s[5];
  Msf duration;
  
  if (time == Current)						    /* Display the current playing time on the CD */
  {
    if (timetype == Absolute)					    /* Get absolute time information */
    {
      cdrom_get_absinfo(fd, &track, &min, &sec, &frame);
    }
    else							    /* Get relative time information */
    {
      cdrom_get_relinfo(fd, &track, &min, &sec, &frame);
    }
  }
  else if (time == Whole)					    /* Display the total time and number of tracks in the CD */
  {
    currentTrack = -1;
    track = numTracks;
    sprintf(t, "Total tracks : %02d", track);
    min = toc->total_msf->min;
    sec = toc->total_msf->sec;
  }
  else 
  {
    track = 0;							    /* Set the time to all zeros.  Disc has been ejected. */
    min = 0;
    sec = 0;
  }
  sprintf(m, "Time : %02d:%02d", min, sec);
  durationMin = 0;
  durationSec = 0;
  if (time == Current) 
  {
    if (timetype == Relative)					    /* Get the duration of the track, in order to update */
    {								    /* the duration slider later (if necessary) */
      duration = get_track_duration(toc, track);
      if (duration != NULL) 
      {
	durationMin = duration->min;
	durationSec = duration->sec;
      }
    }
    else 
    {
      durationMin = toc->total_msf->min;			    /* Set the duration to the disc duration (absolute duration) */
      durationSec = toc->total_msf->sec;
    }
  }
  if (time != Current)						    /* min, sec is now used to update the duration time display.  They will only */
  {								    /* be relevant if time == Current, else they are set to zero. */
    min = 0;
    sec = 0;
  }
  xv_set(cdEdit_window1->TimeMsg,				    /* Update the time display field */
	 PANEL_LABEL_STRING, m,
	 NULL);
  if (track != currentTrack || timetype != displayType)		    /* Update the track and time display if either the track varies or  */
  {								    /* the display time varies (whether absolute or relative) */
    if (timetype == Relative)					    /* Set the label for the duration slider */
    {
      xv_set(cdEdit_window1->DurationLabelMsg,
	     PANEL_LABEL_STRING, "Track Duration",
	     NULL);
    }
    else 
    {
      xv_set(cdEdit_window1->DurationLabelMsg,
	     PANEL_LABEL_STRING, "Disc Duration",
	     NULL);
    }
    if (time != Whole)						    /* Update the track display field */
    {
      sprintf(t, "Track : %02d", track);
    }
    xv_set(cdEdit_window1->TotalTracksMsg,
	   PANEL_LABEL_STRING, t,
	   NULL);
    if (time == Current) 
    {
      xv_set(cdEdit_window1->DurationSlider,			    /* Set the min, max range for the duration slider */
	     PANEL_MIN_VALUE, 0,
	     PANEL_MAX_VALUE, durationSec + durationMin*60,
	     NULL);
      sprintf(d, "%02d:%02d", durationMin, durationSec);
      currentTrack = track;					    /* Set the global flags to the current values */
    }
    else							    /* Ignore the duration slider since we are not in play mode */
    {
      sprintf(d, "00:00");
    }
    xv_set(cdEdit_window1->DurationTimeMsg,			    /* Set the duration slider time (either track duration or disc duration) */
	   PANEL_LABEL_STRING, d,
	   NULL);
    displayType = timetype;					    /* Keeps track of whether the player is in relative or absolute mode */
  }
  if (playerState != DurationMode) 
  {
    xv_set(cdEdit_window1->DurationSlider,			    /* Set the duration slider to the correct position */
	   PANEL_VALUE, sec + min*60,
	   NULL);
    sprintf(s, "%02d:%02d", min, sec);
    xv_set(cdEdit_window1->TempDurationMsg,			    /* Update the duration slider time display */
	   PANEL_LABEL_STRING, s,
	   NULL);
  }
}

/*
 * This is the timer notify function that is called every second.
 * First, it checks if there is a need to hightlight the play or pause button in green, depending on what mode the player is in.
 * It will then either update the time display field with the current time, or show the total duration time of the disc, depending on which state the
 * CD player is in.  It calls the function UpdateTime (see above) to do the updating.
 */
Notify_value TimerNotify()
{
  int track;
  
  if ((cdrom_playing(fd, &track) || playerState == PauseMode) && flag) /* Highlight the play button in green or black (alternating) */
  {
    xv_set(cdEdit_window1->PlayButton, PANEL_ITEM_COLOR, gcm_color_index("green"), NULL);
    if (playerState != PauseMode) 
    {
      xv_set(cdEdit_window1->PauseButton, PANEL_ITEM_COLOR, gcm_color_index("black"), NULL);
    }
    flag = 0;
  }
  else if (playerState == PauseMode) 
  {
    xv_set(cdEdit_window1->PlayButton, PANEL_ITEM_COLOR, gcm_color_index("black"), NULL);
    xv_set(cdEdit_window1->PauseButton, PANEL_ITEM_COLOR, gcm_color_index("green"), NULL);
    if (flag) 
      flag = 0;
    else
      flag = 1;
  }
  else 
  {
    xv_set(cdEdit_window1->PlayButton, PANEL_ITEM_COLOR, gcm_color_index("black"), NULL);
    xv_set(cdEdit_window1->PauseButton, PANEL_ITEM_COLOR, gcm_color_index("black"), NULL);
    flag = 1;
  }
  if (cdrom_playing(fd, &track) ||				    /* CD player is currently playing.  Update the current time. */
      playerState == PauseMode || playerState == DurationMode)	    /* CD player is in pause mode, or user is currently dragging the slider bar */
  {								    /* whether CD is playing.  Also update the current time */
    UpdateTime(Current);
  }
  else if (playerState == Nothing)				    /* Clear the time display */
  {
    UpdateTime(Clear);
    InitializeCD();						    /* Check if a disc has been inserted */
  }
  else								    /* Player is in stop mode.  Show the total time of the disc */
  {
    UpdateTime(Whole);
    playerState = StopMode;
  }    
  return NOTIFY_DONE;
}


/*
 * Menu handler for `DisplayTimeMenu (Relative)'.
 */
Menu_item RelativeTimeHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    timetype = Relative;
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Menu handler for `DisplayTimeMenu (Absolute)'.
 */
Menu_item AbsoluteTimeHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    timetype = Absolute;
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

