/*
 * Copyright (c) 1990, 1991, 1992 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/VideoObject/RCS/videoObj.h,v 0.31 92/09/03 10:24:53 drapeau Exp $ */
/* $Log:	videoObj.h,v $
 * Revision 0.31  92/09/03  10:24:53  drapeau
 * Changed communications settings for the NEC driver configuration.
 * 
 * Revision 0.30  92/09/02  17:13:15  drapeau
 * Added support and documentation for a new field in the Config structure,
 * "flowControl".  Modified the appropriate data structures (e.g.,
 * device-specific Config structs).
 * 
 * Revision 0.29  92/09/01  17:03:30  drapeau
 * Made several changes:
 * * Updated copyright notice.
 * * Modified documentation to reflect addition of the two new
 *   methods, DevRecord() and DevRecordFromTo().
 * * Made code more ANSI-compliant through use of function prototypes.
 * * Cosmetic changes to make code more readable.
 * 
 * Revision 0.28  92/07/30  15:13:28  drapeau
 * Removed window system-specific include files.
 * Also, defined "FrameRate", a default definition for frame rates to be used
 * by all drivers in the library.  This definition may be overridden by the
 * Makefile variables.
 * 
 * Revision 0.27  92/06/17  01:02:09  drapeau
 * Changed termio usage, used more portable "termio" structure and
 * associated calls instead of "termios".
 * 
 * Revision 0.26  92/06/16  23:35:19  drapeau
 * Added AppClient for use by XView notify_set_input_func.
 * 
 * Revision 0.25  92/01/03  16:56:09  drapeau
 * Removed inclusion of string.h so as not to conflict with other files
 * that use this driver.
 * 
 * Revision 0.24  91/09/30  17:09:19  lim
 * Added DevPing, DefaultPing, PlayerVerified.
 * 
 * Revision 0.23  91/09/27  14:32:29  lim
 * Removed serialLine as a global.
 * 
 * Revision 0.22  91/08/24  17:50:33  lim
 * Added comments at the end with configurations of devices.
 * 
 * Revision 0.21  91/08/24  13:59:43  lim
 *  CheckImplemented corrected to return an enum Boolean.
 * 
 * Revision 0.20  91/08/24  13:39:36  lim
 * 1. Clear Marker() removed as part of video Object.
 * 
 * Revision 0.19  91/08/17  20:43:59  lim
 * 1. Removed 'interruptibleSegmentPlay' from video object. All stops and
 * >> pauses must be implemented such that they can interrupt segment play.
 * >> .
 * 2. Removed 'interruptible' because no longer used.
 * 
 * Revision 0.18  91/08/08  17:20:09  lim
 * 1. CalcSpeed() - one parameter added, 'playMode'. If playMode is 1, it is used to calculate
 *    device speed for segment play. If playMode is 0, it is used to calculate device speed for
 *    normal play.
 * 2. QueryMedium() - one parameter added, 'result'. Used to return non-integer results.
 * 
 * Revision 0.17  91/08/07  13:32:23  lim
 * 1. Added _VideoObjectHeader_
 * 2. Added <X11/resource.h>
 * 3. Added 'MyAbs' and 'MyRound'.
 * 4. Removed ErrorDecode from VideoObject.
 * 5. Config has new field, fd, for the file descriptor for the object.
 * 
 * Revision 0.16  91/08/02  13:06:39  lim
 * 1. Defines for CheckImplemented() arguments.
 * 2. Max speed now part of configuration structure.
 * 3. VideoObject now includes configuration structure and
 * Pause() has been removed.
 * 4. allDevices changed from array of device configurations to array
 * of names of all devices supported.
 * 5.
 * Added : GetDeviceMaxSpeed
 *         CheckImplemented
 * 
 * Revision 0.15  91/07/29  22:33:26  lim
 * If there is no eol char, it now means non-canonical input.
 * 
 * Revision 0.14  91/07/27  22:30:18  lim
 * Changed speed from double to int.
 * 
 * Revision 0.13  91/07/23  16:01:12  lim
 * Added field interruptibleSegmentPlay.
 * Changed static variables to globals.
 * Changed floats to doubles.
 * 
 * Revision 0.12  91/07/19  14:16:28  lim
 * Removed old SetSerialPort function.
 * 
 * 
 * Revision 0.11  91/07/17  22:54:46  lim
 * Removed static variable currDevice.
 * 
 * Revision 0.10  91/07/12  16:25:53  lim
 * Initial revision implementing VideoObject.
 *  */

/* IMPORTANT : NAMING CONVENTION 
 * ... All functions for the driver should be prefixed by ...
 * ... the name of the model so as to avoid clashes with other modules ...
 * ... already written.
 */

#ifndef _VideoObjectHeader_
#define _VideoObjectHeader_

#include <errno.h>
#include <stdio.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/termio.h>
#include <sys/file.h>

#define MaxNameLength 30
#define MaxPathLength 256
#define MaxDevices    20
#define Ok            0
#define InitError     -1
#define TimeOut       10					    /* Set Ping timeout to 10 seconds */
#define AppClient     200
#define FrameRate	30					    /* NTSC Frame Rate; override in Makefile */

#define MyAbs(n)          ((n) > 0? (n) : (-n))
#define MyRound(n, d)     ((n%d)/d > 0.5? (n/d+1) : (n/d))

/* Definition of functions of the VideoObject */
/* This is used in CheckImplemented, so that the user can check if a certain...
 * ... VideoObject function has been implemented by the current video object. */

#define PlayFunction                     1
#define PlayFromToFunction               2
#define FastForwardFunction              3
#define ReverseFunction                  4
#define CalcSpeedFunction                5
#define PlayAtSpeedDirFunction           6
#define StepFunction                     7
#define StillFunction                    8
#define StopFunction                     9
#define SetDefaultsFunction             10
#define SetAudioFunction                11
#define SetVideoFunction                12
#define SetAddModeFunction              13
#define SetAddressDisplayFunction       14
#define EjectFunction                   15
#define PowerFunction                   16
#define QueryFrameFunction              17
#define QueryChapterFunction            18
#define QueryAudioFunction              19
#define QueryVideoFunction              20
#define QueryMediumFunction             21
#define QueryStatusFunction             22
#define PingFunction                    23
#define RecordFunction			24
#define RecordFromToFunction		25

enum Parity {Odd, Even, Any, NoParity};
enum Boolean {No, Yes};
enum Direction {Forward, Reverse};

/* Definition of VideoObject */
/* DEFINING NEW OBJECTS 
 * 1. Implement each relevant function for the device in your application.
 * 2. Include file "videoObjects.c" into your application.
 * 3. Modify DevInit() to include the new device. This means writing the new object
 *    including the configuration structure, and setting 'myVideo' to the new
 *    object if the new device is selected. 
 * NOTE: Do not remove any calls made in the function, because they set up the data
 *     structures for the video objects.
 */


/* Definition of Configuration */
/*    The structure Config stores the configuration for the player, including serial line
 * settings. This structure is stored with the video object associated with the player.
 * To allow easy change of serial port connection, the serial port may be set in ~/.Xdefaults,
 * using "<deviceName>.serialPort".  This setting overrides the setting in the configuration
 * structure. "videoObjects.c" has to be changed and the application recompiled if there are
 * changes to be made for other settings in the configuration.
 *
 *
 * Representation of a device in Config: 
 *   modelName  serialPort  baudRate  eolChar  charSize  parity  igParError flowControl maxSpeed  fileDescriptor
 *
 *  where : modelName is a string denoting the name of the device
 *          serialPort is a string denoting the serial port to which the device is attached 
 *          baudRate is an integer denoting the baud rate at which data transfer is carried out 
 *          eolChar represents the ASCII value (decimal) of the character sent by the 
 *             device denoting the end of a line and that a read() should be performed.
 *          charSize can be 5, 6, 7 or 8
 *          parity can either be Even, Odd, Any, or None 
 *          igParError can be either No or Yes
 *	    flowControl can be either No or Yes
 *          maxSpeed is an integer denoting the maximum speed in terms of frames/second that
 *             the device can play
 *          fileDescriptor is an integer denoting the file descriptor for the device (always initialized to -1,
 *             or any negative number)
 * 
 *  NOTE: 
 *  1. If the device does not return an end-of-line character, enter a negative number in its place.
 *     It is assumed that non-canonical input processing will be used if a negative number is entered
 *     for the end-of-line character. 
 *     Struct termio defaults: c_cc[VMIN] = 1 and c_cc[VTIME] = 0.
 *  2. If serialPort is not fixed (determined by ~/.Xdefaults file), enter 'na'.
 *
 * eg. 
 
      static Config Pioneer4200Config = 
      {
        "Pioneer4200",
        "/dev/ttya",
	4800,
	13, 
	8,
	Even,
	Yes,
	Yes,
	120,
	-1
	}
	;

      static Config Sony1550Config = 
      {
        "Sony1550",
        "na",							   <= serial port set in ~/.Xdefaults
	2400,
	-1,							   <= non-canonical input processing
	8,
	Even,
	Yes,
	Yes,
	90, 
	-1
	}
	;

     ASCII value table (Decimal) - Character

     |  0 NUL|  1 SOH|  2 STX|  3 ETX|  4 EOT|  5 ENQ|  6 ACK|  7 BEL|
     |  8 BS |  9 HT | 10 NL | 11 VT | 12 NP | 13 CR | 14 SO | 15 SI |
     | 16 DLE| 17 DC1| 18 DC2| 19 DC3| 20 DC4| 21 NAK| 22 SYN| 23 ETB|
     | 24 CAN| 25 EM | 26 SUB| 27 ESC| 28 FS | 29 GS | 30 RS | 31 US |
     | 32 SP | 33  ! | 34  " | 35  # | 36  $ | 37  % | 38  & | 39  ' |
     | 40  ( | 41  ) | 42  * | 43  + | 44  , | 45  - | 46  . | 47  / |
     | 48  0 | 49  1 | 50  2 | 51  3 | 52  4 | 53  5 | 54  6 | 55  7 |
     | 56  8 | 57  9 | 58  : | 59  ; | 60  < | 61  = | 62  > | 63  ? |
     | 64  @ | 65  A | 66  B | 67  C | 68  D | 69  E | 70  F | 71  G |
     | 72  H | 73  I | 74  J | 75  K | 76  L | 77  M | 78  N | 79  O |
     | 80  P | 81  Q | 82  R | 83  S | 84  T | 85  U | 86  V | 87  W |
     | 88  X | 89  Y | 90  Z | 91  [ | 92  \ | 93  ] | 94  ^ | 95  _ |
     | 96  ` | 97  a | 98  b | 99  c |100  d |101  e |102  f |103  g |
     |104  h |105  i |106  j |107  k |108  l |109  m |110  n |111  o |
     |112  p |113  q |114  r |115  s |116  t |117  u |118  v |119  w |
     |120  x |121  y |122  z |123  { |124  | |125  } |126  ~ |127 DEL|

*/

typedef struct _Config 
{
  char		modelName[MaxNameLength];
  char		serialPort[15];
  int		baudRate;
  int		eolChar;
  int		charSize;
  enum Parity	parity;
  enum Boolean	igParError;
  enum Boolean	flowControl;
  int		maxSpeed;
  int		fd;
} Config;

  
/* COMPOSITION OF A VIDEO OBJECT
 * Data structures
 * 1.  Config configuration - stores the device configurations (described above).
 *
 * Functions
 * 1.  int (*DevPlayPtr) (VideoObject* theObject) 
 *     Pointer to function that puts device in 'play' mode. Returns status code.
 * 2.  int (*DevPlayFromTo) (VideoObject* theObject, int startAddress, int endAddress, 
 *                           int speedInFramesPerSecond)
 *     Pointer to function that plays a segment. This function is used in 4 different ways.
 *     (i)   startAddress == endAddress 
 *           Searches to 'startAddress' and stills.
 *     (ii)  startAddress != NULL, endAddress == NULL
 *           Searches to 'startAddress', and continues playing at 'speedInFramesPerSecond'.
 *     (iii) startAddress == NULL, endAddress != NULL
 *          Play from current position (no search) until 'endAddress' at 'speedInFramesPerSecond'.
 *     (iv)  startAddress != endAddress, and startAddress != NULL, and endAddress != NULL
 *           Play from 'startAddress' to 'endAddress' at 'speedInFramesPerSecond'.
 *     'startAddress' and 'endAddress' are in terms of frames if the player is in frame mode, 
 *     and in terms of chapters if the player is in chapter mode.
 *     For devices that have indices rather than chapters, and have relative search of indices,
 *     'endAddress' is used to indicate direction. As usual, 0 is Forward, 1 is Reverse.
 *     Returns status code.
 * 3.  int (*DevFastForwardPtr) (VideoObject* theObject)
 *     Pointer to function that plays forward at higher than normal (device-dependent) speed.
 *     It may continue to play at this speed until some other command is issued, or it may
 *     return to normal playback after a certain number of frames (device-dependent).
 *     Returns status code.
 * 4.  int (*DevReversePtr) (VideoObject* theObject)
 *     Pointer to function that plays reverse at higher than normal (device-dependent) speed.
 *     It may continue to play at this speed until some other command is issued, or it may
 *     return to normal playback after a certain number of frames (device-dependent).
 *     Returns status code.
 * 5.  int (*DevCalcSpeedPtr) (VideoObject* theObject, int inputValue, int playMode)
 *     Pointer to function that returns speed (in number of frames per second) that can be 
 *     played by the device, given 'inputValue'.  'playMode' is 0 for normal playback and 1 for
 *     segment play. Returns status code.
 * 6.  int (*DevPlayAtSpeedDirPtr) (VideoObject* theObject, int speedInFramesPerSecond, 
 *                                  enum Direction direction)
 *     Pointer to function that plays at 'speedInFramesPerSecond' in 'direction'. 
 *     'speedInFramesPerSecond' can either be converted to a speed the device can play, or 
 *     can be any arbitrary positive speed.
 *     Returns status code.
 * 7.  int (*DevStepPtr) (VideoObject* theObject, enum Direction direction)
 *     Pointer to function that steps one frame in 'direction.
 *     Returns status code.
 * 8.  int (*DevStillPtr) (VideoObject* theObject)
 *     Pointer to function that puts player into 'still' mode, meaning frame is frozen,
 *     but video is not muted. This function should be able to execute in any mode, which
 *     means that if segment play is not interruptible by a normal pause command, an 
 *     interrupt command should be used to implement pause.
 *     Returns status code.
 * 9.  int (*DevStopPtr) (VideoObject* theObject)
 *     Pointer to function that puts player into 'stop' mode, meaning playback is stopped
 *     and video is muted. It does NOT shut down the disc motor. This function should be able to
 *     execute in any mode, which means that if segment play is not interruptible by a normal
 *     stop command, an interrupt command should be used to implement stop.
 *     Returns status code.
 * 10. int (*DevSetDefaultsPtr) (VideoObject* theObject)
 *     Pointer to function that puts player into desired mode when starting it up. Examples
 *     of features that are set can be : audio, frame display, etc.
 *     Returns status code.
 * 11. int (*DevSetAudioPtr) (VideoObject* theObject, int mode)
 *     Pointer to function that sets audio to 'mode' on player.
 *     Returns status code.
 * 12. int (*DevSetVideoPtr) (VideoObject* theObject, int mode)
 *     Pointer to function that sets video to 'mode' on player.
 *     Returns status code.
 * 13. int (*DevSetAddModePtr) (VideoObject* theObject, int mode)
 *     Pointer to function that sets addressing mode to 'mode' on player.
 *     Returns status code.
 * 14. int (*DevSetAddressDisplayPtr) (VideoObject* theObject, int onOff, int mode)
 *     Pointer to function that either turns on-screen address display to 'on' or 'off' 
 *     depending on value of 'onOff' and sets the display to 'mode'.
 *     Returns status code.
 * 15. int (*DevEjectPtr) (VideoObject* theObject)
 *     Pointer to function that ejects the medium from the player.
 *     Returns status code.
 * 16. int (*DevPowerPtr) (VideoObject* theObject, int mode)
 *     Pointer to function that turns power either on or off, depending on 'mode'.
 *     Returns status code.
 * 17. int (*DevQueryFrame) (VideoObject* theObject)
 *     Pointer to function that asks the player for the current address of the medium. 
 *     Returns the address in terms of frames.
 * 18. int (*DevQueryChapter) (VideoObject* theObject)
 *     Pointer to function that asks the player for the current chapter of the medium, and
 * 19. int (*DevQueryAudio) (VideoObject* theObject)
 *     Pointer to function that asks the player for status of audio output. Returns
 *     audio status.
 * 20. int (*DevQueryVideo) (VideoObject* theObject)
 *     Pointer to function that asks the player for status of video output. Returns
 *     video status.
 * 21. int (*DevQueryMedium) (VideoObject* theObject, char* result)
 *     Pointer to function that asks the player for medium ID. Returns the medium ID if integer,
 *     or if a string, returns in 'result'.
 * 22. int (*DevQueryStatus) (VideoObject* theObject)
 *     Pointer to function that asks the player for its status, eg play, pause, search, etc.
 *     Returns status.
 * 23. int (*DevPing) (VideoObject* theObject)
 *     Pointer to function that sends a command to the player that is guaranteed to elicit
 *     some response from the player. It returns the number of characters read.
 *     This function is used only during the installation of a driver, so that the application
 *     can check if the player is correctly selected. 
 * 24. int (*DevRecord) (VideoObject* theObject)
 *     Pointer to function that begins normal recording on the player.  In video
 *     terminology, this is known as "assemble edit".
 *     Returns status code.
 * 25. int (DevRecordFromTo) (VideoObject* theObject, int startAddress, int endAddress,
 *			      int speedInFramesPerSecond)
 *     Pointer to function that records a segment.  This function is currently used in 2
 *     different ways:
 *     (i)	startAddress == NULL, endAddress == NULL
 *		Puts player into "Insert Edit" mode, allowing precise time-code recording
 *		to occur.
 *     (ii)	startAddress == 0, endAddress != 0
 *		Records from current position (no search) until 'endAddress' is reached.
 *     Returns status code.
 *
 *     Notes on DevRecordFromTo:
 *	1)   Although "speedInFramesPerSecond" field is here, as with DevPlayFromTo(), it is
 *     	     unlikely that players will allow recording at other than 1x speed.  This field
 *	     is here to accommodate the possibility of future devices that can record at other
 *	     than normal speed.
 *	2)   Before use (ii) takes into effect, the player must first be put into
 *	     insert edit mode, by using (i).  In other words, the calling application must call
 *	     "DevPlayFromTo(theObject, NULL, NULL, 30)" to turn on insert edit mode, after which
 *	     the application may call "DevPlayFromTo(theObject, 0, endAddress, 30) to begin insert
 *	     edit recording.  This is done in two separate calls since some devices experience a
 *	     long setup time to enter insert edit mode.  The separate calls allow for better
 *	     synchronization between playback and record decks.
 *	3)   To record a single frame of video (e.g., frame 100), call DevRecordFromTo in this
 *	     manner: DevRecordFromTo(theObject, 100, 100, 30).
 */

typedef struct _VideoObject 
{
  Config* DevConfig;
  int (*DevPlayPtr) (struct _VideoObject*);
  int (*DevPlayFromToPtr) (struct _VideoObject*, int, int, int);
  int (*DevFastForwardPtr) (struct _VideoObject*);
  int (*DevReversePtr) (struct _VideoObject*);
  int (*DevCalcSpeedPtr) (struct _VideoObject*, int, int);
  int (*DevPlayAtSpeedDirPtr) (struct _VideoObject*, int, enum Direction);
  int (*DevStepPtr) (struct _VideoObject*, enum Direction);
  int (*DevStillPtr) (struct _VideoObject*);
  int (*DevStopPtr) (struct _VideoObject*);
  int (*DevSetDefaultsPtr) (struct _VideoObject*, int, int, int, int);
  int (*DevSetAudioPtr) (struct _VideoObject*, int);
  int (*DevSetVideoPtr) (struct _VideoObject*, int);
  int (*DevSetAddModePtr) (struct _VideoObject*, int);
  int (*DevSetAddressDisplayPtr) (struct _VideoObject*, int, int);
  int (*DevEjectPtr) (struct _VideoObject*);
  int (*DevPowerPtr) (struct _VideoObject*, int);
  int (*DevQueryFramePtr) (struct _VideoObject*);
  int (*DevQueryChapterPtr) (struct _VideoObject*);
  int (*DevQueryAudioPtr) (struct _VideoObject*);
  int (*DevQueryVideoPtr) (struct _VideoObject*);
  int (*DevQueryMediumPtr) (struct _VideoObject*, char*);
  int (*DevQueryStatusPtr) (struct _VideoObject*);
  int (*DevPingPtr) (struct _VideoObject*);
  int (*DevRecordPtr) (struct _VideoObject*);
  int (*DevRecordFromToPtr) (struct _VideoObject*, int, int, int);
} VideoObject;



/* DECLARATIONS */

char allDevices[MaxDevices][MaxNameLength];			    /* Array of names of all devices supported */
int numDevices;							    /* Total number of devices supported so far */


int GetDeviceMaxSpeed(VideoObject*);
enum Boolean CheckImplemented(VideoObject*, int);

/* Default functions for the video object */

int DefaultPlay			(VideoObject*);
int DefaultPlayFromTo		(VideoObject*, int, int, int);
int DefaultFastForward		(VideoObject*);
int DefaultReverse		(VideoObject*);
int DefaultCalcSpeed		(VideoObject*, int, int);
int DefaultPlayAtSpeedDir	(VideoObject*, int, enum Direction);
int DefaultStep			(VideoObject*, enum Direction);
int DefaultStill		(VideoObject*);
int DefaultStop			(VideoObject*);
int DefaultSetDefaults		(VideoObject*, int, int, int, int);
int DefaultSetAudio		(VideoObject*, int);
int DefaultSetVideo		(VideoObject*, int);
int DefaultSetAddMode		(VideoObject*, int);
int DefaultSetAddressDisplay	(VideoObject*, int, int);
int DefaultEject		(VideoObject*);
int DefaultPower		(VideoObject*, int);
int DefaultQueryFrame		(VideoObject*);
int DefaultQueryChapter		(VideoObject*);
int DefaultQueryAudio		(VideoObject*);
int DefaultQueryVideo		(VideoObject*);
int DefaultQueryMedium		(VideoObject*, char*);
int DefaultQueryStatus		(VideoObject*);
int DefaultPing			(VideoObject*);
int DefaultRecord		(VideoObject*);
int DefaultRecordFromTo		(VideoObject*, int, int, int);


/* Video functions called by applications */

int DevPlay			(VideoObject*);
int DevPlayFromTo		(VideoObject*, int, int, int);
int DevFastForward		(VideoObject*);
int DevReverse			(VideoObject*);
int DevCalcSpeed		(VideoObject*, int, int);
int DevPlayAtSpeedDir		(VideoObject*, int, enum Direction);
int DevStep			(VideoObject*, enum Direction);
int DevStill			(VideoObject*);
int DevStop			(VideoObject*);
int DevSetDevs			(VideoObject*, int, int, int, int);
int DevSetAudio			(VideoObject*, int);
int DevSetVideo			(VideoObject*, int);
int DevSetAddMode		(VideoObject*, int);
int DevSetAddressDisplay	(VideoObject*, int, int);
int DevEject			(VideoObject*);
int DevPower			(VideoObject*, int);
int DevQueryFrame		(VideoObject*);
int DevQueryChapter		(VideoObject*);
int DevQueryAudio		(VideoObject*);
int DevQueryVideo		(VideoObject*);
int DevQueryMedium		(VideoObject*, char*);
int DevQueryStatus		(VideoObject*);
int DevPing			(VideoObject*);
int DevRecord			(VideoObject*);
int DevRecordFromTo		(VideoObject*, int, int, int);


int PlayerVerified(VideoObject*, char*);
int SetSerialLine(Config*, char*);
VideoObject* BuildVideoObject(VideoObject*);

#endif

/* Configurations of video objects thus supported - Reference */

/* Video Object for Pioneer4200 */
/*
static Config Pioneer4200Config = 
{
  "Pioneer4200",
  "na",
  4800,
  10,
  8,
  Even,
  Yes,
  Yes,
  120,
  -1
  }
;
*/

/* Video Object for Panasonic 2026/2027 */
/*
static Config PanasonicOptConfig = 
{
  "Panasonic2026",
  "/dev/ttyb",
  1200,
  10,
  8,
  Even,
  Yes,
  Yes,
  300, 
  -1
  }
;
*/

/* Video Object for Pioneer6000 */
/*
static Config Pioneer6000Config = 
{
  "Pioneer6000",
  "/dev/ttyb",
  1200,
  10,
  8,
  Even,
  Yes,
  Yes,
  90,
  -1
  }
;
*/

/* Video Object for Sony1550 */
/*
static Config Sony1550Config = 
{
  "Sony1550",
  "/dev/ttyb",
  2400,
  -1,
  8,
  Even,
  Yes,
  Yes,
  90,
  -1
  }
;
*/

/* Video Object for NEC PC VCR */
/*
static Config NECConfig = 
{
  "NEC",
  "na",
  1200,
  13,
  8,
  NoParity,
  No,
  No,
  270,
  -1
  }
;
*/


/* Video Object for CVD1000 */
/*
static Config CVD1000Config = 
{
  "CVD1000",
  "/dev/ttya",
  9600,
  -1,
  8,
  NoParity,
  No,
  No,
  60,
  -1
  }
;
*/
