/*
 * Copyright (c) 1990, 1991, 1992 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/VideoObject/RCS/PanasonicOptDriver.c,v 0.26 92/09/28 12:39:41 drapeau Exp $ */
/* $Log:	PanasonicOptDriver.c,v $
 * Revision 0.26  92/09/28  12:39:41  drapeau
 * PlayFromTo() was modified to reflect the new semantics of the VideoObject
 * library.
 * 
 * Revision 0.25  92/09/12  14:06:24  drapeau
 * Removed the function PanasonicOptReadResponse() since it was not being used.
 * Because of this, also removed references to XView toolkit header files.
 * 
 * Revision 0.24  92/07/30  15:21:58  drapeau
 * Several changes:
 * * Re-formatted all function declarations to conform to ANSI function
 *   prototype standards.
 * * Replaced hard-coded references to 30 frame-per-second frame rates with
 *   new definition "FrameRate".  All math is based on this definition now,
 *   allowing the driver to be used in places where the frame rate is not
 *   30 fps.
 * * Improved diagnostic messages.  Diagnostics now report the serial port
 *   being used for the command when possible.
 * 
 * Revision 0.23  92/06/16  23:40:57  drapeau
 * Changed the way asynchronous calls are handled, to remove XView-dependent
 * code from the driver.  This might have to be changed yet again, when the
 * new toolkit-independent asynchronous reading scheme is implemented for
 * all appropriate drivers.
 * 
 * Revision 0.22  92/01/03  16:51:17  drapeau
 * Modified PanasonicOptReadResponse() to return the more proper
 * XView return value "NOTIFY_DONE" rather than NULL.
 * 
 * Revision 0.21  91/09/30  17:05:55  lim
 * Implemented PanasonicOptPing.
 * 
 * Revision 0.20  91/08/24  17:49:54  lim
 * Implemented PrintDiagnostics.
 * 
 * Revision 0.19  91/08/24  13:37:59  lim
 * 1. Updated to use status codes in new PlayerStatus.h
 * 2. Clear Marker() removed as part of video object.
 * 
 * Revision 0.18  91/08/17  20:38:37  lim
 * 1. Removed 'stopped'. Video mute is turned off whenever a command
 * requiring video output is issued.
 * 2. Pause After Search is now implemented.
 * 3. Scan has been fixed to scan forward 250 frames at each call.
 * 4. Stop and Pause implemented with interrupt capability.
 * 5. Clear Marker implemented to do as specs in videoObj.h require.
 * 
 * Revision 0.17  91/08/15  12:52:05  lim
 * Changed interrupt command to use "AC" which clears buffer and puts player
 * in still playback mode.
 * 
 * Revision 0.16  91/08/13  22:57:18  lim
 * Changed PlayAtSpeedDir to perform playback when speed is 30 frames/sec and
 * direction is forward
 * 
 * Revision 0.15  91/08/08  17:13:43  lim
 * 1. PlayAtSpeedDir() - try to input device-dependent speed rather than an arbitrary speed.
 *    ie use CalcSpeed before calling PlayAtSpeedDir()
 * 2. CalcSpeed() - one parameter added, 'playMode'. If playMode is 1, it is used to calculate
 *    device speed for segment play. If playMode is 0, it is used to calculate device speed for
 *    normal play.
 * 3. QueryMedium() - one parameter added, 'result'. Used to return non-integer results.
 * 
 * Revision 0.14  91/08/07  13:01:37  lim
 * 1. Made as part of VideoLib.a. Removed references to vEdit application.
 * 2. Added instance pointer, "theObject" to all public functions. 
 * 3. ErrorDecode() is now a private function. It calls DisplayError() 
 * (in 'videoObjects.c') to display the error message. 
 * 4. notify_set_input_func uses a PanasonicClient as first argument. The
 * value is set in 'PanasonicOptDriver.h'.
 * 
 * Revision 0.13  91/08/02  12:48:17  lim
 * 1. SetDefault function changed to take in 4 parameters:
 * audio, addressingMode, addressDisplayOnOff and DisplayMode
 * 2. xv_set is no longer called within driver code. Called
 *    within vEdit instead.
 * 3. Pause is removed and replaced by Still.
 * 4. SetAddressDisplay now has 2 parameters, so as to be consistent
 * with the default function.
 * 
 * Revision 0.12  91/07/27  22:26:23  lim
 * Changed all code regarding speed settings to deal with int rather than double.
 * Rounding macro, "MyRound" has been changed.
 * 
 * Revision 0.11  91/07/24  11:00:39  lim
 * Ran through xsaber, and fixed warnings.
 * 
 * Revision 0.10  91/07/23  16:13:09  lim
 * Initial revision.
 *  */

#include "PanasonicOptDriver.h"

static char panasonicRcsid[] = "$Header: /Source/Media/collab/VideoObject/RCS/PanasonicOptDriver.c,v 0.26 92/09/28 12:39:41 drapeau Exp $";

static int	endFrame = 0;					    /* marks the end of segment to be played */
int		waitConsumeResponse = 0;			    /* Set to 1 if we do not wait to consume end response for the
								     command immediately after issuing it */
static char	diagMsg[64];


int 
  PanasonicOptSendCode(VideoObject*	theObject,
		       char*		code,
		       int		param1,			    /* Negative => no parameters */
		       int		param2,			    /* Negative => only one param */
		       int		respMode)		    /* 0 means read execution end, 1 means return so ...
								       ... that the app is not waiting for command to ...
								       ... finish execution */
{
  int	charRead1;
  int	charRead2;
  int	error;
  int	data;
  char	command[20];
  char	response1[20];
  char	response2[20];
  char	errorMsg[50];
  
  command[0] = Stx;
  
  if ((param1 >= 0) && (param2 >= 0))				    /* 2 params */
    sprintf(&command[1], "%s%d:%d:", code, param1, param2);
  else if (param1 >= 0)						    /* 1 param */
    sprintf(&command[1], "%s%d:", code, param1);
  else								    /* 0 params */
    sprintf(&command[1], "%s", code);
  
  sprintf(diagMsg, "%s :\tCommand's length = %d.\n",
	  theObject->DevConfig->serialPort,
	  strlen(command));
  PrintDiagnostics(diagMsg);
  
  command[strlen(command)] = Etx;
  command[strlen(command) + 1] = '\r';
  
  sprintf(diagMsg, "%s :\tSending %s.\n",
	  theObject->DevConfig->serialPort,
	  &command[1]);
  PrintDiagnostics(diagMsg);
  write(theObject->DevConfig->fd, command, strlen(command));
  
  if (waitConsumeResponse)					    /* If response from previous command is not consumed... */
  {								    /* ... consume it here first */
    if (strcmp(command, "AC") != 0)
    {
      sprintf(diagMsg, "%s :\tConsumed asynch end response.\n",
	      theObject->DevConfig->serialPort);
      PrintDiagnostics(diagMsg);
      charRead1 = read(theObject->DevConfig->fd, response1, 20);
    }
    waitConsumeResponse = 0;
  }
  
  charRead1 = read(theObject->DevConfig->fd, response1, 20);
  response1[charRead1 - 2] = '\0';
  
  if (response1[0] == Ack)					    /* ACK is received - 1st response*/
  {
    sprintf(diagMsg, "%s :\tRead an ACK.\n",
	    theObject->DevConfig->serialPort);
    PrintDiagnostics(diagMsg);
    if (respMode == 1)
      return PlayerOk;
    else
    {
      charRead2 = read(theObject->DevConfig->fd, response2, 20);    /* Read 2nd response */
      response2[charRead2 - 1] = '\0';
      
      if (response2[1] == 'E')					    /* Error in execution */
      {
	sscanf(&response2[2], "%d", &error);
	sprintf(diagMsg, "%s :\t2nd read - error: %d.\n",
		theObject->DevConfig->serialPort,
		error);
	PrintDiagnostics(diagMsg);
	PanasonicOptErrorDecode(theObject, -error, errorMsg);
	return(-error);
      }
      else if (response2[1] == 'N')				    /* Returning frame address */
      {
	sscanf(&response2[3], "%d", &data);
	sprintf(diagMsg, "%s :\t2nd read - address: %d.\n",
		theObject->DevConfig->serialPort,
		data);
	PrintDiagnostics(diagMsg);
	return (data);
      }
      else if (response2[1] == 'Z')				    /* Returning player status */
      {
	sprintf(diagMsg, "%s :\t2nd read - status: %s.\n",
		theObject->DevConfig->serialPort,
		&response2[2]);
	PrintDiagnostics(diagMsg);
	if (strncmp(&response2[2], "PF", 2) == 0)
	  return PlayerForwardPlay;
	else if (strncmp(&response2[2], "PR", 2) == 0)
	  return PlayerReversePlay;
	else if (strncmp(&response2[2], "TF", 2) == 0)
	  return PlayerForwardStep;
	else if (strncmp(&response2[2], "TR", 2) == 0)
	  return PlayerReverseStep;
	else if (strncmp(&response2[2], "LF", 2) == 0)
	  return PlayerForwardSlow;
	else if (strncmp(&response2[2], "LR", 2) == 0)
	  return PlayerReverseSlow;
	else if (strncmp(&response2[2], "FF", 2) == 0)
	  return PlayerForwardFast;
	else if (strncmp(&response2[2], "FR", 2) == 0)
	  return PlayerReverseFast;
	else if (strncmp(&response2[2], "RA", 2) == 0)
	  return PlayerRecording;
	else if (strncmp(&response2[2], "RM", 2) == 0)
	  return PlayerRecording;
	else if (strncmp(&response2[2], "GS", 2) == 0)
	  return PlayerRecording;
	else if (strncmp(&response2[2], "RE", 2) == 0)
	  return PlayerRecording;
	else if (strncmp(&response2[2], "AE", 2) == 0)
	  return PlayerRecording;
	else if (strncmp(&response2[2], "SR", 2) == 0)
	  return PlayerSearch;
	else if (strncmp(&response2[2], "LD", 2) == 0)
	  return PlayerLoad;
	else if (strncmp(&response2[2], "EJ", 2) == 0)
	  return PlayerEject;
	else if (strncmp(&response2[2], "HT", 2) == 0)
	  return PlayerStop;
	else if (strncmp(&response2[2], "WT", 2) == 0)
	  return PlayerWait;
	else if (strncmp(&response2[2], "SD", 2) == 0)
	  return PlayerSystemDown;
      }
      else if (response2[0] == Stx)				    /* Normal execution end response */
      {
	sprintf(diagMsg, "%s :\t2nd read - end execution %s.\n",
		theObject->DevConfig->serialPort,
		&response2[1]);
	PrintDiagnostics(diagMsg);
	return PlayerOk;
      }
    }
  }								   /* end of ACK 2nd response handling */
  else
  {
    if (response1[0] == Nak)					    /* NAK is received - 1st response */
    {
      sprintf(diagMsg, "%s :\tRead a NAK.\n",
	      theObject->DevConfig->serialPort);
      PrintDiagnostics(diagMsg);
      sscanf(&response1[1], "%d", &error);
      PanasonicOptErrorDecode(theObject, -error, errorMsg);
      return (-error);
    }
    else if ((response1[1] == 'E') && (response1[2] != 'J'))	    /* Error in instruction */
    {
      sscanf(&response1[2], "%d", &error);
      sprintf(diagMsg, "%s :\tRead error : %d.\n",
	      theObject->DevConfig->serialPort,
	      error);
      PrintDiagnostics(diagMsg);
      PanasonicOptErrorDecode(theObject, -error, errorMsg);
      return (-error);
    }
  }								    /* end of NAK 1st response */
}								    /* end function PanasonicOptSendCode */


int 
  PanasonicOptSetDefaults(VideoObject*	theObject,
			  int		audio,
			  int		addressingMode,		    /* Unused */
			  int		addressDisplayOnOff,
			  int		displayMode)		    /* Unused */
{
  PanasonicOptSendCode(theObject, "ON", 0, -1, 0);		    /* set player OnLine */
  PanasonicOptPlay(theObject);					    /* Put player in ready state */
  PanasonicOptStill(theObject);
  PanasonicOptSetAudio(theObject, audio);			    /* Set audio */
  PanasonicOptSetAddressDisplay(theObject, 
				addressDisplayOnOff, displayMode);  /* Set address display */
}								    /* end function PanasonicOptSetDefaults */


int 
  PanasonicOptPlay(VideoObject* theObject)
{
  PanasonicOptSetVideo(theObject, 1);				    /* get rid of video mute */
  return(PanasonicOptSendCode(theObject, "PF", -1, -1, 0));
}								    /* end function PanasonicOptPlay */


int 
  PanasonicOptPlayFromTo(VideoObject*	theObject,
			 int		from,
			 int		to,
			 int		speed)
{
  int status;
  int deviceSpeed;
  
  PanasonicOptSetVideo(theObject, 1);				    /* get rid of video mute */
  if ((from == to) || ((from > 0) && (to == 0)))		    /* Case a & b: search to "from" and still (non-blocking... */
  {								    /* ...search doesn't apply here since search time is quick) */
    PanasonicOptSendCode(theObject, "SR", from,  -1, 0);	    /* Search to address "from" */
    if (status != PlayerOk)
      return (status);
    return (PanasonicOptSendCode(theObject, "AC", -1,  -1, 0));	    /* Stills */
  }    
  if ((from == 0) && (to > 0))					    /* Case c: play from current position until address... */
  {								    /* ..."to" has been reached. */
    endFrame = to;
    deviceSpeed = PanasonicOptCalcSpeed(theObject, speed, 0);
    status = PanasonicOptPlayAtSpeedDir(theObject, deviceSpeed,	    /* This function examines "endFrame" to determine when... */
					Forward);		    /* ...to stop playback. */
    endFrame = 0;						    /* reset endFrame back to 0 */
    return (status);
  }
  if ((from > 0) && (to > 0) && (from < to))			    /* Case d: play from "from" to "to" */
  {
    PanasonicOptSendCode(theObject, "SR", from,  -1, 0);	    /* Search to address "from" */
    if (status != PlayerOk)
      return (status);
    PanasonicOptSendCode(theObject, "AC", -1,  -1, 0);		    /* Stills */
    endFrame = to;
    deviceSpeed = PanasonicOptCalcSpeed(theObject, speed, 0);
    status = PanasonicOptPlayAtSpeedDir(theObject, deviceSpeed,	    /* This function examines "endFrame" to determine when... */
					Forward);		    /* ...to stop playback. */
    endFrame = 0;						    /* reset endFrame back to 0 */
    return (status);
  }
  return(PlayerReturnError);					    /* If this code is reached, incorrect from & to were requested */
}								    /* end function PanasonicOptPlayFromTo */


int 
  PanasonicOptFastForward(VideoObject* theObject)
{
  return(PanasonicOptSendCode(theObject, "CF", 0, -1, 0));
}								    /* end function PanasonicOptFastForward */


int 
  PanasonicOptReverse(VideoObject* theObject)
{ 
  return(PanasonicOptSendCode(theObject, "CR", 0, -1, 0));
}								    /* end function PanasonicOptReverse */


/* Returns number of frames per second (device-capable) */
int
  PanasonicOptCalcSpeed(VideoObject*	theObject,
			int		sliderValue,
			int		playMode)		    /* unused */
{
  int denom;
  int deviceSpeed;						    /* in frames per second */
  
  if (sliderValue == 0)
    return (0);
  
  if (MyAbs(sliderValue) < FrameRate)
  {
    denom = MyRound(FrameRate, sliderValue);			    /* Bring to the nearest rounded denom */
    deviceSpeed = (FrameRate / denom);
  }
  else
  {
    deviceSpeed = MyRound(sliderValue, FrameRate) * FrameRate;	    /* Bring to nearest whole number */
    if (deviceSpeed > (10 * FrameRate))
      deviceSpeed = 10 * FrameRate;
  }
  sprintf(diagMsg, "%s :\tDevice speed : %d.\n",
	  theObject->DevConfig->serialPort,
	  deviceSpeed);
  PrintDiagnostics(diagMsg);
  return (deviceSpeed);
}								    /* end function PanasonicOptCalcSpeed */


int 
  PanasonicOptPlayAtSpeedDir(VideoObject*	theObject,
			     int		inputSpeed,
			     enum Direction	dir)
{
  int		denom;
  int		speedTimes;
  int		sendStatus;
  static int	lastSpeed = 0;
  
  if (inputSpeed == lastSpeed)					    /* Was a new speed requested? */
    return(PlayerOk);						    /* No, don't bother sending a new command to the player */
  
  lastSpeed = inputSpeed;					    /* Update last speed requested */
  PanasonicOptSetVideo(theObject, 1);				    /* get rid of video mute */
  
  if ((inputSpeed == FrameRate) && (dir == Forward))
    if (endFrame)
      sendStatus = PanasonicOptSendCode(theObject, "PF", endFrame, -1, 1);
    else
      sendStatus = PanasonicOptSendCode(theObject, "PF", -1, -1, 0);
  
  if (inputSpeed < FrameRate)
  {
    denom = MyRound(FrameRate, inputSpeed);
    if (dir == Forward)
    {
      if (endFrame)
	sendStatus = PanasonicOptSendCode(theObject, "LF", denom, endFrame, 1);
      else
	sendStatus = PanasonicOptSendCode(theObject, "LF", denom, -1, 0);
    }
    else
    {
      if (endFrame)
	sendStatus = PanasonicOptSendCode(theObject, "LR", denom, endFrame, 1);
      else
	sendStatus = PanasonicOptSendCode(theObject, "LR", denom, -1, 0);
    }
  }
  else
  {
    speedTimes = MyRound(inputSpeed, FrameRate);
    if (speedTimes > 10)
      speedTimes = 10;
    if (dir == Forward)
    {
      if (endFrame)
	sendStatus = PanasonicOptSendCode(theObject, "FF", speedTimes, 
					  endFrame, 1);
      else
	sendStatus = PanasonicOptSendCode(theObject, "FF", speedTimes, -1, 0);
    }
    else
    {
      if (endFrame)
	sendStatus = PanasonicOptSendCode(theObject, "FR", speedTimes, 
					  endFrame, 1);
      else
	sendStatus = PanasonicOptSendCode(theObject, "FR", speedTimes, 
					  -1, 0);
    }
  }  
  
  if ((sendStatus >= 0) && (endFrame))				    /* No errors in sending */
    waitConsumeResponse = 1;					    /* Set flag because end response is not consumed here */
  return sendStatus;
}								    /* end function PanasonicOptPlayAtSpeedDir */



int 
  PanasonicOptStep(VideoObject*		theObject,
		   enum Direction	dir)
{
  PanasonicOptSetVideo(theObject, 1);				    /* get rid of video mute */
  if (dir == Forward)
    return (PanasonicOptSendCode(theObject, "TF", -1, -1, 0));
  else
    return (PanasonicOptSendCode(theObject, "TR", -1, -1, 0));
}								    /* end function PanasonicOptStep */


int 
  PanasonicOptStill(VideoObject* theObject)
{
  PanasonicOptSetVideo(theObject, 1);				    /* get rid of video mute */
  return(PanasonicOptSendCode(theObject, "AC", -1, -1, 0));	    /* Interrupts play and stills */
  
}								    /* end function PanasonicOptStill */


int 
  PanasonicOptStop(VideoObject*	theObject)
{
  PanasonicOptSendCode(theObject, "AC", -1, -1, 0);		    /* Interrupts play and stills */
  return(PanasonicOptSetVideo(theObject, 0));			    /* Set video off */
}								    /* end function PanasonicOptStop */


int 
  PanasonicOptSetAudio(VideoObject*	theObject,
		       int		mode)
{
  switch (mode)
  {
   case PlayerAudioMute:
    PanasonicOptSendCode(theObject, "A1", 0, -1, 0);
    return(PanasonicOptSendCode(theObject, "A2", 0, -1, 0));
   case PlayerAudioLeft:
    PanasonicOptSendCode(theObject, "A1", 34, -1, 0);
    return(PanasonicOptSendCode(theObject, "A2", 0, -1, 0));
   case PlayerAudioRight:
    PanasonicOptSendCode(theObject, "A2", 34, -1, 0);
    return(PanasonicOptSendCode(theObject, "A1", 0, -1, 0));
   case PlayerAudioStereo:
    PanasonicOptSendCode(theObject, "A1", 34, -1, 0);
    return(PanasonicOptSendCode(theObject, "A2", 34, -1, 0));
  }
}								    /* end function PanasonicOptSetAudio */


int 
  PanasonicOptSetVideo(VideoObject*	theObject,
		       int		mode)			    /* 0 for mute, 1 for on */
{
  if (mode)
    return(PanasonicOptSendCode(theObject, "VS", -1, -1, 0));
  else
    return(PanasonicOptSendCode(theObject, "VR", -1, -1, 0));
}								    /* end function PanasonicOptSetVideo */


int 
  PanasonicOptSetAddressDisplay(VideoObject*	theObject,
				int		onOff,		    /* 0 for off, 1 for on */
				int		mode)		    /* Unused */
{
  if (onOff)
    return(PanasonicOptSendCode(theObject, "DS", -1, -1, 0));
  else
    return(PanasonicOptSendCode(theObject, "DR", -1, -1, 0));
  
}								    /* end function PanasonicOptSetAddressDisplay */



int 
  PanasonicOptEject(VideoObject* theObject)
{
  return (PanasonicOptSendCode(theObject, "EJ", -1, -1, 0));
}								    /* end function PanasonicOptEject */


int 
  PanasonicOptQueryFrame(VideoObject* theObject)
{
  return(PanasonicOptSendCode(theObject, "NO", -1, -1, 0));
}								    /* end function PanasonicOptQueryFrame */


int								    /* right now, don't know if can be used */
  PanasonicOptQueryMedium(VideoObject*	theObject,
			  char*		result)
{
  return;
}								    /* end function PanasonicOptQueryMedium */


int 
  PanasonicOptQueryStatus(VideoObject* theObject)
{
  return(PanasonicOptSendCode(theObject, "PS", -1, -1, 0));
}								    /* end function PanasonicOptQueryStatus */


int 
  PanasonicOptPing(VideoObject* theObject)
{
  int	charRead;
  char	command[5];
  char	response[7];
  
  command[0] = Stx;
  strcpy(&command[1], "PS");					    /* Player status command */
  command[3] = Etx;
  command[4] = '\r';
  sprintf(diagMsg, "%s :\tSent ping.\n",
	  theObject->DevConfig->serialPort);
  PrintDiagnostics(diagMsg);
  
  write(theObject->DevConfig->fd, command, 5);
  
  charRead = read(theObject->DevConfig->fd, response, 1);
  if (charRead)
  {
    sprintf(diagMsg, "%s :\tSuccessful ping.\n",
	    theObject->DevConfig->serialPort);
    PrintDiagnostics(diagMsg);
  }
  else
  {
    sprintf(diagMsg, "%s :\tPing unsuccessful.\n",
	    theObject->DevConfig->serialPort);
    PrintDiagnostics(diagMsg);
  }  
  return charRead;
}								    /* end function PanasonicOptPing */



void 
  PanasonicOptErrorDecode(VideoObject*	theObject,
			  int		errorCode,
			  char*		errorMsg)
{
  switch (errorCode)
  {
   case InvalidInstruction:
    strcpy(errorMsg, "Invalid instruction");
    break;
   case DiskCheck:
    strcpy(errorMsg, "Please check disc");
    break;
   case NoDisk:
    strcpy(errorMsg, "There is no disc loaded");
    break;
   case CantFindAddress:
    strcpy(errorMsg, "Can't find address");
    break;
   case OutOfFocus:
    strcpy(errorMsg, "Focus is off");
    break;
   case DiscMotorSyncNG:
    strcpy(errorMsg, "Disk motor out of sync/speed low");
    break;
   case OptHeadOffEnd:
    strcpy(errorMsg, "Optical head is off start/end of disc");
    break;
   case RecordError:
    strcpy(errorMsg, "Attempting to set recording area from recorded track");
    break;
   case ExtVideoError:
    strcpy(errorMsg, "Video signal not correct/ not input");
    break;
   case LoadError:
    strcpy(errorMsg, "Load error. Perform inverting operation");
    break;
   case Moisture:
    strcpy(errorMsg, "Moisture : Optical system is dewed");
    break;
   case RcvBuffOverflow:
    strcpy(errorMsg, "Receiver buffer overflow");
    break;
   case CommError:
    strcpy(errorMsg, "Communication error");
    break;
   case InvalidCmdFormat:
    strcpy(errorMsg, "Inaccurate format of command");
    break;
   case ConcentricDisc:
    strcpy(errorMsg, "Cannot proceed with concentric disk");
    break;
   case Battery:
    strcpy(errorMsg, "Stored program cannot be protected.");
    break;
   case NestingErr:
    strcpy(errorMsg, "Subroutine depth > level 15");
    break;
   case NoProgram:
    strcpy(errorMsg, "Program not loaded/stored program cannot be protected");
    break;
   case ProgramExceedMem:
    strcpy(errorMsg, "Program exceeds memory");
    break;
   case NoiseInUnit:
    strcpy(errorMsg, "Noise detected by unit");
    break;
   default:
    strcpy(errorMsg, "Unknown error code");
    break;
  }
  DisplayError(errorMsg, " ");
  return;
}								    /* end function PanasonicOptErrorDecode */
