/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/VideoEdit/RCS/VideoEditProtocol.c,v 0.37 92/05/29 12:38:57 drapeau Exp $ */
/* $Log:	VideoEditProtocol.c,v $
 * Revision 0.37  92/05/29  12:38:57  drapeau
 * Modified code to track new name of the MAEstro "Selection" structure;
 * it is now named "MAESelection".
 * 
 * Revision 0.36  92/05/13  00:22:10  drapeau
 * Minor changes:
 * * Minor formatting changes to conform to coding standards.
 * * Changed case of the document "Untitled".
 * 
 * Revision 0.35  92/01/07  14:11:35  drapeau
 * Minor cosmetic change to make code easier to read.
 * 
 * Revision 0.34  92/01/03  15:51:16  drapeau
 * Changed occurrances of "NULL" in calls to Browse() to use "0" instead.
 * This is due to the ANSI definition of NULL as "(void*)0".
 * 
 * Revision 0.33  91/11/14  10:14:41  collab
 * Modified GetAppIcon() to reflect new location of the application icon, in
 * the "icons/" directory.
 * 
 * Revision 0.32  91/09/26  12:58:20  lim
 * *** empty log message ***
 * 
 * Revision 0.31  91/09/26  12:29:09  lim
 * *** empty log message ***
 * 
 * Revision 0.30  91/09/26  11:59:32  lim
 * Returns from protocol functions immediately if there
 * is no initialized driver.
 * 
 * Revision 0.29  91/09/24  21:37:07  lim
 * Changed 'vEdit' to 'VideoEdit' everywhere.
 * 
 * Revision 0.28  91/08/30  15:44:24  lim
 * Don't send error message if sender is NULL.
 * Create receiver only when sender not null.
 * 
 * Revision 0.27  91/08/24  17:55:45  lim
 * Implemented PrintDiagnostics. 
 * Using command line flag '-d', the user can choose to launch the
 * app with diagnostic messages. Otherwise, no diag messages will
 * be printed.
 * 
 * Revision 0.26  91/08/23  17:16:43  lim
 * Document format has been changed.
 * 
 * Revision 0.25  91/08/21  14:31:32  lim
 * Implemented GetAppIcon().
 * 
 * Revision 0.24  91/08/20  16:31:15  lim
 * Now SetSelection blanks out screen and the screen wil
 * show only on Perform selection.
 * 
 * Revision 0.23  91/08/20  16:30:33  lim
 * Parameters for SetVideo : 0 - Video mute
 * 		             1 - Video on
 * 
 * Revision 0.22  91/08/20  15:46:46  lim
 *  Get selection now returns a duration of -1 if no edit has been selected.
 * 
 * 
 * Revision 0.21  91/08/20  10:07:53  lim
 * HaltSelection and PauseSelection now call DevStop() and DevStill()
 * directly.
 * 
 * Revision 0.20  91/08/17  20:52:11  lim
 * 1. All stops and pauses are capable of interrupt.
 * 2. OpenPanel is now Browse.
 * 
 * Revision 0.19  91/08/16  13:12:23  lim
 * 1. No more editFilename. Filename now put in title bar. 
 * 2. Call Browse() with OPENPANEL_CHECK for open doc.
 * 3. Functions check for validity before proceeding.
 * 
 * Revision 0.18  91/08/08  17:19:10  lim
 * 1. CalcSpeed() - one parameter added, 'playMode'. If playMode is 1, it is used to calculate
 *    device speed for segment play. If playMode is 0, it is used to calculate device speed for
 *    normal play.
 * 
 * Revision 0.17  91/08/07  13:29:36  lim
 * 1.Added instance pointer, "theObject" to all public function calls.
 * 2. Added 'Round' (same as 'MyRound')
 * 
 * Revision 0.16  91/07/29  22:31:44  lim
 * Updated status codes to conform with specs.
 * 
 * Revision 0.15  91/07/27  22:29:51  lim
 * Changed speed from double to int.
 * 
 * Revision 0.14  91/07/23  15:57:32  lim
 * 1. Offset corrected to take into account the speed at which the
 * segment will be played.
 * 2. Fixed duration calculation.
 * 3. For PauseSelection and HaltSelection, we check if the device
 * has interruptible segment play. If so, then the regular Pause/Halt 
 * operation is performed. Otherwise, we have to interrupt the player.
 * In such devices, ClearMarker will not exist (this is what allows
 * devices to be interruptible, by setting a stop marker and returning)
 * and so we use ClearMarker to perform the interrupt action.
 * 
 * Revision 0.13  91/07/20  10:06:58  lim
 * SetSelection now places new start (according to offset) on the
 * textfield "editStartTxt" rather than what is in startframe[editnum].
 * 
 * Revision 0.12  91/07/19  14:15:27  lim
 * Added audio and speed settings.
 * 
 * Revision 0.11  91/07/12  17:29:21  lim
 * Set icon for TimeLine.
 * 
 * Revision 0.10  91/07/12  16:22:01  lim
 * Initial revision implementing VideoObject.
 *  */

#include "VideoEdit.h"
#define Round(n, d)          ((n%d)/d > 0.5? (n/d+1) : (n/d))

void OpenDoc();
void SetSelection();
void PerformSelection();
char** GetDoc();
MAESelection* GetSelection();
void HaltSelection();
void PauseSelection();
void ResumeSelection();
void HideApplication();
void ShowApplication();
IconData* GetAppIcon();

static char* canonFilename;
static char lrcsid[] = "$Header: /Source/Media/collab/VideoEdit/RCS/VideoEditProtocol.c,v 0.37 92/05/29 12:38:57 drapeau Exp $";

Sender* sender;
Receiver* receiver;

void 
InitNetwork(h) 
     char* h;
{
  Port senderPort;
  char diagMsg[30];
  
  static DispatchTable  DT = 
  {
    OpenDoc,
    GetDoc,
    GetSelection,
    SetSelection,
    PerformSelection,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    HaltSelection,
    PauseSelection,
    ResumeSelection,
    HideApplication,
    ShowApplication,
    GetAppIcon
    };
  
  lines = 0;
  clearframe = 0;
  change = 0;
  search = 0;
  editnum = -1;
  
  /* Networking stuff */
  
  senderPort.hostName = h;
  senderPort.portNumber = PortMgrPortNumber;
  sender = NewSender(&senderPort);
  if (sender)							    /* Create receiver only if sender is alive */
  {
    receiver = NewReceiver(sender, "VideoEdit", receiverPort);
    sprintf(diagMsg, "Listening on port %d.\n", receiver->transport->xp_port);
    PrintDiagnostics(diagMsg);
  
    BuildDispatchTable (&DT);
  }
  
  canonFilename = (char*) malloc(MAX_PATHLENGTH);
  
  (void) notify_enable_rpc_svc (TRUE);
  
}



void
  OpenDoc(filename)
char** filename;
{
  int result;
  char fullFilename[MAX_PATHLENGTH];
  char* currentFile = (char*) xv_get(VideoEdit_editPopup->editPopup,
				     XV_LABEL);
  
  if (strcmp(*filename, "Untitled") != 0) 
  {								    /* check that filename is not 'untitled' */
    if (strlen(currentFile) == 0) strcpy (currentFile, ".");	    /* If no filename specified in panel, set to '.' (directory) */
    
    if (realpath(currentFile, fullFilename) != NULL)
      strcpy(fullFilename, (char*) realpath(currentFile, fullFilename));
    else 
      strcpy(fullFilename, ".");
    
    if (strcmp(fullFilename, *filename) != 0 &&			    /* check if filename given by link editor is the same... */
	strcmp(currentFile, *filename) != 0)			    /* ... as that on on the panel textfield */
    {								    
      result = NOTICE_NO;
      if ((change) && (lines > 0))				    /* check for changes in edit list */
      {
	result = notice_prompt(VideoEdit_editPopup->editPopup, NULL,
			       NOTICE_MESSAGE_STRINGS,
			       "Unsaved changes exist in the video edit list.",
			       "Loading a file will erase all changes.",
			       "Go ahead and load the file?",
			       NULL,
			       NOTICE_BUTTON_NO,       "Yes",
			       NOTICE_BUTTON_YES,      "No",
			       NULL);
      }
      if (result == NOTICE_NO) 
      {
	change = 0;
	Browse(*filename, BrowseCheckOpen, 0, "#Video Edit Document#", "VideoEdit");
      }
    }								    /* end of if (strcmp(fullFilename, *filename) != 0 && ...
								       ... strcmp(currentFile, *filename) != 0) */
  }								    /* end of if (strcmp(*filename, "untitled") != 0)  */
}


void
SetSelection(MAESelection* selection)
{
  int start;  
  char s[30];
  char audio[7];
  int speedInDevice;
  char diagMsg[30];
  
  if (myVideo == NULL)						    /* Do nothing if no device selected */
    return;
  
  if ((selection->start > 0) && (selection->start <= lines))
  {
    if (editnum != -1)						    /* Deselect any current selection */
      xv_set(VideoEdit_editPopup->editScrollList,
	     PANEL_LIST_SELECT, editnum, FALSE,
	     NULL);
    
    editnum = selection->start - 1;
    sprintf(s, "Current Selection : Edit #%d", editnum+1);
    speedInDevice = DevCalcSpeed(myVideo, speedStg[editnum], 1);
    sprintf(diagMsg, "SpeedInDevice = %d\n", speedInDevice);
    PrintDiagnostics(diagMsg);
    
    start = startframe[editnum] + 
      (selection->offset)/1000*speedInDevice;			    /* Add offset and get new start address */
   
    /* load the appropriate values into the textfields */
    xv_set(VideoEdit_editPopup->editNewEditMsg, PANEL_LABEL_STRING, s, NULL);
    xv_set(VideoEdit_editPopup->editStartTxt, PANEL_VALUE, start, NULL);
    xv_set(VideoEdit_editPopup->editEndTxt, PANEL_VALUE, endframe[editnum], NULL);
    xv_set(VideoEdit_editPopup->editLabelTxt, PANEL_VALUE, label[editnum], NULL);
    xv_set(VideoEdit_editPopup->editSpeedTxt, PANEL_VALUE, speedStg[editnum], NULL);
    switch (audioStg[editnum])
    {
     case Stereo:
      strcpy(audio, "Stereo");
      break;
     case Right:
      strcpy(audio, "Right");
      break;
     case Left:
      strcpy(audio, "Left");
      break;
     case Mute:
      strcpy(audio, "Mute");
      break;
    }
    xv_set(VideoEdit_editPopup->editAudioMenuButton, PANEL_LABEL_STRING, audio, NULL);
    SetDuration();
       
    PlayMode(PlayerFrameMode, start, start);			    /* Seek to correct position */
    DevSetVideo(myVideo, 0);					    /* Video muted */
    
    /* make the modify, delete buttons active */
    xv_set(VideoEdit_editPopup->editModButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set(VideoEdit_editPopup->editDelButton, PANEL_INACTIVE, FALSE, NULL);
    
    xv_set(VideoEdit_editPopup->editScrollList,
	   PANEL_LIST_SELECT, editnum, TRUE,
	   NULL);
    
    
  }								    /* end of if ((selection->start > 0) &&
								       (selection->start <= lines)) */
  
}								    /* end function SetSelection */

void
PerformSelection()
{
  int value;
  
  if ((editnum == -1) || (!lines) || (!myVideo))
    return;

  DevSetVideo(myVideo, 1);					    /* Video mute off */
  PlayMode(PlayerFrameMode, NULL, endframe[editnum]);		    /* Play to end with correct settings */
   
  /* check if search status is frame or chapter */
  value = xv_get(VideoEdit_optionsPopup->optionsSearchStg, 
		 PANEL_VALUE);
  if (value == 1) 
    DevSetAddMode(myVideo, PlayerChapterMode);
  
}

char**
GetDoc(unusedArg)
     void* unusedArg;
{
  static char* doc;
  
  doc = (char*) xv_get(VideoEdit_editPopup->editPopup, XV_LABEL);
  if ((strlen(doc) == 0) || (realpath(doc, canonFilename) == NULL))
    strcpy(canonFilename, "Untitled");
  else
    strcpy(canonFilename, (char*) realpath(doc, canonFilename));
  return (&canonFilename);
}

MAESelection*
GetSelection(unusedArg)
     void *unusedArg;
{
  static MAESelection select;
  int speedInDevice;
  
  if ((editnum == -1) || (!lines))	
  {
    select.duration = -1;			    
    return (&select);
  }
  
  select.start = editnum + 1;
  select.end = editnum + 1; 
  if (myVideo)
    speedInDevice = DevCalcSpeed(myVideo, speedStg[editnum], 1);    
  else
    speedInDevice = speedStg[editnum];
  
  if (speedInDevice)
    select.duration = (1000 * Round((endframe[editnum] - startframe[editnum]), speedInDevice));
  else
    select.duration = -1;

  select.offset = 0;
  if (strlen(label[editnum]))
    strcpy(select.label, label[editnum]);   
  else
    strcpy(select.label, "NO LABEL");
  
  return (&select);
}


/* Halts playback of Selection as directed by TimeLine */
void
HaltSelection()
{
  if ((editnum == -1) || (!lines) || (!myVideo))
    return;
  
  DevStop(myVideo);
}

/* Pauses playback of Selection as directed by TimeLine */
void
PauseSelection()
{
  if ((editnum == -1) || (!lines) || (!myVideo))
    return;
  
  DevStill(myVideo);
}

/* Resumes playback of Selection as directed by TimeLine */
void
ResumeSelection()
{
  if ((editnum == -1) || (!lines) || (!myVideo))
    return;
  paused = 0;
  
  DevPlayFromTo(myVideo, NULL, endframe[editnum], speedStg[editnum]);
  
}

/* Iconifies application as directed by TimeLine */
void
HideApplication()
{
  xv_set(VideoEdit_window1->window1, FRAME_CLOSED, TRUE, NULL);
  XFlush((Display*) xv_get(VideoEdit_window1->window1, XV_DISPLAY));
  
}

/* Deiconifies application as directed by TimeLine */
void
ShowApplication()
{
  xv_set(VideoEdit_window1->window1, FRAME_CLOSED, FALSE, NULL);
  xv_set(VideoEdit_window1->window1, XV_SHOW, TRUE, NULL);		   
  
  /* Show all popups that were up originally */

  if (xv_get(VideoEdit_optionsPopup->optionsPopup, XV_SHOW) == TRUE)		    
  {
    xv_set(VideoEdit_optionsPopup->optionsPopup, XV_SHOW, TRUE, NULL);
  }
  if (xv_get(VideoEdit_previewPopup->previewPopup, XV_SHOW) == TRUE) 
  {
    xv_set(VideoEdit_previewPopup->previewPopup, XV_SHOW, TRUE, NULL);
  }
  if (xv_get(VideoEdit_editPopup->editPopup, XV_SHOW) == TRUE) 
  {
    xv_set(VideoEdit_editPopup->editPopup, XV_SHOW, TRUE, NULL);
  }  
  if (xv_get(VideoEdit_infoPopup->infoPopup, XV_SHOW) == TRUE) 
  {
    xv_set(VideoEdit_infoPopup->infoPopup, XV_SHOW, TRUE, NULL);
  }

  XFlush((Display*) xv_get(VideoEdit_window1->window1, XV_DISPLAY));    
}

IconData *
GetAppIcon()
{
  static IconData returnVal;
  static unsigned short baseWindow_bits[] = {
#include "icons/VideoEditIcon"
        };

  returnVal.iconData = (char *)malloc(sizeof(baseWindow_bits));
  bcopy(baseWindow_bits, returnVal.iconData, sizeof(baseWindow_bits));
  if (returnVal.iconData)
    returnVal.dataLength = sizeof(baseWindow_bits);
  return(&returnVal);
}

