/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/TimeLine/RCS/stop.c,v 1.11 92/10/01 14:53:28 drapeau Exp Locker: drapeau $ */
/* $Log:	stop.c,v $
 * Revision 1.11  92/10/01  14:53:28  drapeau
 * One minor logic fix, in the PauseResume() function.  Before sending a
 * PauseSelection() message, TimeLine needs to assure that messaging is
 * synchronous so that it can be aware of which applications were able to
 * pause their current selections.  So before sending the PauseSelection
 * message, TimeLine now sets synchronous messaging to "On".
 * In addition, a number of minor cosmetic changes were made, and some slight
 * rearrangement of if-then tests was made to clean up the code and improve
 * readability.
 * 
 * Revision 1.10  92/09/24  17:29:03  drapeau
 * Minor code changes were made to support documents with synchronization
 * information.  When documents with "sync hints" are performed, TimeLine
 * uses asynchronous messaging.  However, when stopping performance of a
 * document, synchronous messaging must be used or TimeLine will "believe" that
 * an error occurred in sending Halt messages (the error is a side effect of the
 * way asynchronous messaging is done in Sun RPC).  To solution is to enable
 * synchronous messaging before sending Halt messages.  This is correct from
 * an interface standpoint as well, since the author will want to be sure that
 * all the components of a document have actually stopped when they were sent
 * Halt messages; asynchronous messaging will not guarantee this.
 * 
 * A number of cosmetic changes were made to improve readability and ANSI
 * compliance.
 * 
 * Revision 1.0  91/09/30  17:03:01  chua
 * Update to version 1.0
 * 
 * Revision 0.46  91/09/26  18:05:50  chua
 * In Stop and PauseResume, before sending any network messages, check that the
 * application is still open by calling CheckAppOpen.
 * 
 * Revision 0.45  91/09/25  13:51:20  chua
 * Changed the instrument field, instInfo, to editInfo.
 * Changed InstrumentInfo to EditInfo.
 * 
 * Revision 0.44  91/09/23  17:20:48  chua
 * Changed StopHandler to Stop and PauseResumeHandler to PauseResume, as these
 * are now notify functions for buttons instead of menu items.
 * 
 * In PauseResume, do not call PlayStop anymore.  The timer still runs when
 * in pause mode.
 * 
 * Revision 0.43  91/09/19  17:29:09  chua
 * Make sure that variables are initialized properly.  Change formatting slightly,
 * so that (if, for, while) statements with only one statement in them will not have
 * braces.
 * 
 * Revision 0.42  91/08/20  16:21:15  chua
 * In the StopHandler and PauseResumeHandler, send a Halt or Pause message to an application
 * only if it is currently in the middle of performing a note.
 * 
 * Revision 0.41  91/08/16  17:07:50  chua
 * 
 * 
 * Revision 0.40  91/08/16  13:49:37  chua
 * Contains the StopHandler and PauseResumeHandler routines.
 *  */

static char stoprcsid[] = "$Header: /Source/Media/collab/TimeLine/RCS/stop.c,v 1.11 92/10/01 14:53:28 drapeau Exp Locker: drapeau $";

#include "main.h"

/*
 * Notify callback function for `stopButton'.  This function will
 * send the SenderHaltSelection message to the applications.  The Halt
 * message is sent to only those applications which are currently in the
 * middle of performing a note.  The PlayStop routine is then called.
 */
void Stop(Panel_item	item,
	  Event*	event)
{
  int				i, done;
  int				result;
  TimeLineFramePtr		tlFrame;
  TimeLine_window_objects*	ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  Instrument*			instrument;
  
  tlFrame = TimeLineWindow[xv_get(ip->controls, PANEL_CLIENT_DATA)];
  instrument = (Instrument *) FindInstrument(startApp, tlFrame);
  for (i=0; i < numApps; i++, instrument = instrument->next) 
  {
    if (xv_get(instrument->editInfo->MuteChoice,		    /* If this instrument should not or cannot be played, then... */
	       PANEL_VALUE) != 0 ||				    /* ...go on to the next instrument. */
	CheckAppOpen(tlFrame, instrument, 2) != OK)
      continue;
    done = 0;							    /* Either go through the whole instrument list or a... */
    instrument->currentNote = instrument->firstNote;		    /* ...subset of it, depending on the startApp &... */
    while (instrument->currentNote != NULL && !done)		    /* ...numApps values */
    {
      if (instrument->currentNote->start  < tlFrame->lastX &&	    /* Keep searching while the whole note is before.... */
	  instrument->currentNote->end < tlFrame->lastX)	    /* ...the virtual clock */
	instrument->currentNote = instrument->currentNote->next;
      else 
	done = 1;
    }
    if (instrument->currentNote != NULL)		
    {
      if (instrument->currentNote->start <= tlFrame->lastX) 
      {
	SenderSynchronousMessaging(instrument->sender, On);	    /* Assure that TimeLine blocks until HaltSelection completes */
	result = SenderHaltSelection(instrument->sender);
	if (result == -1) 
	{
	  AlertMessage(tlFrame, "Error in sending a halt message.", 
		       "Check if the application is still running.", NULL);
	  return;
	}
      }
    }
  }								    /* end for */
  PlayStop();
  return;
}								    /* end function Stop */



/*
 * Notify callback function for `pauseButton'.
 * If the TimeLine is in pause mode, PlayStop will be called, and a Pause
 * message sent to the applications.  The Pause message is sent to only
 * those applications which are currently in the middle of performing a
 * note.  If the TimeLine is resume mode, a check is made to see what
 * state the TimeLine was in when Pause was called, meaning whether it
 * was doing a PlaySelected or Play/PlayFrom.  The appropriate Play
 * routine will then be called to resume play.
 */

void PauseResume(Panel_item	item,
		 Event		*event)
{
  Instrument*			instrument;
  int				i, done;
  int				result;
  TimeLineFramePtr		tlFrame;
  TimeLine_window_objects*	ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  
  tlFrame = TimeLineWindow[xv_get(ip->controls, PANEL_CLIENT_DATA)];
  if ((tlFrame->status != PauseMode) &&
      (tlFrame->status != PauseSelectedMode))
  {
    xv_set(tlFrame->TimeLine_window->pauseButton,
	   PANEL_ITEM_COLOR, gcm_color_index("green"), NULL);
    instrument = (Instrument *) FindInstrument(startApp, tlFrame);
    for (i=0; i < numApps; i++)					    /* Either go through the whole instrument list or a... */
    {								    /* ...subset of it, depending on startApp & numApps values*/
      if (xv_get(instrument->editInfo->MuteChoice,		    /* If this instrument should not or cannot be played, then... */
		 PANEL_VALUE) != 0 ||				    /* ...go on to the next instrument. */
	  CheckAppOpen(tlFrame, instrument, 2) != OK)
	continue;
      done = 0;
      instrument->currentNote = instrument->firstNote;
      while (instrument->currentNote != NULL && !done)
      {
	if (instrument->currentNote->start  < tlFrame->lastX &&
	    instrument->currentNote->end < tlFrame->lastX)	    /* Keep searching while whole note is before virtual clock. */
	  instrument->currentNote = instrument->currentNote->next;
	else 
	  done = 1;
      }
      if (instrument->currentNote != NULL)		
	if (instrument->currentNote->start <= tlFrame->lastX) 
	{
	  SenderSynchronousMessaging(instrument->sender, On);	    /* Assure that TimeLine blocks until PauseSelection completes */
	  result = SenderPauseSelection(instrument->sender);
	  if (result == -1) 
	  {
	    AlertMessage(tlFrame, "Error in sending a pause message.", 
			 "Check if the application is still running.", NULL);
	    return;
	  }
	  instrument->playNote = 1;				    /* Remind TimerNotify that this instrument has a note to... */
	}							    /* ...be played */
      instrument = instrument->next;
    }								    /* end for */
    if (tlFrame->status == PlaySelectedMode ||			    /* Set the correct status */
	tlFrame->status == ResumeSelectedMode)
      tlFrame->status = PauseSelectedMode;
    else 
      tlFrame->status = PauseMode;
  }
  else								    /* Resume is to be done */
  {
    xv_set(tlFrame->TimeLine_window->pauseButton,
	   PANEL_ITEM_COLOR, gcm_color_index("black"), NULL);
    if (tlFrame->status == PauseSelectedMode)
    {
      tlFrame->status = ResumeSelectedMode;
      PlaySelectedHandler(item, MENU_NOTIFY);
    }
    else 
    {
      tlFrame->status = ResumeMode;
      PlayFromHandler(item, MENU_NOTIFY);
    }
  }
}								    /* end function PauseResume */

