/*
    lffhdr - read and display the header block of a .lff image file

    Usage: lffhdr [-o offset] file [[ -o offset] file]
*/
#include <stdio.h>
#include <sys/file.h>
#ifdef sys5
#include <fcntl.h>
#endif
#include "pixlib.h"

char Usage[] = "Usage: %s [--] lff_file [lff_file]\n";
char *Example[] = {
    "\tlffhdr reads the header blocks of one or more Lucas-Film Format\n",
    "\timage files and prints the relevant fields on standard ouput.\n",
    NullPtr(char)
};

main(argc,argv)
int argc;
char *argv[];
{
    int fd;
    LFF_HDR hdr;
    int format, i;
    int offset;

    if ((argc < 2) || !strcmp(argv[1],"--"))
	userhelp(argv[0],Usage,Example);

    for (i = 1; i < argc; i++) {
	offset = 0;
	if (!strcmp(argv[i], "-o")) {
	    if (!sscanf(argv[++i], "%d", &offset)) {
		printf("Bad offset %s\n", argv[i]);
		continue;
	    } else
		i++;
	}

	printf("----------\n");
	printf("%s\n",argv[i]);
	printf("----------\n");

	fd = open(argv[i], O_RDONLY, 0644);
	if (fd == -1) {
	    printf("\tCan't open file %s\n",argv[i]);
	    continue;
	}

	if (offset != 0) {
	    off_t pos = lseek(fd, (off_t)offset, L_SET);
	    if (pos != offset) {
		printf("\tCan't seek to offset %d in %s\n",
		    offset, argv[i]);
		close(fd);
		continue;
	    }
	}

	if (get_lffhdr(&hdr, fd, (off_t)offset) != 0)
	    printf("\tPremature EOF - could not read header block\n");
	else if (hdr.hdr_magic == MAGIC_CMAP)
	    printf("\t%s is a color map file\n",argv[i]);
	else if (hdr.hdr_magic != MAGIC_LFF)
	    printf("\t%s has a bad magic # 0%o\n",
		argv[i],hdr.hdr_magic);
	else {
	    int vtiles, htiles, v, table_length;
	    TILEPTR *tile_table;

	    printf("\tVersion:     %d\n",hdr.hdr_version);
	    printf("\tLabel:       %s\n",hdr.hdr_label);
	    printf("\tHeight:      %d\n",hdr.hdr_height);
	    printf("\tWidth:       %d\n",hdr.hdr_width);
	    printf("\tTile Height: %d\n",hdr.hdr_tileheight);
	    printf("\tTile Width:  %d\n",hdr.hdr_tilewidth);
	    printf("\tFormat:      ");
	    format = hdr.hdr_format;
	    if (format == FORMAT_BINARY)
		printf("binary, padded to %d bytes/scanline\n", hdr.hdr_tilepad);
	    else {
		if (format & FORMAT_R) putchar('r');
		if (format & FORMAT_G) putchar('g');
		if (format & FORMAT_B) putchar('b');
		if (format & FORMAT_A) putchar('a');
		putchar('\n');
	    }
	    printf("\tStorage:     %s\n",
		 hdr.hdr_storage == STORAGE_DUMP ? "dump" :
		(hdr.hdr_storage == STORAGE_RLE ? "rle" : "BAD VALUE"));
	    printf("\tBlocking:    %d\n",hdr.hdr_blocking);
	    printf("\tAlpha mode:  %s\n",
		hdr.hdr_alphamode ? "unassociated" : "matted to black");
	    printf("\tXoffset:     %d\n",hdr.hdr_xoffset);
	    printf("\tYoffset:     %d\n",hdr.hdr_yoffset);
	    printf("\tMap ptr:     %d\n",hdr.hdr_map_ptr);
	    printf("\tMap name:    %s\n",hdr.hdr_map_name);
	    printf("\n\t----- Tile pointer table -----\n");

	    if (hdr.hdr_tileheight == 0 || hdr.hdr_tilewidth == 0) {
		printf("Tile height or width is invalid (0)\n");
		close(fd);
		continue;
	    }

	    vtiles = hdr.hdr_height / hdr.hdr_tileheight;
	    htiles = hdr.hdr_width / hdr.hdr_tilewidth;
	    printf("\n\t%d vertical tile%s, %d horizontal tile%s\n",
		vtiles,vtiles > 1 ? "s" : "",htiles,htiles > 1 ? "s" : "");

	    table_length = vtiles * htiles * sizeof(TILEPTR);
	    tile_table = (TILEPTR *)malloc(table_length);
	    if (tile_table == NullPtr(TILEPTR)) {
		printf("Error allocating tile pointer table\n");
		continue;
	    }

	    if (read(fd, (char *)tile_table, table_length) != table_length) {
		printf("Premature EOF reading tile pointer table\n");
		continue;
	    }

	    tile_table_to_host(tile_table,vtiles * htiles);
	    printf("\tPointer\t\tLength\n");
	    for (v = 0; v < vtiles; v++) {
		int h;

		for (h = 0; h < htiles; h++)
		    printf("\t%-10d\t%d\n",
			tile_table[v*htiles+h].tileptr,
			tile_table[v*htiles+h].tilelength);
	    }
	}
	close(fd);
    }
}

/* Read lucasfilm header from file and fixup for host byte order.
   Retruns 0 on success, -1 on failure (premature EOF)
*/
get_lffhdr(hdr, fd, offset)
LFF_HDR *hdr;
int fd;
off_t offset;
{
    if (read(fd, (char *)hdr, sizeof(LFF_HDR)) != sizeof(LFF_HDR))
	return -1;
    hdr_to_host(hdr);
    return 0;
}
