#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "pixlib.h"

char Usage[] =
"%s [--] file1 file2 < scriptfile\n";

char *Example[] = {
    "\tlffblend blends two lucasfilm files according to a supplied script\n",
    "\t\tThe script should contain lines of the form\n",
    "\t\t\talpha output-file\n",
    "\t\twhere alpha is a real number between 0 and 1. The transformation\n",
    "\t\toutput-file[i,j] = alpha * file1[i,j] + (1 - alpha) * file2[i,j]\n",
    "\t\tis performed at every pixel and output-file written.\n",
    "\n",
    "\t\tThe two input files must be of the same size.\n",
    NullPtr(char)
};

main (argc, argv)
int	argc;
char   *argv[];
{
    int  i, status;
    LFF_HDR hdr[2];
    RGBPIXEL *image[2], *get_lff_image();
    static char *file[] = { NULL, NULL };
    char buf[120];

    for (i = 1; i < argc; i++) {
	if (!strcmp(argv[i],"--"))
	    userhelp(argv[0],Usage,Example);
	else if (file[0] == NULL)
	    file[0] = argv[i];
	else if (file[1] == NULL)
	    file[1] = argv[i];
	else {
	    gr_error("Unknown option (or extra image file name) %s\n", argv[i]);
	    userhelp(argv[0],Usage,Example);
	}
    }

    if (file[0] == NULL || file[1] == NULL)
	gr_bomb("Need two input images specified\n");

    /* Read both images */
    for (i = 0; i < 2; i++) {
	printf("\nReading input file %s\n", file[i]);
	status = open_lff(file[i], &hdr[i]);
	if (status  !=	LFF_OK)
	    gr_bomb("Can't open '%s', open_lff returns: %s\n", file[i],
		pixerrmess(status));

	if (i > 0) {
	    verify_lff("height", file[0], hdr[0].hdr_height,
				 file[i], hdr[i].hdr_height);
	    verify_lff("width" , file[0], hdr[0].hdr_width,
				 file[i], hdr[i].hdr_width);
	}

	image[i] = get_lff_image(&hdr[i]);
	close_lff();
    }

    /* Now for each input line, blend the images */
    while (fgets(buf, 120, stdin)) {
	double alpha;
	char out_name[120];

	if (sscanf(buf, "%lf %s", &alpha, out_name) != 2) {
	    buf[strlen(buf)-1] = '\0';
	    gr_error("Error on line: '%s'\n", buf);
	} else
	    blend_images(out_name, file[0], file[1], alpha,
			 hdr[0].hdr_height, hdr[0].hdr_width,
			 image[0], image[1]);
    }
}

/*
    Read an lff image and return a pointer to its unpacked pixels.
    Assumes the image has just been opened and that hdr points
	to the returned lff header.
*/
RGBPIXEL *get_lff_image(hdr)
LFF_HDR *hdr;
{
    RGBPIXEL *image;
    int i, status;

    /* Allocate memory for the image in core */
    i = hdr->hdr_height * hdr->hdr_width * sizeof(RGBPIXEL);
    image = (RGBPIXEL *)malloc(i);
    if (image == (RGBPIXEL *)NULL)
	gr_bomb("Failed to allocate %d bytes for incore buffer\n",i);

    /* Scan each tile in order and save in the incore buffer */
    {
	int j, vtiles,htiles;

	vtiles = hdr->hdr_height / hdr->hdr_tileheight;
	htiles = hdr->hdr_width / hdr->hdr_tilewidth;
	for (i = 0; i < vtiles; i++) {
	    int yoff = i * hdr->hdr_tileheight;

	    for (j = 0; j < htiles; j++) {
		int y, xoff = j * hdr->hdr_tilewidth;

		if (lff_seek_tile(i,j) != LFF_OK)
		    gr_bomb("Error reading tile %d, row %d, column %d\n",
			i*htiles+j,i,j);
		for (y = yoff; y < yoff+hdr->hdr_tileheight; y++) {
		    int newy;
		    RGBPIXEL *ptr = &image[hdr->hdr_width * y + xoff];

		    status = lff_read_line(ptr,&newy);
		    if (status != LFF_OK)
			gr_bomb("lff_read_line(y=%d) returns: %s\n",
			    newy,pixerrmess(status));
		}
	    }
	}
    }

    return image;
}

blend_images(out_name, in_1, in_2, alpha, height, width, image1, image2)
char *out_name, *in_1, *in_2;
double alpha;
int height, width;
RGBPIXEL *image1, *image2;
{
    int i, length, status, getpixfun();
    double beta;
    RGBPIXEL *out_image;

    printf("Blending %s = %g * %s + %g * %s\n",
	out_name, alpha, in_1, 1 - alpha, in_2);

    length = height * width;
    out_image = (RGBPIXEL *)malloc(length * sizeof(RGBPIXEL));
    if (out_image == NULL)
	gr_bomb("Can't allocate %d bytes of memory for output image!\n",
	    length * sizeof(RGBPIXEL));

    /* Now blend each pixel */
    beta = 1.0 - alpha;

    for (i = 0; i < length; i++) {
	register int r, g, b;
	register RGBPIXEL pixel1, pixel2;

	pixel1 = image1[i];
	pixel2 = image2[i];
	r = alpha * GETRED(pixel1) + beta * GETRED(pixel2);
	g = alpha * GETGRN(pixel1) + beta * GETGRN(pixel2);
	b = alpha * GETBLU(pixel1) + beta * GETBLU(pixel2);
	out_image[i] = RGB8(r, g, b);
	/*
	if (i % width == 0)
	*/
    }

    /* Now write the image */
    printf("Writing %s\n", out_name);
    pixinit(out_image, height, width);
    status = write_lff(
	out_name,
	height, width,
	1, 1,
	FORMAT_RGB,
	STORAGE_DUMP,
	0, 0,
	LFF_VERSION,
	0,
	getpixfun);
    if (status != LFF_OK)
	gr_error("Error writing output file %s: %s\n",
	    out_name, pixerrmess(status));
    else
	printf("Finished writing %s\n", out_name);
    free(out_image);
}

static RGBPIXEL *Image;
static int Height, Width;

pixinit(image, height, width)
RGBPIXEL *image;
int height, width;
{
    Image = image;
    Height = height;
    Width = width;
}

/*
    Function called by write_lff, which returns the next line from the
	input lucasfilm file.
*/
int getpixfun(row,col,length,pixels)
int row, col, length;
RGBPIXEL pixels[];
{
    RGBPIXEL *pixsrc;

    pixsrc = &Image[row * Width + col];

    bcopy( (char *)pixsrc,(char *)pixels,length * sizeof(RGBPIXEL) );
    return length;
}

verify_lff(token, file1, parm1, file2, parm2)
char *token, *file1, *file2;
int parm1, parm2;
{
    if (parm1 != parm2) {
	gr_error("Fatal error: images are not conformable\n");
	gr_bomb("\t%s(%s) = %d != %s(%s) = %d\n",
	    token, file1, parm1,
	    token, file2, parm2);
    }
}
