/*
    Routines to read a .pix format file into memory, one scan line at a time

    int open_pix(name,hdr)
    char *name;
    PIX_HDR *hdr;

	The specified .pix file 'name' is opened and the header  block
	is read into the passed structure.  If successful, 0  (PIX_OK)
	is returned; otherwise, one of the following error codes:

	    PIX_BADMAGIC    Bad magic number
	    PIX_BADFORMAT   Bad file format
	    PIX_OPENERR     Can't open file (examine errno)
	    PIX_EOF	    Premature end of file
	    PIX_REOPEN	    A .pix file is currently open

    int pix_read_line(line,y)
    RGBPIXEL line[];
    int *y;

	The next sequential scanline is read and placed in  the  array
	line[],  which	must  be  long	enough.   Y  is  set  to   the
	y-coordinate  of  the  scan  line  which  was  read  in.    If
	successful,  PIX_OK  is  returned;  otherwise,	one   of   the
	following error code:

	    PIX_EOF	    End of file
*/
#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include "pixlib.h"

/* Globally accessible symbols */
int open_pix();
int pix_read_line();
void close_pix();

/* Private variables */
static FILE    *Pixfp = NullPtr(FILE);	/* Pointer to currently open file */
static PIX_HDR Pixhdr;			/* Header block of current file */
static int     Mask;			/* Image mask of file */
static int     Y_current;		/* Current scanline coordinate */
static off_t   Rpos, Gpos, Bpos;	/* Pointers into file for color planes */

/* Private functions */
static int   pix_read();
static off_t pix_get_pixels();
static int   rle_read();
static off_t rle_get_pixels();
static int   packed_read();
static void  merge_pixels();

int open_pix(name,hdr)
char *name;
PIX_HDR *hdr;
{
    if (Pixfp != NullPtr(FILE))
	return PIX_REOPEN;

    Pixfp = fopen(name,"r");
    if (Pixfp == NullPtr(FILE))
	return PIX_OPENERR;

    if (fread (hdr, sizeof(PIX_HDR), 1, Pixfp) != 1) {
	close_pix();
	return PIX_EOF;
    }
    pix_hdr_to_host(hdr);

    /* Verify magic number */
    if (hdr->magic != PIX_MAGIC) {
	close_pix();
	return PIX_BADMAGIC;
    }

    /* Verify file format code */
    if (hdr->format != GRIN_PIX &&
	hdr->format != GRIN_PACKED &&
	hdr->format != GRIN_RLE) {
	    close_pix();
	    return PIX_BADFORMAT;
    }

    /* Initialize pointers into file */
    switch (hdr->format) {
	case GRIN_PIX: {
		off_t nblocks, offset;

		nblocks = (hdr->height * hdr->width + PIXBLKSIZE - 1) / PIXBLKSIZE;
		offset = nblocks * PIXBLKSIZE;
		Rpos = hdr->block * PIXBLKSIZE;
		Gpos = Rpos + offset;
		Bpos = Gpos + offset;
	    }
	    break;
	case GRIN_PACKED:
	    Rpos = hdr->block * PIXBLKSIZE;
	    break;
	case GRIN_RLE: {
		long *dict, dictlen;

		/* Read the scanline dictionary in */
		dictlen = 3 * hdr->height * sizeof(long);
		dict = (long *)malloc(dictlen);
		fseek(Pixfp, hdr->block * PIXBLKSIZE, L_SET);
		if (fread(dict,dictlen,1,Pixfp) != 1) {
		    close_pix();
		    free(dict);
		    return PIX_EOF;
		}
		Rpos = dict[0];
		Gpos = dict[hdr->height];
		Bpos = dict[2 * hdr->height];
		free(dict);
	    }
	    break;
    }
    Y_current = 0;
    /* Copy header block to static block for reading file */
    Pixhdr = *hdr;
    Mask = Pixhdr.mask;

    return PIX_OK;
}

int pix_read_line(line,y)
RGBPIXEL line[];
int *y;
{
    if (Pixfp == NullPtr(FILE))
	return PIX_OPENERR;

    /* Cannot read beyond number of lines in the file */
    if (Y_current > Pixhdr.height)
	return PIX_EOF;

    *y = Y_current++;
    switch (Pixhdr.format) {
	case GRIN_PIX:	  return pix_read(line);
	case GRIN_PACKED: return packed_read(line);
	case GRIN_RLE:	  return rle_read(line);
    }
}

/*
    Read the next scanline from the .pix file (in PIXEL  format)  into
    the passed array. Return an error code to pix_read_line() above.
*/
static int pix_read(line)
RGBPIXEL line[];
{
    RPIXEL red[MAXLINE], grn[MAXLINE], blu[MAXLINE];
    off_t pix_get_pixels();

    if (Mask & BW || Mask & RED)
	Rpos = pix_get_pixels(red,Rpos);
    if (Mask & GRN)
	Gpos = pix_get_pixels(grn,Gpos);
    if (Mask & BLU)
	Bpos = pix_get_pixels(blu,Bpos);

    if (Rpos == -1 || Gpos == -1 || Bpos == -1)
	return PIX_EOF;

    merge_pixels(line,red,grn,blu);
    return PIX_OK;
}

static off_t pix_get_pixels(array,offset)
RPIXEL array[];
off_t offset;
{
    fseek(Pixfp,offset,L_SET);
    if (fread(array,Pixhdr.width,1,Pixfp) != 1)
	return -1;
    return offset + Pixhdr.width;
}

/*
    Read the next scanline from the .pix file (in  run-length  format)
    into the passed array. Return an  error  code  to  pix_read_line()
    above.
*/
static int rle_read(line)
RGBPIXEL line[];
{
    RPIXEL red[MAXLINE], grn[MAXLINE], blu[MAXLINE];
    off_t rle_get_pixels();

    if (Mask & BW || Mask & RED)
	Rpos = rle_get_pixels(red,Rpos);
    if (Mask & GRN)
	Gpos = rle_get_pixels(grn,Gpos);
    if (Mask & BLU)
	Bpos = rle_get_pixels(blu,Bpos);

    if (Rpos == -1 || Gpos == -1 || Bpos == -1)
	return PIX_EOF;

    merge_pixels(line,red,grn,blu);
    return PIX_OK;
}

static off_t rle_get_pixels(array,offset)
register RPIXEL *array;
off_t offset;
{
    register int count, col;
    register RPIXEL value;

    fseek(Pixfp,offset,L_SET);

    for (col = 1; col <= Pixhdr.width; ) {
	count = getc(Pixfp);
	value = getc(Pixfp);

	if (count == EOF)
	    return -1;
	count++;
	col += count;
	while (count--)
	    *array++ = value;
    }
    return ftell(Pixfp);
}

/*
    Read the next scanline from the .pix file (in packed format)  into
    the passed array. Return an error code to pix_read_line() above.
*/
static int packed_read(line)
RGBPIXEL line[];
{
    u_char red[MAXLINE], grn[MAXLINE], blu[MAXLINE];
    off_t pix_get_pixels();

    if (Mask & BW || Mask & RED)
	Rpos = pix_get_pixels(red,Rpos);
    if (Mask & GRN)
	Rpos = pix_get_pixels(grn,Rpos);
    if (Mask & BLU)
	Rpos = pix_get_pixels(blu,Rpos);

    if (Rpos == -1 || Gpos == -1 || Bpos == -1)
	return PIX_EOF;

    merge_pixels(line,red,grn,blu);
    return PIX_OK;
}

/*
    Merge the passed pixel  arrays  (based  on	the  file  mask)  into
	longword pixels. If some color plane is not  present,  garbage
	will be in the corresponding byte of each pixel.
*/
static void merge_pixels(line,red,grn,blu)
register RGBPIXEL *line;
register RPIXEL *red, *grn, *blu;
{
    register int i;

    if (Mask & BW) {
	for (i = 0; i < Pixhdr.width; i++,red++)
	    *line++ = RGB8(*red,*red,*red);
    } else {
	for (i = 0; i < Pixhdr.width; i++,red++,grn++,blu++)
	    *line++ = RGB8(*red,*grn,*blu);
    }
}

void close_pix()
{
    if (Pixfp != NullPtr(FILE)) {
	fclose(Pixfp);
	Pixfp = NullPtr(FILE);
    }
}
